import os
from typing import Dict, List, Literal, Optional, Union

import streamlit.components.v1 as components

_RELEASE = True
COMPONENT_NAME = "streamlit_message"

if _RELEASE:  # use the build instead of development if release is true
    root_dir = os.path.dirname(os.path.abspath(__file__))
    build_dir = os.path.join(root_dir, "frontend/build")

    _streamlit_message = components.declare_component(
        COMPONENT_NAME,
        path = build_dir
    )
else:
    _streamlit_message = components.declare_component(
        COMPONENT_NAME,
        url = "http://localhost:3000"
    )

# data type for avatar style
AvatarStyle = Literal[ 
    "adventurer", 
    "adventurer-neutral", 
    "avataaars",
    "big-ears",
    "big-ears-neutral",
    "big-smile",
    "bottts", 
    "croodles",
    "croodles-neutral",
    "female",
    "gridy",
    "human",
    "identicon",
    "initials",
    "jdenticon",
    "male",
    "micah",
    "miniavs",
    "pixel-art",
    "pixel-art-neutral",
    "personas",
]

def message(message: Union[List[str], str], 
            is_user: Optional[bool] = False, 
            is_picture: Optional[Union[bool, List[bool]]] = False,
            avatar_style: Optional[AvatarStyle] = None,
            seed: Optional[Union[int, str]] = 42,
            answers: Optional[Dict[int, str]] = None,
            key: Optional[str] = None):
    """
    Creates a new instance of streamlit-message component

    Parameters
    ----------
    message: str or list
        The message(s) to be displayed in the component
    is_user: bool 
        if the sender of the message is user, if `True` will align the 
        message to right, default is False.
    is_picture: bool or list
        if the message is a picture, if `True` will add an img tag to the
        message, default is False.
    avatar_style: Literal or None
        The style for the avatar of the sender of message, default is bottts
        for not user, and pixel-art-neutral for user.
        st-message uses https://avatars.dicebear.com/styles for the avatar
    seed: int or str
        The seed for choosing the avatar to be used, default is 42.
    answers: dict of int => str
        Which message does this one answers.
        Puts an "answer bubble" next to the message
    key: str or None
        An optional key that uniquely identifies this component. If this is
        None, and the component's arguments are changed, the component will
        be re-mounted in the Streamlit frontend and lose its current state.

    Returns: None
    """
    if not avatar_style:
        avatar_style = "pixel-art-neutral" if is_user else "bottts"

    _streamlit_message(message=message, seed=seed, isUser=is_user, isPicture=is_picture, avatarStyle=avatar_style, key=key, answers=answers)


if not _RELEASE:
    import streamlit as st

    # testing
    long_message = """
    A chatbot or chatterbot is a software application used to conduct an on-line chat conversation via text or text-to-speech, in lieu of providing direct contact with a live human agent. Designed to convincingly simulate the way a human would behave as a conversational partner, chatbot systems typically require continuous tuning and testing, and many in production remain unable to adequately converse, while none of them can pass the standard Turing test. The term "ChatterBot" was originally coined by Michael Mauldin (creator of the first Verbot) in 1994 to describe these conversational programs.
    """

    message("Hello, how are you?", key=1)

    message([
        "Really good",
        "you?",
    ], is_user=True, key=2, answers={1: "Hello, how are you?"})
