import os
import datetime
import logging
from typing import Optional

from dbsavior.db.DbManagerInterface import DbManagerInterface
from dbsavior.db.PostgresDbManager import PostgresDbManager
from util import get_date_format, get_date_limit_of_backup_file

from dbsavior.storage.LocalStorageEngine import LocalStorageEngine
from dbsavior.storage.S3StorageEngine import S3StorageEngine
from dbsavior.storage.StorageEngineInterface import StorageEngineInterface

log = logging.getLogger(__name__)
log.setLevel(logging.INFO)
handler = logging.StreamHandler()
formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
handler.setFormatter(formatter)
log.addHandler(handler)


class DbBackupRestoreBot:
    def __init__(self, storage_engine: StorageEngineInterface, db_manager: DbManagerInterface):
        self.storage_engine = storage_engine
        self.db_manager = db_manager

    ################################ Backup methods ##################################################################
    def make_auto_backup(self, db_name: str, backup_file_parent_dir: str, tmp_dir='/tmp') -> Optional[str]:
        # generate backup file name base on current time stamp and database name
        current_date = datetime.datetime.now().strftime(get_date_format())
        backup_file_name = f"{current_date}_{db_name}_pg_bck.sql"
        return self.make_backup_with_custom_file_name(db_name, backup_file_parent_dir, backup_file_name, tmp_dir)

    def make_backup_with_custom_file_name(self, db_name: str, backup_file_parent_dir: str, backup_file_name: str,
                                          tmp_dir='/tmp') -> Optional[str]:
        # step 1: check if the target db exist
        if self.db_manager.has_db(db_name):
            # if db exist, start step 2: build the backup file
            tmp_bk_file_path = self.db_manager.backup_db(db_name, tmp_dir, backup_file_name)
            if tmp_bk_file_path:
                # if backup success, start step 3: upload backup file
                destination_path = f"{backup_file_parent_dir}/{backup_file_name}"
                if self.storage_engine.upload_data(tmp_bk_file_path, destination_path):
                    log.info(f"Backup complete. Creat backup file {destination_path} for db {db_name}")
                    return destination_path
                else:
                    log.error(f"Unable to upload backup file {tmp_bk_file_path} to {backup_file_parent_dir}")
                    return None
            else:
                log.error(f"Unable to create backup for database {db_name}")
                return None
        else:
            log.exception(f"The input database {db_name} does not exist in the target server")
            return None

    ################################ Restore methods ##################################################################
    def list_available_backups(self, storage_path: str):
        return self.storage_engine.list_dir(storage_path)

    @staticmethod
    def is_backup_of_db(file_name, db_name):
        return db_name in file_name

    def get_latest_backup_name(self, storage_path: str, db_name: str) -> Optional[str]:
        """
        This function search the latest backup of the given database name that are generated by the DbBackupBot

        :param storage_path: the parent path of storage that stores the backup of the database
        :param db_name: the target database name
        :return: return the full path and name of the latest backup
        """
        all_backups = self.list_available_backups(storage_path)
        backup_list = [f for f in all_backups if self.is_backup_of_db(f, db_name)]
        if backup_list:
            current_date = datetime.datetime.now()
            latest_backup_path = ""
            min_time = datetime.timedelta(days=get_date_limit_of_backup_file())
            for raw_file_name in backup_list:
                # as each storage engine has different definition of file name, we call the storage_engine
                # implementation's get_short_file_name to get the shortest correct file name that discards
                # all path and protocol
                file_name = self.storage_engine.get_short_file_name(raw_file_name)
                try:
                    # for the backup file that does not match the required pattern, we log the info
                    # and skip the backup file.
                    time_stamp = datetime.datetime.strptime(file_name.split("_")[0], get_date_format())
                except ValueError as e:
                    log.info(f"can't parse the raw file name with given timestamp format. {e}")
                    time_stamp = None
                # arithmetic operations on datetime will return a timedelta type
                if time_stamp and (min_time > current_date - time_stamp):
                    min_time = current_date - time_stamp
                    latest_backup_path = os.path.join(storage_path, file_name)
            return latest_backup_path
        else:
            return None

    def restore_db_backup(self, db_name: str, backup_file_path: str) -> bool:
        # create db if the given db_name does not exist yet
        if self.db_manager.has_db(db_name):
            log.info(f"There is already a database with name {db_name}, you may have conflict")
        else:
            self.db_manager.create_db(db_name)
            log.info(f"Creating database {db_name}")
        # in our case, we name the postgres custom format with extension .pgdump; the plain text sql format with .sql
        if backup_file_path.endswith(".pgdump"):
            log.info("restore_db_backup with custom format")
            return self.db_manager.restore_db(db_name, backup_file_path, backup_format="psql")
        elif backup_file_path.endswith(".sql"):
            log.info("restore_db_backup with sql plain text format")
            return self.db_manager.restore_db(db_name, backup_file_path, backup_format="sql")

    def restore_db_with_latest_backup(self, db_name, backup_root_path: str) -> bool:
        """
        This method gets the latest backup of the given database name from the given root_path. If it finds one,
        it downloads the backup to a local temporal path. Then call the restore_db_backup

        :param db_name:
        :param backup_root_path:
        :return:
        """
        # get the path of the latest backup
        latest_backup = self.get_latest_backup_name(backup_root_path, db_name)
        # start the restore process if we found a backup file
        if latest_backup:
            log.info(f"Find the latest backup file {latest_backup} for db {db_name} ")
            # step1 download data from remote storage to local
            local_path = f"/tmp/latest_{db_name}_backup.sql"
            if self.storage_engine.download_data(latest_backup, local_path):
                # step2: restore db with the backup file
                return self.restore_db_backup(db_name, local_path)
            else:
                log.error("Fail to download the backup file")
                return False
        else:
            log.exception(f"There is no backup file for the given db name {db_name} in directory {backup_root_path}")
            return False

    def populate_db_with_sql_dump(self, db_name, backup_file_path: str) -> bool:
        """
        This method gets the latest backup of the given database name from the given root_path. If it finds one,
        it downloads the backup to a local temporal path. Then call the restore_db_backup

        :param backup_file_path:
        :param db_name:
        :return:
        """

        # step1 download data from remote storage to local
        local_path = f"/tmp/{db_name}_backup.sql"
        if self.storage_engine.download_data(backup_file_path, local_path):
            # step2: restore db with the backup file
            if self.restore_db_backup(db_name, local_path):
                log.info(f"Populate the database {db_name} with file {backup_file_path} is complete")
                return True
            else:
                log.error(f"Fail to populate the database {db_name} with file {backup_file_path}")
                return False
        else:
            log.error(f"Fail to download the backup file {backup_file_path}")
            return False
