"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dashboard = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_elasticache_1 = require("aws-cdk-lib/aws-elasticache");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const constructs_1 = require("constructs");
const auto_scaling_1 = require("./auto-scaling");
const load_balancer_1 = require("./load-balancer");
const rds_1 = require("./rds");
const redis_1 = require("./redis");
/**
 * An AWS CloudWatch Dashboard.
 *
 * @example
 * // create a dashboard for AutoScaling
 * new Dashboard(this, 'dashboard', {
 *    autoScalingName: 'my-auto-scaling',
 *    autoScalingMaxCapacity: 10
 * });
 *
 */
class Dashboard extends constructs_1.Construct {
    /**
     * Creates a Dashboard based on the Halloumi best practices.
     *
     * @param scope the scope into which to import this dashboard
     * @param id the logical ID of the returned dashboard construct
     */
    constructor(scope, id, props) {
        super(scope, id);
        let start = props?.start ? props.start : '-PT24H';
        let periodOverride = props?.periodOverride ? props.periodOverride : aws_cloudwatch_1.PeriodOverride.INHERIT;
        const dashboard = new aws_cloudwatch_1.Dashboard(this, id, {
            dashboardName: props?.dashboardName,
            start: start,
            end: props?.end,
            periodOverride: periodOverride,
            widgets: props?.widgets,
        });
        if (props?.loadBalancer) {
            for (let i = 0; i < props.loadBalancer.length; i++) {
                let loadBalancer = props.loadBalancer[i];
                let name;
                let full_name;
                if (loadBalancer instanceof aws_elasticloadbalancingv2_1.BaseLoadBalancer) {
                    name = loadBalancer.loadBalancerName;
                    full_name = loadBalancer.loadBalancerFullName;
                }
                else if (loadBalancer instanceof aws_elasticloadbalancingv2_1.CfnLoadBalancer) {
                    name = loadBalancer.attrLoadBalancerName;
                    full_name = loadBalancer.attrLoadBalancerFullName;
                }
                else {
                    name = loadBalancer.name;
                    full_name = loadBalancer.full_name;
                }
                const lbWidgets = load_balancer_1.LoadBalancer.metrics(name, full_name);
                lbWidgets.forEach(widget => {
                    dashboard.addWidgets(widget);
                });
            }
        }
        if (props?.autoScaling) {
            for (let i = 0; i < props.autoScaling.length; i++) {
                let auto_scaling_group = props.autoScaling[i];
                let maxCapacity;
                let name;
                if (auto_scaling_group instanceof aws_autoscaling_1.CfnAutoScalingGroup) {
                    name = auto_scaling_group.ref;
                    maxCapacity = parseInt(auto_scaling_group.maxSize);
                    dashboard.node.addDependency(auto_scaling_group);
                }
                else if (auto_scaling_group instanceof aws_autoscaling_1.AutoScalingGroup) {
                    name = auto_scaling_group.autoScalingGroupName;
                    dashboard.node.addDependency(auto_scaling_group);
                }
                else {
                    name = auto_scaling_group.name;
                    maxCapacity = auto_scaling_group.max_capacity;
                }
                let autoScalingWidgets = auto_scaling_1.AutoScaling.metrics(name, maxCapacity);
                autoScalingWidgets.forEach(widget => {
                    dashboard.addWidgets(widget);
                });
            }
        }
        if (props?.elasticache) {
            for (let i = 0; i < props.elasticache.length; i++) {
                let elasticache = props.elasticache[i];
                let name;
                let numNodes;
                if (elasticache instanceof aws_elasticache_1.CfnReplicationGroup) {
                    name = elasticache.ref;
                    numNodes = elasticache.numNodeGroups || 1;
                }
                else {
                    name = elasticache.name;
                    numNodes = elasticache.nodes || 1;
                }
                for (let node = 1; node <= numNodes; node++) {
                    let nodeId = name + '-' + node.toString().padStart(3, '0');
                    let elasticacheWidgets = redis_1.Redis.metrics(nodeId);
                    elasticacheWidgets.forEach(widget => {
                        dashboard.addWidgets(widget);
                    });
                }
            }
        }
        if (props?.rds) {
            for (let i = 0; i < props.rds.length; i++) {
                let rds_cluster = props.rds[i];
                let name = rds_cluster.ref;
                let rdsWidgets = rds_1.Rds.metrics(name);
                rdsWidgets.forEach(widget => {
                    dashboard.addWidgets(widget);
                });
            }
        }
    }
}
exports.Dashboard = Dashboard;
_a = JSII_RTTI_SYMBOL_1;
Dashboard[_a] = { fqn: "halloumi-cloudwatch-dashboard.Dashboard", version: "2.0.0" };
//# sourceMappingURL=data:application/json;base64,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