"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeoRestriction = void 0;
/**
 * Controls the countries in which content is distributed.
 */
class GeoRestriction {
    /**
     * Creates an instance of GeoRestriction for internal use
     *
     * @param restrictionType Specifies the restriction type to impose (whitelist or blacklist)
     * @param locations Two-letter, uppercase country code for a country
     * that you want to whitelist/blacklist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    constructor(restrictionType, locations) {
        this.restrictionType = restrictionType;
        this.locations = locations;
    }
    /**
     * Whitelist specific countries which you want CloudFront to distribute your content.
     *
     * @param locations Two-letter, uppercase country code for a country
     * that you want to whitelist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    static whitelist(...locations) {
        return new GeoRestriction('whitelist', GeoRestriction.validateLocations(locations));
    }
    /**
     * Blacklist specific countries which you don't want CloudFront to distribute your content.
     *
     * @param locations Two-letter, uppercase country code for a country
     * that you want to blacklist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    static blacklist(...locations) {
        return new GeoRestriction('blacklist', GeoRestriction.validateLocations(locations));
    }
    static validateLocations(locations) {
        if (locations.length === 0) {
            throw new Error('Should provide at least 1 location');
        }
        locations.forEach(location => {
            if (!GeoRestriction.LOCATION_REGEX.test(location)) {
                // eslint-disable-next-line max-len
                throw new Error(`Invalid location format for location: ${location}, location should be two-letter and uppercase country ISO 3166-1-alpha-2 code`);
            }
        });
        return locations;
    }
}
exports.GeoRestriction = GeoRestriction;
GeoRestriction.LOCATION_REGEX = /^[A-Z]{2}$/;
//# sourceMappingURL=data:application/json;base64,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