"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
const k8s_patch_1 = require("../lib/k8s-patch");
module.exports = {
    'applies a patch to k8s'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        // WHEN
        new k8s_patch_1.KubernetesPatch(stack, 'MyPatch', {
            cluster,
            applyPatch: { patch: { to: 'apply' } },
            restorePatch: { restore: { patch: 123 } },
            resourceName: 'myResourceName',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            ServiceToken: {
                'Fn::GetAtt': [
                    'awscdkawseksKubectlProviderNestedStackawscdkawseksKubectlProviderNestedStackResourceA7AEBA6B',
                    'Outputs.awscdkawseksKubectlProviderframeworkonEvent0A650005Arn',
                ],
            },
            ResourceName: 'myResourceName',
            ResourceNamespace: 'default',
            ApplyPatchJson: '{"patch":{"to":"apply"}}',
            RestorePatchJson: '{"restore":{"patch":123}}',
            ClusterName: {
                Ref: 'MyCluster8AD82BF8',
            },
            RoleArn: {
                'Fn::GetAtt': [
                    'MyClusterCreationRoleB5FA4FF3',
                    'Arn',
                ],
            },
        }));
        test.done();
    },
    'defaults to "strategic" patch type if no patchType is specified'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        // WHEN
        new k8s_patch_1.KubernetesPatch(stack, 'MyPatch', {
            cluster,
            applyPatch: { patch: { to: 'apply' } },
            restorePatch: { restore: { patch: 123 } },
            resourceName: 'myResourceName',
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            PatchType: 'strategic',
        }));
        test.done();
    },
    'uses specified to patch type if specified'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        // WHEN
        new k8s_patch_1.KubernetesPatch(stack, 'jsonPatch', {
            cluster,
            applyPatch: { patch: { to: 'apply' } },
            restorePatch: { restore: { patch: 123 } },
            resourceName: 'jsonPatchResource',
            patchType: k8s_patch_1.PatchType.JSON,
        });
        new k8s_patch_1.KubernetesPatch(stack, 'mergePatch', {
            cluster,
            applyPatch: { patch: { to: 'apply' } },
            restorePatch: { restore: { patch: 123 } },
            resourceName: 'mergePatchResource',
            patchType: k8s_patch_1.PatchType.MERGE,
        });
        new k8s_patch_1.KubernetesPatch(stack, 'strategicPatch', {
            cluster,
            applyPatch: { patch: { to: 'apply' } },
            restorePatch: { restore: { patch: 123 } },
            resourceName: 'strategicPatchResource',
            patchType: k8s_patch_1.PatchType.STRATEGIC,
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            ResourceName: 'jsonPatchResource',
            PatchType: 'json',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            ResourceName: 'mergePatchResource',
            PatchType: 'merge',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            ResourceName: 'strategicPatchResource',
            PatchType: 'strategic',
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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