"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
module.exports = {
    'can be added to a cluster'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            selectors: [{ namespace: 'default' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                selectors: [{ namespace: 'default' }],
            },
        }));
        test.done();
    },
    'supports specifying a profile name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            fargateProfileName: 'MyProfileName',
            selectors: [{ namespace: 'default' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                selectors: [{ namespace: 'default' }],
                fargateProfileName: 'MyProfileName',
            },
        }));
        test.done();
    },
    'supports custom execution role'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        const myRole = new iam.Role(stack, 'MyRole', { assumedBy: new iam.AnyPrincipal() });
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            podExecutionRole: myRole,
            selectors: [{ namespace: 'default' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyRoleF48FFE04', 'Arn'] },
                selectors: [{ namespace: 'default' }],
            },
        }));
        test.done();
    },
    'supports tags through aspects'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            selectors: [{ namespace: 'default' }],
        });
        core_1.Tag.add(stack, 'aspectTag', 'hello');
        core_1.Tag.add(cluster, 'propTag', '123');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                selectors: [{ namespace: 'default' }],
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                tags: {
                    propTag: '123',
                    aspectTag: 'hello',
                },
            },
        }));
        test.done();
    },
    'supports specifying vpc'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        const vpc = ec2.Vpc.fromVpcAttributes(stack, 'MyVpc', {
            vpcId: 'vpc123',
            availabilityZones: ['az1'],
            privateSubnetIds: ['priv1'],
        });
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            selectors: [{ namespace: 'default' }],
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                selectors: [{ namespace: 'default' }],
                subnets: ['priv1'],
            },
        }));
        test.done();
    },
    'fails if there are no selectors or if there are more than 5'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        // THEN
        test.throws(() => cluster.addFargateProfile('MyProfile', { selectors: [] }));
        test.throws(() => cluster.addFargateProfile('MyProfile', {
            selectors: [
                { namespace: '1' },
                { namespace: '2' },
                { namespace: '3' },
                { namespace: '4' },
                { namespace: '5' },
                { namespace: '6' },
            ],
        }));
        test.done();
    },
    'FargateCluster creates an EKS cluster fully managed by Fargate'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            ResourceName: 'deployment/coredns',
            ResourceNamespace: 'kube-system',
            ApplyPatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"fargate"}}}}}',
            RestorePatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"ec2"}}}}}',
            ClusterName: {
                Ref: 'FargateCluster019F03E8',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: {
                    Ref: 'FargateCluster019F03E8',
                },
                podExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'FargateClusterfargateprofiledefaultPodExecutionRole66F2610E',
                        'Arn',
                    ],
                },
                selectors: [
                    { namespace: 'default' },
                    { namespace: 'kube-system' },
                ],
            },
        }));
        test.done();
    },
    'can create FargateCluster with a custom profile'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster', {
            defaultProfile: {
                fargateProfileName: 'my-app', selectors: [{ namespace: 'foo' }, { namespace: 'bar' }],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: {
                    Ref: 'FargateCluster019F03E8',
                },
                fargateProfileName: 'my-app',
                podExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'FargateClusterfargateprofilemyappPodExecutionRole875B4635',
                        'Arn',
                    ],
                },
                selectors: [
                    { namespace: 'foo' },
                    { namespace: 'bar' },
                ],
            },
        }));
        test.done();
    },
    'custom profile name is "custom" if no custom profile name is provided'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster', {
            defaultProfile: {
                selectors: [{ namespace: 'foo' }, { namespace: 'bar' }],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: {
                    Ref: 'FargateCluster019F03E8',
                },
                podExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'FargateClusterfargateprofilecustomPodExecutionRoleDB415F19',
                        'Arn',
                    ],
                },
                selectors: [
                    { namespace: 'foo' },
                    { namespace: 'bar' },
                ],
            },
        }));
        test.done();
    },
    'multiple Fargate profiles added to a cluster are processed sequentially'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster');
        // WHEN
        cluster.addFargateProfile('MyProfile1', {
            selectors: [{ namespace: 'namespace1' }],
        });
        cluster.addFargateProfile('MyProfile2', {
            selectors: [{ namespace: 'namespace2' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfile1PodExecutionRole794E9E37', 'Arn'] },
                selectors: [{ namespace: 'namespace1' }],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Properties: {
                ServiceToken: { 'Fn::GetAtt': [
                        'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                        'Outputs.awscdkawseksClusterResourceProviderframeworkonEventEA97AA31Arn',
                    ] },
                AssumeRoleArn: { 'Fn::GetAtt': ['MyClusterCreationRoleB5FA4FF3', 'Arn'] },
                Config: {
                    clusterName: { Ref: 'MyCluster8AD82BF8' },
                    podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfile2PodExecutionRoleD1151CCF', 'Arn'] },
                    selectors: [{ namespace: 'namespace2' }],
                },
            },
            DependsOn: [
                'MyClusterfargateprofileMyProfile1PodExecutionRole794E9E37',
                'MyClusterfargateprofileMyProfile1879D501A',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'fargate role is added to RBAC'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesResource', {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'FargateClusterfargateprofiledefaultPodExecutionRole66F2610E',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"system:node:{{SessionName}}\\",\\"groups\\":[\\"system:bootstrappers\\",\\"system:nodes\\",\\"system:node-proxier\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'cannot be added to a cluster without kubectl enabled'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { kubectlEnabled: false });
        // WHEN
        test.throws(() => new eks.FargateProfile(stack, 'MyFargateProfile', {
            cluster,
            selectors: [{ namespace: 'default' }],
        }), /unsupported/);
        test.done();
    },
    'allow cluster creation role to iam:PassRole on fargate pod execution role'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'iam:PassRole',
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'FargateClusterRole8E36B33A',
                                'Arn',
                            ],
                        },
                    },
                    {
                        Action: [
                            'ec2:DescribeSubnets',
                            'ec2:DescribeRouteTables',
                        ],
                        Effect: 'Allow',
                        Resource: '*',
                    },
                    {
                        Action: [
                            'eks:CreateCluster',
                            'eks:DescribeCluster',
                            'eks:DescribeUpdate',
                            'eks:DeleteCluster',
                            'eks:UpdateClusterVersion',
                            'eks:UpdateClusterConfig',
                            'eks:CreateFargateProfile',
                            'eks:TagResource',
                            'eks:UntagResource',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            '*',
                        ],
                    },
                    {
                        Action: [
                            'eks:DescribeFargateProfile',
                            'eks:DeleteFargateProfile',
                        ],
                        Effect: 'Allow',
                        Resource: '*',
                    },
                    {
                        Action: 'iam:GetRole',
                        Effect: 'Allow',
                        Resource: '*',
                    },
                    {
                        Action: 'iam:CreateServiceLinkedRole',
                        Effect: 'Allow',
                        Resource: '*',
                    },
                    {
                        Action: 'iam:PassRole',
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'FargateClusterfargateprofiledefaultPodExecutionRole66F2610E',
                                'Arn',
                            ],
                        },
                    },
                ],
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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