/*
 * Decompiled with CFR 0.152.
 */
package trieUtils;

import basicUtils.Utils;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Hashtable;
import trieUtils.TrieDB;

public class EfficientNMerCounter {
    public static String UsageInfo = "trieUtils.EfficientNMerCounter\nCounts the number of appearances of N-mers for the purpose of estimating compression\nRequired:\n -r [FILE] Trie file of proteins for counting\n -n [NUM] size of n-mers\nOptional:\n -w [FILE] Write fasta file of nMer queries\n";
    private TrieDB trieFile;
    private int nMerSize;
    private String outFileName;
    public static int ALPHABET_SIZE = 20;
    private int[][] trieNodes;

    public EfficientNMerCounter(String inFileName, int size, String outFileName) {
        if (!Utils.IsFile(inFileName)) {
            System.err.println("ERROR: Invalid input file name " + inFileName);
            System.exit(-1);
        }
        if (size <= 0) {
            System.err.println("ERROR: Invalid n-mer size '" + size + "'!!");
            System.exit(-1);
        }
        this.nMerSize = size;
        this.trieFile = new TrieDB(inFileName);
        System.out.println("Trie size: " + this.trieFile.GetNumAA());
        System.out.println("MAX SIZE : 2147483647");
        if (this.trieFile.GetNumAA() > Integer.MAX_VALUE) {
            System.err.println("ERROR: Implementation failure.  Can only store addresses up to 2147483647, but DB contains " + this.trieFile.GetNumAA());
        }
        if (outFileName == null) {
            this.trieNodes = new int[(int)this.trieFile.GetNumAA() / 2][ALPHABET_SIZE + 1];
        } else {
            this.outFileName = outFileName;
        }
        System.out.println("Constructor complete");
    }

    private void resizeTrieNodes() {
        int[][] newArray = new int[this.trieNodes.length * 2][ALPHABET_SIZE + 1];
        int i = 0;
        while (i < this.trieNodes.length) {
            int j = 0;
            while (j < this.trieNodes[i].length) {
                newArray[i][j] = this.trieNodes[i][j];
                ++j;
            }
            ++i;
        }
        this.trieNodes = newArray;
    }

    public static void main(String[] args) {
        String[] Commands = new String[]{"-r", "-n", "-w"};
        boolean[] Values = new boolean[]{true, true, true};
        Hashtable Options = Utils.ParseCommandLine(args, Commands, Values);
        if (!Options.containsKey("-r") || !Options.containsKey("-n")) {
            System.err.println("ERROR: Must specify an input trie file and a word size!!");
            System.out.println(UsageInfo);
            return;
        }
        String inFileName = (String)Options.get("-r");
        int nSize = Integer.parseInt((String)Options.get("-n"));
        String outFile = null;
        if (Options.containsKey("-w")) {
            outFile = (String)Options.get("-w");
        }
        EfficientNMerCounter c = new EfficientNMerCounter(inFileName, nSize, outFile);
        c.count();
    }

    private void count() {
        if (this.outFileName != null) {
            this.writeQueries();
            return;
        }
        boolean debug = false;
        int numProteins = this.trieFile.getNumProteins();
        int nextNodeIndex = 1;
        int i = 0;
        while (i < numProteins) {
            String proteinSeq;
            if (i != 0 && i % 1000 == 0) {
                System.out.println("[" + i + "/" + numProteins + "]");
            }
            if ((proteinSeq = this.trieFile.getProteinSequence(i)).length() < this.nMerSize) {
                System.err.println("WARNING: Found a protein of length " + proteinSeq.length() + " < " + this.nMerSize);
            } else {
                int len = proteinSeq.length() - this.nMerSize + 1;
                int j = 0;
                while (j < len) {
                    int currNode = 0;
                    String nMer = proteinSeq.substring(j, j + this.nMerSize);
                    if (debug) {
                        System.out.println("DEBUG: Adding nmer " + nMer);
                    }
                    int k = 0;
                    while (k < this.nMerSize) {
                        char currChar = nMer.charAt(k);
                        int trans = this.getTrans(currChar);
                        if (debug) {
                            System.out.println("DEBUG: At node " + currNode + " transition with " + currChar + " (" + trans + ")");
                        }
                        if (this.trieNodes[currNode][trans] != 0) {
                            if (debug) {
                                System.out.println("DEBUG: Already seen this transition!");
                            }
                            currNode = this.trieNodes[currNode][trans];
                        } else {
                            if (debug) {
                                System.out.println("DEBUG: New prefix, adding transition to new node " + nextNodeIndex);
                            }
                            if (nextNodeIndex >= this.trieNodes.length) {
                                System.out.println("DEBUG: Need to resize memory, old size " + this.trieNodes.length + "!!");
                                Runtime r = Runtime.getRuntime();
                                this.resizeTrieNodes();
                                r.gc();
                                System.out.println("DEBUG: Free memory: " + r.freeMemory() + " new size " + this.trieNodes.length);
                            }
                            this.trieNodes[currNode][trans] = nextNodeIndex++;
                            currNode = this.trieNodes[currNode][trans];
                        }
                        ++k;
                    }
                    int[] nArray = this.trieNodes[currNode];
                    nArray[0] = nArray[0] + 1;
                    if (debug) {
                        System.out.println("DEBUG: Finished adding nMer, ended at " + currNode + " with new count " + this.trieNodes[currNode][0]);
                        Utils.WaitForEnter();
                    }
                    ++j;
                }
            }
            ++i;
        }
        int totalNMers = 0;
        int uniqueNMers = 0;
        int occurrencesOfRepeatedNMers = 0;
        int i2 = 0;
        while (i2 < this.trieNodes.length) {
            if (this.trieNodes[i2][0] != 0) {
                if (this.trieNodes[i2][0] == 1) {
                    ++totalNMers;
                    ++uniqueNMers;
                } else {
                    ++totalNMers;
                    occurrencesOfRepeatedNMers += this.trieNodes[i2][0];
                }
            }
            ++i2;
        }
        System.out.println("Total nMers: " + totalNMers);
        System.out.println("Unique nMers (only 1 location): " + uniqueNMers);
        System.out.println("Occurences of repeated nMers: " + occurrencesOfRepeatedNMers);
    }

    private void writeQueries() {
        int numProteins = this.trieFile.getNumProteins();
        FileWriter f = null;
        try {
            f = new FileWriter(this.outFileName);
        }
        catch (IOException E) {
            E.printStackTrace();
            System.exit(-1);
        }
        int i = 0;
        while (i < numProteins) {
            String proteinSeq;
            if (i != 0 && i % 1000 == 0) {
                System.out.println("[" + i + "/" + numProteins + "]");
            }
            if ((proteinSeq = this.trieFile.getProteinSequence(i)).length() < this.nMerSize) {
                System.err.println("WARNING: Found a protein of length " + proteinSeq.length() + " < " + this.nMerSize);
            } else {
                int len = proteinSeq.length() - this.nMerSize + 1;
                int j = 0;
                while (j < len) {
                    boolean currNode = false;
                    String nMer = proteinSeq.substring(j, j + this.nMerSize);
                    try {
                        f.write(">" + i + "." + j + "\n");
                        f.write(String.valueOf(nMer) + "\n");
                    }
                    catch (IOException E) {
                        E.printStackTrace();
                        System.exit(-1);
                    }
                    ++j;
                }
            }
            ++i;
        }
        try {
            f.close();
        }
        catch (IOException E) {
            E.printStackTrace();
            System.exit(-1);
        }
    }

    private int getTrans(char currChar) {
        switch (currChar) {
            case 'A': 
            case 'a': {
                return 1;
            }
            case 'C': 
            case 'c': {
                return 2;
            }
            case 'D': 
            case 'd': {
                return 3;
            }
            case 'E': 
            case 'e': {
                return 4;
            }
            case 'F': 
            case 'f': {
                return 5;
            }
            case 'G': 
            case 'g': {
                return 6;
            }
            case 'H': 
            case 'h': {
                return 7;
            }
            case 'I': 
            case 'i': {
                return 8;
            }
            case 'K': 
            case 'k': {
                return 9;
            }
            case 'L': 
            case 'l': {
                return 10;
            }
            case 'M': 
            case 'm': {
                return 11;
            }
            case 'N': 
            case 'n': {
                return 12;
            }
            case 'P': 
            case 'p': {
                return 13;
            }
            case 'Q': 
            case 'q': {
                return 14;
            }
            case 'R': 
            case 'r': {
                return 15;
            }
            case 'S': 
            case 's': {
                return 16;
            }
            case 'T': 
            case 't': {
                return 17;
            }
            case 'V': 
            case 'v': {
                return 18;
            }
            case 'W': 
            case 'w': {
                return 19;
            }
            case 'Y': 
            case 'y': {
                return 20;
            }
        }
        return -1;
    }
}

