"use strict";
const assert_1 = require("@aws-cdk/assert");
const certificatemanager = require("@aws-cdk/aws-certificatemanager");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'distribution with custom origin adds custom origin'(test) {
        const stack = new cdk.Stack();
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    originHeaders: {
                        'X-Custom-Header': 'somevalue',
                    },
                    customOriginSource: {
                        domainName: 'myorigin.com',
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'AnAmazingWebsiteProbablyCFDistribution47E3983B': {
                    'Type': 'AWS::CloudFront::Distribution',
                    'Properties': {
                        'DistributionConfig': {
                            'DefaultCacheBehavior': {
                                'AllowedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'CachedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'ForwardedValues': {
                                    'Cookies': {
                                        'Forward': 'none',
                                    },
                                    'QueryString': false,
                                },
                                'TargetOriginId': 'origin1',
                                'ViewerProtocolPolicy': 'redirect-to-https',
                                'Compress': true,
                            },
                            'DefaultRootObject': 'index.html',
                            'Enabled': true,
                            'HttpVersion': 'http2',
                            'IPV6Enabled': true,
                            'Origins': [
                                {
                                    'CustomOriginConfig': {
                                        'HTTPPort': 80,
                                        'HTTPSPort': 443,
                                        'OriginKeepaliveTimeout': 5,
                                        'OriginProtocolPolicy': 'https-only',
                                        'OriginReadTimeout': 30,
                                        'OriginSSLProtocols': [
                                            'TLSv1.2',
                                        ],
                                    },
                                    'DomainName': 'myorigin.com',
                                    'Id': 'origin1',
                                    'OriginCustomHeaders': [
                                        {
                                            'HeaderName': 'X-Custom-Header',
                                            'HeaderValue': 'somevalue',
                                        },
                                    ],
                                },
                            ],
                            'PriceClass': 'PriceClass_100',
                            'ViewerCertificate': {
                                'CloudFrontDefaultCertificate': true,
                            },
                        },
                    },
                },
            },
        });
        test.done();
    },
    'most basic distribution'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket,
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Bucket83908E77': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
                'AnAmazingWebsiteProbablyCFDistribution47E3983B': {
                    'Type': 'AWS::CloudFront::Distribution',
                    'Properties': {
                        'DistributionConfig': {
                            'DefaultRootObject': 'index.html',
                            'Origins': [
                                {
                                    'DomainName': {
                                        'Fn::GetAtt': [
                                            'Bucket83908E77',
                                            'RegionalDomainName',
                                        ],
                                    },
                                    'Id': 'origin1',
                                    'S3OriginConfig': {},
                                },
                            ],
                            'ViewerCertificate': {
                                'CloudFrontDefaultCertificate': true,
                            },
                            'PriceClass': 'PriceClass_100',
                            'DefaultCacheBehavior': {
                                'AllowedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'CachedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'TargetOriginId': 'origin1',
                                'ViewerProtocolPolicy': 'redirect-to-https',
                                'ForwardedValues': {
                                    'QueryString': false,
                                    'Cookies': { 'Forward': 'none' },
                                },
                                'Compress': true,
                            },
                            'Enabled': true,
                            'IPV6Enabled': true,
                            'HttpVersion': 'http2',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'distribution with trusted signers on default distribution'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket,
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                            trustedSigners: ['1234'],
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Bucket83908E77': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
                'AnAmazingWebsiteProbablyCFDistribution47E3983B': {
                    'Type': 'AWS::CloudFront::Distribution',
                    'Properties': {
                        'DistributionConfig': {
                            'DefaultRootObject': 'index.html',
                            'Origins': [
                                {
                                    'DomainName': {
                                        'Fn::GetAtt': [
                                            'Bucket83908E77',
                                            'RegionalDomainName',
                                        ],
                                    },
                                    'Id': 'origin1',
                                    'S3OriginConfig': {},
                                },
                            ],
                            'ViewerCertificate': {
                                'CloudFrontDefaultCertificate': true,
                            },
                            'PriceClass': 'PriceClass_100',
                            'DefaultCacheBehavior': {
                                'AllowedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'CachedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'TargetOriginId': 'origin1',
                                'ViewerProtocolPolicy': 'redirect-to-https',
                                'ForwardedValues': {
                                    'QueryString': false,
                                    'Cookies': { 'Forward': 'none' },
                                },
                                'TrustedSigners': [
                                    '1234',
                                ],
                                'Compress': true,
                            },
                            'Enabled': true,
                            'IPV6Enabled': true,
                            'HttpVersion': 'http2',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'distribution with ViewerProtocolPolicy set to a non-default value'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            viewerProtocolPolicy: lib_1.ViewerProtocolPolicy.ALLOW_ALL,
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket,
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Bucket83908E77': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
                'AnAmazingWebsiteProbablyCFDistribution47E3983B': {
                    'Type': 'AWS::CloudFront::Distribution',
                    'Properties': {
                        'DistributionConfig': {
                            'DefaultRootObject': 'index.html',
                            'Origins': [
                                {
                                    'DomainName': {
                                        'Fn::GetAtt': [
                                            'Bucket83908E77',
                                            'RegionalDomainName',
                                        ],
                                    },
                                    'Id': 'origin1',
                                    'S3OriginConfig': {},
                                },
                            ],
                            'ViewerCertificate': {
                                'CloudFrontDefaultCertificate': true,
                            },
                            'PriceClass': 'PriceClass_100',
                            'DefaultCacheBehavior': {
                                'AllowedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'CachedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'TargetOriginId': 'origin1',
                                'ViewerProtocolPolicy': 'allow-all',
                                'ForwardedValues': {
                                    'QueryString': false,
                                    'Cookies': { 'Forward': 'none' },
                                },
                                'Compress': true,
                            },
                            'Enabled': true,
                            'IPV6Enabled': true,
                            'HttpVersion': 'http2',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'distribution with disabled compression'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket,
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                            compress: false,
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Bucket83908E77': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
                'AnAmazingWebsiteProbablyCFDistribution47E3983B': {
                    'Type': 'AWS::CloudFront::Distribution',
                    'Properties': {
                        'DistributionConfig': {
                            'DefaultRootObject': 'index.html',
                            'Origins': [
                                {
                                    'DomainName': {
                                        'Fn::GetAtt': [
                                            'Bucket83908E77',
                                            'RegionalDomainName',
                                        ],
                                    },
                                    'Id': 'origin1',
                                    'S3OriginConfig': {},
                                },
                            ],
                            'ViewerCertificate': {
                                'CloudFrontDefaultCertificate': true,
                            },
                            'PriceClass': 'PriceClass_100',
                            'DefaultCacheBehavior': {
                                'AllowedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'CachedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'TargetOriginId': 'origin1',
                                'ViewerProtocolPolicy': 'redirect-to-https',
                                'ForwardedValues': {
                                    'QueryString': false,
                                    'Cookies': { 'Forward': 'none' },
                                },
                                'Compress': false,
                            },
                            'Enabled': true,
                            'IPV6Enabled': true,
                            'HttpVersion': 'http2',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'distribution with resolvable lambda-association'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        const lambdaFunction = new lambda.SingletonFunction(stack, 'Lambda', {
            uuid: 'xxxx-xxxx-xxxx-xxxx',
            code: lambda.Code.inline('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket,
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                            lambdaFunctionAssociations: [{
                                    eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                                    lambdaFunction: lambdaFunction.latestVersion,
                                }],
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
            'DistributionConfig': {
                'DefaultCacheBehavior': {
                    'LambdaFunctionAssociations': [
                        {
                            'EventType': 'origin-request',
                            'LambdaFunctionARN': {
                                'Fn::Join': [
                                    '',
                                    [
                                        { 'Fn::GetAtt': ['SingletonLambdaxxxxxxxxxxxxxxxx69D4268A', 'Arn'] },
                                        ':$LATEST',
                                    ],
                                ],
                            },
                        },
                    ],
                },
            },
        }));
        test.done();
    },
    'distribution has a defaultChild'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        const distribution = new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket,
                    },
                    behaviors: [{ isDefaultBehavior: true }],
                },
            ],
        });
        test.ok(distribution.node.defaultChild instanceof lib_1.CfnDistribution);
        test.done();
    },
    'allows multiple aliasConfiguration CloudFrontWebDistribution per stack'(test) {
        const stack = new cdk.Stack();
        const s3BucketSource = new s3.Bucket(stack, 'Bucket');
        const originConfigs = [{
                s3OriginSource: { s3BucketSource },
                behaviors: [{ isDefaultBehavior: true }],
            }];
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs,
            aliasConfiguration: { acmCertRef: 'acm_ref', names: ['www.example.com'] },
        });
        new lib_1.CloudFrontWebDistribution(stack, 'AnotherAmazingWebsiteProbably', {
            originConfigs,
            aliasConfiguration: { acmCertRef: 'another_acm_ref', names: ['ftp.example.com'] },
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
            'DistributionConfig': {
                'Aliases': ['www.example.com'],
                'ViewerCertificate': {
                    'AcmCertificateArn': 'acm_ref',
                    'SslSupportMethod': 'sni-only',
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
            'DistributionConfig': {
                'Aliases': ['ftp.example.com'],
                'ViewerCertificate': {
                    'AcmCertificateArn': 'another_acm_ref',
                    'SslSupportMethod': 'sni-only',
                },
            },
        }));
        test.done();
    },
    'viewerCertificate': {
        'acmCertificate': {
            'base usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                const certificate = new certificatemanager.Certificate(stack, 'cert', {
                    domainName: 'example.com',
                });
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromAcmCertificate(certificate),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': [],
                        'ViewerCertificate': {
                            'AcmCertificateArn': {
                                'Ref': 'cert56CA94EB',
                            },
                            'SslSupportMethod': 'sni-only',
                        },
                    },
                }));
                test.done();
            },
            'imported certificate fromCertificateArn'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                const certificate = certificatemanager.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromAcmCertificate(certificate),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': [],
                        'ViewerCertificate': {
                            'AcmCertificateArn': 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d',
                            'SslSupportMethod': 'sni-only',
                        },
                    },
                }));
                test.done();
            },
            'advanced usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                const certificate = new certificatemanager.Certificate(stack, 'cert', {
                    domainName: 'example.com',
                });
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromAcmCertificate(certificate, {
                        securityPolicy: lib_1.SecurityPolicyProtocol.SSL_V3,
                        sslMethod: lib_1.SSLMethod.VIP,
                        aliases: ['example.com', 'www.example.com'],
                    }),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': ['example.com', 'www.example.com'],
                        'ViewerCertificate': {
                            'AcmCertificateArn': {
                                'Ref': 'cert56CA94EB',
                            },
                            'MinimumProtocolVersion': 'SSLv3',
                            'SslSupportMethod': 'vip',
                        },
                    },
                }));
                test.done();
            },
        },
        'iamCertificate': {
            'base usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromIamCertificate('test'),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': [],
                        'ViewerCertificate': {
                            'IamCertificateId': 'test',
                            'SslSupportMethod': 'sni-only',
                        },
                    },
                }));
                test.done();
            },
            'advanced usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromIamCertificate('test', {
                        securityPolicy: lib_1.SecurityPolicyProtocol.TLS_V1,
                        sslMethod: lib_1.SSLMethod.VIP,
                        aliases: ['example.com'],
                    }),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': ['example.com'],
                        'ViewerCertificate': {
                            'IamCertificateId': 'test',
                            'MinimumProtocolVersion': 'TLSv1',
                            'SslSupportMethod': 'vip',
                        },
                    },
                }));
                test.done();
            },
        },
        'cloudFrontDefaultCertificate': {
            'base usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromCloudFrontDefaultCertificate(),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': [],
                        'ViewerCertificate': {
                            'CloudFrontDefaultCertificate': true,
                        },
                    },
                }));
                test.done();
            },
            'aliases are set'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromCloudFrontDefaultCertificate('example.com', 'www.example.com'),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': ['example.com', 'www.example.com'],
                        'ViewerCertificate': {
                            'CloudFrontDefaultCertificate': true,
                        },
                    },
                }));
                test.done();
            },
        },
        'errors': {
            'throws if both deprecated aliasConfiguration and viewerCertificate'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                test.throws(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                        originConfigs: [{
                                s3OriginSource: { s3BucketSource: sourceBucket },
                                behaviors: [{ isDefaultBehavior: true }],
                            }],
                        aliasConfiguration: { acmCertRef: 'test', names: ['ftp.example.com'] },
                        viewerCertificate: lib_1.ViewerCertificate.fromCloudFrontDefaultCertificate('example.com', 'www.example.com'),
                    });
                }, /You cannot set both aliasConfiguration and viewerCertificate properties/);
                test.done();
            },
            'throws if invalid security policy for SSL method'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                test.throws(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                        originConfigs: [{
                                s3OriginSource: { s3BucketSource: sourceBucket },
                                behaviors: [{ isDefaultBehavior: true }],
                            }],
                        viewerCertificate: lib_1.ViewerCertificate.fromIamCertificate('test', {
                            securityPolicy: lib_1.SecurityPolicyProtocol.TLS_V1_1_2016,
                            sslMethod: lib_1.SSLMethod.VIP,
                        }),
                    });
                }, /TLSv1.1_2016 is not compabtible with sslMethod vip./);
                test.done();
            },
            // FIXME https://github.com/aws/aws-cdk/issues/4724
            'does not throw if acmCertificate explicitly not in us-east-1'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                const certificate = certificatemanager.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:eu-west-3:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromAcmCertificate(certificate),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': [],
                        'ViewerCertificate': {
                            'AcmCertificateArn': 'arn:aws:acm:eu-west-3:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d',
                            'SslSupportMethod': 'sni-only',
                        },
                    },
                }));
                test.done();
            },
        },
    },
    'edgelambda.amazonaws.com is added to the trust policy of lambda'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        const fn = new lambda.Function(stack, 'Lambda', {
            code: lambda.Code.fromInline('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const lambdaVersion = new lambda.Version(stack, 'LambdaVersion', { lambda: fn });
        // WHEN
        new lib_1.CloudFrontWebDistribution(stack, 'MyDistribution', {
            originConfigs: [
                {
                    s3OriginSource: { s3BucketSource: sourceBucket },
                    behaviors: [
                        {
                            isDefaultBehavior: true, lambdaFunctionAssociations: [
                                {
                                    eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                                    lambdaFunction: lambdaVersion,
                                },
                            ],
                        },
                    ],
                },
            ],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                'Statement': [
                    {
                        'Action': 'sts:AssumeRole',
                        'Effect': 'Allow',
                        'Principal': {
                            'Service': 'lambda.amazonaws.com',
                        },
                    },
                    {
                        'Action': 'sts:AssumeRole',
                        'Effect': 'Allow',
                        'Principal': {
                            'Service': 'edgelambda.amazonaws.com',
                        },
                    },
                ],
                'Version': '2012-10-17',
            },
        }));
        test.done();
    },
    'edgelambda.amazonaws.com is not added to lambda role for imported functions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        const lambdaVersion = lambda.Version.fromVersionArn(stack, 'Version', 'arn:my-version');
        // WHEN
        new lib_1.CloudFrontWebDistribution(stack, 'MyDistribution', {
            originConfigs: [
                {
                    s3OriginSource: { s3BucketSource: sourceBucket },
                    behaviors: [
                        {
                            isDefaultBehavior: true, lambdaFunctionAssociations: [
                                {
                                    eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                                    lambdaFunction: lambdaVersion,
                                },
                            ],
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Role'));
        test.done();
    },
    'geo restriction': {
        'success': {
            'whitelist'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    geoRestriction: lib_1.GeoRestriction.whitelist('US', 'UK'),
                });
                assert_1.expect(stack).toMatch({
                    'Resources': {
                        'Bucket83908E77': {
                            'Type': 'AWS::S3::Bucket',
                            'DeletionPolicy': 'Retain',
                            'UpdateReplacePolicy': 'Retain',
                        },
                        'AnAmazingWebsiteProbablyCFDistribution47E3983B': {
                            'Type': 'AWS::CloudFront::Distribution',
                            'Properties': {
                                'DistributionConfig': {
                                    'DefaultRootObject': 'index.html',
                                    'Origins': [
                                        {
                                            'DomainName': {
                                                'Fn::GetAtt': [
                                                    'Bucket83908E77',
                                                    'RegionalDomainName',
                                                ],
                                            },
                                            'Id': 'origin1',
                                            'S3OriginConfig': {},
                                        },
                                    ],
                                    'ViewerCertificate': {
                                        'CloudFrontDefaultCertificate': true,
                                    },
                                    'PriceClass': 'PriceClass_100',
                                    'DefaultCacheBehavior': {
                                        'AllowedMethods': [
                                            'GET',
                                            'HEAD',
                                        ],
                                        'CachedMethods': [
                                            'GET',
                                            'HEAD',
                                        ],
                                        'TargetOriginId': 'origin1',
                                        'ViewerProtocolPolicy': 'redirect-to-https',
                                        'ForwardedValues': {
                                            'QueryString': false,
                                            'Cookies': { 'Forward': 'none' },
                                        },
                                        'Compress': true,
                                    },
                                    'Enabled': true,
                                    'IPV6Enabled': true,
                                    'HttpVersion': 'http2',
                                    'Restrictions': {
                                        'GeoRestriction': {
                                            'Locations': ['US', 'UK'],
                                            'RestrictionType': 'whitelist',
                                        },
                                    },
                                },
                            },
                        },
                    },
                });
                test.done();
            },
            'blacklist'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    geoRestriction: lib_1.GeoRestriction.blacklist('US'),
                });
                assert_1.expect(stack).toMatch({
                    'Resources': {
                        'Bucket83908E77': {
                            'Type': 'AWS::S3::Bucket',
                            'DeletionPolicy': 'Retain',
                            'UpdateReplacePolicy': 'Retain',
                        },
                        'AnAmazingWebsiteProbablyCFDistribution47E3983B': {
                            'Type': 'AWS::CloudFront::Distribution',
                            'Properties': {
                                'DistributionConfig': {
                                    'DefaultRootObject': 'index.html',
                                    'Origins': [
                                        {
                                            'DomainName': {
                                                'Fn::GetAtt': [
                                                    'Bucket83908E77',
                                                    'RegionalDomainName',
                                                ],
                                            },
                                            'Id': 'origin1',
                                            'S3OriginConfig': {},
                                        },
                                    ],
                                    'ViewerCertificate': {
                                        'CloudFrontDefaultCertificate': true,
                                    },
                                    'PriceClass': 'PriceClass_100',
                                    'DefaultCacheBehavior': {
                                        'AllowedMethods': [
                                            'GET',
                                            'HEAD',
                                        ],
                                        'CachedMethods': [
                                            'GET',
                                            'HEAD',
                                        ],
                                        'TargetOriginId': 'origin1',
                                        'ViewerProtocolPolicy': 'redirect-to-https',
                                        'ForwardedValues': {
                                            'QueryString': false,
                                            'Cookies': { 'Forward': 'none' },
                                        },
                                        'Compress': true,
                                    },
                                    'Enabled': true,
                                    'IPV6Enabled': true,
                                    'HttpVersion': 'http2',
                                    'Restrictions': {
                                        'GeoRestriction': {
                                            'Locations': ['US'],
                                            'RestrictionType': 'blacklist',
                                        },
                                    },
                                },
                            },
                        },
                    },
                });
                test.done();
            },
        },
        'error': {
            'throws if locations is empty array'(test) {
                test.throws(() => {
                    lib_1.GeoRestriction.whitelist();
                }, 'Should provide at least 1 location');
                test.throws(() => {
                    lib_1.GeoRestriction.blacklist();
                }, 'Should provide at least 1 location');
                test.done();
            },
            'throws if locations format is wrong'(test) {
                test.throws(() => {
                    lib_1.GeoRestriction.whitelist('us');
                }, 'Invalid location format for location: us, location should be two-letter and uppercase country ISO 3166-1-alpha-2 code');
                test.throws(() => {
                    lib_1.GeoRestriction.blacklist('us');
                }, 'Invalid location format for location: us, location should be two-letter and uppercase country ISO 3166-1-alpha-2 code');
                test.done();
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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