"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Volume = exports.EbsDeviceVolumeType = exports.BlockDeviceVolume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * Describes a block device mapping for an EC2 instance or Auto Scaling group.
 *
 * @stability stable
 */
class BlockDeviceVolume {
    /**
     * @param ebsDevice EBS device info.
     * @param virtualName Virtual device name.
     * @stability stable
     */
    constructor(ebsDevice, virtualName) {
        this.ebsDevice = ebsDevice;
        this.virtualName = virtualName;
    }
    /**
     * Creates a new Elastic Block Storage device.
     *
     * @param volumeSize The volume size, in Gibibytes (GiB).
     * @param options additional device options.
     * @stability stable
     */
    static ebs(volumeSize, options = {}) {
        return new this({ ...options, volumeSize });
    }
    /**
     * Creates a new Elastic Block Storage device from an existing snapshot.
     *
     * @param snapshotId The snapshot ID of the volume to use.
     * @param options additional device options.
     * @stability stable
     */
    static ebsFromSnapshot(snapshotId, options = {}) {
        return new this({ ...options, snapshotId });
    }
    /**
     * Creates a virtual, ephemeral device.
     *
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index.
     * @stability stable
     */
    static ephemeral(volumeIndex) {
        if (volumeIndex < 0) {
            throw new Error(`volumeIndex must be a number starting from 0, got "${volumeIndex}"`);
        }
        return new this(undefined, `ephemeral${volumeIndex}`);
    }
}
exports.BlockDeviceVolume = BlockDeviceVolume;
_a = JSII_RTTI_SYMBOL_1;
BlockDeviceVolume[_a] = { fqn: "@aws-cdk/aws-ec2.BlockDeviceVolume", version: "1.90.1" };
/**
 * Supported EBS volume types for blockDevices.
 *
 * @stability stable
 */
var EbsDeviceVolumeType;
(function (EbsDeviceVolumeType) {
    EbsDeviceVolumeType["STANDARD"] = "standard";
    EbsDeviceVolumeType["IO1"] = "io1";
    EbsDeviceVolumeType["IO2"] = "io2";
    EbsDeviceVolumeType["GP2"] = "gp2";
    EbsDeviceVolumeType["GP3"] = "gp3";
    EbsDeviceVolumeType["ST1"] = "st1";
    EbsDeviceVolumeType["SC1"] = "sc1";
    /**
     * General purpose SSD volume (GP2) that balances price and performance for a wide variety of workloads.
     */
    EbsDeviceVolumeType["GENERAL_PURPOSE_SSD"] = "gp2";
    /**
     * General purpose SSD volume (GP3) that balances price and performance for a wide variety of workloads.
     */
    EbsDeviceVolumeType["GENERAL_PURPOSE_SSD_GP3"] = "gp3";
    /**
     * Highest-performance SSD volume (IO1) for mission-critical low-latency or high-throughput workloads.
     */
    EbsDeviceVolumeType["PROVISIONED_IOPS_SSD"] = "io1";
    /**
     * Highest-performance SSD volume (IO2) for mission-critical low-latency or high-throughput workloads.
     */
    EbsDeviceVolumeType["PROVISIONED_IOPS_SSD_IO2"] = "io2";
    /**
     * Low-cost HDD volume designed for frequently accessed, throughput-intensive workloads.
     */
    EbsDeviceVolumeType["THROUGHPUT_OPTIMIZED_HDD"] = "st1";
    /**
     * Lowest cost HDD volume designed for less frequently accessed workloads.
     */
    EbsDeviceVolumeType["COLD_HDD"] = "sc1";
    /**
     * Magnetic volumes are backed by magnetic drives and are suited for workloads where data is accessed infrequently, and scenarios where low-cost
     * storage for small volume sizes is important.
     */
    EbsDeviceVolumeType["MAGNETIC"] = "standard";
})(EbsDeviceVolumeType = exports.EbsDeviceVolumeType || (exports.EbsDeviceVolumeType = {}));
/**
 * Common behavior of Volumes. Users should not use this class directly, and instead use ``Volume``.
 */
class VolumeBase extends core_1.Resource {
    /**
     * Grants permission to attach this Volume to an instance.
     *
     * CAUTION: Granting an instance permission to attach to itself using this method will lead to
     * an unresolvable circular reference between the instance role and the instance.
     * Use {@link IVolume.grantAttachVolumeToSelf} to grant an instance permission to attach this
     * volume to itself.
     *
     * @stability stable
     */
    grantAttachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:AttachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        if (this.encryptionKey) {
            // When attaching a volume, the EC2 Service will need to grant to itself permission
            // to be able to decrypt the encryption key. We restrict the CreateGrant for principle
            // of least privilege, in accordance with best practices.
            // See: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-permissions
            const kmsGrant = this.encryptionKey.grant(grantee, 'kms:CreateGrant');
            kmsGrant.principalStatement.addConditions({
                Bool: { 'kms:GrantIsForAWSResource': true },
                StringEquals: {
                    'kms:ViaService': `ec2.${core_1.Stack.of(this).region}.amazonaws.com`,
                    'kms:GrantConstraintType': 'EncryptionContextSubset',
                },
            });
        }
        return result;
    }
    /**
     * Grants permission to attach the Volume by a ResourceTag condition.
     *
     * If you are looking to
     * grant an Instance, AutoScalingGroup, EC2-Fleet, SpotFleet, ECS host, etc the ability to attach
     * this volume to **itself** then this is the method you want to use.
     *
     * This is implemented by adding a Tag with key `VolumeGrantAttach-<suffix>` to the given
     * constructs and this Volume, and then conditioning the Grant such that the grantee is only
     * given the ability to AttachVolume if both the Volume and the destination Instance have that
     * tag applied to them.
     *
     * @stability stable
     */
    grantAttachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantAttach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantAttachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    /**
     * Grants permission to detach this Volume from an instance CAUTION: Granting an instance permission to detach from itself using this method will lead to an unresolvable circular reference between the instance role and the instance.
     *
     * Use {@link IVolume.grantDetachVolumeFromSelf} to grant an instance permission to detach this
     * volume from itself.
     *
     * @stability stable
     */
    grantDetachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:DetachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        // Note: No encryption key permissions are required to detach an encrypted volume.
        return result;
    }
    /**
     * Grants permission to detach the Volume by a ResourceTag condition.
     *
     * This is implemented via the same mechanism as {@link IVolume.grantAttachVolumeByResourceTag},
     * and is subject to the same conditions.
     *
     * @stability stable
     */
    grantDetachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantDetach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantDetachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    collectGrantResourceArns(instances) {
        const stack = core_1.Stack.of(this);
        const resourceArns = [
            `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:volume/${this.volumeId}`,
        ];
        const instanceArnPrefix = `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:instance`;
        if (instances) {
            instances.forEach(instance => resourceArns.push(`${instanceArnPrefix}/${instance === null || instance === void 0 ? void 0 : instance.instanceId}`));
        }
        else {
            resourceArns.push(`${instanceArnPrefix}/*`);
        }
        return resourceArns;
    }
    calculateResourceTagValue(constructs) {
        const md5 = crypto.createHash('md5');
        constructs.forEach(construct => md5.update(core_1.Names.uniqueId(construct)));
        return md5.digest('hex');
    }
}
/**
 * Creates a new EBS Volume in AWS EC2.
 *
 * @stability stable
 */
class Volume extends VolumeBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g;
        super(scope, id, {
            physicalName: props.volumeName,
        });
        this.validateProps(props);
        const resource = new ec2_generated_1.CfnVolume(this, 'Resource', {
            availabilityZone: props.availabilityZone,
            autoEnableIo: props.autoEnableIo,
            encrypted: props.encrypted,
            kmsKeyId: (_c = props.encryptionKey) === null || _c === void 0 ? void 0 : _c.keyArn,
            iops: props.iops,
            multiAttachEnabled: (_d = props.enableMultiAttach) !== null && _d !== void 0 ? _d : false,
            size: (_e = props.size) === null || _e === void 0 ? void 0 : _e.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }),
            snapshotId: props.snapshotId,
            volumeType: (_f = props.volumeType) !== null && _f !== void 0 ? _f : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD,
        });
        this.volumeId = resource.ref;
        this.availabilityZone = props.availabilityZone;
        this.encryptionKey = props.encryptionKey;
        if (this.encryptionKey) {
            // Per: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-requirements
            const principal = new aws_kms_1.ViaServicePrincipal(`ec2.${core_1.Stack.of(this).region}.amazonaws.com`, new aws_iam_1.AccountRootPrincipal()).withConditions({
                StringEquals: {
                    'kms:CallerAccount': core_1.Stack.of(this).account,
                },
            });
            const grant = this.encryptionKey.grant(principal, 
            // Describe & Generate are required to be able to create the CMK-encrypted Volume.
            'kms:DescribeKey', 'kms:GenerateDataKeyWithoutPlainText');
            if (props.snapshotId) {
                // ReEncrypt is required for when re-encrypting from an encrypted snapshot.
                (_g = grant.principalStatement) === null || _g === void 0 ? void 0 : _g.addActions('kms:ReEncrypt*');
            }
        }
    }
    /**
     * Import an existing EBS Volume into the Stack.
     *
     * @param scope the scope of the import.
     * @param id the ID of the imported Volume in the construct tree.
     * @param attrs the attributes of the imported Volume.
     * @stability stable
     */
    static fromVolumeAttributes(scope, id, attrs) {
        class Import extends VolumeBase {
            constructor() {
                super(...arguments);
                this.volumeId = attrs.volumeId;
                this.availabilityZone = attrs.availabilityZone;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        // Check that the provided volumeId looks like it could be valid.
        if (!core_1.Token.isUnresolved(attrs.volumeId) && !/^vol-[0-9a-fA-F]+$/.test(attrs.volumeId)) {
            throw new Error('`volumeId` does not match expected pattern. Expected `vol-<hexadecmial value>` (ex: `vol-05abe246af`) or a Token');
        }
        return new Import(scope, id);
    }
    /**
     * @stability stable
     */
    validateProps(props) {
        var _c, _d, _e;
        if (!(props.size || props.snapshotId)) {
            throw new Error('Must provide at least one of `size` or `snapshotId`');
        }
        if (props.snapshotId && !core_1.Token.isUnresolved(props.snapshotId) && !/^snap-[0-9a-fA-F]+$/.test(props.snapshotId)) {
            throw new Error('`snapshotId` does match expected pattern. Expected `snap-<hexadecmial value>` (ex: `snap-05abe246af`) or Token');
        }
        if (props.encryptionKey && !props.encrypted) {
            throw new Error('`encrypted` must be true when providing an `encryptionKey`.');
        }
        if (props.volumeType &&
            [
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2,
            ].includes(props.volumeType) &&
            !props.iops) {
            throw new Error('`iops` must be specified if the `volumeType` is `PROVISIONED_IOPS_SSD` or `PROVISIONED_IOPS_SSD_IO2`.');
        }
        if (props.iops) {
            const volumeType = (_c = props.volumeType) !== null && _c !== void 0 ? _c : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            if (![
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2,
                EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3,
            ].includes(volumeType)) {
                throw new Error('`iops` may only be specified if the `volumeType` is `PROVISIONED_IOPS_SSD`, `PROVISIONED_IOPS_SSD_IO2` or `GENERAL_PURPOSE_SSD_GP3`.');
            }
            // Enforce minimum & maximum IOPS:
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ebs-volume.html
            const iopsRanges = {};
            iopsRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3] = { Min: 3000, Max: 16000 };
            iopsRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = { Min: 100, Max: 64000 };
            iopsRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2] = { Min: 100, Max: 64000 };
            const { Min, Max } = iopsRanges[volumeType];
            if (props.iops < Min || props.iops > Max) {
                throw new Error(`\`${volumeType}\` volumes iops must be between ${Min} and ${Max}.`);
            }
            // Enforce maximum ratio of IOPS/GiB:
            // https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html
            const maximumRatios = {};
            maximumRatios[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3] = 500;
            maximumRatios[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = 50;
            maximumRatios[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2] = 500;
            const maximumRatio = maximumRatios[volumeType];
            if (props.size && (props.iops > maximumRatio * props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }))) {
                throw new Error(`\`${volumeType}\` volumes iops has a maximum ratio of ${maximumRatio} IOPS/GiB.`);
            }
        }
        if (props.enableMultiAttach) {
            const volumeType = (_d = props.volumeType) !== null && _d !== void 0 ? _d : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            if (![
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2,
            ].includes(volumeType)) {
                throw new Error('multi-attach is supported exclusively on `PROVISIONED_IOPS_SSD` and `PROVISIONED_IOPS_SSD_IO2` volumes.');
            }
        }
        if (props.size) {
            const size = props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL });
            // Enforce minimum & maximum volume size:
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ebs-volume.html
            const sizeRanges = {};
            sizeRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD] = { Min: 1, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3] = { Min: 1, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = { Min: 4, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2] = { Min: 4, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.THROUGHPUT_OPTIMIZED_HDD] = { Min: 125, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.COLD_HDD] = { Min: 125, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.MAGNETIC] = { Min: 1, Max: 1024 };
            const volumeType = (_e = props.volumeType) !== null && _e !== void 0 ? _e : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            const { Min, Max } = sizeRanges[volumeType];
            if (size < Min || size > Max) {
                throw new Error(`\`${volumeType}\` volumes must be between ${Min} GiB and ${Max} GiB in size.`);
            }
        }
    }
}
exports.Volume = Volume;
_b = JSII_RTTI_SYMBOL_1;
Volume[_b] = { fqn: "@aws-cdk/aws-ec2.Volume", version: "1.90.1" };
//# sourceMappingURL=data:application/json;base64,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