"use strict";
var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitSource = exports.InitService = exports.InitPackage = exports.InitUser = exports.InitGroup = exports.InitFile = exports.InitCommand = exports.InitCommandWaitDuration = exports.InitElement = exports.InitServiceRestartHandle = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * An object that represents reasons to restart an InitService.
 *
 * Pass an instance of this object to the `InitFile`, `InitCommand`,
 * `InitSource` and `InitPackage` objects, and finally to an `InitService`
 * itself to cause the actions (files, commands, sources, and packages)
 * to trigger a restart of the service.
 *
 * For example, the following will run a custom command to install Nginx,
 * and trigger the nginx service to be restarted after the command has run.
 *
 * ```ts
 * const handle = new ec2.InitServiceRestartHandle();
 * ec2.CloudFormationInit.fromElements(
 *    ec2.InitCommand.shellCommand('/usr/bin/custom-nginx-install.sh', { serviceRestartHandles: [handle] }),
 *    ec2.InitService.enable('nginx', { serviceRestartHandle: handle }),
 * );
 * ```
 *
 * @stability stable
 */
class InitServiceRestartHandle {
    constructor() {
        this.commands = new Array();
        this.files = new Array();
        this.sources = new Array();
        this.packages = {};
    }
    /**
     * Add a command key to the restart set
     * @internal
     */
    _addCommand(key) {
        return this.commands.push(key);
    }
    /**
     * Add a file key to the restart set
     * @internal
     */
    _addFile(key) {
        return this.files.push(key);
    }
    /**
     * Add a source key to the restart set
     * @internal
     */
    _addSource(key) {
        return this.sources.push(key);
    }
    /**
     * Add a package key to the restart set
     * @internal
     */
    _addPackage(packageType, key) {
        if (!this.packages[packageType]) {
            this.packages[packageType] = [];
        }
        this.packages[packageType].push(key);
    }
    /**
     * Render the restart handles for use in an InitService declaration
     * @internal
     */
    _renderRestartHandles() {
        const nonEmpty = (x) => x.length > 0 ? x : undefined;
        return {
            commands: nonEmpty(this.commands),
            files: nonEmpty(this.files),
            packages: Object.keys(this.packages).length > 0 ? this.packages : undefined,
            sources: nonEmpty(this.sources),
        };
    }
}
exports.InitServiceRestartHandle = InitServiceRestartHandle;
_a = JSII_RTTI_SYMBOL_1;
InitServiceRestartHandle[_a] = { fqn: "@aws-cdk/aws-ec2.InitServiceRestartHandle", version: "1.90.1" };
/**
 * Base class for all CloudFormation Init elements.
 *
 * @stability stable
 */
class InitElement {
}
exports.InitElement = InitElement;
_b = JSII_RTTI_SYMBOL_1;
InitElement[_b] = { fqn: "@aws-cdk/aws-ec2.InitElement", version: "1.90.1" };
/**
 * Represents a duration to wait after a command has finished, in case of a reboot (Windows only).
 *
 * @stability stable
 */
class InitCommandWaitDuration {
    /**
     * Wait for a specified duration after a command.
     *
     * @stability stable
     */
    static of(duration) {
        return new class extends InitCommandWaitDuration {
            /** @internal */
            _render() { return duration.toSeconds(); }
        }();
    }
    /**
     * Do not wait for this command.
     *
     * @stability stable
     */
    static none() {
        return InitCommandWaitDuration.of(core_1.Duration.seconds(0));
    }
    /**
     * cfn-init will exit and resume only after a reboot.
     *
     * @stability stable
     */
    static forever() {
        return new class extends InitCommandWaitDuration {
            /** @internal */
            _render() { return 'forever'; }
        }();
    }
}
exports.InitCommandWaitDuration = InitCommandWaitDuration;
_c = JSII_RTTI_SYMBOL_1;
InitCommandWaitDuration[_c] = { fqn: "@aws-cdk/aws-ec2.InitCommandWaitDuration", version: "1.90.1" };
/**
 * Command to execute on the instance.
 *
 * @stability stable
 */
class InitCommand extends InitElement {
    constructor(command, options) {
        super();
        this.command = command;
        this.options = options;
        /**
         * Returns the init element type for this element.
         *
         * @stability stable
         */
        this.elementType = cfn_init_internal_1.InitElementType.COMMAND.toString();
    }
    /**
     * Run a shell command.
     *
     * Remember that some characters like `&`, `|`, `;`, `>` etc. have special meaning in a shell and
     * need to be preceded by a `\` if you want to treat them as part of a filename.
     *
     * @stability stable
     */
    static shellCommand(shellCommand, options = {}) {
        return new InitCommand(shellCommand, options);
    }
    /**
     * Run a command from an argv array.
     *
     * You do not need to escape space characters or enclose command parameters in quotes.
     *
     * @stability stable
     */
    static argvCommand(argv, options = {}) {
        if (argv.length === 0) {
            throw new Error('Cannot define argvCommand with an empty arguments');
        }
        return new InitCommand(argv, options);
    }
    /** @internal */
    _bind(options) {
        var _l, _m;
        const commandKey = this.options.key || `${options.index}`.padStart(3, '0'); // 001, 005, etc.
        if (options.platform !== cfn_init_internal_1.InitPlatform.WINDOWS && this.options.waitAfterCompletion !== undefined) {
            throw new Error(`Command '${this.command}': 'waitAfterCompletion' is only valid for Windows systems.`);
        }
        for (const handle of (_l = this.options.serviceRestartHandles) !== null && _l !== void 0 ? _l : []) {
            handle._addCommand(commandKey);
        }
        return {
            config: {
                [commandKey]: {
                    command: this.command,
                    env: this.options.env,
                    cwd: this.options.cwd,
                    test: this.options.testCmd,
                    ignoreErrors: this.options.ignoreErrors,
                    waitAfterCompletion: (_m = this.options.waitAfterCompletion) === null || _m === void 0 ? void 0 : _m._render(),
                },
            },
        };
    }
}
exports.InitCommand = InitCommand;
_d = JSII_RTTI_SYMBOL_1;
InitCommand[_d] = { fqn: "@aws-cdk/aws-ec2.InitCommand", version: "1.90.1" };
/**
 * Create files on the EC2 instance.
 *
 * @stability stable
 */
class InitFile extends InitElement {
    /**
     * @stability stable
     */
    constructor(fileName, options) {
        super();
        this.fileName = fileName;
        this.options = options;
        /**
         * Returns the init element type for this element.
         *
         * @stability stable
         */
        this.elementType = cfn_init_internal_1.InitElementType.FILE.toString();
    }
    /**
     * Use a literal string as the file content.
     *
     * @stability stable
     */
    static fromString(fileName, content, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        content,
                        encoding: this.options.base64Encoded ? 'base64' : 'plain',
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Write a symlink with the given symlink target.
     *
     * @stability stable
     */
    static symlink(fileName, target, options = {}) {
        const { mode, ...otherOptions } = options;
        if (mode && mode.slice(0, 3) !== '120') {
            throw new Error('File mode for symlinks must begin with 120XXX');
        }
        return InitFile.fromString(fileName, target, { mode: (mode || '120644'), ...otherOptions });
    }
    /**
     * Use a JSON-compatible object as the file content, write it to a JSON file.
     *
     * May contain tokens.
     *
     * @stability stable
     */
    static fromObject(fileName, obj, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        content: obj,
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Read a file from disk and use its contents.
     *
     * The file will be embedded in the template, so care should be taken to not
     * exceed the template size.
     *
     * If options.base64encoded is set to true, this will base64-encode the file's contents.
     *
     * @stability stable
     */
    static fromFileInline(targetFileName, sourceFileName, options = {}) {
        const encoding = options.base64Encoded ? 'base64' : 'utf8';
        const fileContents = fs.readFileSync(sourceFileName).toString(encoding);
        return InitFile.fromString(targetFileName, fileContents, options);
    }
    /**
     * Download from a URL at instance startup time.
     *
     * @stability stable
     */
    static fromUrl(fileName, url, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: url,
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Download a file from an S3 bucket at instance startup time.
     *
     * @stability stable
     */
    static fromS3Object(fileName, bucket, key, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                bucket.grantRead(bindOptions.instanceRole, key);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: bucket.urlForObject(key),
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, bucket.bucketName),
                };
            }
        }(fileName, options);
    }
    /**
     * Create an asset from the given file.
     *
     * This is appropriate for files that are too large to embed into the template.
     *
     * @stability stable
     */
    static fromAsset(targetFileName, path, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                const asset = new s3_assets.Asset(bindOptions.scope, `${targetFileName}Asset`, {
                    path,
                    ...options,
                });
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: asset.httpUrl,
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetFileName, options);
    }
    /**
     * Use a file from an asset at instance startup time.
     *
     * @stability stable
     */
    static fromExistingAsset(targetFileName, asset, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: asset.httpUrl,
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetFileName, options);
    }
    /** @internal */
    _bind(bindOptions) {
        var _l;
        for (const handle of (_l = this.options.serviceRestartHandles) !== null && _l !== void 0 ? _l : []) {
            handle._addFile(this.fileName);
        }
        return this._doBind(bindOptions);
    }
    /**
     * Render the standard config block, given content vars
     * @internal
     */
    _standardConfig(fileOptions, platform, contentVars) {
        if (platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            if (fileOptions.group || fileOptions.owner || fileOptions.mode) {
                throw new Error('Owner, group, and mode options not supported for Windows.');
            }
            return {
                [this.fileName]: { ...contentVars },
            };
        }
        return {
            [this.fileName]: {
                ...contentVars,
                mode: fileOptions.mode || '000644',
                owner: fileOptions.owner || 'root',
                group: fileOptions.group || 'root',
            },
        };
    }
}
exports.InitFile = InitFile;
_e = JSII_RTTI_SYMBOL_1;
InitFile[_e] = { fqn: "@aws-cdk/aws-ec2.InitFile", version: "1.90.1" };
/**
 * Create Linux/UNIX groups and assign group IDs.
 *
 * Not supported for Windows systems.
 *
 * @stability stable
 */
class InitGroup extends InitElement {
    /**
     * @stability stable
     */
    constructor(groupName, groupId) {
        super();
        this.groupName = groupName;
        this.groupId = groupId;
        /**
         * Returns the init element type for this element.
         *
         * @stability stable
         */
        this.elementType = cfn_init_internal_1.InitElementType.GROUP.toString();
    }
    /**
     * Create a group from its name, and optionally, group id.
     *
     * @stability stable
     */
    static fromName(groupName, groupId) {
        return new InitGroup(groupName, groupId);
    }
    /** @internal */
    _bind(options) {
        if (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            throw new Error('Init groups are not supported on Windows');
        }
        return {
            config: {
                [this.groupName]: this.groupId !== undefined ? { gid: this.groupId } : {},
            },
        };
    }
}
exports.InitGroup = InitGroup;
_f = JSII_RTTI_SYMBOL_1;
InitGroup[_f] = { fqn: "@aws-cdk/aws-ec2.InitGroup", version: "1.90.1" };
/**
 * Create Linux/UNIX users and to assign user IDs.
 *
 * Users are created as non-interactive system users with a shell of
 * /sbin/nologin. This is by design and cannot be modified.
 *
 * Not supported for Windows systems.
 *
 * @stability stable
 */
class InitUser extends InitElement {
    /**
     * @stability stable
     */
    constructor(userName, userOptions) {
        super();
        this.userName = userName;
        this.userOptions = userOptions;
        /**
         * Returns the init element type for this element.
         *
         * @stability stable
         */
        this.elementType = cfn_init_internal_1.InitElementType.USER.toString();
    }
    /**
     * Create a user from user name.
     *
     * @stability stable
     */
    static fromName(userName, options = {}) {
        return new InitUser(userName, options);
    }
    /** @internal */
    _bind(options) {
        if (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            throw new Error('Init users are not supported on Windows');
        }
        return {
            config: {
                [this.userName]: {
                    uid: this.userOptions.userId,
                    groups: this.userOptions.groups,
                    homeDir: this.userOptions.homeDir,
                },
            },
        };
    }
}
exports.InitUser = InitUser;
_g = JSII_RTTI_SYMBOL_1;
InitUser[_g] = { fqn: "@aws-cdk/aws-ec2.InitUser", version: "1.90.1" };
/**
 * A package to be installed during cfn-init time.
 *
 * @stability stable
 */
class InitPackage extends InitElement {
    /**
     * @stability stable
     */
    constructor(type, versions, packageName, serviceHandles) {
        super();
        this.type = type;
        this.versions = versions;
        this.packageName = packageName;
        this.serviceHandles = serviceHandles;
        /**
         * Returns the init element type for this element.
         *
         * @stability stable
         */
        this.elementType = cfn_init_internal_1.InitElementType.PACKAGE.toString();
    }
    /**
     * Install an RPM from an HTTP URL or a location on disk.
     *
     * @stability stable
     */
    static rpm(location, options = {}) {
        return new InitPackage('rpm', [location], options.key, options.serviceRestartHandles);
    }
    /**
     * Install a package using Yum.
     *
     * @stability stable
     */
    static yum(packageName, options = {}) {
        var _l;
        return new InitPackage('yum', (_l = options.version) !== null && _l !== void 0 ? _l : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install a package from RubyGems.
     *
     * @stability stable
     */
    static rubyGem(gemName, options = {}) {
        var _l;
        return new InitPackage('rubygems', (_l = options.version) !== null && _l !== void 0 ? _l : [], gemName, options.serviceRestartHandles);
    }
    /**
     * Install a package from PyPI.
     *
     * @stability stable
     */
    static python(packageName, options = {}) {
        var _l;
        return new InitPackage('python', (_l = options.version) !== null && _l !== void 0 ? _l : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install a package using APT.
     *
     * @stability stable
     */
    static apt(packageName, options = {}) {
        var _l;
        return new InitPackage('apt', (_l = options.version) !== null && _l !== void 0 ? _l : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install an MSI package from an HTTP URL or a location on disk.
     *
     * @stability stable
     */
    static msi(location, options = {}) {
        // The MSI package version must be a string, not an array.
        return new class extends InitPackage {
            renderPackageVersions() { return location; }
        }('msi', [location], options.key, options.serviceRestartHandles);
    }
    /** @internal */
    _bind(options) {
        var _l;
        if ((this.type === 'msi') !== (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS)) {
            if (this.type === 'msi') {
                throw new Error('MSI installers are only supported on Windows systems.');
            }
            else {
                throw new Error('Windows only supports the MSI package type');
            }
        }
        if (!this.packageName && !['rpm', 'msi'].includes(this.type)) {
            throw new Error('Package name must be specified for all package types besides RPM and MSI.');
        }
        const packageName = this.packageName || `${options.index}`.padStart(3, '0');
        for (const handle of (_l = this.serviceHandles) !== null && _l !== void 0 ? _l : []) {
            handle._addPackage(this.type, packageName);
        }
        return {
            config: {
                [this.type]: {
                    [packageName]: this.renderPackageVersions(),
                },
            },
        };
    }
    /**
     * @stability stable
     */
    renderPackageVersions() {
        return this.versions;
    }
}
exports.InitPackage = InitPackage;
_h = JSII_RTTI_SYMBOL_1;
InitPackage[_h] = { fqn: "@aws-cdk/aws-ec2.InitPackage", version: "1.90.1" };
/**
 * A services that be enabled, disabled or restarted when the instance is launched.
 *
 * @stability stable
 */
class InitService extends InitElement {
    constructor(serviceName, serviceOptions) {
        super();
        this.serviceName = serviceName;
        this.serviceOptions = serviceOptions;
        /**
         * Returns the init element type for this element.
         *
         * @stability stable
         */
        this.elementType = cfn_init_internal_1.InitElementType.SERVICE.toString();
    }
    /**
     * Enable and start the given service, optionally restarting it.
     *
     * @stability stable
     */
    static enable(serviceName, options = {}) {
        var _l;
        const { enabled, ensureRunning, ...otherOptions } = options;
        return new InitService(serviceName, {
            enabled: enabled !== null && enabled !== void 0 ? enabled : true,
            ensureRunning: (_l = ensureRunning !== null && ensureRunning !== void 0 ? ensureRunning : enabled) !== null && _l !== void 0 ? _l : true,
            ...otherOptions,
        });
    }
    /**
     * Disable and stop the given service.
     *
     * @stability stable
     */
    static disable(serviceName) {
        return new InitService(serviceName, { enabled: false, ensureRunning: false });
    }
    /** @internal */
    _bind(options) {
        var _l;
        const serviceManager = options.platform === cfn_init_internal_1.InitPlatform.LINUX ? 'sysvinit' : 'windows';
        return {
            config: {
                [serviceManager]: {
                    [this.serviceName]: {
                        enabled: this.serviceOptions.enabled,
                        ensureRunning: this.serviceOptions.ensureRunning,
                        ...(_l = this.serviceOptions.serviceRestartHandle) === null || _l === void 0 ? void 0 : _l._renderRestartHandles(),
                    },
                },
            },
        };
    }
}
exports.InitService = InitService;
_j = JSII_RTTI_SYMBOL_1;
InitService[_j] = { fqn: "@aws-cdk/aws-ec2.InitService", version: "1.90.1" };
/**
 * Extract an archive into a directory.
 *
 * @stability stable
 */
class InitSource extends InitElement {
    /**
     * @stability stable
     */
    constructor(targetDirectory, serviceHandles) {
        super();
        this.targetDirectory = targetDirectory;
        this.serviceHandles = serviceHandles;
        /**
         * Returns the init element type for this element.
         *
         * @stability stable
         */
        this.elementType = cfn_init_internal_1.InitElementType.SOURCE.toString();
    }
    /**
     * Retrieve a URL and extract it into the given directory.
     *
     * @stability stable
     */
    static fromUrl(targetDirectory, url, options = {}) {
        return new class extends InitSource {
            _doBind() {
                return {
                    config: { [this.targetDirectory]: url },
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Extract a GitHub branch into a given directory.
     *
     * @stability stable
     */
    static fromGitHub(targetDirectory, owner, repo, refSpec, options = {}) {
        return InitSource.fromUrl(targetDirectory, `https://github.com/${owner}/${repo}/tarball/${refSpec !== null && refSpec !== void 0 ? refSpec : 'master'}`, options);
    }
    /**
     * Extract an archive stored in an S3 bucket into the given directory.
     *
     * @stability stable
     */
    static fromS3Object(targetDirectory, bucket, key, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                bucket.grantRead(bindOptions.instanceRole, key);
                return {
                    config: { [this.targetDirectory]: bucket.urlForObject(key) },
                    authentication: standardS3Auth(bindOptions.instanceRole, bucket.bucketName),
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Create an InitSource from an asset created from the given path.
     *
     * @stability stable
     */
    static fromAsset(targetDirectory, path, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                const asset = new s3_assets.Asset(bindOptions.scope, `${targetDirectory}Asset`, {
                    path,
                    ...bindOptions,
                });
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: { [this.targetDirectory]: asset.httpUrl },
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Extract a directory from an existing directory asset.
     *
     * @stability stable
     */
    static fromExistingAsset(targetDirectory, asset, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: { [this.targetDirectory]: asset.httpUrl },
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /** @internal */
    _bind(options) {
        var _l;
        for (const handle of (_l = this.serviceHandles) !== null && _l !== void 0 ? _l : []) {
            handle._addSource(this.targetDirectory);
        }
        // Delegate actual bind to subclasses
        return this._doBind(options);
    }
}
exports.InitSource = InitSource;
_k = JSII_RTTI_SYMBOL_1;
InitSource[_k] = { fqn: "@aws-cdk/aws-ec2.InitSource", version: "1.90.1" };
/**
 * Render a standard S3 auth block for use in AWS::CloudFormation::Authentication
 *
 * This block is the same every time (modulo bucket name), so it has the same
 * key every time so the blocks are merged into one in the final render.
 */
function standardS3Auth(role, bucketName) {
    return {
        S3AccessCreds: {
            type: 'S3',
            roleName: role.roleName,
            buckets: [bucketName],
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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