# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fleece',
 'fleece.cli',
 'fleece.cli.build',
 'fleece.cli.config',
 'fleece.cli.run',
 'fleece.handlers']

package_data = \
{'': ['*']}

install_requires = \
['boto3>=1.0.0', 'requests>=2.9.1', 'structlog>=15.3.0', 'wrapt>=1.10.10']

extras_require = \
{'cli': ['docker>=3.5.1', 'PyYAML>=3.12', 'ruamel.yaml>=0.15.34'],
 'connexion': ['connexion>=1.1.9', 'Flask>=1.1.1', 'Werkzeug>=0.15.5'],
 'wsgi': ['Werkzeug>=0.15.5']}

entry_points = \
{'console_scripts': ['fleece = fleece.cli.main:main']}

setup_kwargs = {
    'name': 'fleece',
    'version': '1.1.3',
    'description': 'Wrap the lamb...da',
    'long_description': '# Fleece\n\n## Logging\n\nTo start using fleece with a lambda project you will need to make 2 small updates to your project.\n\n* Where you would normally import `logging.get_logger` or `logging.getLogger` use`fleece.log.get_logger` or `fleece.log.getLogger`\n\n* In the file with your primary lambda handler include `fleece.log.setup_root_logger()`prior to setting up any additional logging.\n\nThis should ensure that all handlers on the root logger are cleaned up and one with appropriate stream handlers is in place.\n\n### Retry logging calls\n\nA retry wrapper for logging handlers that occasionally fail is also provided. This wrapper can be useful in preventing crashes when logging calls to external services such as CloudWatch fail.\n\nFor example, consider the following handler for CloudWatch using watchtower:\n\n```python\nlogger.addHandler(\n    watchtower.CloudWatchLogHandler(log_group=\'WORKER-POLL\',\n                                    stream_name=str(uuid.uuid4()),\n                                    use_queues=False))\n```\n\nIf the CloudWatch service is down, or rate limits the client, that will cause logging calls to raise an exception, which may interrupt the script. To avoid that, the watchtower handler can be wrapped in a `RetryHandler` as follows:\n\n```python\nlogger.addHandler(\n    fleece.log.RetryHandler(\n        watchtower.CloudWatchLogHandler(log_group=\'WORKER-POLL\',\n                                        stream_name=str(uuid.uuid4()),\n                                        use_queues=False)))\n```\n\nIn the above example, logging calls that fail will be retried up to 5 times, using an exponential backoff algorithm to increasingly space out retries. If all retries fail, then the logging call will, by default, give up silently and return, allowing the program to continue. See the documentation for the `RetryHandler` class for information on how to customize the retry strategy.\n\n### Logging request IDs for each log event\n\nIt might be very helpful to have the API Gateway and/or Lambda request IDs present in each log event, so that troubleshooting problematic requests becomes easy. If you apply the `fleece.log.inject_request_ids_into_environment` decorator to the Lambda handler function, an `api_request_id` (only if the event source is API Gateway) and a `lambda_request_id` attribute will be added to the log event dictionary.\n\n## boto3 wrappers\n\nThis project includes `fleece.boto3.client()` and `fleece.boto3.resource()` wrappers that support a friendly format for setting less conservative timeouts than the default 60 seconds used by boto. The following additional arguments are accepted to set these timeouts:\n\n- `connect_timeout`: timeout for socket connections in seconds.\n- `read_timeout`: timeout for socket read operations in seconds.\n- `timeout`: convenience timeout that sets both of the above to the same value.\n\nAlso for convenience, timeouts can be set globally by calling `fleece.boto3.set_default_timeout()` at startup. Globally set timeouts are then applied to all clients, unless explicitly overriden. Default timeouts set via the `set_default_timeout()` function apply to all threads, and for that reason it is a good idea to only call this function during start up, before any additional threads are spawn.\n\nAs an example, the following code written against the original boto3 package uses the default 60 second socket timeouts:\n\n```python\nimport boto3\n# ...\nlambda = boto3.client(\'lambda\')\n```\n\nIf you wanted to use 15 second timeouts instead, you can simply switch to the fleece wrappers as follows:\n\n```python\nfrom fleece import boto3\nboto3.set_default_timeout(15)\n# ...\nlambda = boto3.client(\'lambda\')\n```\n\nYou can import other boto3 attributes, but only `client()` and `resource()` accept the additional arguments documented in this section.\n\n## requests wrappers\n\nThis project also includes a wrapper for the requests package. When using `fleece.requests`, convenient access to set timeouts and retries is provided.\n\nThe high-level request functions such as `requests.get()` and `requests.post()` accept the following arguments:\n\n- `timeout`: a network timeout, or a tuple containing the connection and\n             read timeouts, in seconds. Note that this is functionality that\n             exists in the requests package.\n- `retries`: a retry mechanism to use with this request. This argument can be\n             of several types: if it is `None`, then the default retry\n             mechanism installed by the `set_default_retries` function is used;\n             if it is an integer, it is the number of retries to use; if it is\n             a dictionary, it must have the arguments to a urllib3 `Retry`\n             instance. Alternatively, this argument can be a Retry instance as\n             well.\n\nThe `Session` class is also wrapped. A session instance from this module also accepts the two arguments above, and passes them on to any requests it issues.\n\nFinally, it is also possible to install global timeout and retry defaults that are used for any requests that don\'t specify them explicitly. This enables existing code to take advantage of retries and timeouts after changing the imports to point to this wrapped version of requests. Below is an example that sets global timeouts and retries:\n\n```python\nfrom fleece import requests\n\n# 15 second timeout\nrequests.set_default_timeout(15)\n\n# 5 retries with exponential backoff, also retry 429 and 503 responses\nrequests.set_default_retries(total=5, backoff_factor=1,\n                             status_forcelist=[429, 503])\n\n# the defaults above apply to any regular requests, no need to make\n# changes to existing code.\nr = requests.get(\'https://...\')\n\n# a request can override the defaults if desired\nr = requests.put(\'https://...\', timeout=25, retries=2)\n\n# sessions are also supported\nwith requests.Session() as session:\n    session.get(\'https://...\')\n```\n\n## X-Ray integration\n\nThis project also bridges the gap of missing Python support in the [AWS X-Ray](https://aws.amazon.com/xray/) [Lambda integration](http://docs.aws.amazon.com/xray/latest/devguide/xray-services-lambda.html).\n\n### Prerequisites\n\n 1. Make sure you add the following permissions to the Lambda execution role of your function: `xray:PutTraceSegments` and `xray:PutTelemetryRecords`.\n 2. Enable active tracing under Advanced settings on the Configuration tab of your Lambda function in the AWS Console (or using the [`update_function_configuration` API call](http://boto3.readthedocs.io/en/latest/reference/services/lambda.html#Lambda.Client.update_function_configuration)).\n\n### Features\n\nYou can mark any function or method for tracing by using the `@trace_xray_subsegment` decorator. You can apply the decorator to any number of functions and methods, the resulting trace will be properly nested. You have to decorate all the methods you want traced (e.g. if you decorate your handler function only, no other functions will be traced that it calls).\n\nThis module also provides wrappers for `boto` and `requests` so that any AWS API call, or HTTP request will be automatically traced by X-Ray, but you have to explicitly allow this behavior by calling `monkey_patch_botocore_for_xray` and/or `monkey_patch_requests_for_xray`. The best place to do this would be the main handler module where the Lambda entry point is defined.\n\n### A quick example (`handler.py`)\n\n```python\nfrom fleece import boto3\nfrom fleece.xray import (monkey_patch_botocore_for_xray,\n                         trace_xray_subsegment)\n\nmonkey_patch_botocore_for_xray()\n\n\n@trace_xray_subsegment()\ndef lambda_handler(event, context):\n    return get_user()\n\n\ndef get_user():\n    # This function doesn\'t have to be decorated, because the API call to IAM\n    # will be traced thanks to the monkey-patching.\n    iam = boto3.client(\'iam\')\n    return iam.get_user()\n```\n\n**Note:** the monkey-patched tracing will also work with the wrappers described above.\n\n## Connexion integration\n\nSummary about what [Connexion](https://github.com/zalando/connexion) exactly is (from their project page):\n\n > Connexion is a framework on top of [Flask](http://flask.pocoo.org/) that automagically handles HTTP requests based on [OpenAPI 2.0 Specification](https://github.com/OAI/OpenAPI-Specification/blob/master/versions/2.0.md) (formerly known as Swagger Spec) of your API described in [YAML format](https://github.com/OAI/OpenAPI-Specification/blob/master/versions/2.0.md#format). Connexion allows you to write a Swagger specification, then maps the endpoints to your Python functions; this makes it unique, as many tools generate the specification based on your Python code. You can describe your REST API in as much detail as you want; then Connexion guarantees that it will work as you specified.\n\nIt\'s the perfect glue between your API Gateway API specification and your Lambda function. Fleece makes it very easy to use Connexion:\n\n```python\nfrom fleece.connexion import call_api\nfrom fleece.log import get_logger\n\nlogger = get_logger(__name__)\n\n\ndef lambda_handler(event, context):\n    return call_api(event, \'myapi\', \'swagger.yml\', logger)\n```\n\nYou just have to make sure that the `swagger.yml` file is included in the Lambda bundle. For the API Gateway integration, we assume the [request template defined by yoke](https://github.com/rackerlabs/yoke/blob/master/yoke/templates.py#L60-L132) for now.\n\nUsing this integration has the added benefit of being able to run your API locally, by adding something like this to your Lambda handler:\n\n```python\nfrom fleece.connexion import get_connexion_app\n\n[...]\n\nif __name__ == \'__main__\':\n    app = get_connexion_app(\'myapi\', \'swagger.yml\')\n    app.run(8080)\n```\n\n## Fleece CLI\n\nFleece offers a limited functionality CLI to help build Lambda packages and run commands in a shell environment with AWS credentials from a Rackspace Fanatical AWS Account. The CLI functionality is not installed by default but can be installed as an extras package. NOTE: Package building with Fleece requires Docker.\n\n### Installation\n\n```\npip install fleece[cli]\n```\n\n### `fleece build`\n\n```\nusage: fleece build [-h] [--python36] [--rebuild]\n                    [--requirements REQUIREMENTS]\n                    [--dependencies DEPENDENCIES] [--target TARGET]\n                    [--source SOURCE]\n                    [--exclude [EXCLUDE [EXCLUDE ...]]]\n                    service_dir\n\nSimple Lambda builder.\n\npositional arguments:\n  service_dir           directory where the service is located (default: $pwd)\n\noptional arguments:\n  -h, --help            show this help message and exit\n  --python36, -3        use Python 3.6 (default: Python 2.7)\n  --rebuild             rebuild Python dependencies\n  --requirements REQUIREMENTS, -r REQUIREMENTS\n                        requirements.txt file with dependencies (default:\n                        $service_dir/src/requirements.txt)\n  --dependencies DEPENDENCIES, -d DEPENDENCIES\n                        comma separated list of system dependencies\n  --target TARGET, -t TARGET\n                        target directory for lambda_function.zip (default\n                        $service_dir/dist)\n  --source SOURCE, -s SOURCE\n                        source directory to include in lambda_function.zip\n                        (default: $service_dir/src)\n  --exclude [EXCLUDE [EXCLUDE ...]], -e [EXCLUDE [EXCLUDE ...]]\n                        glob pattern to exclude\n```\n\nTo build a lambda package from the service\'s top-level directory:\n\n```\n$ fleece build .\n```\n\nThe assumptions made with the above command are that the source code of the service is in `./src`, the requirements file is in `./src/requirements.txt` and the output zip file will be written to `./dist`. These defaults can be changed with the `--source`, `--requirements` and `--target` options respectively.\n\nThe build process will run in a Docker container based on the Amazon Linux image. If there are any additional dependencies that need to be installed on the container prior to installing the Python requirements, those can be given with the `--dependencies` option. Any environment variables recognized by `pip`, such as `PIP_INDEX_URL`, are passed on to the container.\n\n### `fleece run`\n\n```\nusage: fleece run [-h] [--username USERNAME] [--apikey APIKEY]\n                  [--config CONFIG] [--account ACCOUNT]\n                  [--environment ENVIRONMENT] [--role ROLE]\n                  command\n\nRun command in environment with AWS credentials from Rackspace FAWS API\n\npositional arguments:\n  command               Command to execute\n\noptional arguments:\n  -h, --help            show this help message and exit\n  --username USERNAME, -u USERNAME\n                        Rackspace username. Can also be set via RS_USERNAME\n                        environment variable\n  --apikey APIKEY, -k APIKEY\n                        Rackspace API key. Can also be set via RS_API_KEY\n                        environment variable\n  --config CONFIG, -c CONFIG\n                        Path to YAML config file with defined accounts and\n                        aliases. Default is ./environments.yml\n  --account ACCOUNT, -a ACCOUNT\n                        AWS account number. Cannot be used with\n                        `--environment`\n  --environment ENVIRONMENT, -e ENVIRONMENT\n                        Environment alias to AWS account defined in config\n                        file. Cannot be used with `--account`\n  --role ROLE, -r ROLE  Role name to assume after obtaining credentials from\n                        FAWS API\n```\n\n```\n# fleece run --username $username --apikey $apikey --account $account \'aws s3 ls\'\n2017-10-02 12:03:18 bucket1\n2017-06-08 14:31:07 bucket2\n2017-08-10 17:28:47 bucket3\n2017-08-10 17:21:58 bucket4\n2017-08-15 20:33:02 bucket5\n```\n\nYou can also setup an environments file to reduce command-line flags. Ensure accounts are quoted to ensure they are not interperted incorrectly as ints or octals:\n\n```\n# cat environments.yml\nenvironments:\n  - name: development\n    account: \'123456789012\'\n  - name: staging\n    account: \'123456789012\'\n    rs_username_var: MY_RS_USERNAME\n    rs_apikey_var: MY_RS_APIKEY\n  - name: testing\n    account: \'123456789012\'\n  - name: production\n    account: \'123456789012\'\n    role: LambdaDeployRole\n\n# fleece run --username $username --apikey $apikey --environment testing \'aws s3 ls\'\n2017-10-02 12:03:18 bucket1\n2017-06-08 14:31:07 bucket2\n2017-08-10 17:28:47 bucket3\n2017-08-10 17:21:58 bucket4\n2017-08-15 20:33:02 bucket5\n```\n\nNote the `staging` environment example above, which provides a custom pair of\nenvironment variables from where the Rackspace username and API key are sourced.\nThese would be used only if credentials are not explicitly given as part of\nthe command.\n\n### `fleece config`\n\n```\nusage: fleece config [-h] [--config CONFIG] [--username USERNAME]\n                     [--apikey APIKEY] [--environments ENVIRONMENTS]\n                     {import,export,edit,render} ...\n\nConfiguration management\n\npositional arguments:\n  {import,export,edit,render}\n                        Sub-command help\n    import              Import configuration from stdin\n    export              Export configuration to stdout\n    edit                Edit configuration\n    render              Render configuration for an environment\n\noptional arguments:\n  -h, --help            show this help message and exit\n  --config CONFIG, -c CONFIG\n                        Config file (default is config.yml)\n  --username USERNAME, -u USERNAME\n                        Rackspace username. Can also be set via RS_USERNAME\n                        environment variable\n  --apikey APIKEY, -k APIKEY\n                        Rackspace API key. Can also be set via RS_API_KEY\n                        environment variable\n  --environments ENVIRONMENTS, -e ENVIRONMENTS\n                        Path to YAML config file with defined accounts and\n                        environment names. Defaults to ./environments.yml\n```\n\nThe `fleece config` command has a few sub-commands that work with configuration files. There are a number of arguments that apply to all commands:\n\n- `--config` sets the configuration file. This is the file that holds the configuration, in a format that is appropriate to commit to source control (i.e. sensitive variables are encrypted).\n- `--username` and `--apikey` are the Rackspace credentials, used to obtain temporary AWS access credentials from FAWS. For convenience, these can be set in environment variables.\n- `--environments` is an environments.yml file that defines the different environments and the associated AWS accounts for each. The format is as described in the `fleece run` command.\n\nThe config commands work with two types of config files. The `config.yml` file is a "closed" config file, where all sensitive values are encrypted. Developers typically do not edit this file but instead export it to a temporary "open" configuration file where sensitive variables appear in plain text for editing. As soon as changes are made, the open config file is imported back into the closed `config.yml`.\n\nThe open configuration format is as follows:\n\n```\nstages:                                 # stage definitions\n  prod:                                 # stage name\n    environment: prod                   # environment associated with this stage\n    key: prod-key-here                  # KMS key, ARN or name with or without the "alias/" prefix are all valid\n  /.*/:                                 # regular expressions for custom stage names\n    environment: dev\n    key: dev-key-here\nconfig:\n  foo: bar                              # plain text variable\n  password:                             # per-stage values, encrypted\n    +dev: :encrypt:my-dev-password      # per-stage keys must have a "+" prefix so they are\n    +prod: :encrypt:my-prod-password    # not taken as a nested dict\n    +/.*/: :encrypt:my-custom-password\n  nested:                               # nested dictionaries\n    inner_var: value\n    a_list:                             # list of dictionaries\n      - username1:                      # per-stage values, without encryption\n          +prod: bob-prod\n          +/.*/: bob-dev\n        password1:                      # per-stage values, encrypted\n          +prod: :encrypt:bob-prod-pw\n          +/.*/: :encrypt:bob-dev-pw\n      - username2: user2\n        password2:\n          +prod: :encrypt:prod-pw2\n          +/.*/: :encrypt:dev-pw2\n```\n\nThe `stages` section defines the available stages, along with their association to an environment and a KMS key. The environment, which must be defined in the `environments.yml`, links the stage to a AWS account. The KMS key can be given as an ARN or as an alias. The alias can be given with or without the `alias/` prefix. Stage names can be given explicitly or as a regular expression (surrounded by `/`s). When fleece needs to match a stage name given in one of its commands, it will first attempt to do an equality match, and only when that fails it will try the regular expression based stage names. The regular expression stage names are evaluated in random order until one succeeds, so it is important to avoid ambiguities in the regex patterns.\n\nThe `config` section is where configuration variables are defined. A standard key/value pair in this section represents a plaintext variable that will be made available for all stages. A variable can be given per-stage values by making its value a sub-dictionary where the keys are the stage names prefixed by `+`. Regex patterns for stage names are supported here as well.\n\nAny variables that are sensitive and need to be encrypted must have per-stage values, and these values must have the `:encrypt:` prefix so that fleece knows to encrypt them when the configuration is imported and stored in `config.yml`.\n\nThe available sub-commands are:\n\n#### `fleece config import`\n\nReads a source configuration file from `stdin` and writes a `config.yml` file. The input data can be in YAML or JSON format.\n\n#### `fleece config export [--json]`\n\nWrites the contents of `config.yml` to `stdout` in the open format for editing. By default this command outputs a YAML file. Use `--json` to output in JSON format.\n\n#### `fleece config edit [--json] [--editor EDITOR]`\n\nThis command exports the configuration to a temp file, then starts a text editor (`vi` by default) on this file. After the editor is closed, the modified file is re-imported. This is the most convenient workflow to edit the configuration.\n\n#### `fleece config render [--environment] [--json] [--encrypt] [--python] [--parameter-store PARAMETER_STORE_PREFIX] [--ssm-kms-key SSM_KMS_KEY] <stage>`\n\nWrites the configuration variables for the given environment to stdout or uploads them to parameter store. There are four output options: YAML plaintext (the default), JSON plaintext (with `--json`), JSON encrypted (with `--encrypt`) and an encrypted Python module (with `--python`).\n\nParameters uploaded into SSM are encrypted by default, using the default SSM encryption key. If you want to use a custom KMS key to encrypt parameters, use the `--ssm-kms-key` option.\nFor this value, you can pass in a KMS key ID, ARN, alias name, or alias ARN. This feature enables a use case where parameters are copied from SSM into a Lambda Function environment configuration at deploy time, and a custom KMS key is configured for that function to decrypt config at runtime.\n\nThe encrypted configuration consists on a list of encrypted buffers that need to be decrypted and appended. The result of this operation is the JSON plaintext configuration. The following output is the output of `--python`, which includes the decrypt and decode logic:\n\n```python\nENCRYPTED_CONFIG = [\'... encrypted blob here ...\']\nimport base64\nimport boto3\nimport json\n\ndef load_config():\n    config_json = \'\'\n    kms = boto3.client(\'kms\')\n    for buffer in ENCRYPTED_CONFIG:\n        r = kms.decrypt(CiphertextBlob=base64.b64decode(buffer.encode(\n            \'utf-8\')))\n        config_json += r[\'Plaintext\'].decode(\'utf-8\')\n    return json.loads(config_json)\n\nCONFIG = load_config()\n```\n\nIf this is saved as `fleece_config.py` in the source directory, the configuration can be imported with:\n\n```python\nfrom fleece_config import CONFIG\n```\n\nIf `--parameter-store` is specified, the next argument needs to be a prefix used for all variables that will be uploaded to parameter store. This should start with a slash.\n\nFor example, if the arguments are `--parameter-store /super-service/some-id` and the config has a value called `foo`, then fleece will create or overwrite a secure string parameter store value named `/super-service/some-id/foo` with the value being the decrypted config value of `foo`.\n\nAll values are converted to strings before being saved to parameter store. If the config has a nested dictionary, then multiple parameter store values will be saved (so inthe example above, the field `nested` with a value of `inner` would be saved as `/super-service/some-id/nested/inner`).\n',
    'author': 'Bruce Stringer',
    'author_email': 'bruce.stringer@rackspace.com',
    'maintainer': 'Bruce Stringer',
    'maintainer_email': 'bruce.stringer@rackspace.com',
    'url': 'https://github.com/rackerlabs/fleece',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
