# Extend the builtin 'threading.Thread' class to add cooperative pause, unpause, and kill
# capability to python threads.
#
# By: David Smith
# License: MIT
# 3/10/2021

import copy
import threading
import time
from typing import Callable, Optional



class Threadify(threading.Thread):
    """
    Extend the builtin python 'threading.Thread' class to add cooperative pause, unpause, and kill
    capability to python threads.
    """

    VERSION = "1.0.0"

    # Enable printing debug info
    ENABLE_DEBUG_OUTPUT = False


    def __init__(self, task: Optional[Callable] = None, storage: Optional[dict] = None, *, name: str = None,
                 daemon: bool = True, deep_copy: bool = True, ignore_task_exceptions: bool = False,
                 start: bool = False):
        """
        :param task: The callable to be repeatedly executed by the thread in the thread's context.
        :param storage: Dictionary containing data for 'task'. It is persistent and mutable across invocations of 'task'
                        by the thread for the life of the thread. The task can access, modify, and add variables to
                        the dictionary and have them persist across each task invocation (which happens repeatedly).
        :param name:    Name for the thread or None for an autogenerated default name.
        :param daemon:  True - Run as a daemon thread (ie: if main program exits, thread exits);
                        False - Thread continues to run even if the main program that created it exits.
        :param deep_copy: True - Make independent, deep copy of storage for use by thread; False - Shallow copy. A
                        deep copy may require less programmer care since independent copies are made. A shallow
                        copy is potentially faster, but requires the programmer to be careful not to create
                        data contention between various threads of execution. Items that can't be pickled
                        can't be deep-copied.
        :param ignore_task_exceptions: True - Ignore unhandled exceptions raised in task and continue;
                        False - Re-raise task exception thereby terminating the thread.
        :param start:   True - Automatically start thread after construction; False - Thread must be manually started
                        by calling its builtin 'start' method.
        """
        super().__init__(name=name, daemon=daemon)

        # If no storage passed, create default empty dict
        if not storage:
            storage = {}
            deep_copy = False

        # Deep copy requires less programmer care since truly independent copies are made; shallow copy is potentially
        # faster, but requires application programmer not to create data contention.
        # Note: Items that can't be pickled can't be deep-copied.
        if deep_copy:
            try:
                self.task_storage = copy.deepcopy(storage)
            except TypeError as ex:
                raise TypeError("Storage contains an item that cannot be deep-copied.") from ex

        else:
            self.task_storage = copy.copy(storage)

        # Select task that the thread continuously executes
        if task:
            if callable(task):
                # User-supplied task
                self.task = task
            else:
                raise TypeError("The 'task' parameter must be a callable or None.")

        else:
            # Default do-nothing-but-sleep task
            self.task = Threadify.task

        # Specify how to handle task exceptions
        self.ignore_task_exceptions = ignore_task_exceptions


        # ## Create and configure thread controls ##

        # Control for terminating the thread
        self._killthread_event = threading.Event()
        self._killthread_event.clear()

        # Control for cooperative pausing/running of thread: start with set so thread runs as soon as start is called
        self._do_run_event = threading.Event()
        self._do_run_event.set()

        # Flag indicates when pause has taken effect (thread is actually paused instead of just been told to pause)
        self._is_paused_event = threading.Event()
        self._is_paused_event.clear()

        # Automatically start the thread?
        if start:
            self.start()


    def pause(self, wait_until_paused: bool = False, timeout_secs: Optional[int] = None):
        """
        Use to cooperatively pause the thread. Note that unless 'wait_until_paused' is True, this
        method can return before the pause has taken effect since thread pausing is affected by the responsiveness
        of and the blocking in the user task.
        :param wait_until_paused: True - wait until thread has paused before returning; False - return immediately.
        :param timeout_secs: 'None' or maximum number of seconds to wait for thread to pause when
                            'wait_until_paused' is True; None means ignore timeout and wait as long as required
                            for thread to pause.
        :return: True - Thread paused before return; False - Thread not yet paused before return
        """
        # Clear run event to signal a pause request to the run function
        self._do_run_event.clear()

        # If wait desired, wait until pause has occurred (or timed out if timeout_secs is not None)
        timeout = (time.time() + timeout_secs) if timeout_secs else time.time()
        while all([wait_until_paused,
                   not self._is_paused_event.is_set(),
                   ((timeout_secs is None) or (time.time() < timeout))]):
            time.sleep(0.010)

        return self._is_paused_event.is_set()


    def unpause(self):
        """
        Unpause a paused thread.
        :return: None
        """
        # Set event to wake up the paused thread
        self._do_run_event.set()


    def is_paused(self) -> bool:
        """
        Indicate if the thread is currently paused. This represents the current actual state of the thread - not
        whether or not a pause was requested.
        :return: True - Thread is currently paused; False - Thread is not paused
        """
        return self._is_paused_event.is_set()


    def kill(self, wait_until_dead: bool = False, timeout_secs: Optional[int] = None):
        """
        Cooperatively end execution of the thread.
        :param wait_until_dead: True - Wait with timeout for thread to terminate; False - Return immediately
        :param timeout_secs: 'None' or maximum number of seconds to wait for termination when
                            'wait_until_dead' is True; None means ignore timeout and wait as long as required
                            for thread to terminate.
        :return: True - Thread terminated before return; False - Thread not yet terminated before returning
        """
        # Signal thread to terminate
        self._killthread_event.set()

        # If thread was paused, wake it so it can proceed with termination
        if not self._do_run_event.is_set():
            self._do_run_event.set()

        # If waiting for termination, sleep until killed (or timed out if timeout_secs is not None)
        timeout = (time.time() + timeout_secs) if timeout_secs else time.time()
        while all([wait_until_dead, self.is_alive(), ((timeout_secs is None) or (time.time() < timeout))]):
            time.sleep(0.010)

        return not self.is_alive()


    def run(self):
        """
        The template callable executed by the thread. It implements cooperative pause/restart and kill features.
        It acts as the superloop that repeatedly calls the user task.
        :return: None
        """
        if self.ENABLE_DEBUG_OUTPUT:
            print("<{:s} Setup>".format(self.name), flush=True)

        try:
            # Loop until killed
            while not self._killthread_event.is_set():
                try:
                    # If cooperative pause is signaled, handle it here
                    was_paused = self._handle_pause_request()

                    # If resuming from a pause, proceed back to top of loop to test the kill condition to
                    # allow terminating a paused thread.
                    if was_paused:
                        continue

                    # ** Execute Task **
                    task_running = True
                    try:
                        task_running = self.task(self.task_storage)
                    except Exception as ex:
                        if self.ENABLE_DEBUG_OUTPUT:
                            print("Exception in task: "+str(ex), flush=True)

                        # If not ignoring task exceptions, raise and kill thread
                        if not self.ignore_task_exceptions:
                            raise Exception("Exception raised from task with 'ignore_task_exceptions' False.") from ex

                    # Check if task requests thread termination
                    if not task_running:
                        # Directly set kill event
                        self._killthread_event.set()

                except Exception as ex:
                    raise

        finally:
            if self.ENABLE_DEBUG_OUTPUT:
                print("\n<{:s} Cleanup>".format(self.name), flush=True)


    @staticmethod
    def task(storage: dict) -> bool:
        """
        The periodic work to be done by the thread. This stub routine is replaced by the task callable passed
        by the user when the initial object is created. Blocking affects the responsiveness to cooperative
        pause and kill signals; however, at least some small sleep delay (ex: time.sleep(0.010) ) or IO blocking
        should be included to allow opportunities for context-switches for other threads. Note that changes made to
        'storage' persist across each invocation of task for the life of the thread.
        :param storage: Dict to provide persistent, mutable task variable storage.
        :returns: True - continue to run; False - kill thread
        """
        # **********************************
        # ** This is an example task body **
        # **********************************

        # Demonstrate the use of the 'storage' parameter
        symbol = storage.get("symbol", ".")[0]      # Get symbol if passed during thread creation, otherwise use '.'
        count = storage.get("count", 0)             # Demonstrate persistent variable storage across calls to task

        # Do something observable
        print(symbol, sep=" ", end="", flush=True)

        # Line wrap after 25 symbols and update 'count' value in storage
        if count > 24:
            storage["count"] = 0
            print()
        else:
            storage["count"] = count + 1

        # Sleep allows other threads to execute so as not to hog the processor, but it also controls the responsiveness
        # of this thread to commands like pause and kill.
        time.sleep(.25)

        # True signals thread to continue running
        return True


    def _handle_pause_request(self) -> bool:
        """
        Called from run function to include/implement cooperative pause functionality.
        :return: True - a pause occurred; False - no pause occurred
        """
        # Test for cooperative pause and sleep here if pause indicated
        pause_occurred = False
        if not self._do_run_event.is_set():
            pause_occurred = True
            self._is_paused_event.set()          # Set flag indicating that cooperative pause is taking effect now
            self._do_run_event.wait()           # Sleeps here until '_do_run_event' is set
            self._is_paused_event.clear()        # After thread is re-awakened, indicate no longer paused

        return pause_occurred
