"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateGrpcGatewayRouteMatch = exports.validateGrpcRouteMatch = exports.validateGrpcMatchArrayLength = exports.validateHttpMatchArrayLength = exports.renderMeshOwner = exports.renderListenerTlsOptions = exports.renderTlsClientPolicy = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * This is the helper method to render TLS property of client policy.
 */
function renderTlsClientPolicy(scope, tlsClientPolicy) {
    const certificate = tlsClientPolicy?.mutualTlsCertificate?.bind(scope).tlsCertificate;
    const sans = tlsClientPolicy?.validation.subjectAlternativeNames;
    return tlsClientPolicy
        ? {
            certificate: certificate,
            ports: tlsClientPolicy.ports,
            enforce: tlsClientPolicy.enforce,
            validation: {
                subjectAlternativeNames: sans
                    ? {
                        match: sans.bind(scope).subjectAlternativeNamesMatch,
                    }
                    : undefined,
                trust: tlsClientPolicy.validation.trust.bind(scope).tlsValidationTrust,
            },
        }
        : undefined;
}
exports.renderTlsClientPolicy = renderTlsClientPolicy;
/**
 * This is the helper method to render the TLS config for a listener.
 */
function renderListenerTlsOptions(scope, listenerTls) {
    const tlsValidation = listenerTls?.mutualTlsValidation;
    return listenerTls
        ? {
            certificate: listenerTls.certificate.bind(scope).tlsCertificate,
            mode: listenerTls.mode,
            validation: tlsValidation
                ? {
                    subjectAlternativeNames: tlsValidation.subjectAlternativeNames
                        ? {
                            match: tlsValidation.subjectAlternativeNames.bind(scope).subjectAlternativeNamesMatch,
                        }
                        : undefined,
                    trust: tlsValidation.trust.bind(scope).tlsValidationTrust,
                }
                : undefined,
        }
        : undefined;
}
exports.renderListenerTlsOptions = renderListenerTlsOptions;
/**
 * This is the helper method to populate mesh owner when it is a shared mesh scenario
 */
function renderMeshOwner(resourceAccount, meshAccount) {
    const comparison = core_1.Token.compareStrings(resourceAccount, meshAccount);
    return comparison === core_1.TokenComparison.DIFFERENT || comparison === core_1.TokenComparison.ONE_UNRESOLVED
        ? meshAccount
        : undefined;
}
exports.renderMeshOwner = renderMeshOwner;
/**
 * This is the helper method to validate the length of HTTP match array when it is specified.
 */
function validateHttpMatchArrayLength(headers, queryParameters) {
    const MIN_LENGTH = 1;
    const MAX_LENGTH = 10;
    if (headers && (headers.length < MIN_LENGTH || headers.length > MAX_LENGTH)) {
        throw new Error(`Number of headers provided for matching must be between ${MIN_LENGTH} and ${MAX_LENGTH}, got: ${headers.length}`);
    }
    if (queryParameters && (queryParameters.length < MIN_LENGTH || queryParameters.length > MAX_LENGTH)) {
        throw new Error(`Number of query parameters provided for matching must be between ${MIN_LENGTH} and ${MAX_LENGTH}, got: ${queryParameters.length}`);
    }
}
exports.validateHttpMatchArrayLength = validateHttpMatchArrayLength;
/**
 * This is the helper method to validate the length of gRPC match array when it is specified.
 */
function validateGrpcMatchArrayLength(metadata) {
    const MIN_LENGTH = 1;
    const MAX_LENGTH = 10;
    if (metadata && (metadata.length < MIN_LENGTH || metadata.length > MAX_LENGTH)) {
        throw new Error(`Number of metadata provided for matching must be between ${MIN_LENGTH} and ${MAX_LENGTH}, got: ${metadata.length}`);
    }
}
exports.validateGrpcMatchArrayLength = validateGrpcMatchArrayLength;
/**
 * This is the helper method to validate at least one of gRPC route match type is defined.
 */
function validateGrpcRouteMatch(match) {
    if (match.serviceName === undefined && match.metadata === undefined && match.methodName === undefined) {
        throw new Error('At least one gRPC route match property must be provided');
    }
}
exports.validateGrpcRouteMatch = validateGrpcRouteMatch;
/**
 * This is the helper method to validate at least one of gRPC gateway route match type is defined.
 */
function validateGrpcGatewayRouteMatch(match) {
    if (match.serviceName === undefined && match.metadata === undefined && match.hostname === undefined) {
        throw new Error('At least one gRPC gateway route match property beside rewriteRequestHostname must be provided');
    }
}
exports.validateGrpcGatewayRouteMatch = validateGrpcGatewayRouteMatch;
//# sourceMappingURL=data:application/json;base64,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