"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3DeployAction = exports.CacheControl = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const case_1 = require("case");
const action_1 = require("../action");
const common_1 = require("../common");
// Class copied verbatim from the aws-s3-deployment module.
// Yes, it sucks that we didn't abstract this properly in a common class,
// but having 2 different CacheControl classes that behave differently would be worse I think.
// Something to do when CDK 2.0.0 comes out.
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 *
 * Use the provided static factory methods to construct instances of this class.
 * Used in the {@link S3DeployActionProps.cacheControl} property.
 *
 * @see https://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.9
 * @stability stable
 */
class CacheControl {
    /** @param value the actual text value of the created directive */
    constructor(value) {
        this.value = value;
    }
    /**
     * The 'must-revalidate' cache control directive.
     *
     * @stability stable
     */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /**
     * The 'no-cache' cache control directive.
     *
     * @stability stable
     */
    static noCache() { return new CacheControl('no-cache'); }
    /**
     * The 'no-transform' cache control directive.
     *
     * @stability stable
     */
    static noTransform() { return new CacheControl('no-transform'); }
    /**
     * The 'public' cache control directive.
     *
     * @stability stable
     */
    static setPublic() { return new CacheControl('public'); }
    /**
     * The 'private' cache control directive.
     *
     * @stability stable
     */
    static setPrivate() { return new CacheControl('private'); }
    /**
     * The 'proxy-revalidate' cache control directive.
     *
     * @stability stable
     */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /**
     * The 'max-age' cache control directive.
     *
     * @stability stable
     */
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    /**
     * The 's-max-age' cache control directive.
     *
     * @stability stable
     */
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    /**
     * Allows you to create an arbitrary cache control directive, in case our support is missing a method for a particular directive.
     *
     * @stability stable
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
/**
 * Deploys the sourceArtifact to Amazon S3.
 *
 * @stability stable
 */
class S3DeployAction extends action_1.Action {
    /**
     * @stability stable
     */
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'S3',
            artifactBounds: common_1.deployArtifactBounds(),
            inputs: [props.input],
        });
        this.props = props;
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @stability stable
     */
    bound(_scope, _stage, options) {
        // pipeline needs permissions to write to the S3 bucket
        this.props.bucket.grantWrite(options.role);
        if (this.props.accessControl !== undefined) {
            // we need to modify the ACL settings of objects within the Bucket,
            // so grant the Action's Role permissions to do that
            this.props.bucket.grantPutAcl(options.role);
        }
        // the Action Role also needs to read from the Pipeline's bucket
        options.bucket.grantRead(options.role);
        const acl = this.props.accessControl;
        return {
            configuration: {
                BucketName: this.props.bucket.bucketName,
                Extract: this.props.extract === false ? 'false' : 'true',
                ObjectKey: this.props.objectKey,
                CannedACL: acl ? case_1.kebab(acl.toString()) : undefined,
                CacheControl: this.props.cacheControl && this.props.cacheControl.map(ac => ac.value).join(', '),
            },
        };
    }
}
exports.S3DeployAction = S3DeployAction;
//# sourceMappingURL=data:application/json;base64,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