"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.jsonPathString = exports.recurseObject = exports.findReferencedPaths = exports.renderObject = exports.JsonPathToken = void 0;
const core_1 = require("@aws-cdk/core");
const JSON_PATH_TOKEN_SYMBOL = Symbol.for('@aws-cdk/aws-stepfunctions.JsonPathToken');
class JsonPathToken {
    constructor(path) {
        this.path = path;
        this.creationStack = core_1.captureStackTrace();
        this.displayHint = path.replace(/^[^a-zA-Z]+/, '');
        Object.defineProperty(this, JSON_PATH_TOKEN_SYMBOL, { value: true });
    }
    static isJsonPathToken(x) {
        return x[JSON_PATH_TOKEN_SYMBOL] === true;
    }
    resolve(_ctx) {
        return this.path;
    }
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.JsonPathToken = JsonPathToken;
/**
 * Deep render a JSON object to expand JSON path fields, updating the key to end in '.$'
 */
function renderObject(obj) {
    return recurseObject(obj, {
        handleString: renderString,
        handleList: renderStringList,
        handleNumber: renderNumber,
        handleBoolean: renderBoolean,
    });
}
exports.renderObject = renderObject;
/**
 * Return all JSON paths that are used in the given structure
 */
function findReferencedPaths(obj) {
    const found = new Set();
    recurseObject(obj, {
        handleString(_key, x) {
            const path = jsonPathString(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleList(_key, x) {
            const path = jsonPathStringList(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleNumber(_key, x) {
            const path = jsonPathNumber(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleBoolean(_key, _x) {
            return {};
        },
    });
    return found;
}
exports.findReferencedPaths = findReferencedPaths;
function recurseObject(obj, handlers, visited = []) {
    if (obj === undefined) {
        return undefined;
    }
    if (visited.includes(obj)) {
        return {};
    }
    else {
        visited.push(obj);
    }
    const ret = {};
    for (const [key, value] of Object.entries(obj)) {
        if (typeof value === 'string') {
            Object.assign(ret, handlers.handleString(key, value));
        }
        else if (typeof value === 'number') {
            Object.assign(ret, handlers.handleNumber(key, value));
        }
        else if (Array.isArray(value)) {
            Object.assign(ret, recurseArray(key, value, handlers, visited));
        }
        else if (typeof value === 'boolean') {
            Object.assign(ret, handlers.handleBoolean(key, value));
        }
        else if (value === null || value === undefined) {
            // Nothing
        }
        else if (typeof value === 'object') {
            ret[key] = recurseObject(value, handlers, visited);
        }
    }
    return ret;
}
exports.recurseObject = recurseObject;
/**
 * Render an array that may or may not contain a string list token
 */
function recurseArray(key, arr, handlers, visited = []) {
    if (isStringArray(arr)) {
        const path = jsonPathStringList(arr);
        if (path !== undefined) {
            return handlers.handleList(key, arr);
        }
        // Fall through to correctly reject encoded strings inside an array.
        // They cannot be represented because there is no key to append a '.$' to.
    }
    return {
        [key]: arr.map(value => {
            if ((typeof value === 'string' && jsonPathString(value) !== undefined)
                || (typeof value === 'number' && jsonPathNumber(value) !== undefined)
                || (isStringArray(value) && jsonPathStringList(value) !== undefined)) {
                throw new Error('Cannot use JsonPath fields in an array, they must be used in objects');
            }
            if (typeof value === 'object' && value !== null) {
                return recurseObject(value, handlers, visited);
            }
            return value;
        }),
    };
}
function isStringArray(x) {
    return Array.isArray(x) && x.every(el => typeof el === 'string');
}
/**
 * Render a parameter string
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderString(key, value) {
    const path = jsonPathString(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter string list
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderStringList(key, value) {
    const path = jsonPathStringList(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter number
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderNumber(key, value) {
    const path = jsonPathNumber(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter boolean
 */
function renderBoolean(key, value) {
    return { [key]: value };
}
/**
 * If the indicated string is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathString(x) {
    const fragments = core_1.Tokenization.reverseString(x);
    const jsonPathTokens = fragments.tokens.filter(JsonPathToken.isJsonPathToken);
    if (jsonPathTokens.length > 0 && fragments.length > 1) {
        throw new Error(`Field references must be the entire string, cannot concatenate them (found '${x}')`);
    }
    if (jsonPathTokens.length > 0) {
        return jsonPathTokens[0].path;
    }
    return undefined;
}
exports.jsonPathString = jsonPathString;
/**
 * If the indicated string list is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathStringList(x) {
    return pathFromToken(core_1.Tokenization.reverseList(x));
}
/**
 * If the indicated number is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathNumber(x) {
    return pathFromToken(core_1.Tokenization.reverseNumber(x));
}
function pathFromToken(token) {
    return token && (JsonPathToken.isJsonPathToken(token) ? token.path : undefined);
}
//# sourceMappingURL=data:application/json;base64,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