"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseQuery = void 0;
const path = require("path");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib");
const customresources = require("aws-cdk-lib/custom-resources");
const cluster_1 = require("../cluster");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const constructs_1 = require("constructs");
class DatabaseQuery extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const adminUser = this.getAdminUser(props);
        const handler = new lambda.SingletonFunction(this, 'Handler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'database-query-provider'), {
                exclude: ['*.ts'],
            }),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            timeout: cdk.Duration.minutes(1),
            uuid: '3de5bea7-27da-4796-8662-5efb56431b5f',
            lambdaPurpose: 'Query Redshift Database',
        });
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['redshift-data:DescribeStatement', 'redshift-data:ExecuteStatement'],
            resources: ['*'],
        }));
        adminUser.grantRead(handler);
        const provider = new customresources.Provider(this, 'Provider', {
            onEventHandler: handler,
        });
        const queryHandlerProps = {
            handler: props.handler,
            clusterName: props.cluster.clusterName,
            adminUserArn: adminUser.secretArn,
            databaseName: props.databaseName,
            ...props.properties,
        };
        this.resource = new cdk.CustomResource(this, 'Resource', {
            resourceType: 'Custom::RedshiftDatabaseQuery',
            serviceToken: provider.serviceToken,
            removalPolicy: props.removalPolicy,
            properties: queryHandlerProps,
        });
        this.grantPrincipal = handler.grantPrincipal;
        this.ref = this.resource.ref;
    }
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    getAtt(attributeName) {
        return this.resource.getAtt(attributeName);
    }
    getAttString(attributeName) {
        return this.resource.getAttString(attributeName);
    }
    getAdminUser(props) {
        const cluster = props.cluster;
        let adminUser = props.adminUser;
        if (!adminUser) {
            if (cluster instanceof cluster_1.Cluster) {
                if (cluster.secret) {
                    adminUser = cluster.secret;
                }
                else {
                    throw new Error('Administrative access to the Redshift cluster is required but an admin user secret was not provided and the cluster did not generate admin user credentials (they were provided explicitly)');
                }
            }
            else {
                throw new Error('Administrative access to the Redshift cluster is required but an admin user secret was not provided and the cluster was imported');
            }
        }
        return adminUser;
    }
}
exports.DatabaseQuery = DatabaseQuery;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGF0YWJhc2UtcXVlcnkuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJkYXRhYmFzZS1xdWVyeS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2QkFBNkI7QUFDN0IsMkNBQTJDO0FBQzNDLGlEQUFpRDtBQUVqRCxtQ0FBbUM7QUFDbkMsZ0VBQWdFO0FBQ2hFLHdDQUFxQztBQUlyQyxpR0FBaUc7QUFDakcsOERBQThEO0FBQzlELDJDQUF1QztBQWF2QyxNQUFhLGFBQTRCLFNBQVEsc0JBQVM7SUFNeEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF1QztRQUMvRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDM0MsTUFBTSxPQUFPLEdBQUcsSUFBSSxNQUFNLENBQUMsaUJBQWlCLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUM1RCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUseUJBQXlCLENBQUMsRUFBRTtnQkFDM0UsT0FBTyxFQUFFLENBQUMsTUFBTSxDQUFDO2FBQ2xCLENBQUM7WUFDRixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLE9BQU8sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDaEMsSUFBSSxFQUFFLHNDQUFzQztZQUM1QyxhQUFhLEVBQUUseUJBQXlCO1NBQ3pDLENBQUMsQ0FBQztRQUNILE9BQU8sQ0FBQyxlQUFlLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQzlDLE9BQU8sRUFBRSxDQUFDLGlDQUFpQyxFQUFFLGdDQUFnQyxDQUFDO1lBQzlFLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNqQixDQUFDLENBQUMsQ0FBQztRQUNKLFNBQVMsQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFN0IsTUFBTSxRQUFRLEdBQUcsSUFBSSxlQUFlLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDOUQsY0FBYyxFQUFFLE9BQU87U0FDeEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxpQkFBaUIsR0FBNkM7WUFDbEUsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQ3RCLFdBQVcsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLFdBQVc7WUFDdEMsWUFBWSxFQUFFLFNBQVMsQ0FBQyxTQUFTO1lBQ2pDLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtZQUNoQyxHQUFHLEtBQUssQ0FBQyxVQUFVO1NBQ3BCLENBQUM7UUFDRixJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3ZELFlBQVksRUFBRSwrQkFBK0I7WUFDN0MsWUFBWSxFQUFFLFFBQVEsQ0FBQyxZQUFZO1lBQ25DLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtZQUNsQyxVQUFVLEVBQUUsaUJBQWlCO1NBQzlCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxjQUFjLEdBQUcsT0FBTyxDQUFDLGNBQWMsQ0FBQztRQUM3QyxJQUFJLENBQUMsR0FBRyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDO0tBQzlCO0lBRU0sa0JBQWtCLENBQUMsTUFBeUI7UUFDakQsSUFBSSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztLQUMxQztJQUVNLE1BQU0sQ0FBQyxhQUFxQjtRQUNqQyxPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0tBQzVDO0lBRU0sWUFBWSxDQUFDLGFBQXFCO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsYUFBYSxDQUFDLENBQUM7S0FDbEQ7SUFFTyxZQUFZLENBQUMsS0FBc0I7UUFDekMsTUFBTSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM5QixJQUFJLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ2hDLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDZCxJQUFJLE9BQU8sWUFBWSxpQkFBTyxFQUFFO2dCQUM5QixJQUFJLE9BQU8sQ0FBQyxNQUFNLEVBQUU7b0JBQ2xCLFNBQVMsR0FBRyxPQUFPLENBQUMsTUFBTSxDQUFDO2lCQUM1QjtxQkFBTTtvQkFDTCxNQUFNLElBQUksS0FBSyxDQUNiLDZMQUE2TCxDQUM5TCxDQUFDO2lCQUNIO2FBQ0Y7aUJBQU07Z0JBQ0wsTUFBTSxJQUFJLEtBQUssQ0FDYixrSUFBa0ksQ0FDbkksQ0FBQzthQUNIO1NBQ0Y7UUFDRCxPQUFPLFNBQVMsQ0FBQztLQUNsQjtDQUNGO0FBaEZELHNDQWdGQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBzZWNyZXRzbWFuYWdlciBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc2VjcmV0c21hbmFnZXInO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCAqIGFzIGN1c3RvbXJlc291cmNlcyBmcm9tICdhd3MtY2RrLWxpYi9jdXN0b20tcmVzb3VyY2VzJztcbmltcG9ydCB7IENsdXN0ZXIgfSBmcm9tICcuLi9jbHVzdGVyJztcbmltcG9ydCB7IERhdGFiYXNlT3B0aW9ucyB9IGZyb20gJy4uL2RhdGFiYXNlLW9wdGlvbnMnO1xuaW1wb3J0IHsgRGF0YWJhc2VRdWVyeUhhbmRsZXJQcm9wcyB9IGZyb20gJy4vaGFuZGxlci1wcm9wcyc7XG5cbi8vIGtlZXAgdGhpcyBpbXBvcnQgc2VwYXJhdGUgZnJvbSBvdGhlciBpbXBvcnRzIHRvIHJlZHVjZSBjaGFuY2UgZm9yIG1lcmdlIGNvbmZsaWN0cyB3aXRoIHYyLW1haW5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1kdXBsaWNhdGUtaW1wb3J0cywgaW1wb3J0L29yZGVyXG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuZXhwb3J0IGludGVyZmFjZSBEYXRhYmFzZVF1ZXJ5UHJvcHM8SGFuZGxlclByb3BzPiBleHRlbmRzIERhdGFiYXNlT3B0aW9ucyB7XG4gIHJlYWRvbmx5IGhhbmRsZXI6IHN0cmluZztcbiAgcmVhZG9ubHkgcHJvcGVydGllczogSGFuZGxlclByb3BzO1xuICAvKipcbiAgICogVGhlIHBvbGljeSB0byBhcHBseSB3aGVuIHRoaXMgcmVzb3VyY2UgaXMgcmVtb3ZlZCBmcm9tIHRoZSBhcHBsaWNhdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgY2RrLlJlbW92YWxQb2xpY3kuRGVzdHJveVxuICAgKi9cbiAgcmVhZG9ubHkgcmVtb3ZhbFBvbGljeT86IGNkay5SZW1vdmFsUG9saWN5O1xufVxuXG5leHBvcnQgY2xhc3MgRGF0YWJhc2VRdWVyeTxIYW5kbGVyUHJvcHM+IGV4dGVuZHMgQ29uc3RydWN0IGltcGxlbWVudHMgaWFtLklHcmFudGFibGUge1xuICByZWFkb25seSBncmFudFByaW5jaXBhbDogaWFtLklQcmluY2lwYWw7XG4gIHJlYWRvbmx5IHJlZjogc3RyaW5nO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgcmVzb3VyY2U6IGNkay5DdXN0b21SZXNvdXJjZTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRGF0YWJhc2VRdWVyeVByb3BzPEhhbmRsZXJQcm9wcz4pIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3QgYWRtaW5Vc2VyID0gdGhpcy5nZXRBZG1pblVzZXIocHJvcHMpO1xuICAgIGNvbnN0IGhhbmRsZXIgPSBuZXcgbGFtYmRhLlNpbmdsZXRvbkZ1bmN0aW9uKHRoaXMsICdIYW5kbGVyJywge1xuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdkYXRhYmFzZS1xdWVyeS1wcm92aWRlcicpLCB7XG4gICAgICAgIGV4Y2x1ZGU6IFsnKi50cyddLFxuICAgICAgfSksXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTJfWCxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgIHRpbWVvdXQ6IGNkay5EdXJhdGlvbi5taW51dGVzKDEpLFxuICAgICAgdXVpZDogJzNkZTViZWE3LTI3ZGEtNDc5Ni04NjYyLTVlZmI1NjQzMWI1ZicsXG4gICAgICBsYW1iZGFQdXJwb3NlOiAnUXVlcnkgUmVkc2hpZnQgRGF0YWJhc2UnLFxuICAgIH0pO1xuICAgIGhhbmRsZXIuYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsncmVkc2hpZnQtZGF0YTpEZXNjcmliZVN0YXRlbWVudCcsICdyZWRzaGlmdC1kYXRhOkV4ZWN1dGVTdGF0ZW1lbnQnXSxcbiAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgfSkpO1xuICAgIGFkbWluVXNlci5ncmFudFJlYWQoaGFuZGxlcik7XG5cbiAgICBjb25zdCBwcm92aWRlciA9IG5ldyBjdXN0b21yZXNvdXJjZXMuUHJvdmlkZXIodGhpcywgJ1Byb3ZpZGVyJywge1xuICAgICAgb25FdmVudEhhbmRsZXI6IGhhbmRsZXIsXG4gICAgfSk7XG5cbiAgICBjb25zdCBxdWVyeUhhbmRsZXJQcm9wczogRGF0YWJhc2VRdWVyeUhhbmRsZXJQcm9wcyAmIEhhbmRsZXJQcm9wcyA9IHtcbiAgICAgIGhhbmRsZXI6IHByb3BzLmhhbmRsZXIsXG4gICAgICBjbHVzdGVyTmFtZTogcHJvcHMuY2x1c3Rlci5jbHVzdGVyTmFtZSxcbiAgICAgIGFkbWluVXNlckFybjogYWRtaW5Vc2VyLnNlY3JldEFybixcbiAgICAgIGRhdGFiYXNlTmFtZTogcHJvcHMuZGF0YWJhc2VOYW1lLFxuICAgICAgLi4ucHJvcHMucHJvcGVydGllcyxcbiAgICB9O1xuICAgIHRoaXMucmVzb3VyY2UgPSBuZXcgY2RrLkN1c3RvbVJlc291cmNlKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIHJlc291cmNlVHlwZTogJ0N1c3RvbTo6UmVkc2hpZnREYXRhYmFzZVF1ZXJ5JyxcbiAgICAgIHNlcnZpY2VUb2tlbjogcHJvdmlkZXIuc2VydmljZVRva2VuLFxuICAgICAgcmVtb3ZhbFBvbGljeTogcHJvcHMucmVtb3ZhbFBvbGljeSxcbiAgICAgIHByb3BlcnRpZXM6IHF1ZXJ5SGFuZGxlclByb3BzLFxuICAgIH0pO1xuXG4gICAgdGhpcy5ncmFudFByaW5jaXBhbCA9IGhhbmRsZXIuZ3JhbnRQcmluY2lwYWw7XG4gICAgdGhpcy5yZWYgPSB0aGlzLnJlc291cmNlLnJlZjtcbiAgfVxuXG4gIHB1YmxpYyBhcHBseVJlbW92YWxQb2xpY3kocG9saWN5OiBjZGsuUmVtb3ZhbFBvbGljeSk6IHZvaWQge1xuICAgIHRoaXMucmVzb3VyY2UuYXBwbHlSZW1vdmFsUG9saWN5KHBvbGljeSk7XG4gIH1cblxuICBwdWJsaWMgZ2V0QXR0KGF0dHJpYnV0ZU5hbWU6IHN0cmluZyk6IGNkay5SZWZlcmVuY2Uge1xuICAgIHJldHVybiB0aGlzLnJlc291cmNlLmdldEF0dChhdHRyaWJ1dGVOYW1lKTtcbiAgfVxuXG4gIHB1YmxpYyBnZXRBdHRTdHJpbmcoYXR0cmlidXRlTmFtZTogc3RyaW5nKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy5yZXNvdXJjZS5nZXRBdHRTdHJpbmcoYXR0cmlidXRlTmFtZSk7XG4gIH1cblxuICBwcml2YXRlIGdldEFkbWluVXNlcihwcm9wczogRGF0YWJhc2VPcHRpb25zKTogc2VjcmV0c21hbmFnZXIuSVNlY3JldCB7XG4gICAgY29uc3QgY2x1c3RlciA9IHByb3BzLmNsdXN0ZXI7XG4gICAgbGV0IGFkbWluVXNlciA9IHByb3BzLmFkbWluVXNlcjtcbiAgICBpZiAoIWFkbWluVXNlcikge1xuICAgICAgaWYgKGNsdXN0ZXIgaW5zdGFuY2VvZiBDbHVzdGVyKSB7XG4gICAgICAgIGlmIChjbHVzdGVyLnNlY3JldCkge1xuICAgICAgICAgIGFkbWluVXNlciA9IGNsdXN0ZXIuc2VjcmV0O1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAgICdBZG1pbmlzdHJhdGl2ZSBhY2Nlc3MgdG8gdGhlIFJlZHNoaWZ0IGNsdXN0ZXIgaXMgcmVxdWlyZWQgYnV0IGFuIGFkbWluIHVzZXIgc2VjcmV0IHdhcyBub3QgcHJvdmlkZWQgYW5kIHRoZSBjbHVzdGVyIGRpZCBub3QgZ2VuZXJhdGUgYWRtaW4gdXNlciBjcmVkZW50aWFscyAodGhleSB3ZXJlIHByb3ZpZGVkIGV4cGxpY2l0bHkpJyxcbiAgICAgICAgICApO1xuICAgICAgICB9XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgJ0FkbWluaXN0cmF0aXZlIGFjY2VzcyB0byB0aGUgUmVkc2hpZnQgY2x1c3RlciBpcyByZXF1aXJlZCBidXQgYW4gYWRtaW4gdXNlciBzZWNyZXQgd2FzIG5vdCBwcm92aWRlZCBhbmQgdGhlIGNsdXN0ZXIgd2FzIGltcG9ydGVkJyxcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIGFkbWluVXNlcjtcbiAgfVxufVxuIl19