from power_bdd.bdd import BDD, WeightedGame
import unittest

class TestBDD(unittest.TestCase):
    def test_electoral_college_1996(self):
        # https://en.wikipedia.org/wiki/United_States_Electoral_College
        # example in https://www.orsj.or.jp/~archive/pdf/e_mag/Vol.43_01_071.pdf
        w = [54, 33, 32, 25, 23, 22, 21, 18, 15, 14, 13, 13,
                    12, 12, 11, 11, 11, 11, 10, 10, 9, 9, 8, 8, 8, 8, 8, 8, 7, 7,
                    7, 6, 6, 5, 5, 5, 5, 4, 4, 4, 4, 4, 4, 3, 3, 3, 3, 3, 3, 3, 3]                
        q = 270
        game = WeightedGame(q, w)
        bdd = BDD(game)
        bdd.create_BDD()
        bdd.calc_banzhaf()
        bdd.calc_shapley()

        self.assertEqual(sum(map(lambda x: len(x), bdd.uniqueTable)), 4655)
        beta = [0, 0.4664541346829614, 0.25958612004997317, 0.2512143267270046, 0.19386430412490352, 0.17784420367909792, 0.16988528488751609, 0.16195807015004515, 0.13834904167561657, 0.1149639030998193, 0.10721120484408697, 0.09947716139126062, 0.09947716139126062, 0.0917603106390894, 0.0917603106390894, 0.08405922092784923, 0.08405922092784923, 0.08405922092784923, 0.08405922092784923, 0.07637248687519982, 0.07637248687519982, 0.0686987274492834, 0.0686987274492834, 0.061036584694352136, 0.061036584694352136, 0.061036584694352136, 0.061036584694352136, 0.061036584694352136, 0.061036584694352136, 0.0533847189530432, 0.0533847189530432, 0.0533847189530432, 0.045741808640666015, 0.045741808640666015, 0.03810654673017755, 0.03810654673017755, 0.03810654673017755, 0.03810654673017755, 0.030477639570128723, 0.030477639570128723, 0.030477639570128723, 0.030477639570128723, 0.030477639570128723, 0.030477639570128723, 0.0228538034563881, 0.0228538034563881, 0.0228538034563881, 0.0228538034563881, 0.0228538034563881, 0.0228538034563881, 0.0228538034563881, 0.0228538034563881]
        phi  = [0, 0.10812540951779513, 0.06296832700852449, 0.06093272191760149, 0.04693092721554598, 0.043006330162357095, 0.041056127565584306, 0.03911385143347975, 0.03333359769343866, 0.027621240480289428, 0.025731793751168748, 0.023849541579004027, 0.023849541579004027, 0.02197440228337113, 0.02197440228337113, 0.020106295306354327, 0.020106295306354327, 0.020106295306354327, 0.020106295306354327, 0.01824514249273979, 0.01824514249273979, 0.016390867225524578, 0.016390867225524578, 0.01454339424426838, 0.01454339424426838, 0.01454339424426838, 0.01454339424426838, 0.01454339424426838, 0.01454339424426838, 0.012702650342316452, 0.012702650342316452, 0.012702650342316452, 0.01086856361234286, 0.01086856361234286, 0.009041063713444781, 0.009041063713444781, 0.009041063713444781, 0.009041063713444781, 0.00722008153290024, 0.00722008153290024, 0.00722008153290024, 0.00722008153290024, 0.00722008153290024, 0.00722008153290024, 0.005405549637386588, 0.005405549637386588, 0.005405549637386588, 0.005405549637386588, 0.005405549637386588, 0.005405549637386588, 0.005405549637386588, 0.005405549637386588]        
        for i in range(1,game.n + 1):
            self.assertAlmostEqual(bdd.beta[i], beta[i], places=14)
            self.assertAlmostEqual(bdd.phi[i],  phi[i],  places=14)
        
    def test_us_federal_system_2020(self):
        # U.S. federal system represented by weighted voting systems, according to
        # https://www.fernuni-hagen.de/stochastik/downloads/voting.pdf.
        # G = (G1 and G2 and G3) or (G1 and G4 and G5 and G3) or (G6 and G7)
        game1 = WeightedGame(218, [0, 0] + [0] * 100 + [1] * 435)
        game2 = WeightedGame(51,  [0, 0] + [1] * 100 + [0] * 435)
        game3 = WeightedGame(1,   [1, 0] + [0] * 100 + [0] * 435)
        game4 = WeightedGame(50,  [0, 0] + [1] * 100 + [0] * 435)
        game5 = WeightedGame(1,   [0, 1] + [0] * 100 + [0] * 435)
        game6 = WeightedGame(290, [0, 0] + [0] * 100 + [1] * 435)
        game7 = WeightedGame(67,  [0, 0] + [1] * 100 + [0] * 435)
    
        federal_system = BDD(game1) & BDD(game2) & BDD(game3) | BDD(game1) & BDD(game4) & BDD(game5) & BDD(game3) | BDD(game6) & BDD(game7)
        
        beta = [0, 0.2499999999999993, 0.019897309346794692, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.01989730934679494, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 
            0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 
            0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256, 0.009569407217589256]

        federal_system.calc_banzhaf()
        self.assertEqual(sum(map(lambda x: len(x), federal_system.uniqueTable)), 63718)
        for i in range(1, federal_system.n + 1):
            self.assertAlmostEqual(federal_system.beta[i], beta[i], places=14)

        # Proofing that the U.S. federal system has dimension 2: Show that both BDDs are the same:
        # G = G1 and G2 
        # page 29 of https://www.math.ust.hk/~maykwok/courses/MATH4823/MATH4823_Topic1.pdf
        game1 = WeightedGame(290, [72, 0] + [0] * 100 + [1] * 435)
        game2 = WeightedGame(134, [33, 1] + [2] * 100 + [0] * 435)

        federal_system_2 = BDD(game1) & BDD(game2)
        self.assertEqual(sum(map(lambda x: len(x),federal_system_2.uniqueTable)), 63718)

        federal_system_2.calc_banzhaf()
        for i in range(1, federal_system_2.n + 1):
            self.assertAlmostEqual(federal_system_2.beta[i], beta[i], places=14)
        
        self.assertEqual(federal_system, federal_system_2)
    
    def test_council_of_european_union_2019(self):
        # https://www.sciencedirect.com/science/article/abs/pii/S0377221705008805
        # https://ec.europa.eu/eurostat/tgm/refreshTableAction.do?tab=table&plugin=1&pcode=tps00001&language=en
        game1 = WeightedGame(16, [1] * 28)
        game2 = WeightedGame(333763100, [493559, 613894, 875898, 1324820, 1919968, 2080908, 2794184, 4076246, 4904226,5450421, 5517919, 5806081, 7000039, 8858775,9772756, 10230185, 10276617, 10649800, 10722287, 11467923, 17282163, 19401658, 37972812,46934632, 60359546, 66647112,67028048, 83019214])
        game3 = WeightedGame(25, [1] * 28)
        euc = BDD(game1) & (BDD(game2) | BDD(game3))
        euc.calc_banzhaf()
        euc.calc_shapley()
        beta = [0, 0.02993607521057129, 0.030137911438941956, 0.030573546886444092, 0.03132106363773346, 0.03231021761894226, 0.0325763076543808, 0.03375901281833649, 0.03587421774864197, 0.03724370896816254, 0.03814743459224701, 0.03825753927230835, 0.03873381018638611, 0.04070255160331726, 0.043756380677223206, 0.04526035487651825, 0.04601353406906128, 0.046089619398117065, 0.04670339822769165, 0.04682156443595886, 0.04804880917072296, 0.057645514607429504, 0.061206281185150146, 0.08346369862556458, 0.10274338722229004, 0.12771199643611908, 0.13851484656333923, 0.13922742009162903, 0.17020191252231598]
        phi =  [0, 0.00883986193482761, 0.009002234136841751, 0.009357572974479695, 0.009975079747929395, 0.01078847600553079, 0.01101647765196022, 0.012007360905597543, 0.013785377848845343, 0.014940288917944054, 0.015713165640141007, 0.015808291968273123, 0.016225354627225675, 0.017904512379879188, 0.020593884784892994, 0.021878553213187754, 0.022539183557153675, 0.02261653846221083, 0.023149908102795432, 0.023266568437520115, 0.024335834182718022, 0.033006553130392136, 0.03638113536653044, 0.06241470459265586, 0.0750470799074245, 0.09954163583184986, 0.11131426309142017, 0.11209421991189389, 0.14645588268787893]
        for i in range(1, euc.n + 1):
            self.assertAlmostEqual(euc.beta[i], beta[i], places=14)
            self.assertAlmostEqual(euc.phi[i],  phi[i],  places=14)

        self.assertEqual(sum(map(lambda x: len(x), euc.uniqueTable)), 18733)
  
if __name__ == '__main__':
    unittest.main()
