import numpy as np
import pandas as pd


def load_optotrak(file, max_rows=None) -> pd.DataFrame:
    """Load data from an Optotrak tab-separated values file. 

    TSV files generated by the NDI software are a little weird, and need special
    handling.
    """
    # Load the numeric data, replacing missing values with NaN. The final column
    # of data has an extraneous tab; this makes 'genfromtxt' think there's an
    # extra column that isn't there.
    data = np.genfromtxt(
        file,
        delimiter='\t',
        skip_header=5,
        max_rows=max_rows,
        missing_values=[''],
        filling_values=[float('nan')],
    )[:, :-1]

    headers = pd.Series(np.loadtxt(
        file,
        delimiter='\t',
        skiprows=4,
        max_rows=1,
        dtype=object,
    )).str.replace(' ', '')

    df = pd.DataFrame(data, columns=headers)
    df['Frame'] = df['Frame'].astype('uint')
    df.set_index('Frame', inplace=True)

    return df
