"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ToolkitCleaner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecr_assets_1 = require("aws-cdk-lib/aws-ecr-assets");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
const constructs_1 = require("constructs");
const clean_images_function_1 = require("./clean-images-function");
const clean_objects_function_1 = require("./clean-objects-function");
const extract_template_hashes_function_1 = require("./extract-template-hashes-function");
const get_stack_names_function_1 = require("./get-stack-names-function");
/**
 * Clean unused S3 and ECR assets from your CDK Toolkit.
 *
 * @stability stable
 */
class ToolkitCleaner extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c;
        super(scope, id);
        // Dummy assets to reference S3 bucket and ECR repository
        const fileAsset = new aws_s3_assets_1.Asset(this, 'FileAsset', {
            path: path.join(__dirname, '..', '..', 'assets', 'toolkit-cleaner', 'docker', 'dummy.txt'),
        });
        const dockerImageAsset = new aws_ecr_assets_1.DockerImageAsset(this, 'DockerImageAsset', {
            directory: path.join(__dirname, '..', '..', 'assets', 'toolkit-cleaner', 'docker'),
        });
        const getStackNamesFunction = new get_stack_names_function_1.GetStackNamesFunction(this, 'GetStackNamesFunction');
        getStackNamesFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['cloudformation:DescribeStacks'],
            resources: ['*'],
        }));
        const getStackNames = new tasks.LambdaInvoke(this, 'GetStackNames', {
            lambdaFunction: getStackNamesFunction,
            payloadResponseOnly: true,
        });
        const stacksMap = new sfn.Map(this, 'StacksMap', {
            maxConcurrency: 1,
            resultSelector: {
                AssetHashes: sfn.JsonPath.stringAt('$'),
            },
        });
        const extractTemplateHashesFunction = new extract_template_hashes_function_1.ExtractTemplateHashesFunction(this, 'ExtractTemplateHashesFunction', {
            timeout: aws_cdk_lib_1.Duration.seconds(30),
        });
        extractTemplateHashesFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['cloudformation:GetTemplate'],
            resources: ['*'],
        }));
        const extractTemplateHashes = new tasks.LambdaInvoke(this, 'ExtractTemplateHashes', {
            lambdaFunction: extractTemplateHashesFunction,
            payloadResponseOnly: true,
        }).addRetry({
            errors: ['Throttling'],
        });
        const flattenHashes = new tasks.EvaluateExpression(this, 'FlattenHashes', {
            expression: '[...new Set(($.AssetHashes).flat())]',
        });
        const cleanObjectsFunction = new clean_objects_function_1.CleanObjectsFunction(this, 'CleanObjectsFunction', {
            timeout: aws_cdk_lib_1.Duration.minutes(5),
        });
        cleanObjectsFunction.addEnvironment('BUCKET_NAME', fileAsset.bucket.bucketName);
        fileAsset.bucket.grantRead(cleanObjectsFunction);
        fileAsset.bucket.grantDelete(cleanObjectsFunction);
        const cleanObjects = new tasks.LambdaInvoke(this, 'CleanObjects', {
            lambdaFunction: cleanObjectsFunction,
            payloadResponseOnly: true,
        });
        const cleanImagesFunction = new clean_images_function_1.CleanImagesFunction(this, 'CleanImagesFunction', {
            timeout: aws_cdk_lib_1.Duration.minutes(5),
        });
        cleanImagesFunction.addEnvironment('REPOSITORY_NAME', dockerImageAsset.repository.repositoryName);
        dockerImageAsset.repository.grant(cleanImagesFunction, 'ecr:DescribeImages', 'ecr:BatchDeleteImage');
        const cleanImages = new tasks.LambdaInvoke(this, 'CleanImages', {
            lambdaFunction: cleanImagesFunction,
            payloadResponseOnly: true,
        });
        if (!props.dryRun) {
            cleanObjectsFunction.addEnvironment('RUN', 'true');
            cleanImagesFunction.addEnvironment('RUN', 'true');
        }
        if (props.retainAssetsNewerThan) {
            const retainMilliseconds = props.retainAssetsNewerThan.toMilliseconds().toString();
            cleanObjectsFunction.addEnvironment('RETAIN_MILLISECONDS', retainMilliseconds);
            cleanImagesFunction.addEnvironment('RETAIN_MILLISECONDS', retainMilliseconds);
        }
        const sumReclaimed = new tasks.EvaluateExpression(this, 'SumReclaimed', {
            expression: '({ Deleted: $[0].Deleted + $[1].Deleted, Reclaimed: $[0].Reclaimed + $[1].Reclaimed })',
        });
        const stateMachine = new sfn.StateMachine(this, 'Resource', {
            definition: getStackNames
                .next(stacksMap.iterator(extractTemplateHashes))
                .next(flattenHashes)
                .next(new sfn.Parallel(this, 'Clean')
                .branch(cleanObjects)
                .branch(cleanImages))
                .next(sumReclaimed),
        });
        const rule = new aws_events_1.Rule(this, 'Rule', {
            enabled: (_b = props.scheduleEnabled) !== null && _b !== void 0 ? _b : true,
            schedule: (_c = props.schedule) !== null && _c !== void 0 ? _c : aws_events_1.Schedule.rate(aws_cdk_lib_1.Duration.days(1)),
        });
        rule.addTarget(new aws_events_targets_1.SfnStateMachine(stateMachine));
    }
}
exports.ToolkitCleaner = ToolkitCleaner;
_a = JSII_RTTI_SYMBOL_1;
ToolkitCleaner[_a] = { fqn: "cloudstructs.ToolkitCleaner", version: "0.4.19" };
//# sourceMappingURL=data:application/json;base64,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