"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeyPair = exports.KeyLength = void 0;
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const statement = require("cdk-iam-floyd");
const path = require("path");
const resourceType = 'Custom::EC2-Key-Pair';
const ID = `CFN::Resource::${resourceType}`;
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
var KeyLength;
(function (KeyLength) {
    KeyLength[KeyLength["L2048"] = 2048] = "L2048";
    KeyLength[KeyLength["L4096"] = 4096] = "L4096";
})(KeyLength = exports.KeyLength || (exports.KeyLength = {}));
/**
 * An EC2 Key Pair
 */
class KeyPair extends cdk.Construct {
    /**
     * Defines a new EC2 Key Pair. The private key will be stored in AWS Secrets Manager
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * ARN of the private key in AWS Secrets Manager
         */
        this.arn = '';
        /**
         * Name of the Key Pair
         */
        this.name = '';
        this.prefix = '';
        if (props.removePrivateKeyAfterDays &&
            (props.removePrivateKeyAfterDays < 0 ||
                (props.removePrivateKeyAfterDays > 0 &&
                    props.removePrivateKeyAfterDays < 7) ||
                props.removePrivateKeyAfterDays > 30)) {
            scope.node.addError(`Parameter removePrivateKeyAfterDays must be 0 or between 7 and 30. Got ${props.removePrivateKeyAfterDays}`);
        }
        const stack = cdk.Stack.of(this).stackName;
        this.prefix = props.resourcePrefix || stack;
        this.lambda = this.ensureLambda();
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'Custom::EC2-Key-Pair');
        this.tags.setTag('CreatedBy', ID);
        const key = new cfn.CustomResource(this, `EC2-Key-Pair-${props.name}`, {
            provider: cfn.CustomResourceProvider.fromLambda(this.lambda),
            resourceType: resourceType,
            properties: {
                Name: props.name,
                Description: props.description || '',
                KeyLength: props.keyLength || KeyLength.L2048,
                Kms: ((_a = props.kms) === null || _a === void 0 ? void 0 : _a.keyArn) || 'alias/aws/secretsmanager',
                RemovePrivateKeyAfterDays: props.removePrivateKeyAfterDays || 0,
                SecretPrefix: props.secretPrefix || 'ec2-private-key/',
                StackName: stack,
                Tags: cdk.Lazy.anyValue({
                    produce: () => this.tags.renderTags(),
                }),
            },
        });
        if (typeof props.kms !== 'undefined') {
            props.kms.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kms);
            key.node.addDependency(this.lambda.role);
        }
        this.arn = key.getAttString('PrivateKeyARN');
        this.name = key.getAttString('KeyPairName');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'EC2-Key-Name-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'EC2-Key-Pair-Manager-Policy', {
            managedPolicyName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            statements: [
                new statement.Ec2()
                    .allow()
                    .describeKeyPairs()
                    .createKeyPair()
                    .deleteKeyPair(),
                new statement.Secretsmanager().allow().listSecrets(),
                new statement.Secretsmanager()
                    .allow()
                    .createSecret()
                    .tagResource()
                    .ifAwsRequestTag('CreatedBy', ID),
                new statement.Secretsmanager()
                    .allow()
                    .allActions(/^(Describe|Delete|Put|Update)/)
                    .getResourcePolicy()
                    .restoreSecret()
                    .listSecretVersionIds()
                    .untagResource()
                    .ifResourceTag('CreatedBy', ID),
            ],
        });
        const role = new iam.Role(stack, 'EC2-Key-Pair-Manager-Role', {
            roleName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${this.prefix}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage EC2 Key Pairs',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout),
        });
        return fn;
    }
    /**
     * Grants read access to the private key in AWS Secrets Manager
     */
    grantRead(grantee) {
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'secretsmanager:DescribeSecret',
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:ListSecretVersionIds',
            ],
            resourceArns: [this.arn],
            scope: this,
        });
        return result;
    }
}
exports.KeyPair = KeyPair;
//# sourceMappingURL=data:application/json;base64,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