"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DualAlbFargateService = exports.LoadBalancerAccessibility = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const route53 = require("@aws-cdk/aws-route53");
const targets = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
/**
 * The load balancer accessibility.
 *
 * @stability stable
 */
var LoadBalancerAccessibility;
(function (LoadBalancerAccessibility) {
    LoadBalancerAccessibility["EXTERNAL_ONLY"] = "EXTERNAL_ONLY";
    LoadBalancerAccessibility["INTERNAL_ONLY"] = "INTERNAL_ONLY";
})(LoadBalancerAccessibility = exports.LoadBalancerAccessibility || (exports.LoadBalancerAccessibility = {}));
/**
 * @stability stable
 */
class DualAlbFargateService extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j;
        super(scope, id);
        this.hasExternalLoadBalancer = false;
        this.hasInternalLoadBalancer = false;
        this.vpcSubnets = { subnetType: ec2.SubnetType.PRIVATE };
        /**
         * determine if vpcSubnets are all public ones
         */
        this.isPublicSubnets = false;
        this.enableLoadBalancerAlias = ((_b = props.route53Ops) === null || _b === void 0 ? void 0 : _b.enableLoadBalancerAlias) != false;
        this.vpc = (_c = props.vpc) !== null && _c !== void 0 ? _c : getOrCreateVpc(this),
            this.service = [];
        if (props.vpcSubnets) {
            this.vpcSubnets = props.vpcSubnets;
            this.validateSubnets(this.vpc, this.vpcSubnets);
        }
        // determine whether we need the external LB
        props.tasks.forEach(t => {
            // determine the accessibility
            if (t.accessibility != LoadBalancerAccessibility.INTERNAL_ONLY) {
                this.hasExternalLoadBalancer = true;
            }
            if (t.accessibility != LoadBalancerAccessibility.EXTERNAL_ONLY) {
                this.hasInternalLoadBalancer = true;
            }
        });
        if (this.hasExternalLoadBalancer) {
            this.externalAlb = new elbv2.ApplicationLoadBalancer(this, 'ExternalAlb', {
                vpc: this.vpc,
                internetFacing: true,
            });
        }
        if (this.hasInternalLoadBalancer) {
            this.internalAlb = new elbv2.ApplicationLoadBalancer(this, 'InternalAlb', {
                vpc: this.vpc,
                internetFacing: false,
            });
        }
        const cluster = new ecs.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
        });
        const spotOnlyStrategy = [
            {
                capacityProvider: 'FARGATE_SPOT',
                base: 0,
                weight: 1,
            },
            {
                capacityProvider: 'FARGATE',
                base: 0,
                weight: 0,
            },
        ];
        props.tasks.forEach(t => {
            var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
            const defaultContainerName = (_b = t.task.defaultContainer) === null || _b === void 0 ? void 0 : _b.containerName;
            const svc = new ecs.FargateService(this, `${defaultContainerName}Service`, {
                taskDefinition: t.task,
                cluster,
                capacityProviderStrategies: (_c = t.capacityProviderStrategy) !== null && _c !== void 0 ? _c : (props.spot ? spotOnlyStrategy : undefined),
                desiredCount: t.desiredCount,
                enableExecuteCommand: (_d = props.enableExecuteCommand) !== null && _d !== void 0 ? _d : false,
                vpcSubnets: this.vpcSubnets,
                assignPublicIp: this.isPublicSubnets,
            });
            this.service.push(svc);
            // default scaling policy
            const scaling = svc.autoScaleTaskCount({ maxCapacity: (_f = (_e = t.scalingPolicy) === null || _e === void 0 ? void 0 : _e.maxCapacity) !== null && _f !== void 0 ? _f : 10 });
            scaling.scaleOnCpuUtilization('CpuScaling', {
                targetUtilizationPercent: (_h = (_g = t.scalingPolicy) === null || _g === void 0 ? void 0 : _g.targetCpuUtilization) !== null && _h !== void 0 ? _h : 50,
            });
            if (t.accessibility != LoadBalancerAccessibility.INTERNAL_ONLY) {
                const exttg = new elbv2.ApplicationTargetGroup(this, `${defaultContainerName}ExtTG`, {
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    vpc: this.vpc,
                    healthCheck: t.healthCheck,
                });
                // listener for the external ALB
                new elbv2.ApplicationListener(this, `ExtAlbListener${t.listenerPort}`, {
                    loadBalancer: this.externalAlb,
                    open: true,
                    port: t.listenerPort,
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    defaultTargetGroups: [exttg],
                });
                scaling.scaleOnRequestCount('RequestScaling', {
                    requestsPerTarget: (_k = (_j = t.scalingPolicy) === null || _j === void 0 ? void 0 : _j.requestPerTarget) !== null && _k !== void 0 ? _k : 1000,
                    targetGroup: exttg,
                });
                exttg.addTarget(svc);
            }
            if (t.accessibility != LoadBalancerAccessibility.EXTERNAL_ONLY) {
                const inttg = new elbv2.ApplicationTargetGroup(this, `${defaultContainerName}IntTG`, {
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    vpc: this.vpc,
                    healthCheck: t.healthCheck,
                });
                // listener for the internal ALB
                new elbv2.ApplicationListener(this, `IntAlbListener${t.listenerPort}`, {
                    loadBalancer: this.internalAlb,
                    open: true,
                    port: t.listenerPort,
                    protocol: elbv2.ApplicationProtocol.HTTP,
                    defaultTargetGroups: [inttg],
                });
                // extra scaling policy
                scaling.scaleOnRequestCount('RequestScaling2', {
                    requestsPerTarget: (_m = (_l = t.scalingPolicy) === null || _l === void 0 ? void 0 : _l.requestPerTarget) !== null && _m !== void 0 ? _m : 1000,
                    targetGroup: inttg,
                });
                inttg.addTarget(svc);
            }
        });
        // Route53
        const zoneName = (_e = (_d = props.route53Ops) === null || _d === void 0 ? void 0 : _d.zoneName) !== null && _e !== void 0 ? _e : 'svc.local';
        const externalAlbRecordName = (_g = (_f = props.route53Ops) === null || _f === void 0 ? void 0 : _f.externalAlbRecordName) !== null && _g !== void 0 ? _g : 'external';
        const internalAlbRecordName = (_j = (_h = props.route53Ops) === null || _h === void 0 ? void 0 : _h.internalAlbRecordName) !== null && _j !== void 0 ? _j : 'internal';
        if (this.enableLoadBalancerAlias) {
            const zone = new route53.PrivateHostedZone(this, 'HostedZone', {
                zoneName,
                vpc: this.vpc,
            });
            if (this.hasInternalLoadBalancer) {
                new route53.ARecord(this, 'InternalAlbAlias', {
                    zone,
                    recordName: internalAlbRecordName,
                    target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(this.internalAlb)),
                });
            }
            if (this.hasExternalLoadBalancer) {
                new route53.ARecord(this, 'ExternalAlbAlias', {
                    zone,
                    recordName: externalAlbRecordName,
                    target: route53.RecordTarget.fromAlias(new targets.LoadBalancerTarget(this.externalAlb)),
                });
            }
            if (this.hasExternalLoadBalancer) {
                new cdk.CfnOutput(this, 'ExternalEndpoint', { value: `http://${this.externalAlb.loadBalancerDnsName}` });
                new cdk.CfnOutput(this, 'ExternalEndpointPrivate', { value: `http://${externalAlbRecordName}.${zoneName}` });
            }
            if (this.hasInternalLoadBalancer) {
                new cdk.CfnOutput(this, 'InternalEndpoint', { value: `http://${this.internalAlb.loadBalancerDnsName}` });
                new cdk.CfnOutput(this, 'InternalEndpointPrivate', { value: `http://${internalAlbRecordName}.${zoneName}` });
            }
        }
        else {
            if (this.hasExternalLoadBalancer) {
                new cdk.CfnOutput(this, 'ExternalEndpoint', { value: `http://${this.externalAlb.loadBalancerDnsName}` });
            }
            if (this.hasInternalLoadBalancer) {
                new cdk.CfnOutput(this, 'InternalEndpoint', { value: `http://${this.internalAlb.loadBalancerDnsName}` });
            }
        }
    }
    validateSubnets(vpc, vpcSubnets) {
        const subnets = vpc.selectSubnets(vpcSubnets);
        // get all subnets in the VPC
        const allsubnetIds = vpc.publicSubnets.concat(vpc.privateSubnets).concat(vpc.isolatedSubnets).map(x => x.subnetId);
        // validate the given subnets
        subnets.subnetIds.forEach(s => {
            if (!allsubnetIds.includes(s)) {
                throw new Error(`${s} does not exist in the VPC`);
            }
            if (vpc.isolatedSubnets.map(i => i.subnetId).includes(s)) {
                throw new Error(`Isolated subnet ${s} is not allowed`);
            }
        });
        const hasPublic = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
        const hasPrivate = subnets.subnetIds.some(s => new Set(vpc.privateSubnets.map(x => x.subnetId)).has(s));
        if (hasPublic && hasPrivate) {
            throw new Error('You should provide either all public or all private subnets, not both.');
        }
        this.isPublicSubnets = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
    }
}
exports.DualAlbFargateService = DualAlbFargateService;
_a = JSII_RTTI_SYMBOL_1;
DualAlbFargateService[_a] = { fqn: "cdk-fargate-patterns.DualAlbFargateService", version: "0.1.4" };
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1'
        || process.env.CDK_USE_DEFAULT_VPC === '1' ? ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
//# sourceMappingURL=data:application/json;base64,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