"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJsonPath = exports.renderList = exports.State = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
const fields_1 = require("../fields");
const types_1 = require("../types");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * Base class for all other state classes
 */
class State extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.branches = [];
        this.retries = [];
        this.catches = [];
        this.choices = [];
        this.prefixes = [];
        /**
         * States with references to this state.
         *
         * Used for finding complete connected graph that a state is part of.
         */
        this.incomingStates = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_StateProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.startState = this;
        this.comment = props.comment;
        this.inputPath = props.inputPath;
        this.parameters = props.parameters;
        this.outputPath = props.outputPath;
        this.resultPath = props.resultPath;
        this.resultSelector = props.resultSelector;
    }
    /**
     * Add a prefix to the stateId of all States found in a construct tree
     */
    static prefixStates(root, prefix) {
        const queue = [root];
        while (queue.length > 0) {
            const el = queue.splice(0, 1)[0];
            if (isPrefixable(el)) {
                el.addPrefix(prefix);
            }
            queue.push(...constructs_1.Node.of(el).children);
        }
    }
    /**
     * Find the set of states reachable through transitions from the given start state.
     * This does not retrieve states from within sub-graphs, such as states within a Parallel state's branch.
     */
    static findReachableStates(start, options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_State(start);
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_FindStateOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.findReachableStates);
            }
            throw error;
        }
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            queue.push(...outgoing);
            ret.add(state);
        }
        return Array.from(ret);
    }
    /**
     * Find the set of end states states reachable through transitions from the given start state
     */
    static findReachableEndStates(start, options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_State(start);
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_FindStateOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.findReachableEndStates);
            }
            throw error;
        }
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            if (outgoing.length > 0) {
                // We can continue
                queue.push(...outgoing);
            }
            else {
                // Terminal state
                ret.add(state);
            }
        }
        return Array.from(ret);
    }
    /**
     * Return only the states that allow chaining from an array of states
     */
    static filterNextables(states) {
        return states.filter(isNextable);
    }
    get id() {
        return this.node.id;
    }
    /**
     * Tokenized string that evaluates to the state's ID
     */
    get stateId() {
        return this.prefixes.concat(this.id).join('');
    }
    /**
     * Add a prefix to the stateId of this state
     */
    addPrefix(x) {
        if (x !== '') {
            this.prefixes.splice(0, 0, x);
        }
    }
    /**
     * Register this state as part of the given graph
     *
     * Don't call this. It will be called automatically when you work
     * with states normally.
     */
    bindToGraph(graph) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_StateGraph(graph);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bindToGraph);
            }
            throw error;
        }
        if (this.containingGraph === graph) {
            return;
        }
        if (this.containingGraph) {
            // eslint-disable-next-line max-len
            throw new Error(`Trying to use state '${this.stateId}' in ${graph}, but is already in ${this.containingGraph}. Every state can only be used in one graph.`);
        }
        this.containingGraph = graph;
        this.whenBoundToGraph(graph);
        for (const incoming of this.incomingStates) {
            incoming.bindToGraph(graph);
        }
        for (const outgoing of this.outgoingTransitions({ includeErrorHandlers: true })) {
            outgoing.bindToGraph(graph);
        }
        for (const branch of this.branches) {
            branch.registerSuperGraph(this.containingGraph);
        }
        if (!!this.iteration) {
            this.iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a retrier to the retry list of this state
     * @internal
     */
    _addRetry(props = {}) {
        var _b;
        validateErrors(props.errors);
        this.retries.push({
            ...props,
            errors: (_b = props.errors) !== null && _b !== void 0 ? _b : [types_1.Errors.ALL],
        });
    }
    /**
     * Add an error handler to the catch list of this state
     * @internal
     */
    _addCatch(handler, props = {}) {
        var _b;
        validateErrors(props.errors);
        this.catches.push({
            next: handler,
            props: {
                errors: (_b = props.errors) !== null && _b !== void 0 ? _b : [types_1.Errors.ALL],
                resultPath: props.resultPath,
            },
        });
        handler.addIncoming(this);
        if (this.containingGraph) {
            handler.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default transition of this state
     */
    makeNext(next) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_State(next);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.makeNext);
            }
            throw error;
        }
        // Can't be called 'setNext' because of JSII
        if (this._next) {
            throw new Error(`State '${this.id}' already has a next state`);
        }
        this._next = next;
        next.addIncoming(this);
        if (this.containingGraph) {
            next.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a choice branch to this state
     */
    addChoice(condition, next) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_Condition(condition);
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_State(next);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addChoice);
            }
            throw error;
        }
        this.choices.push({ condition, next });
        next.startState.addIncoming(this);
        if (this.containingGraph) {
            next.startState.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a paralle branch to this state
     */
    addBranch(branch) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_StateGraph(branch);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addBranch);
            }
            throw error;
        }
        this.branches.push(branch);
        if (this.containingGraph) {
            branch.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a map iterator to this state
     */
    addIterator(iteration) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_StateGraph(iteration);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addIterator);
            }
            throw error;
        }
        this.iteration = iteration;
        if (this.containingGraph) {
            iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default choice transition of this state
     */
    makeDefault(def) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_State(def);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.makeDefault);
            }
            throw error;
        }
        // Can't be called 'setDefault' because of JSII
        if (this.defaultChoice) {
            throw new Error(`Choice '${this.id}' already has a default next state`);
        }
        this.defaultChoice = def;
    }
    /**
     * Render the default next state in ASL JSON format
     */
    renderNextEnd() {
        if (this._next) {
            return { Next: this._next.stateId };
        }
        else {
            return { End: true };
        }
    }
    /**
     * Render the choices in ASL JSON format
     */
    renderChoices() {
        var _b;
        return {
            Choices: renderList(this.choices, renderChoice),
            Default: (_b = this.defaultChoice) === null || _b === void 0 ? void 0 : _b.stateId,
        };
    }
    /**
     * Render InputPath/Parameters/OutputPath in ASL JSON format
     */
    renderInputOutput() {
        return {
            InputPath: renderJsonPath(this.inputPath),
            Parameters: this.parameters,
            OutputPath: renderJsonPath(this.outputPath),
        };
    }
    /**
     * Render parallel branches in ASL JSON format
     */
    renderBranches() {
        return {
            Branches: this.branches.map(b => b.toGraphJson()),
        };
    }
    /**
     * Render map iterator in ASL JSON format
     */
    renderIterator() {
        if (!this.iteration) {
            throw new Error('Iterator must not be undefined !');
        }
        return {
            Iterator: this.iteration.toGraphJson(),
        };
    }
    /**
     * Render error recovery options in ASL JSON format
     */
    renderRetryCatch() {
        return {
            Retry: renderList(this.retries, renderRetry, (a, b) => compareErrors(a.errors, b.errors)),
            Catch: renderList(this.catches, renderCatch, (a, b) => compareErrors(a.props.errors, b.props.errors)),
        };
    }
    /**
     * Render ResultSelector in ASL JSON format
     */
    renderResultSelector() {
        return fields_1.FieldUtils.renderObject({
            ResultSelector: this.resultSelector,
        });
    }
    /**
     * Called whenever this state is bound to a graph
     *
     * Can be overridden by subclasses.
     */
    whenBoundToGraph(graph) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_StateGraph(graph);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.whenBoundToGraph);
            }
            throw error;
        }
        graph.registerState(this);
    }
    /**
     * Add a state to the incoming list
     */
    addIncoming(source) {
        this.incomingStates.push(source);
    }
    /**
     * Return all states this state can transition to
     */
    outgoingTransitions(options) {
        const ret = new Array();
        if (this._next) {
            ret.push(this._next);
        }
        if (this.defaultChoice) {
            ret.push(this.defaultChoice);
        }
        for (const c of this.choices) {
            ret.push(c.next);
        }
        if (options.includeErrorHandlers) {
            for (const c of this.catches) {
                ret.push(c.next);
            }
        }
        return ret;
    }
}
exports.State = State;
_a = JSII_RTTI_SYMBOL_1;
State[_a] = { fqn: "@aws-cdk/aws-stepfunctions.State", version: "1.153.0" };
/**
 * Render a choice transition
 */
function renderChoice(c) {
    return {
        ...c.condition.renderCondition(),
        Next: c.next.stateId,
    };
}
/**
 * Render a Retry object to ASL
 */
function renderRetry(retry) {
    return {
        ErrorEquals: retry.errors,
        IntervalSeconds: retry.interval && retry.interval.toSeconds(),
        MaxAttempts: retry.maxAttempts,
        BackoffRate: retry.backoffRate,
    };
}
/**
 * Render a Catch object to ASL
 */
function renderCatch(c) {
    return {
        ErrorEquals: c.props.errors,
        ResultPath: renderJsonPath(c.props.resultPath),
        Next: c.next.stateId,
    };
}
/**
 * Compares a list of Errors to move Errors.ALL last in a sort function
 */
function compareErrors(a, b) {
    if (a === null || a === void 0 ? void 0 : a.includes(types_1.Errors.ALL)) {
        return 1;
    }
    if (b === null || b === void 0 ? void 0 : b.includes(types_1.Errors.ALL)) {
        return -1;
    }
    return 0;
}
/**
 * Validates an errors list
 */
function validateErrors(errors) {
    if ((errors === null || errors === void 0 ? void 0 : errors.includes(types_1.Errors.ALL)) && errors.length > 1) {
        throw new Error(`${types_1.Errors.ALL} must appear alone in an error list`);
    }
}
/**
 * Render a list or return undefined for an empty list
 */
function renderList(xs, mapFn, sortFn) {
    if (xs.length === 0) {
        return undefined;
    }
    let list = xs;
    if (sortFn) {
        list = xs.sort(sortFn);
    }
    return list.map(mapFn);
}
exports.renderList = renderList;
/**
 * Render JSON path, respecting the special value DISCARD
 */
function renderJsonPath(jsonPath) {
    if (jsonPath === undefined) {
        return undefined;
    }
    if (jsonPath === fields_1.JsonPath.DISCARD) {
        return null;
    }
    if (!jsonPath.startsWith('$')) {
        throw new Error(`Expected JSON path to start with '$', got: ${jsonPath}`);
    }
    return jsonPath;
}
exports.renderJsonPath = renderJsonPath;
/**
 * Whether an object is a Prefixable
 */
function isPrefixable(x) {
    return typeof (x) === 'object' && x.addPrefix;
}
/**
 * Whether an object is INextable
 */
function isNextable(x) {
    return typeof (x) === 'object' && x.next;
}
//# sourceMappingURL=data:application/json;base64,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