"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const appmesh = require("../lib");
module.exports = {
    'When an existing VirtualNode': {
        'with existing backends, adds new backend': {
            'should add resource with service backends'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                const service1 = new appmesh.VirtualService(stack, 'service-1', {
                    virtualServiceName: 'service1.domain.local',
                    mesh,
                });
                const service2 = new appmesh.VirtualService(stack, 'service-2', {
                    virtualServiceName: 'service2.domain.local',
                    mesh,
                });
                const node = new appmesh.VirtualNode(stack, 'test-node', {
                    mesh,
                    dnsHostName: 'test',
                    listener: {},
                    backends: [service1],
                });
                node.addBackends(service2);
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::VirtualNode', {
                    Spec: {
                        Backends: [
                            {
                                VirtualService: {
                                    VirtualServiceName: {
                                        'Fn::GetAtt': ['service1A48078CF', 'VirtualServiceName'],
                                    },
                                },
                            },
                            {
                                VirtualService: {
                                    VirtualServiceName: {
                                        'Fn::GetAtt': ['service27C65CF7D', 'VirtualServiceName'],
                                    },
                                },
                            },
                        ],
                    },
                }));
                test.done();
            },
        },
        'when a single portmapping is added': {
            'should add the portmapping to the resoource'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                const mesh = new appmesh.Mesh(stack, 'mesh', {
                    meshName: 'test-mesh',
                });
                const node = mesh.addVirtualNode('test-node', {
                    dnsHostName: 'test',
                });
                node.addListeners({
                    portMapping: {
                        port: 8081,
                        protocol: appmesh.Protocol.TCP,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::VirtualNode', {
                    Spec: {
                        Listeners: [
                            {
                                PortMapping: {
                                    Port: 8081,
                                    Protocol: 'tcp',
                                },
                            },
                        ],
                    },
                }));
                test.done();
            },
        },
    },
    'Can export and import VirtualNode and perform actions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const mesh = new appmesh.Mesh(stack, 'mesh', {
            meshName: 'test-mesh',
        });
        const node = mesh.addVirtualNode('test-node', {
            dnsHostName: 'test.domain.local',
            listener: {},
        });
        const stack2 = new cdk.Stack();
        const node2 = appmesh.VirtualNode.fromVirtualNodeName(stack2, 'imported-node', mesh.meshName, node.virtualNodeName);
        node2.addListeners({
            portMapping: {
                port: 8081,
                protocol: appmesh.Protocol.TCP,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::VirtualNode', {
            MeshName: {
                'Fn::GetAtt': ['meshACDFE68E', 'MeshName'],
            },
            Spec: {
                Listeners: [
                    {
                        PortMapping: {
                            Port: 8080,
                            Protocol: 'http',
                        },
                    },
                ],
                ServiceDiscovery: {
                    DNS: {
                        Hostname: 'test.domain.local',
                    },
                },
            },
            VirtualNodeName: 'meshtestnode428A9479',
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC52aXJ0dWFsLW5vZGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LnZpcnR1YWwtbm9kZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsNENBQTJEO0FBQzNELHFDQUFxQztBQUdyQyxrQ0FBa0M7QUFFbEMsaUJBQVM7SUFDUCw4QkFBOEIsRUFBRTtRQUM5QiwwQ0FBMEMsRUFBRTtZQUMxQywyQ0FBMkMsQ0FBQyxJQUFVO2dCQUNwRCxRQUFRO2dCQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO2dCQUU5QixPQUFPO2dCQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksT0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO29CQUMzQyxRQUFRLEVBQUUsV0FBVztpQkFDdEIsQ0FBQyxDQUFDO2dCQUVILE1BQU0sUUFBUSxHQUFHLElBQUksT0FBTyxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO29CQUM5RCxrQkFBa0IsRUFBRSx1QkFBdUI7b0JBQzNDLElBQUk7aUJBQ0wsQ0FBQyxDQUFDO2dCQUNILE1BQU0sUUFBUSxHQUFHLElBQUksT0FBTyxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO29CQUM5RCxrQkFBa0IsRUFBRSx1QkFBdUI7b0JBQzNDLElBQUk7aUJBQ0wsQ0FBQyxDQUFDO2dCQUVILE1BQU0sSUFBSSxHQUFHLElBQUksT0FBTyxDQUFDLFdBQVcsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO29CQUN2RCxJQUFJO29CQUNKLFdBQVcsRUFBRSxNQUFNO29CQUNuQixRQUFRLEVBQUUsRUFBRTtvQkFDWixRQUFRLEVBQUUsQ0FBQyxRQUFRLENBQUM7aUJBQ3JCLENBQUMsQ0FBQztnQkFFSCxJQUFJLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO2dCQUUzQixPQUFPO2dCQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQ2QseUJBQWdCLENBQUMsMkJBQTJCLEVBQUU7b0JBQzVDLElBQUksRUFBRTt3QkFDSixRQUFRLEVBQUU7NEJBQ1I7Z0NBQ0UsY0FBYyxFQUFFO29DQUNkLGtCQUFrQixFQUFFO3dDQUNsQixZQUFZLEVBQUUsQ0FBQyxrQkFBa0IsRUFBRSxvQkFBb0IsQ0FBQztxQ0FDekQ7aUNBQ0Y7NkJBQ0Y7NEJBQ0Q7Z0NBQ0UsY0FBYyxFQUFFO29DQUNkLGtCQUFrQixFQUFFO3dDQUNsQixZQUFZLEVBQUUsQ0FBQyxrQkFBa0IsRUFBRSxvQkFBb0IsQ0FBQztxQ0FDekQ7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7aUJBQ0YsQ0FBQyxDQUNILENBQUM7Z0JBRUYsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztTQUNGO1FBQ0Qsb0NBQW9DLEVBQUU7WUFDcEMsNkNBQTZDLENBQUMsSUFBVTtnQkFDdEQsUUFBUTtnQkFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztnQkFFOUIsT0FBTztnQkFDUCxNQUFNLElBQUksR0FBRyxJQUFJLE9BQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtvQkFDM0MsUUFBUSxFQUFFLFdBQVc7aUJBQ3RCLENBQUMsQ0FBQztnQkFFSCxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLFdBQVcsRUFBRTtvQkFDNUMsV0FBVyxFQUFFLE1BQU07aUJBQ3BCLENBQUMsQ0FBQztnQkFFSCxJQUFJLENBQUMsWUFBWSxDQUFDO29CQUNoQixXQUFXLEVBQUU7d0JBQ1gsSUFBSSxFQUFFLElBQUk7d0JBQ1YsUUFBUSxFQUFFLE9BQU8sQ0FBQyxRQUFRLENBQUMsR0FBRztxQkFDL0I7aUJBQ0YsQ0FBQyxDQUFDO2dCQUVILE9BQU87Z0JBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FDZCx5QkFBZ0IsQ0FBQywyQkFBMkIsRUFBRTtvQkFDNUMsSUFBSSxFQUFFO3dCQUNKLFNBQVMsRUFBRTs0QkFDVDtnQ0FDRSxXQUFXLEVBQUU7b0NBQ1gsSUFBSSxFQUFFLElBQUk7b0NBQ1YsUUFBUSxFQUFFLEtBQUs7aUNBQ2hCOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGLENBQUMsQ0FDSCxDQUFDO2dCQUVGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNkLENBQUM7U0FDRjtLQUNGO0lBQ0QsdURBQXVELENBQUMsSUFBVTtRQUNoRSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsT0FBTztRQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksT0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1lBQzNDLFFBQVEsRUFBRSxXQUFXO1NBQ3RCLENBQUMsQ0FBQztRQUVILE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsV0FBVyxFQUFFO1lBQzVDLFdBQVcsRUFBRSxtQkFBbUI7WUFDaEMsUUFBUSxFQUFFLEVBQUU7U0FDYixDQUFDLENBQUM7UUFFSCxNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUUvQixNQUFNLEtBQUssR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDLG1CQUFtQixDQUFDLE1BQU0sRUFBRSxlQUFlLEVBQUUsSUFBSSxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUM7UUFFcEgsS0FBSyxDQUFDLFlBQVksQ0FBQztZQUNqQixXQUFXLEVBQUU7Z0JBQ1gsSUFBSSxFQUFFLElBQUk7Z0JBQ1YsUUFBUSxFQUFFLE9BQU8sQ0FBQyxRQUFRLENBQUMsR0FBRzthQUMvQjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUNkLHlCQUFnQixDQUFDLDJCQUEyQixFQUFFO1lBQzVDLFFBQVEsRUFBRTtnQkFDUixZQUFZLEVBQUUsQ0FBQyxjQUFjLEVBQUUsVUFBVSxDQUFDO2FBQzNDO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLFNBQVMsRUFBRTtvQkFDVDt3QkFDRSxXQUFXLEVBQUU7NEJBQ1gsSUFBSSxFQUFFLElBQUk7NEJBQ1YsUUFBUSxFQUFFLE1BQU07eUJBQ2pCO3FCQUNGO2lCQUNGO2dCQUNELGdCQUFnQixFQUFFO29CQUNoQixHQUFHLEVBQUU7d0JBQ0gsUUFBUSxFQUFFLG1CQUFtQjtxQkFDOUI7aUJBQ0Y7YUFDRjtZQUNELGVBQWUsRUFBRSxzQkFBc0I7U0FDeEMsQ0FBQyxDQUNILENBQUM7UUFFRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4cGVjdCwgaGF2ZVJlc291cmNlTGlrZSB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuXG5pbXBvcnQgKiBhcyBhcHBtZXNoIGZyb20gJy4uL2xpYic7XG5cbmV4cG9ydCA9IHtcbiAgJ1doZW4gYW4gZXhpc3RpbmcgVmlydHVhbE5vZGUnOiB7XG4gICAgJ3dpdGggZXhpc3RpbmcgYmFja2VuZHMsIGFkZHMgbmV3IGJhY2tlbmQnOiB7XG4gICAgICAnc2hvdWxkIGFkZCByZXNvdXJjZSB3aXRoIHNlcnZpY2UgYmFja2VuZHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBjb25zdCBtZXNoID0gbmV3IGFwcG1lc2guTWVzaChzdGFjaywgJ21lc2gnLCB7XG4gICAgICAgICAgbWVzaE5hbWU6ICd0ZXN0LW1lc2gnLFxuICAgICAgICB9KTtcblxuICAgICAgICBjb25zdCBzZXJ2aWNlMSA9IG5ldyBhcHBtZXNoLlZpcnR1YWxTZXJ2aWNlKHN0YWNrLCAnc2VydmljZS0xJywge1xuICAgICAgICAgIHZpcnR1YWxTZXJ2aWNlTmFtZTogJ3NlcnZpY2UxLmRvbWFpbi5sb2NhbCcsXG4gICAgICAgICAgbWVzaCxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IHNlcnZpY2UyID0gbmV3IGFwcG1lc2guVmlydHVhbFNlcnZpY2Uoc3RhY2ssICdzZXJ2aWNlLTInLCB7XG4gICAgICAgICAgdmlydHVhbFNlcnZpY2VOYW1lOiAnc2VydmljZTIuZG9tYWluLmxvY2FsJyxcbiAgICAgICAgICBtZXNoLFxuICAgICAgICB9KTtcblxuICAgICAgICBjb25zdCBub2RlID0gbmV3IGFwcG1lc2guVmlydHVhbE5vZGUoc3RhY2ssICd0ZXN0LW5vZGUnLCB7XG4gICAgICAgICAgbWVzaCxcbiAgICAgICAgICBkbnNIb3N0TmFtZTogJ3Rlc3QnLFxuICAgICAgICAgIGxpc3RlbmVyOiB7fSxcbiAgICAgICAgICBiYWNrZW5kczogW3NlcnZpY2UxXSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgbm9kZS5hZGRCYWNrZW5kcyhzZXJ2aWNlMik7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3Qoc3RhY2spLnRvKFxuICAgICAgICAgIGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXBwTWVzaDo6VmlydHVhbE5vZGUnLCB7XG4gICAgICAgICAgICBTcGVjOiB7XG4gICAgICAgICAgICAgIEJhY2tlbmRzOiBbXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgVmlydHVhbFNlcnZpY2U6IHtcbiAgICAgICAgICAgICAgICAgICAgVmlydHVhbFNlcnZpY2VOYW1lOiB7XG4gICAgICAgICAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbJ3NlcnZpY2UxQTQ4MDc4Q0YnLCAnVmlydHVhbFNlcnZpY2VOYW1lJ10sXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgVmlydHVhbFNlcnZpY2U6IHtcbiAgICAgICAgICAgICAgICAgICAgVmlydHVhbFNlcnZpY2VOYW1lOiB7XG4gICAgICAgICAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbJ3NlcnZpY2UyN0M2NUNGN0QnLCAnVmlydHVhbFNlcnZpY2VOYW1lJ10sXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0pLFxuICAgICAgICApO1xuXG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcbiAgICB9LFxuICAgICd3aGVuIGEgc2luZ2xlIHBvcnRtYXBwaW5nIGlzIGFkZGVkJzoge1xuICAgICAgJ3Nob3VsZCBhZGQgdGhlIHBvcnRtYXBwaW5nIHRvIHRoZSByZXNvb3VyY2UnKHRlc3Q6IFRlc3QpIHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBjb25zdCBtZXNoID0gbmV3IGFwcG1lc2guTWVzaChzdGFjaywgJ21lc2gnLCB7XG4gICAgICAgICAgbWVzaE5hbWU6ICd0ZXN0LW1lc2gnLFxuICAgICAgICB9KTtcblxuICAgICAgICBjb25zdCBub2RlID0gbWVzaC5hZGRWaXJ0dWFsTm9kZSgndGVzdC1ub2RlJywge1xuICAgICAgICAgIGRuc0hvc3ROYW1lOiAndGVzdCcsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIG5vZGUuYWRkTGlzdGVuZXJzKHtcbiAgICAgICAgICBwb3J0TWFwcGluZzoge1xuICAgICAgICAgICAgcG9ydDogODA4MSxcbiAgICAgICAgICAgIHByb3RvY29sOiBhcHBtZXNoLlByb3RvY29sLlRDUCxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdChzdGFjaykudG8oXG4gICAgICAgICAgaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBcHBNZXNoOjpWaXJ0dWFsTm9kZScsIHtcbiAgICAgICAgICAgIFNwZWM6IHtcbiAgICAgICAgICAgICAgTGlzdGVuZXJzOiBbXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgUG9ydE1hcHBpbmc6IHtcbiAgICAgICAgICAgICAgICAgICAgUG9ydDogODA4MSxcbiAgICAgICAgICAgICAgICAgICAgUHJvdG9jb2w6ICd0Y3AnLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgKTtcblxuICAgICAgICB0ZXN0LmRvbmUoKTtcbiAgICAgIH0sXG4gICAgfSxcbiAgfSxcbiAgJ0NhbiBleHBvcnQgYW5kIGltcG9ydCBWaXJ0dWFsTm9kZSBhbmQgcGVyZm9ybSBhY3Rpb25zJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBtZXNoID0gbmV3IGFwcG1lc2guTWVzaChzdGFjaywgJ21lc2gnLCB7XG4gICAgICBtZXNoTmFtZTogJ3Rlc3QtbWVzaCcsXG4gICAgfSk7XG5cbiAgICBjb25zdCBub2RlID0gbWVzaC5hZGRWaXJ0dWFsTm9kZSgndGVzdC1ub2RlJywge1xuICAgICAgZG5zSG9zdE5hbWU6ICd0ZXN0LmRvbWFpbi5sb2NhbCcsXG4gICAgICBsaXN0ZW5lcjoge30sXG4gICAgfSk7XG5cbiAgICBjb25zdCBzdGFjazIgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICBjb25zdCBub2RlMiA9IGFwcG1lc2guVmlydHVhbE5vZGUuZnJvbVZpcnR1YWxOb2RlTmFtZShzdGFjazIsICdpbXBvcnRlZC1ub2RlJywgbWVzaC5tZXNoTmFtZSwgbm9kZS52aXJ0dWFsTm9kZU5hbWUpO1xuXG4gICAgbm9kZTIuYWRkTGlzdGVuZXJzKHtcbiAgICAgIHBvcnRNYXBwaW5nOiB7XG4gICAgICAgIHBvcnQ6IDgwODEsXG4gICAgICAgIHByb3RvY29sOiBhcHBtZXNoLlByb3RvY29sLlRDUCxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhcbiAgICAgIGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXBwTWVzaDo6VmlydHVhbE5vZGUnLCB7XG4gICAgICAgIE1lc2hOYW1lOiB7XG4gICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbJ21lc2hBQ0RGRTY4RScsICdNZXNoTmFtZSddLFxuICAgICAgICB9LFxuICAgICAgICBTcGVjOiB7XG4gICAgICAgICAgTGlzdGVuZXJzOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFBvcnRNYXBwaW5nOiB7XG4gICAgICAgICAgICAgICAgUG9ydDogODA4MCxcbiAgICAgICAgICAgICAgICBQcm90b2NvbDogJ2h0dHAnLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIFNlcnZpY2VEaXNjb3Zlcnk6IHtcbiAgICAgICAgICAgIEROUzoge1xuICAgICAgICAgICAgICBIb3N0bmFtZTogJ3Rlc3QuZG9tYWluLmxvY2FsJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgVmlydHVhbE5vZGVOYW1lOiAnbWVzaHRlc3Rub2RlNDI4QTk0NzknLFxuICAgICAgfSksXG4gICAgKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxufTtcbiJdfQ==