"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualService = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * VirtualService represents a service inside an AppMesh
 *
 * It routes traffic either to a Virtual Node or to a Virtual Router.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
 */
class VirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualServiceName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        if (props.virtualNode && props.virtualRouter) {
            throw new Error('Must provide only one of virtualNode or virtualRouter for the provider');
        }
        this.mesh = props.mesh;
        // Check which provider to use node or router (or neither)
        if (props.virtualRouter) {
            this.virtualServiceProvider = this.addVirtualRouter(props.virtualRouter.virtualRouterName);
        }
        if (props.virtualNode) {
            this.virtualServiceProvider = this.addVirtualNode(props.virtualNode.virtualNodeName);
        }
        const svc = new appmesh_generated_1.CfnVirtualService(this, 'Resource', {
            meshName: this.mesh.meshName,
            virtualServiceName: this.physicalName,
            spec: {
                provider: this.virtualServiceProvider,
            },
        });
        this.virtualServiceName = this.getResourceNameAttribute(svc.attrVirtualServiceName);
        this.virtualServiceArn = this.getResourceArnAttribute(svc.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualService`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualService given an ARN
     */
    static fromVirtualServiceArn(scope, id, virtualServiceArn) {
        return new ImportedVirtualService(scope, id, {
            virtualServiceArn,
        });
    }
    /**
     * Import an existing VirtualService given mesh and service names
     */
    static fromVirtualServiceName(scope, id, meshName, virtualServiceName) {
        return new ImportedVirtualService(scope, id, {
            meshName,
            virtualServiceName,
        });
    }
    addVirtualRouter(name) {
        return {
            virtualRouter: {
                virtualRouterName: name,
            },
        };
    }
    addVirtualNode(name) {
        return {
            virtualNode: {
                virtualNodeName: name,
            },
        };
    }
}
exports.VirtualService = VirtualService;
/**
 * Returns properties that allows a VirtualService to be imported
 */
class ImportedVirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualServiceArn) {
            this.virtualServiceArn = props.virtualServiceArn;
            this.virtualServiceName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualServiceArn).resourceName));
        }
        else if (props.virtualServiceName && props.meshName) {
            this.virtualServiceName = props.virtualServiceName;
            this.virtualServiceArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualService`,
                resourceName: this.virtualServiceName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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