# -*- coding: utf-8 -*-

import numpy as np
import pytest

from doc_ufcn import image


@pytest.mark.parametrize(
    "input_image, size, padding_value, expected_resized_image, expected_padding",
    [
        # Resizing table 9x9 to 8 without padding
        (
            [
                [1, 1, 1, 1, 1, 1, 1, 1, 1],
                [2, 2, 2, 2, 2, 2, 2, 2, 2],
                [3, 3, 3, 3, 3, 3, 3, 3, 3],
                [4, 4, 4, 4, 4, 4, 4, 4, 4],
                [5, 5, 5, 5, 5, 5, 5, 5, 5],
                [6, 6, 6, 6, 6, 6, 6, 6, 6],
                [7, 7, 7, 7, 7, 7, 7, 7, 7],
                [8, 8, 8, 8, 8, 8, 8, 8, 8],
                [9, 9, 9, 9, 9, 9, 9, 9, 9],
            ],
            8,
            50,
            [
                [1.0625, 1.0625, 1.0625, 1.0625, 1.0625, 1.0625, 1.0625, 1.0625],
                [2.1875, 2.1875, 2.1875, 2.1875, 2.1875, 2.1875, 2.1875, 2.1875],
                [3.3125, 3.3125, 3.3125, 3.3125, 3.3125, 3.3125, 3.3125, 3.3125],
                [4.4375, 4.4375, 4.4375, 4.4375, 4.4375, 4.4375, 4.4375, 4.4375],
                [5.5625, 5.5625, 5.5625, 5.5625, 5.5625, 5.5625, 5.5625, 5.5625],
                [6.6875, 6.6875, 6.6875, 6.6875, 6.6875, 6.6875, 6.6875, 6.6875],
                [7.8125, 7.8125, 7.8125, 7.8125, 7.8125, 7.8125, 7.8125, 7.8125],
                [8.9375, 8.9375, 8.9375, 8.9375, 8.9375, 8.9375, 8.9375, 8.9375],
            ],
            [0, 0],
        ),
        # Resizing table 8x8 to 8 (no resizing) without padding
        (
            [
                [1, 1, 1, 1, 1, 1, 1, 1],
                [2, 2, 2, 2, 2, 2, 2, 2],
                [3, 3, 3, 3, 3, 3, 3, 3],
                [4, 4, 4, 4, 4, 4, 4, 4],
                [5, 5, 5, 5, 5, 5, 5, 5],
                [6, 6, 6, 6, 6, 6, 6, 6],
                [7, 7, 7, 7, 7, 7, 7, 7],
                [8, 8, 8, 8, 8, 8, 8, 8],
            ],
            8,
            50,
            [
                [1, 1, 1, 1, 1, 1, 1, 1],
                [2, 2, 2, 2, 2, 2, 2, 2],
                [3, 3, 3, 3, 3, 3, 3, 3],
                [4, 4, 4, 4, 4, 4, 4, 4],
                [5, 5, 5, 5, 5, 5, 5, 5],
                [6, 6, 6, 6, 6, 6, 6, 6],
                [7, 7, 7, 7, 7, 7, 7, 7],
                [8, 8, 8, 8, 8, 8, 8, 8],
            ],
            [0, 0],
        ),
        # Resizing table 7x9 to 8 with padding
        (
            [
                [1, 1, 1, 1, 1, 1, 1, 1, 1],
                [2, 2, 2, 2, 2, 2, 2, 2, 2],
                [3, 3, 3, 3, 3, 3, 3, 3, 3],
                [4, 4, 4, 4, 4, 4, 4, 4, 4],
                [5, 5, 5, 5, 5, 5, 5, 5, 5],
                [6, 6, 6, 6, 6, 6, 6, 6, 6],
                [7, 7, 7, 7, 7, 7, 7, 7, 7],
            ],
            8,
            50,
            [
                [50, 50, 50, 50, 50, 50, 50, 50],
                [1.0833, 1.0833, 1.0833, 1.0833, 1.0833, 1.0833, 1.0833, 1.0833],
                [2.25, 2.25, 2.25, 2.25, 2.25, 2.25, 2.25, 2.25],
                [3.4167, 3.4167, 3.4167, 3.4167, 3.4167, 3.4167, 3.4167, 3.4167],
                [4.5833, 4.5833, 4.5833, 4.5833, 4.5833, 4.5833, 4.5833, 4.5833],
                [5.75, 5.75, 5.75, 5.75, 5.75, 5.75, 5.75, 5.75],
                [6.9167, 6.9167, 6.9167, 6.9167, 6.9167, 6.9167, 6.9167, 6.9167],
                [50, 50, 50, 50, 50, 50, 50, 50],
            ],
            [1, 0],
        ),
        # Resizing table 9x7 to 8 with padding
        (
            [
                [1, 1, 1, 1, 1, 1, 1],
                [2, 2, 2, 2, 2, 2, 2],
                [3, 3, 3, 3, 3, 3, 3],
                [4, 4, 4, 4, 4, 4, 4],
                [5, 5, 5, 5, 5, 5, 5],
                [6, 6, 6, 6, 6, 6, 6],
                [7, 7, 7, 7, 7, 7, 7],
                [8, 8, 8, 8, 8, 8, 8],
                [9, 9, 9, 9, 9, 9, 9],
            ],
            8,
            100,
            [
                [100, 1.0625, 1.0625, 1.0625, 1.0625, 1.0625, 1.0625, 100],
                [100, 2.1875, 2.1875, 2.1875, 2.1875, 2.1875, 2.1875, 100],
                [100, 3.3125, 3.3125, 3.3125, 3.3125, 3.3125, 3.3125, 100],
                [100, 4.4375, 4.4375, 4.4375, 4.4375, 4.4375, 4.4375, 100],
                [100, 5.5625, 5.5625, 5.5625, 5.5625, 5.5625, 5.5625, 100],
                [100, 6.6875, 6.6875, 6.6875, 6.6875, 6.6875, 6.6875, 100],
                [100, 7.8125, 7.8125, 7.8125, 7.8125, 7.8125, 7.8125, 100],
                [100, 8.9375, 8.9375, 8.9375, 8.9375, 8.9375, 8.9375, 100],
            ],
            [0, 1],
        ),
    ],
)
def test_resize(
    input_image, size, padding_value, expected_resized_image, expected_padding
):
    """
    Test of the resize function.
    Check that the input image is correctly resized and padded.
    """
    resized_image, padding = image.resize(
        np.array(input_image, dtype=np.float32), size, padding_value
    )
    assert np.array_equal(
        np.round(resized_image, 4), np.array(expected_resized_image, dtype=np.float32)
    )
    assert padding == expected_padding


@pytest.mark.parametrize(
    "input_image, size, mean, std, expected_preprocessed_image, expected_padding",
    [
        # Preprocessing table 9x9x3 to 8
        (
            [
                [
                    [1, 1, 1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7, 7, 7],
                    [8, 8, 8, 8, 8, 8, 8, 8, 8],
                    [9, 9, 9, 9, 9, 9, 9, 9, 9],
                ],
                [
                    [1, 1, 1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7, 7, 7],
                    [8, 8, 8, 8, 8, 8, 8, 8, 8],
                    [9, 9, 9, 9, 9, 9, 9, 9, 9],
                ],
                [
                    [1, 1, 1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7, 7, 7],
                    [8, 8, 8, 8, 8, 8, 8, 8, 8],
                    [9, 9, 9, 9, 9, 9, 9, 9, 9],
                ],
            ],
            8,
            [1, 1, 1],
            [2, 2, 2],
            [
                [
                    [0.0312, 0.0312, 0.0312, 0.0312, 0.0312, 0.0312, 0.0312, 0.0312],
                    [0.5938, 0.5938, 0.5938, 0.5938, 0.5938, 0.5938, 0.5938, 0.5938],
                    [1.1562, 1.1562, 1.1562, 1.1562, 1.1562, 1.1562, 1.1562, 1.1562],
                    [1.7188, 1.7188, 1.7188, 1.7188, 1.7188, 1.7188, 1.7188, 1.7188],
                    [2.2812, 2.2812, 2.2812, 2.2812, 2.2812, 2.2812, 2.2812, 2.2812],
                    [2.8438, 2.8438, 2.8438, 2.8438, 2.8438, 2.8438, 2.8438, 2.8438],
                    [3.4062, 3.4062, 3.4062, 3.4062, 3.4062, 3.4062, 3.4062, 3.4062],
                    [3.9688, 3.9688, 3.9688, 3.9688, 3.9688, 3.9688, 3.9688, 3.9688],
                ],
                [
                    [0.0312, 0.0312, 0.0312, 0.0312, 0.0312, 0.0312, 0.0312, 0.0312],
                    [0.5938, 0.5938, 0.5938, 0.5938, 0.5938, 0.5938, 0.5938, 0.5938],
                    [1.1562, 1.1562, 1.1562, 1.1562, 1.1562, 1.1562, 1.1562, 1.1562],
                    [1.7188, 1.7188, 1.7188, 1.7188, 1.7188, 1.7188, 1.7188, 1.7188],
                    [2.2812, 2.2812, 2.2812, 2.2812, 2.2812, 2.2812, 2.2812, 2.2812],
                    [2.8438, 2.8438, 2.8438, 2.8438, 2.8438, 2.8438, 2.8438, 2.8438],
                    [3.4062, 3.4062, 3.4062, 3.4062, 3.4062, 3.4062, 3.4062, 3.4062],
                    [3.9688, 3.9688, 3.9688, 3.9688, 3.9688, 3.9688, 3.9688, 3.9688],
                ],
                [
                    [0.0312, 0.0312, 0.0312, 0.0312, 0.0312, 0.0312, 0.0312, 0.0312],
                    [0.5938, 0.5938, 0.5938, 0.5938, 0.5938, 0.5938, 0.5938, 0.5938],
                    [1.1562, 1.1562, 1.1562, 1.1562, 1.1562, 1.1562, 1.1562, 1.1562],
                    [1.7188, 1.7188, 1.7188, 1.7188, 1.7188, 1.7188, 1.7188, 1.7188],
                    [2.2812, 2.2812, 2.2812, 2.2812, 2.2812, 2.2812, 2.2812, 2.2812],
                    [2.8438, 2.8438, 2.8438, 2.8438, 2.8438, 2.8438, 2.8438, 2.8438],
                    [3.4062, 3.4062, 3.4062, 3.4062, 3.4062, 3.4062, 3.4062, 3.4062],
                    [3.9688, 3.9688, 3.9688, 3.9688, 3.9688, 3.9688, 3.9688, 3.9688],
                ],
            ],
            [0, 0],
        ),
        # Preprocessing table 8x8 to 8 (no resizing) without padding
        (
            [
                [
                    [1, 1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7, 7],
                    [8, 8, 8, 8, 8, 8, 8, 8],
                ],
                [
                    [1, 1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7, 7],
                    [8, 8, 8, 8, 8, 8, 8, 8],
                ],
                [
                    [1, 1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7, 7],
                    [8, 8, 8, 8, 8, 8, 8, 8],
                ],
            ],
            8,
            [0, 0, 0],
            [1, 1, 1],
            [
                [
                    [1, 1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7, 7],
                    [8, 8, 8, 8, 8, 8, 8, 8],
                ],
                [
                    [1, 1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7, 7],
                    [8, 8, 8, 8, 8, 8, 8, 8],
                ],
                [
                    [1, 1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7, 7],
                    [8, 8, 8, 8, 8, 8, 8, 8],
                ],
            ],
            [0, 0],
        ),
        # Preprocessing table 7x9 to 8 with padding
        (
            [
                [
                    [1, 1, 1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7, 7, 7],
                ],
                [
                    [1, 1, 1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7, 7, 7],
                ],
                [
                    [1, 1, 1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7, 7, 7],
                ],
            ],
            8,
            [2, 2, 2],
            [1, 1, 1],
            [
                [
                    [0, 0, 0, 0, 0, 0, 0, 0],
                    [
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                    ],
                    [0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25],
                    [1.4167, 1.4167, 1.4167, 1.4167, 1.4167, 1.4167, 1.4167, 1.4167],
                    [2.5833, 2.5833, 2.5833, 2.5833, 2.5833, 2.5833, 2.5833, 2.5833],
                    [3.75, 3.75, 3.75, 3.75, 3.75, 3.75, 3.75, 3.75],
                    [4.9167, 4.9167, 4.9167, 4.9167, 4.9167, 4.9167, 4.9167, 4.9167],
                    [0, 0, 0, 0, 0, 0, 0, 0],
                ],
                [
                    [0, 0, 0, 0, 0, 0, 0, 0],
                    [
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                    ],
                    [0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25],
                    [1.4167, 1.4167, 1.4167, 1.4167, 1.4167, 1.4167, 1.4167, 1.4167],
                    [2.5833, 2.5833, 2.5833, 2.5833, 2.5833, 2.5833, 2.5833, 2.5833],
                    [3.75, 3.75, 3.75, 3.75, 3.75, 3.75, 3.75, 3.75],
                    [4.9167, 4.9167, 4.9167, 4.9167, 4.9167, 4.9167, 4.9167, 4.9167],
                    [0, 0, 0, 0, 0, 0, 0, 0],
                ],
                [
                    [0, 0, 0, 0, 0, 0, 0, 0],
                    [
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                        -0.9167,
                    ],
                    [0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25, 0.25],
                    [1.4167, 1.4167, 1.4167, 1.4167, 1.4167, 1.4167, 1.4167, 1.4167],
                    [2.5833, 2.5833, 2.5833, 2.5833, 2.5833, 2.5833, 2.5833, 2.5833],
                    [3.75, 3.75, 3.75, 3.75, 3.75, 3.75, 3.75, 3.75],
                    [4.9167, 4.9167, 4.9167, 4.9167, 4.9167, 4.9167, 4.9167, 4.9167],
                    [0, 0, 0, 0, 0, 0, 0, 0],
                ],
            ],
            [1, 0],
        ),
        # Preprocessing table 9x7 to 8 with padding
        (
            [
                [
                    [1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7],
                    [8, 8, 8, 8, 8, 8, 8],
                    [9, 9, 9, 9, 9, 9, 9],
                ],
                [
                    [1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7],
                    [8, 8, 8, 8, 8, 8, 8],
                    [9, 9, 9, 9, 9, 9, 9],
                ],
                [
                    [1, 1, 1, 1, 1, 1, 1],
                    [2, 2, 2, 2, 2, 2, 2],
                    [3, 3, 3, 3, 3, 3, 3],
                    [4, 4, 4, 4, 4, 4, 4],
                    [5, 5, 5, 5, 5, 5, 5],
                    [6, 6, 6, 6, 6, 6, 6],
                    [7, 7, 7, 7, 7, 7, 7],
                    [8, 8, 8, 8, 8, 8, 8],
                    [9, 9, 9, 9, 9, 9, 9],
                ],
            ],
            8,
            [2, 2, 2],
            [1, 1, 1],
            [
                [
                    [0, -0.9375, -0.9375, -0.9375, -0.9375, -0.9375, -0.9375, 0],
                    [0, 0.1875, 0.1875, 0.1875, 0.1875, 0.1875, 0.1875, 0],
                    [0, 1.3125, 1.3125, 1.3125, 1.3125, 1.3125, 1.3125, 0],
                    [0, 2.4375, 2.4375, 2.4375, 2.4375, 2.4375, 2.4375, 0],
                    [0, 3.5625, 3.5625, 3.5625, 3.5625, 3.5625, 3.5625, 0],
                    [0, 4.6875, 4.6875, 4.6875, 4.6875, 4.6875, 4.6875, 0],
                    [0, 5.8125, 5.8125, 5.8125, 5.8125, 5.8125, 5.8125, 0],
                    [0, 6.9375, 6.9375, 6.9375, 6.9375, 6.9375, 6.9375, 0],
                ],
                [
                    [0, -0.9375, -0.9375, -0.9375, -0.9375, -0.9375, -0.9375, 0],
                    [0, 0.1875, 0.1875, 0.1875, 0.1875, 0.1875, 0.1875, 0],
                    [0, 1.3125, 1.3125, 1.3125, 1.3125, 1.3125, 1.3125, 0],
                    [0, 2.4375, 2.4375, 2.4375, 2.4375, 2.4375, 2.4375, 0],
                    [0, 3.5625, 3.5625, 3.5625, 3.5625, 3.5625, 3.5625, 0],
                    [0, 4.6875, 4.6875, 4.6875, 4.6875, 4.6875, 4.6875, 0],
                    [0, 5.8125, 5.8125, 5.8125, 5.8125, 5.8125, 5.8125, 0],
                    [0, 6.9375, 6.9375, 6.9375, 6.9375, 6.9375, 6.9375, 0],
                ],
                [
                    [0, -0.9375, -0.9375, -0.9375, -0.9375, -0.9375, -0.9375, 0],
                    [0, 0.1875, 0.1875, 0.1875, 0.1875, 0.1875, 0.1875, 0],
                    [0, 1.3125, 1.3125, 1.3125, 1.3125, 1.3125, 1.3125, 0],
                    [0, 2.4375, 2.4375, 2.4375, 2.4375, 2.4375, 2.4375, 0],
                    [0, 3.5625, 3.5625, 3.5625, 3.5625, 3.5625, 3.5625, 0],
                    [0, 4.6875, 4.6875, 4.6875, 4.6875, 4.6875, 4.6875, 0],
                    [0, 5.8125, 5.8125, 5.8125, 5.8125, 5.8125, 5.8125, 0],
                    [0, 6.9375, 6.9375, 6.9375, 6.9375, 6.9375, 6.9375, 0],
                ],
            ],
            [0, 1],
        ),
    ],
)
def test_preprocess_image(
    input_image, size, mean, std, expected_preprocessed_image, expected_padding
):
    """
    Test of the preprocess_image function.
    Check that the input image is correctly resized, padded and normalized.
    """
    preprocessed_image, padding = image.preprocess_image(
        np.transpose(np.array(input_image, dtype=np.float32), (1, 2, 0)),
        size,
        mean,
        std,
    )
    preprocessed_image = preprocessed_image.numpy()

    # Check the shapes.
    expected_preprocessed_image = np.array(
        expected_preprocessed_image, dtype=np.float32
    )
    assert preprocessed_image.shape == (1, *expected_preprocessed_image.shape)
    # Check the values.
    preprocessed_image = np.round(preprocessed_image[0], 4)
    assert np.array_equal(
        preprocessed_image.astype(np.float32), expected_preprocessed_image
    )
    assert padding == expected_padding
