"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DropSpamReceiptRule = exports.ReceiptRule = exports.TlsPolicy = void 0;
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const ses_generated_1 = require("./ses.generated");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * (experimental) The type of TLS policy for a receipt rule.
 *
 * @experimental
 */
var TlsPolicy;
(function (TlsPolicy) {
    TlsPolicy["OPTIONAL"] = "Optional";
    TlsPolicy["REQUIRE"] = "Require";
})(TlsPolicy = exports.TlsPolicy || (exports.TlsPolicy = {}));
/**
 * (experimental) A new receipt rule.
 *
 * @experimental
 */
class ReceiptRule extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.receiptRuleName,
        });
        this.actions = new Array();
        const resource = new ses_generated_1.CfnReceiptRule(this, 'Resource', {
            after: props.after ? props.after.receiptRuleName : undefined,
            rule: {
                actions: core_1.Lazy.anyValue({ produce: () => this.renderActions() }),
                enabled: props.enabled === undefined ? true : props.enabled,
                name: this.physicalName,
                recipients: props.recipients,
                scanEnabled: props.scanEnabled,
                tlsPolicy: props.tlsPolicy,
            },
            ruleSetName: props.ruleSet.receiptRuleSetName,
        });
        this.receiptRuleName = resource.ref;
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    /**
     * @experimental
     */
    static fromReceiptRuleName(scope, id, receiptRuleName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.receiptRuleName = receiptRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Adds an action to this receipt rule.
     *
     * @experimental
     */
    addAction(action) {
        this.actions.push(action.bind(this));
    }
    renderActions() {
        if (this.actions.length === 0) {
            return undefined;
        }
        return this.actions;
    }
}
exports.ReceiptRule = ReceiptRule;
/**
 * (experimental) A rule added at the top of the rule set to drop spam/virus.
 *
 * @see https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
 * @experimental
 */
class DropSpamReceiptRule extends core_2.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        const fn = new lambda.SingletonFunction(this, 'Function', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = ${dropSpamCode}`),
            uuid: '224e77f9-a32e-4b4d-ac32-983477abba16',
        });
        fn.addPermission('AllowSes', {
            action: 'lambda:InvokeFunction',
            principal: new iam.ServicePrincipal('ses.amazonaws.com'),
            sourceAccount: core_1.Aws.ACCOUNT_ID,
        });
        this.rule = new ReceiptRule(this, 'Rule', {
            actions: [
                {
                    bind: () => ({
                        lambdaAction: {
                            functionArn: fn.functionArn,
                            invocationType: 'RequestResponse',
                        },
                    }),
                },
            ],
            scanEnabled: true,
            ruleSet: props.ruleSet,
        });
    }
}
exports.DropSpamReceiptRule = DropSpamReceiptRule;
// Adapted from https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
/* eslint-disable no-console */
function dropSpamCode(event, _, callback) {
    console.log('Spam filter');
    const sesNotification = event.Records[0].ses;
    console.log('SES Notification:\n', JSON.stringify(sesNotification, null, 2));
    // Check if any spam check failed
    if (sesNotification.receipt.spfVerdict.status === 'FAIL'
        || sesNotification.receipt.dkimVerdict.status === 'FAIL'
        || sesNotification.receipt.spamVerdict.status === 'FAIL'
        || sesNotification.receipt.virusVerdict.status === 'FAIL') {
        console.log('Dropping spam');
        // Stop processing rule set, dropping message
        callback(null, { disposition: 'STOP_RULE_SET' });
    }
    else {
        callback(null, null);
    }
}
//# sourceMappingURL=data:application/json;base64,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