import os
from string import Template

import MySQLdb.cursors
import click
from dragonfly.db.database_migrator import DatabaseMigrator

from config import DATABASE, ROOT_DIR

os.chdir(ROOT_DIR + '/tests/builder')

templates = {}

templates['models'] = Template('''from dragonfly import models


class $name(models.Model):
    pass

''')

templates['middleware'] = Template('''
class $name:

    actions = []

    def before(self):
        pass

    def after(self):
        pass

''')

templates['controllers'] = Template('''

class $name:
    pass

''')

templates['user_controller'] = '''from dragonfly import Controller, view, RedirectResponse, Auth
from dragonfly.utils import Utils
from models.user import User
import hashlib
import os


class UserController(Controller):

    def login(self):
        return view('users.login')

    def register(self):
        return view('users.register')

    def store(self, request_data):

        del request_data['password_confirmation']

        salt = os.urandom(16)
        request_data['salt'] = salt.hex()
        request_data['password'] = hashlib.pbkdf2_hmac('sha256', request_data['password'].encode('utf-8'), salt, 100000,
                                                       dklen=64).hex()

        user = User().create(request_data)
        user.session_id = os.urandom(32).hex()
        user.save()

        response = RedirectResponse(Utils.url('articles'))
        response.header('Set-cookie', f"session_id={user.session_id}; SameSite=Strict;")

        return response

    def authenticate(self, request_data):
        user = User().where('username', '=', request_data['username']).first()

        if user:
            to_check = hashlib.pbkdf2_hmac('sha256', request_data['password'].encode('utf-8'), bytes.fromhex(user.salt),
                                           100000, dklen=64).hex()

            if to_check == user.password:
                user.session_id = os.urandom(32).hex()
                user.save()

                response = RedirectResponse(Utils.url('articles'))
                response.header('Set-cookie', f"session_id={user.session_id}; SameSite=Strict;")

                return response

        return RedirectResponse(Utils.url('login'))

    def deauthenticate(self, request_data):
        user = Auth.user()

        user.session_id = ''
        user.save()

        response = RedirectResponse(Utils.url('login'))
        response.header('Set-cockie', "session_id=''; SameSite=Strict;")

        return response

'''

templates['routes'] = '''
# Auth routes (generated by the 'auth' command)
routes.get('login', 'UserController@login')
routes.get('register', 'UserController@register')

routes.post('login', 'UserController@authenticate')
routes.post('register', 'UserController@store')
routes.post('logout', 'UserController@deauthenticate')
'''

templates['user_index'] = '''<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Login</title>
    <link rel="stylesheet" href="https://bootswatch.com/4/materia/bootstrap.min.css">
</head>
<body>
<div class="navbar navbar-expand-lg fixed-top navbar-dark bg-primary">
    <div class="container">
        <a href="{{ Utils.url('') }}" class="navbar-brand">Dragonfly</a>
    </div>
</div>
<div class="container mt-5 pt-5">
    <div class="row">
        <div class="col-lg-12">
            <div class="card border-secondary mb-3">
                <div class="card-header">Login</div>
                <div class="card-body">
                    <form method="POST" action="{{ Utils.url('login') }}">
                        <div class="form-group">
                            <label for="name">Username</label>
                            <input type="text" class="form-control" name="username">
                        </div>
                        <div class="form-group">
                            <label for="password">Password</label>
                            <input type="password" class="form-control" name="password"></input>
                        </div>
                        <button type="submit" class="btn btn-primary">Submit</button>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>
</body>
</html>'''

templates['user_register'] = '''<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Register</title>
    <link rel="stylesheet" href="https://bootswatch.com/4/materia/bootstrap.min.css">
</head>
<body>
<div class="navbar navbar-expand-lg fixed-top navbar-dark bg-primary">
    <div class="container">
        <a href="{{ Utils.url('') }}" class="navbar-brand">Dragonfly</a>
        <div class="collapse navbar-collapse" id="navbarResponsive">
        </div>
    </div>
</div>
<div class="container mt-5 pt-5">
    <div class="row">
        <div class="col-lg-12">
            <div class="card border-secondary mb-3">
                <div class="card-header">Register</div>
                <div class="card-body">
                    <form method="POST" action="{{ Utils.url('register') }}">
                        <div class="form-group">
                            <label for="name">Username</label>
                            <input type="text" class="form-control" name="username">
                        </div>
                        <div class="form-group">
                            <label for="name">Email</label>
                            <input type="email" class="form-control" name="email">
                        </div>
                        <div class="form-group">
                            <label for="password">Password</label>
                            <input type="password" class="form-control" name="password"></input>
                        </div>
                        <div class="form-group">
                            <label for="password_confirmation">Password Confirmation</label>
                            <input type="password" class="form-control" name="password_confirmation"></input>
                        </div>
                        <button type="submit" class="btn btn-primary">Submit</button>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>
</body>
</html>'''

templates['user'] = '''from dragonfly.db import models


class User(models.Model):
    username = models.VarCharField(null=False)
    email = models.VarCharField(length=50)
    password = models.VarCharField(length=128)
    salt = models.VarCharField(length=32)
    session_id = models.VarCharField(length=64, null=True)
'''

templates['session'] = '''from dragonfly.db import models


class Session(models.Model):
    session_id = models.VarCharField(length=64)
    name = models.VarCharField(length=50)
    value = models.VarCharField(length=50)
'''

templates['user_middleware'] = '''from dragonfly.request import request
from dragonfly.response import RedirectResponse
from dragonfly.utils import Utils
from config import NO_AUTH

from models.user import User


class UserMiddleware:
    actions = "*"

    def before(self):

        if request.path not in NO_AUTH:
            try:
                session_id = request.cookies['session_id']
                user = User().where('session_id', '=', session_id).first()
                if not user:
                    return RedirectResponse(Utils.url('login'))

            except KeyError:
                return RedirectResponse(Utils.url('login'))

    def after(self):
        pass
'''

templates['csrf_middleware'] = '''from dragonfly.constants import DATA_METHODS
from dragonfly.request import request
from dragonfly.response import ErrorResponse
from dragonfly.auth import Auth
from config import NO_AUTH

import os


class CsrfMiddleware:
    actions = '*'

    def before(self):
        # Determine if csrf_token for form request is valid
        if request.method in DATA_METHODS and request.path not in NO_AUTH:

            try:
                token = request.get_data()['csrf_token']
            except KeyError:
                return ErrorResponse('No CSRF token', 500)

            if token != Auth.get('csrf_token'):
                return ErrorResponse('CSRF invalid', 500)

        # Set a csrf_token for the form request
        elif request.path not in NO_AUTH:
            Auth.set('csrf_token', os.urandom(25).hex())

    def after(self):
        pass
'''


@click.group()
def cli():
    pass


@cli.command()
def setup():
    """
    Creates the required directories for the application to work.
    """
    os.makedirs('controllers', exist_ok=True)
    os.makedirs('models', exist_ok=True)
    os.makedirs('storage', exist_ok=True)
    os.makedirs('middleware', exist_ok=True)
    os.makedirs('templates', exist_ok=True)

    click.secho("Successfully created directories!", fg="green")


# generate commands
@cli.command()
@click.option('--type', type=click.Choice(['model', 'middleware', 'controller']),
              help="The type of file you want to generate.")
@click.argument('name', type=click.STRING, )
def generate(type, name):
    """
    Generate the desired file type. The files should be named according to the PEP 8 naming scheme (snake case). Please
    note that the file names are converted to camel case for the class names.
    """
    old_type = type
    if type != 'middleware':
        type += 's'

    with open(f"{type}/{name}.py", 'w') as file:
        file.writelines(templates[type].substitute(name=''.join(x.capitalize() or '_' for x in name.split('_'))))

    click.secho(f"Successfully created {old_type}!", fg="green")


@cli.command()
def migrate():
    """
    Generate the SQL to create the tables for all user created models and run it.
    """

    dbm = DatabaseMigrator()

    db = MySQLdb.connect(**DATABASE, cursorclass=MySQLdb.cursors.DictCursor)
    cursor = db.cursor()

    no_dependency = []
    to_append = []

    dependents = []

    for key, value in dbm.tables.items():
        if not value[0]:
            no_dependency.append(key)

    for key, value in dbm.tables.items():
        if value[0]:
            if set(value[0]).issubset(set(no_dependency)):
                to_append.append(key)

            else:
                dependents.append(key)

    no_dependency.extend(to_append)
    no_dependency.extend(dependents)

    for table in no_dependency:
        click.secho(f"Migrating {table} model", fg="blue")
        cursor.execute(dbm.tables[table][1])
        db.commit()
        click.secho(f"Migrated {table} successfully!", fg="green")

    cursor.close()
    db.close()


@cli.command()
def drop():
    dbm = DatabaseMigrator()

    db = MySQLdb.connect(**DATABASE, cursorclass=MySQLdb.cursors.DictCursor)
    cursor = db.cursor()

    no_dependency = []
    to_append = []

    dependents = []

    for key, value in dbm.tables.items():
        if not value[0]:
            no_dependency.append(key)

    for key, value in dbm.tables.items():
        if value[0]:
            if set(value[0]).issubset(set(no_dependency)):
                to_append.append(key)

            else:
                dependents.append(key)

    no_dependency.extend(to_append)
    no_dependency.extend(dependents)

    no_dependency.reverse()

    for table in no_dependency:
        click.secho(f"Deleting {table} model", fg="blue")
        cursor.execute(f"DROP TABLE {table}")
        db.commit()
        click.secho(f"Deleted {table} successfully!", fg="green")


@cli.command()
def seed():
    dbm = DatabaseMigrator()

    db = MySQLdb.connect(**DATABASE, cursorclass=MySQLdb.cursors.DictCursor)
    cursor = db.cursor()

    cursor.execute("INSERT INTO users (username, email, password, salt) VALUES ('test', 'test', 'test', 'test')")
    db.commit()

    for i in range(100):
        cursor.execute("INSERT INTO articles (name, text, user_id) VALUES ('Article', 'Article text', 1)")
    db.commit()


@cli.command()
def auth():
    """
    Generate the authentication scaffold for the given project.
    """
    with open('controllers/user_controller.py', 'w') as file:
        file.writelines(templates['user_controller'])

    with open('routes.py', 'a+') as file:
        file.writelines(templates['routes'])

    with open('templates/users/login.html', 'w') as file:
        file.writelines(templates['user_index'])

    with open('templates/users/register.html', 'w') as file:
        file.writelines(templates['user_register'])

    with open('models/user.py', 'w') as file:
        file.writelines(templates['user'])

    with open('models/session.py', 'w') as file:
        file.writelines(templates['session'])

    with open('middleware/user_middleware.py', 'w') as file:
        file.writelines(templates['user_middleware'])

    with open('middleware/csrf_middleware.py', 'w') as file:
        file.writelines(templates['csrf_middleware'])

    click.secho(f"Successfully generated auth scaffold!", fg="green")
    click.secho(
        f"Please add 'middleware.csrf_middleware' and 'middleware.user_middleware' to the MIDDLEWARE list in 'config.py'.",
        fg="yellow")
    click.secho(f"Please remigrate your tables.", fg="yellow")


if __name__ == '__main__':
    cli()
