"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyCodeConfig = exports.Function = exports.Tracing = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const aws_codeguruprofiler_1 = require("@aws-cdk/aws-codeguruprofiler");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const function_base_1 = require("./function-base");
const function_hash_1 = require("./function-hash");
const handler_1 = require("./handler");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const runtime_1 = require("./runtime");
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html).
 *
 * @stability stable
 */
var Tracing;
(function (Tracing) {
    Tracing["ACTIVE"] = "Active";
    Tracing["PASS_THROUGH"] = "PassThrough";
    Tracing["DISABLED"] = "Disabled";
})(Tracing = exports.Tracing || (exports.Tracing = {}));
/**
 * Deploys a file from from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 *
 * @stability stable
 */
class Function extends function_base_1.FunctionBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.functionName,
        });
        /**
         * The construct node where permissions are attached.
         *
         * @stability stable
         */
        this.permissionsNode = this.node;
        /**
         * Whether the addPermission() call adds any permissions.
         *
         * True for new Lambdas, false for version $LATEST and imported Lambdas
         * from different accounts.
         *
         * @stability stable
         */
        this.canCreatePermissions = true;
        this.layers = [];
        /**
         * Environment variables for this function
         */
        this.environment = {};
        const managedPolicies = new Array();
        // the arn is in the form of - arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
        managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies,
        });
        this.grantPrincipal = this.role;
        // add additonal managed policies when necessary
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.policies) {
                config.policies.forEach(p => {
                    var _b;
                    (_b = this.role) === null || _b === void 0 ? void 0 : _b.addToPolicy(p);
                });
            }
        }
        for (const statement of (props.initialPolicy || [])) {
            this.role.addToPolicy(statement);
        }
        const code = props.code.bind(this);
        verifyCodeConfig(code, props);
        let profilingGroupEnvironmentVariables = {};
        if (props.profilingGroup && props.profiling !== false) {
            this.validateProfiling(props);
            props.profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: core_1.Stack.of(scope).formatArn({
                    service: 'codeguru-profiler',
                    resource: 'profilingGroup',
                    resourceName: props.profilingGroup.profilingGroupName,
                }),
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        else if (props.profiling) {
            this.validateProfiling(props);
            const profilingGroup = new aws_codeguruprofiler_1.ProfilingGroup(this, 'ProfilingGroup', {
                computePlatform: aws_codeguruprofiler_1.ComputePlatform.AWS_LAMBDA,
            });
            profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: profilingGroup.profilingGroupArn,
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        const env = { ...profilingGroupEnvironmentVariables, ...props.environment };
        for (const [key, value] of Object.entries(env)) {
            this.addEnvironment(key, value);
        }
        this.deadLetterQueue = this.buildDeadLetterQueue(props);
        let fileSystemConfigs = undefined;
        if (props.filesystem) {
            fileSystemConfigs = [{
                    arn: props.filesystem.config.arn,
                    localMountPath: props.filesystem.config.localMountPath,
                }];
        }
        const resource = new lambda_generated_1.CfnFunction(this, 'Resource', {
            functionName: this.physicalName,
            description: props.description,
            code: {
                s3Bucket: code.s3Location && code.s3Location.bucketName,
                s3Key: code.s3Location && code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location && code.s3Location.objectVersion,
                zipFile: code.inlineCode,
                imageUri: (_b = code.image) === null || _b === void 0 ? void 0 : _b.imageUri,
            },
            layers: core_1.Lazy.list({ produce: () => this.layers.map(layer => layer.layerVersionArn) }, { omitEmpty: true }),
            handler: props.handler === handler_1.Handler.FROM_IMAGE ? undefined : props.handler,
            timeout: props.timeout && props.timeout.toSeconds(),
            packageType: props.runtime === runtime_1.Runtime.FROM_IMAGE ? 'Image' : undefined,
            runtime: props.runtime === runtime_1.Runtime.FROM_IMAGE ? undefined : props.runtime.name,
            role: this.role.roleArn,
            // Uncached because calling '_checkEdgeCompatibility', which gets called in the resolve of another
            // Token, actually *modifies* the 'environment' map.
            environment: core_1.Lazy.uncachedAny({ produce: () => this.renderEnvironment() }),
            memorySize: props.memorySize,
            vpcConfig: this.configureVpc(props),
            deadLetterConfig: this.buildDeadLetterConfig(this.deadLetterQueue),
            tracingConfig: this.buildTracingConfig(props),
            reservedConcurrentExecutions: props.reservedConcurrentExecutions,
            imageConfig: undefinedIfNoKeys({
                command: (_c = code.image) === null || _c === void 0 ? void 0 : _c.cmd,
                entryPoint: (_d = code.image) === null || _d === void 0 ? void 0 : _d.entrypoint,
            }),
            kmsKeyArn: (_e = props.environmentEncryption) === null || _e === void 0 ? void 0 : _e.keyArn,
            fileSystemConfigs,
        });
        resource.node.addDependency(this.role);
        this.functionName = this.getResourceNameAttribute(resource.ref);
        this.functionArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'lambda',
            resource: 'function',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.runtime = props.runtime;
        if (props.layers) {
            this.addLayers(...props.layers);
        }
        for (const event of props.events || []) {
            this.addEventSource(event);
        }
        // Log retention
        if (props.logRetention) {
            const logRetention = new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, 'LogGroup', logRetention.logGroupArn);
        }
        props.code.bindToResource(resource);
        // Event Invoke Config
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        this.currentVersionOptions = props.currentVersionOptions;
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.dependency) {
                this.node.addDependency(...config.dependency);
            }
        }
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this Lambda function. A new version will be created every time the function's configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.Function`.
     *
     * @stability stable
     */
    get currentVersion() {
        if (this._currentVersion) {
            return this._currentVersion;
        }
        this._currentVersion = new lambda_version_1.Version(this, 'CurrentVersion', {
            lambda: this,
            ...this.currentVersionOptions,
        });
        // override the version's logical ID with a lazy string which includes the
        // hash of the function itself, so a new version resource is created when
        // the function configuration changes.
        const cfn = this._currentVersion.node.defaultChild;
        const originalLogicalId = this.stack.resolve(cfn.logicalId);
        cfn.overrideLogicalId(core_1.Lazy.uncachedString({
            produce: () => {
                const hash = function_hash_1.calculateFunctionHash(this);
                const logicalId = function_hash_1.trimFromStart(originalLogicalId, 255 - 32);
                return `${logicalId}${hash}`;
            },
        }));
        return this._currentVersion;
    }
    /**
     * @stability stable
     */
    static fromFunctionArn(scope, id, functionArn) {
        return Function.fromFunctionAttributes(scope, id, { functionArn });
    }
    /**
     * Creates a Lambda function object which represents a function not defined within this stack.
     *
     * @param scope The parent construct.
     * @param id The name of the lambda construct.
     * @param attrs the attributes of the function to import.
     * @stability stable
     */
    static fromFunctionAttributes(scope, id, attrs) {
        const functionArn = attrs.functionArn;
        const functionName = extractNameFromArn(attrs.functionArn);
        const role = attrs.role;
        class Import extends function_base_1.FunctionBase {
            constructor(s, i) {
                var _b;
                super(s, i);
                this.functionName = functionName;
                this.functionArn = functionArn;
                this.role = role;
                this.permissionsNode = this.node;
                this.canCreatePermissions = (_b = attrs.sameEnvironment) !== null && _b !== void 0 ? _b : this._isStackAccount();
                this.grantPrincipal = role || new iam.UnknownPrincipal({ resource: this });
                if (attrs.securityGroup) {
                    this._connections = new ec2.Connections({
                        securityGroups: [attrs.securityGroup],
                    });
                }
                else if (attrs.securityGroupId) {
                    this._connections = new ec2.Connections({
                        securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)],
                    });
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for this Lambda.
     *
     * @stability stable
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Lambda',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the number of Errors executing all Lambdas.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllErrors(props) {
        return this.metricAll('Errors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the Duration executing all Lambdas.
     *
     * @default average over 5 minutes
     * @stability stable
     */
    static metricAllDuration(props) {
        return this.metricAll('Duration', props);
    }
    /**
     * Metric for the number of invocations of all Lambdas.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllInvocations(props) {
        return this.metricAll('Invocations', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of throttled invocations of all Lambdas.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllThrottles(props) {
        return this.metricAll('Throttles', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of concurrent executions across all Lambdas.
     *
     * @default max over 5 minutes
     * @stability stable
     */
    static metricAllConcurrentExecutions(props) {
        // Mini-FAQ: why max? This metric is a gauge that is emitted every
        // minute, so either max or avg or a percentile make sense (but sum
        // doesn't). Max is more sensitive to spiky load changes which is
        // probably what you're interested in if you're looking at this metric
        // (Load spikes may lead to concurrent execution errors that would
        // otherwise not be visible in the avg)
        return this.metricAll('ConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas.
     *
     * @default max over 5 minutes
     * @stability stable
     */
    static metricAllUnreservedConcurrentExecutions(props) {
        return this.metricAll('UnreservedConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Adds an environment variable to this Lambda function.
     *
     * If this is a ref to a Lambda function, this operation results in a no-op.
     *
     * @param key The environment variable key.
     * @param value The environment variable's value.
     * @param options Environment variable options.
     * @stability stable
     */
    addEnvironment(key, value, options) {
        this.environment[key] = { value, ...options };
        return this;
    }
    /**
     * Adds one or more Lambda Layers to this Lambda function.
     *
     * @param layers the layers to be added.
     * @stability stable
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayers(...layers) {
        for (const layer of layers) {
            if (this.layers.length === 5) {
                throw new Error('Unable to add layer: this lambda function already uses 5 layers.');
            }
            if (layer.compatibleRuntimes && !layer.compatibleRuntimes.find(runtime => runtime.runtimeEquals(this.runtime))) {
                const runtimes = layer.compatibleRuntimes.map(runtime => runtime.name).join(', ');
                throw new Error(`This lambda function uses a runtime that is incompatible with this layer (${this.runtime.name} is not in [${runtimes}])`);
            }
            this.layers.push(layer);
        }
    }
    /**
     * (deprecated) Add a new version for this Lambda.
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want to
     * deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version.
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda source code, or omit to skip validation.
     * @param description A description for this version.
     * @param provisionedExecutions A provisioned concurrency configuration for a function's version.
     * @param asyncInvokeConfig configuration for this version when it is invoked asynchronously.
     * @returns A new Version object.
     * @deprecated This method will create an AWS::Lambda::Version resource which
     * snapshots the AWS Lambda function *at the time of its creation* and it
     * won't get updated when the function changes. Instead, use
     * `this.currentVersion` to obtain a reference to a version resource that gets
     * automatically recreated when the function configuration (or code) changes.
     */
    addVersion(name, codeSha256, description, provisionedExecutions, asyncInvokeConfig = {}) {
        return new lambda_version_1.Version(this, 'Version' + name, {
            lambda: this,
            codeSha256,
            description,
            provisionedConcurrentExecutions: provisionedExecutions,
            ...asyncInvokeConfig,
        });
    }
    /**
     * The LogGroup where the Lambda function's logs are made available.
     *
     * If either `logRetention` is set or this property is called, a CloudFormation custom resource is added to the stack that
     * pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the correct log retention
     * period (never expire, by default).
     *
     * Further, if the log group already exists and the `logRetention` is not set, the custom resource will reset the log retention
     * to never expire even if it was configured with a different value.
     *
     * @stability stable
     */
    get logGroup() {
        if (!this._logGroup) {
            const logRetention = new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: logs.RetentionDays.INFINITE,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, `${this.node.id}-LogGroup`, logRetention.logGroupArn);
        }
        return this._logGroup;
    }
    /** @internal */
    _checkEdgeCompatibility() {
        // Check env vars
        const envEntries = Object.entries(this.environment);
        for (const [key, config] of envEntries) {
            if (config.removeInEdge) {
                delete this.environment[key];
                core_1.Annotations.of(this).addInfo(`Removed ${key} environment variable for Lambda@Edge compatibility`);
            }
        }
        const envKeys = Object.keys(this.environment);
        if (envKeys.length !== 0) {
            throw new Error(`The function ${this.node.path} contains environment variables [${envKeys}] and is not compatible with Lambda@Edge. \
Environment variables can be marked for removal when used in Lambda@Edge by setting the \'removeInEdge\' property in the \'addEnvironment()\' API.`);
        }
        return;
    }
    renderEnvironment() {
        if (!this.environment || Object.keys(this.environment).length === 0) {
            return undefined;
        }
        const variables = {};
        // Sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it). For backwards compatibility we do not sort environment variables in case
        // _currentVersion is not defined. Otherwise, this would have invalidated
        // the template, and for example, may cause unneeded updates for nested
        // stacks.
        const keys = this._currentVersion
            ? Object.keys(this.environment).sort()
            : Object.keys(this.environment);
        for (const key of keys) {
            variables[key] = this.environment[key].value;
        }
        return { variables };
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    configureVpc(props) {
        var _b;
        if ((props.securityGroup || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if (props.securityGroup && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroup && props.securityGroups) {
            throw new Error('Only one of the function props, securityGroup or securityGroups, is allowed');
        }
        if (props.securityGroups) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Lambda Function ' + core_1.Names.uniqueId(this),
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        if (props.filesystem) {
            if (props.filesystem.config.connections) {
                props.filesystem.config.connections.allowDefaultPortFrom(this);
            }
        }
        const allowPublicSubnet = (_b = props.allowPublicSubnet) !== null && _b !== void 0 ? _b : false;
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        const publicSubnetIds = new Set(props.vpc.publicSubnets.map(s => s.subnetId));
        for (const subnetId of subnetIds) {
            if (publicSubnetIds.has(subnetId) && !allowPublicSubnet) {
                throw new Error('Lambda Functions in a public subnet can NOT access the internet. ' +
                    'If you are aware of this limitation and would still like to place the function int a public subnet, set `allowPublicSubnet` to true');
            }
        }
        // List can't be empty here, if we got this far you intended to put your Lambda
        // in subnets. We're going to guarantee that we get the nice error message by
        // making VpcNetwork do the selection again.
        return {
            subnetIds,
            securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
        };
    }
    buildDeadLetterQueue(props) {
        if (props.deadLetterQueue && props.deadLetterQueueEnabled === false) {
            throw Error('deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false');
        }
        if (!props.deadLetterQueue && !props.deadLetterQueueEnabled) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue || new sqs.Queue(this, 'DeadLetterQueue', {
            retentionPeriod: core_1.Duration.days(14),
        });
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sqs:SendMessage'],
            resources: [deadLetterQueue.queueArn],
        }));
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                targetArn: deadLetterQueue.queueArn,
            };
        }
        else {
            return undefined;
        }
    }
    buildTracingConfig(props) {
        if (props.tracing === undefined || props.tracing === Tracing.DISABLED) {
            return undefined;
        }
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords'],
            resources: ['*'],
        }));
        return {
            mode: props.tracing,
        };
    }
    validateProfiling(props) {
        if (!props.runtime.supportsCodeGuruProfiling) {
            throw new Error(`CodeGuru profiling is not supported by runtime ${props.runtime.name}`);
        }
        if (props.environment && (props.environment.AWS_CODEGURU_PROFILER_GROUP_ARN || props.environment.AWS_CODEGURU_PROFILER_ENABLED)) {
            throw new Error('AWS_CODEGURU_PROFILER_GROUP_ARN and AWS_CODEGURU_PROFILER_ENABLED must not be set when profiling options enabled');
        }
    }
}
exports.Function = Function;
_a = JSII_RTTI_SYMBOL_1;
Function[_a] = { fqn: "@aws-cdk/aws-lambda.Function", version: "1.90.0" };
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the function
 * name from the ARN.
 *
 * Function ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name
 *
 * ..which means that in order to extract the `function-name` component from the ARN, we can
 * split the ARN using ":" and select the component in index 6.
 *
 * @returns `FnSelect(6, FnSplit(':', arn))`
 */
function extractNameFromArn(arn) {
    return core_1.Fn.select(6, core_1.Fn.split(':', arn));
}
function verifyCodeConfig(code, props) {
    // mutually exclusive
    const codeType = [code.inlineCode, code.s3Location, code.image];
    if (codeType.filter(x => !!x).length !== 1) {
        throw new Error('lambda.Code must specify exactly one of: "inlineCode", "s3Location", or "image"');
    }
    if (!!code.image === (props.handler !== handler_1.Handler.FROM_IMAGE)) {
        throw new Error('handler must be `Handler.FROM_IMAGE` when using image asset for Lambda function');
    }
    if (!!code.image === (props.runtime !== runtime_1.Runtime.FROM_IMAGE)) {
        throw new Error('runtime must be `Runtime.FROM_IMAGE` when using image asset for Lambda function');
    }
    // if this is inline code, check that the runtime supports
    if (code.inlineCode && !props.runtime.supportsInlineCode) {
        throw new Error(`Inline source not allowed for ${props.runtime.name}`);
    }
}
exports.verifyCodeConfig = verifyCodeConfig;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).every(val => val === undefined);
    return allUndefined ? undefined : struct;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZnVuY3Rpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJmdW5jdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLHNEQUFzRDtBQUN0RCx3RUFBaUc7QUFDakcsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUV4QywwQ0FBMEM7QUFDMUMsd0NBQXdDO0FBQ3hDLHdDQUEyRjtBQU0zRixtREFBOEU7QUFDOUUsbURBQXVFO0FBQ3ZFLHVDQUFvQztBQUNwQyxxREFBMkQ7QUFDM0QseURBQWlEO0FBR2pELHVDQUFvQzs7Ozs7O0FBS3BDLElBQVksT0FlWDtBQWZELFdBQVksT0FBTztJQUtqQiw0QkFBaUIsQ0FBQTtJQUtqQix1Q0FBNEIsQ0FBQTtJQUk1QixnQ0FBcUIsQ0FBQTtBQUN2QixDQUFDLEVBZlcsT0FBTyxHQUFQLGVBQU8sS0FBUCxlQUFPLFFBZWxCOzs7Ozs7Ozs7Ozs7OztBQXlTRCxNQUFhLFFBQVMsU0FBUSw0QkFBWTs7OztJQXVNeEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFvQjs7UUFDNUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVk7U0FDakMsQ0FBQyxDQUFDOzs7Ozs7UUFuQlcsb0JBQWUsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDOzs7Ozs7Ozs7UUFFekIseUJBQW9CLEdBQUcsSUFBSSxDQUFDO1FBRTlCLFdBQU0sR0FBb0IsRUFBRSxDQUFDO1FBSTlDOztXQUVHO1FBQ0ssZ0JBQVcsR0FBeUMsRUFBRSxDQUFDO1FBVTdELE1BQU0sZUFBZSxHQUFHLElBQUksS0FBSyxFQUFzQixDQUFDO1FBRXhELCtGQUErRjtRQUMvRixlQUFlLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsMENBQTBDLENBQUMsQ0FBQyxDQUFDO1FBRTdHLElBQUksS0FBSyxDQUFDLEdBQUcsRUFBRTtZQUNiLGlEQUFpRDtZQUNqRCxlQUFlLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsOENBQThDLENBQUMsQ0FBQyxDQUFDO1NBQ2xIO1FBRUQsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUMsSUFBSSxJQUFJLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQzFELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQztZQUMzRCxlQUFlO1NBQ2hCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQztRQUVoQyxnREFBZ0Q7UUFDaEQsSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFO1lBQ3BCLE1BQU0sTUFBTSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDO1lBQ3ZDLElBQUksTUFBTSxDQUFDLFFBQVEsRUFBRTtnQkFDbkIsTUFBTSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUU7O29CQUMxQixNQUFBLElBQUksQ0FBQyxJQUFJLDBDQUFFLFdBQVcsQ0FBQyxDQUFDLEVBQUU7Z0JBQzVCLENBQUMsQ0FBQyxDQUFDO2FBQ0o7U0FDRjtRQUVELEtBQUssTUFBTSxTQUFTLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxJQUFJLEVBQUUsQ0FBQyxFQUFFO1lBQ25ELElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1NBQ2xDO1FBRUQsTUFBTSxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDbkMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRTlCLElBQUksa0NBQWtDLEdBQThCLEVBQUUsQ0FBQztRQUN2RSxJQUFJLEtBQUssQ0FBQyxjQUFjLElBQUksS0FBSyxDQUFDLFNBQVMsS0FBSyxLQUFLLEVBQUU7WUFDckQsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQzlCLEtBQUssQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUM3QyxrQ0FBa0MsR0FBRztnQkFDbkMsK0JBQStCLEVBQUUsWUFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxTQUFTLENBQUM7b0JBQ3pELE9BQU8sRUFBRSxtQkFBbUI7b0JBQzVCLFFBQVEsRUFBRSxnQkFBZ0I7b0JBQzFCLFlBQVksRUFBRSxLQUFLLENBQUMsY0FBYyxDQUFDLGtCQUFrQjtpQkFDdEQsQ0FBQztnQkFDRiw2QkFBNkIsRUFBRSxNQUFNO2FBQ3RDLENBQUM7U0FDSDthQUFNLElBQUksS0FBSyxDQUFDLFNBQVMsRUFBRTtZQUMxQixJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDOUIsTUFBTSxjQUFjLEdBQUcsSUFBSSxxQ0FBYyxDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRTtnQkFDaEUsZUFBZSxFQUFFLHNDQUFlLENBQUMsVUFBVTthQUM1QyxDQUFDLENBQUM7WUFDSCxjQUFjLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUN2QyxrQ0FBa0MsR0FBRztnQkFDbkMsK0JBQStCLEVBQUUsY0FBYyxDQUFDLGlCQUFpQjtnQkFDakUsNkJBQTZCLEVBQUUsTUFBTTthQUN0QyxDQUFDO1NBQ0g7UUFFRCxNQUFNLEdBQUcsR0FBRyxFQUFFLEdBQUcsa0NBQWtDLEVBQUUsR0FBRyxLQUFLLENBQUMsV0FBVyxFQUFFLENBQUM7UUFDNUUsS0FBSyxNQUFNLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUU7WUFDOUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUM7U0FDakM7UUFFRCxJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUV4RCxJQUFJLGlCQUFpQixHQUF1RCxTQUFTLENBQUM7UUFDdEYsSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFO1lBQ3BCLGlCQUFpQixHQUFHLENBQUM7b0JBQ25CLEdBQUcsRUFBRSxLQUFLLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxHQUFHO29CQUNoQyxjQUFjLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsY0FBYztpQkFDdkQsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxNQUFNLFFBQVEsR0FBZ0IsSUFBSSw4QkFBVyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDOUQsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQy9CLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztZQUM5QixJQUFJLEVBQUU7Z0JBQ0osUUFBUSxFQUFFLElBQUksQ0FBQyxVQUFVLElBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQyxVQUFVO2dCQUN2RCxLQUFLLEVBQUUsSUFBSSxDQUFDLFVBQVUsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLFNBQVM7Z0JBQ25ELGVBQWUsRUFBRSxJQUFJLENBQUMsVUFBVSxJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYTtnQkFDakUsT0FBTyxFQUFFLElBQUksQ0FBQyxVQUFVO2dCQUN4QixRQUFRLFFBQUUsSUFBSSxDQUFDLEtBQUssMENBQUUsUUFBUTthQUMvQjtZQUNELE1BQU0sRUFBRSxXQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUM7WUFDMUcsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPLEtBQUssaUJBQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLE9BQU87WUFDekUsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUU7WUFDbkQsV0FBVyxFQUFFLEtBQUssQ0FBQyxPQUFPLEtBQUssaUJBQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsU0FBUztZQUN2RSxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU8sS0FBSyxpQkFBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUk7WUFDOUUsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTztZQUN2QixrR0FBa0c7WUFDbEcsb0RBQW9EO1lBQ3BELFdBQVcsRUFBRSxXQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxFQUFFLENBQUM7WUFDMUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO1lBQzVCLFNBQVMsRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQztZQUNuQyxnQkFBZ0IsRUFBRSxJQUFJLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQztZQUNsRSxhQUFhLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQztZQUM3Qyw0QkFBNEIsRUFBRSxLQUFLLENBQUMsNEJBQTRCO1lBQ2hFLFdBQVcsRUFBRSxpQkFBaUIsQ0FBQztnQkFDN0IsT0FBTyxRQUFFLElBQUksQ0FBQyxLQUFLLDBDQUFFLEdBQUc7Z0JBQ3hCLFVBQVUsUUFBRSxJQUFJLENBQUMsS0FBSywwQ0FBRSxVQUFVO2FBQ25DLENBQUM7WUFDRixTQUFTLFFBQUUsS0FBSyxDQUFDLHFCQUFxQiwwQ0FBRSxNQUFNO1lBQzlDLGlCQUFpQjtTQUNsQixDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFdkMsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2hFLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUU7WUFDaEUsT0FBTyxFQUFFLFFBQVE7WUFDakIsUUFBUSxFQUFFLFVBQVU7WUFDcEIsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQy9CLEdBQUcsRUFBRSxHQUFHO1NBQ1QsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBRTdCLElBQUksS0FBSyxDQUFDLE1BQU0sRUFBRTtZQUNoQixJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ2pDO1FBRUQsS0FBSyxNQUFNLEtBQUssSUFBSSxLQUFLLENBQUMsTUFBTSxJQUFJLEVBQUUsRUFBRTtZQUN0QyxJQUFJLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQzVCO1FBRUQsZ0JBQWdCO1FBQ2hCLElBQUksS0FBSyxDQUFDLFlBQVksRUFBRTtZQUN0QixNQUFNLFlBQVksR0FBRyxJQUFJLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtnQkFDL0QsWUFBWSxFQUFFLGVBQWUsSUFBSSxDQUFDLFlBQVksRUFBRTtnQkFDaEQsU0FBUyxFQUFFLEtBQUssQ0FBQyxZQUFZO2dCQUM3QixJQUFJLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtnQkFDNUIsd0JBQXdCLEVBQUUsS0FBSyxDQUFDLHdCQUF5RDthQUMxRixDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsZUFBZSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUUsWUFBWSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1NBQzVGO1FBRUQsS0FBSyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFcEMsc0JBQXNCO1FBQ3RCLElBQUksS0FBSyxDQUFDLFNBQVMsSUFBSSxLQUFLLENBQUMsU0FBUyxJQUFJLEtBQUssQ0FBQyxXQUFXLElBQUksS0FBSyxDQUFDLGFBQWEsS0FBSyxTQUFTLEVBQUU7WUFDaEcsSUFBSSxDQUFDLG9CQUFvQixDQUFDO2dCQUN4QixTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7Z0JBQzFCLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztnQkFDMUIsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO2dCQUM5QixhQUFhLEVBQUUsS0FBSyxDQUFDLGFBQWE7YUFDbkMsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxJQUFJLENBQUMscUJBQXFCLEdBQUcsS0FBSyxDQUFDLHFCQUFxQixDQUFDO1FBRXpELElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRTtZQUNwQixNQUFNLE1BQU0sR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUN2QyxJQUFJLE1BQU0sQ0FBQyxVQUFVLEVBQUU7Z0JBQ3JCLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFDO2FBQy9DO1NBQ0Y7SUFDSCxDQUFDOzs7Ozs7Ozs7SUE3VkQsSUFBVyxjQUFjO1FBQ3ZCLElBQUksSUFBSSxDQUFDLGVBQWUsRUFBRTtZQUN4QixPQUFPLElBQUksQ0FBQyxlQUFlLENBQUM7U0FDN0I7UUFFRCxJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksd0JBQU8sQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUU7WUFDekQsTUFBTSxFQUFFLElBQUk7WUFDWixHQUFHLElBQUksQ0FBQyxxQkFBcUI7U0FDOUIsQ0FBQyxDQUFDO1FBRUgsMEVBQTBFO1FBQzFFLHlFQUF5RTtRQUN6RSxzQ0FBc0M7UUFDdEMsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsWUFBMkIsQ0FBQztRQUNsRSxNQUFNLGlCQUFpQixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQVcsQ0FBQztRQUV0RSxHQUFHLENBQUMsaUJBQWlCLENBQUMsV0FBSSxDQUFDLGNBQWMsQ0FBQztZQUN4QyxPQUFPLEVBQUUsR0FBRyxFQUFFO2dCQUNaLE1BQU0sSUFBSSxHQUFHLHFDQUFxQixDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUN6QyxNQUFNLFNBQVMsR0FBRyw2QkFBYSxDQUFDLGlCQUFpQixFQUFFLEdBQUcsR0FBRyxFQUFFLENBQUMsQ0FBQztnQkFDN0QsT0FBTyxHQUFHLFNBQVMsR0FBRyxJQUFJLEVBQUUsQ0FBQztZQUMvQixDQUFDO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixPQUFPLElBQUksQ0FBQyxlQUFlLENBQUM7SUFDOUIsQ0FBQzs7OztJQUVNLE1BQU0sQ0FBQyxlQUFlLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsV0FBbUI7UUFDN0UsT0FBTyxRQUFRLENBQUMsc0JBQXNCLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxFQUFFLFdBQVcsRUFBRSxDQUFDLENBQUM7SUFDckUsQ0FBQzs7Ozs7Ozs7O0lBVU0sTUFBTSxDQUFDLHNCQUFzQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXlCO1FBQzFGLE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7UUFDdEMsTUFBTSxZQUFZLEdBQUcsa0JBQWtCLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQzNELE1BQU0sSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFFeEIsTUFBTSxNQUFPLFNBQVEsNEJBQVk7WUFTL0IsWUFBWSxDQUFZLEVBQUUsQ0FBUzs7Z0JBQ2pDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7Z0JBVEUsaUJBQVksR0FBRyxZQUFZLENBQUM7Z0JBQzVCLGdCQUFXLEdBQUcsV0FBVyxDQUFDO2dCQUUxQixTQUFJLEdBQUcsSUFBSSxDQUFDO2dCQUNaLG9CQUFlLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQztnQkFFekIseUJBQW9CLFNBQUcsS0FBSyxDQUFDLGVBQWUsbUNBQUksSUFBSSxDQUFDLGVBQWUsRUFBRSxDQUFDO2dCQUt4RixJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksSUFBSSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO2dCQUUzRSxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7b0JBQ3ZCLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxHQUFHLENBQUMsV0FBVyxDQUFDO3dCQUN0QyxjQUFjLEVBQUUsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDO3FCQUN0QyxDQUFDLENBQUM7aUJBQ0o7cUJBQU0sSUFBSSxLQUFLLENBQUMsZUFBZSxFQUFFO29CQUNoQyxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksR0FBRyxDQUFDLFdBQVcsQ0FBQzt3QkFDdEMsY0FBYyxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFLEtBQUssQ0FBQyxlQUFlLENBQUMsQ0FBQztxQkFDdkcsQ0FBQyxDQUFDO2lCQUNKO1lBQ0gsQ0FBQztTQUNGO1FBRUQsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDL0IsQ0FBQzs7Ozs7O0lBS00sTUFBTSxDQUFDLFNBQVMsQ0FBQyxVQUFrQixFQUFFLEtBQWdDO1FBQzFFLE9BQU8sSUFBSSxVQUFVLENBQUMsTUFBTSxDQUFDO1lBQzNCLFNBQVMsRUFBRSxZQUFZO1lBQ3ZCLFVBQVU7WUFDVixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDOzs7Ozs7O0lBTU0sTUFBTSxDQUFDLGVBQWUsQ0FBQyxLQUFnQztRQUM1RCxPQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxHQUFHLEtBQUssRUFBRSxDQUFDLENBQUM7SUFDbEUsQ0FBQzs7Ozs7OztJQU9NLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxLQUFnQztRQUM5RCxPQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzNDLENBQUM7Ozs7Ozs7SUFPTSxNQUFNLENBQUMsb0JBQW9CLENBQUMsS0FBZ0M7UUFDakUsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLGFBQWEsRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZFLENBQUM7Ozs7Ozs7SUFPTSxNQUFNLENBQUMsa0JBQWtCLENBQUMsS0FBZ0M7UUFDL0QsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBQ3JFLENBQUM7Ozs7Ozs7SUFPTSxNQUFNLENBQUMsNkJBQTZCLENBQUMsS0FBZ0M7UUFDMUUsa0VBQWtFO1FBQ2xFLG1FQUFtRTtRQUNuRSxpRUFBaUU7UUFDakUsc0VBQXNFO1FBQ3RFLGtFQUFrRTtRQUNsRSx1Q0FBdUM7UUFDdkMsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLHNCQUFzQixFQUFFLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxHQUFHLEtBQUssRUFBRSxDQUFDLENBQUM7SUFDaEYsQ0FBQzs7Ozs7OztJQU9NLE1BQU0sQ0FBQyx1Q0FBdUMsQ0FBQyxLQUFnQztRQUNwRixPQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsZ0NBQWdDLEVBQUUsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLEdBQUcsS0FBSyxFQUFFLENBQUMsQ0FBQztJQUMxRixDQUFDOzs7Ozs7Ozs7OztJQXlOTSxjQUFjLENBQUMsR0FBVyxFQUFFLEtBQWEsRUFBRSxPQUE0QjtRQUM1RSxJQUFJLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLEdBQUcsT0FBTyxFQUFFLENBQUM7UUFDOUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDOzs7Ozs7OztJQVNNLFNBQVMsQ0FBQyxHQUFHLE1BQXVCO1FBQ3pDLEtBQUssTUFBTSxLQUFLLElBQUksTUFBTSxFQUFFO1lBQzFCLElBQUksSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO2dCQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLGtFQUFrRSxDQUFDLENBQUM7YUFDckY7WUFDRCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxFQUFFO2dCQUM5RyxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDbEYsTUFBTSxJQUFJLEtBQUssQ0FBQyw2RUFBNkUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLGVBQWUsUUFBUSxJQUFJLENBQUMsQ0FBQzthQUM1STtZQUNELElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQ3pCO0lBQ0gsQ0FBQzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7SUE0Qk0sVUFBVSxDQUNmLElBQVksRUFDWixVQUFtQixFQUNuQixXQUFvQixFQUNwQixxQkFBOEIsRUFDOUIsb0JBQThDLEVBQUU7UUFFaEQsT0FBTyxJQUFJLHdCQUFPLENBQUMsSUFBSSxFQUFFLFNBQVMsR0FBRyxJQUFJLEVBQUU7WUFDekMsTUFBTSxFQUFFLElBQUk7WUFDWixVQUFVO1lBQ1YsV0FBVztZQUNYLCtCQUErQixFQUFFLHFCQUFxQjtZQUN0RCxHQUFHLGlCQUFpQjtTQUNyQixDQUFDLENBQUM7SUFDTCxDQUFDOzs7Ozs7Ozs7Ozs7O0lBWUQsSUFBVyxRQUFRO1FBQ2pCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFO1lBQ25CLE1BQU0sWUFBWSxHQUFHLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO2dCQUMvRCxZQUFZLEVBQUUsZUFBZSxJQUFJLENBQUMsWUFBWSxFQUFFO2dCQUNoRCxTQUFTLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxRQUFRO2FBQ3ZDLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxlQUFlLENBQUMsSUFBSSxFQUFFLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLFdBQVcsRUFBRSxZQUFZLENBQUMsV0FBVyxDQUFDLENBQUM7U0FDNUc7UUFDRCxPQUFPLElBQUksQ0FBQyxTQUFTLENBQUM7SUFDeEIsQ0FBQztJQUVELGdCQUFnQjtJQUNULHVCQUF1QjtRQUM1QixpQkFBaUI7UUFDakIsTUFBTSxVQUFVLEdBQUcsTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDcEQsS0FBSyxNQUFNLENBQUMsR0FBRyxFQUFFLE1BQU0sQ0FBQyxJQUFJLFVBQVUsRUFBRTtZQUN0QyxJQUFJLE1BQU0sQ0FBQyxZQUFZLEVBQUU7Z0JBQ3ZCLE9BQU8sSUFBSSxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDN0Isa0JBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLFdBQVcsR0FBRyxxREFBcUQsQ0FBQyxDQUFDO2FBQ25HO1NBQ0Y7UUFDRCxNQUFNLE9BQU8sR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUM5QyxJQUFJLE9BQU8sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0JBQWdCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxvQ0FBb0MsT0FBTzttSkFDb0QsQ0FBQyxDQUFDO1NBQ2hKO1FBRUQsT0FBTztJQUNULENBQUM7SUFFTyxpQkFBaUI7UUFDdkIsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNuRSxPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUVELE1BQU0sU0FBUyxHQUE4QixFQUFFLENBQUM7UUFDaEQsOERBQThEO1FBQzlELHlFQUF5RTtRQUN6RSxnRkFBZ0Y7UUFDaEYseUVBQXlFO1FBQ3pFLHVFQUF1RTtRQUN2RSxVQUFVO1FBQ1YsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGVBQWU7WUFDL0IsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDLElBQUksRUFBRTtZQUN0QyxDQUFDLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7UUFFbEMsS0FBSyxNQUFNLEdBQUcsSUFBSSxJQUFJLEVBQUU7WUFDdEIsU0FBUyxDQUFDLEdBQUcsQ0FBQyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBSyxDQUFDO1NBQzlDO1FBRUQsT0FBTyxFQUFFLFNBQVMsRUFBRSxDQUFDO0lBQ3ZCLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNLLFlBQVksQ0FBQyxLQUFvQjs7UUFDdkMsSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLElBQUksS0FBSyxDQUFDLGdCQUFnQixLQUFLLFNBQVMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRTtZQUMvRSxNQUFNLElBQUksS0FBSyxDQUFDLHNGQUFzRixDQUFDLENBQUM7U0FDekc7UUFFRCxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRTtZQUFFLE9BQU8sU0FBUyxDQUFDO1NBQUU7UUFFckMsSUFBSSxLQUFLLENBQUMsYUFBYSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsS0FBSyxTQUFTLEVBQUU7WUFDL0QsTUFBTSxJQUFJLEtBQUssQ0FBQyx3RUFBd0UsQ0FBQyxDQUFDO1NBQzNGO1FBRUQsSUFBSSxjQUFvQyxDQUFDO1FBRXpDLElBQUksS0FBSyxDQUFDLGFBQWEsSUFBSSxLQUFLLENBQUMsY0FBYyxFQUFFO1lBQy9DLE1BQU0sSUFBSSxLQUFLLENBQUMsNkVBQTZFLENBQUMsQ0FBQztTQUNoRztRQUVELElBQUksS0FBSyxDQUFDLGNBQWMsRUFBRTtZQUN4QixjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQztTQUN2QzthQUFNO1lBQ0wsTUFBTSxhQUFhLEdBQUcsS0FBSyxDQUFDLGFBQWEsSUFBSSxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLGVBQWUsRUFBRTtnQkFDeEYsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO2dCQUNkLFdBQVcsRUFBRSwrQ0FBK0MsR0FBRyxZQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQztnQkFDbkYsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjthQUN6QyxDQUFDLENBQUM7WUFDSCxjQUFjLEdBQUcsQ0FBQyxhQUFhLENBQUMsQ0FBQztTQUNsQztRQUVELElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxHQUFHLENBQUMsV0FBVyxDQUFDLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUU1RCxJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7WUFDcEIsSUFBSSxLQUFLLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxXQUFXLEVBQUU7Z0JBQ3ZDLEtBQUssQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUNoRTtTQUNGO1FBRUQsTUFBTSxpQkFBaUIsU0FBRyxLQUFLLENBQUMsaUJBQWlCLG1DQUFJLEtBQUssQ0FBQztRQUMzRCxNQUFNLEVBQUUsU0FBUyxFQUFFLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2hFLE1BQU0sZUFBZSxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1FBQzlFLEtBQUssTUFBTSxRQUFRLElBQUksU0FBUyxFQUFFO1lBQ2hDLElBQUksZUFBZSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLGlCQUFpQixFQUFFO2dCQUN2RCxNQUFNLElBQUksS0FBSyxDQUFDLG1FQUFtRTtvQkFDakYscUlBQXFJLENBQUMsQ0FBQzthQUMxSTtTQUNGO1FBRUQsK0VBQStFO1FBQy9FLDZFQUE2RTtRQUM3RSw0Q0FBNEM7UUFFNUMsT0FBTztZQUNMLFNBQVM7WUFDVCxnQkFBZ0IsRUFBRSxjQUFjLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQztTQUMvRCxDQUFDO0lBQ0osQ0FBQztJQUVPLG9CQUFvQixDQUFDLEtBQW9CO1FBQy9DLElBQUksS0FBSyxDQUFDLGVBQWUsSUFBSSxLQUFLLENBQUMsc0JBQXNCLEtBQUssS0FBSyxFQUFFO1lBQ25FLE1BQU0sS0FBSyxDQUFDLDRFQUE0RSxDQUFDLENBQUM7U0FDM0Y7UUFFRCxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsSUFBSSxDQUFDLEtBQUssQ0FBQyxzQkFBc0IsRUFBRTtZQUMzRCxPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUVELE1BQU0sZUFBZSxHQUFHLEtBQUssQ0FBQyxlQUFlLElBQUksSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRTtZQUN0RixlQUFlLEVBQUUsZUFBUSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7U0FDbkMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDM0MsT0FBTyxFQUFFLENBQUMsaUJBQWlCLENBQUM7WUFDNUIsU0FBUyxFQUFFLENBQUMsZUFBZSxDQUFDLFFBQVEsQ0FBQztTQUN0QyxDQUFDLENBQUMsQ0FBQztRQUVKLE9BQU8sZUFBZSxDQUFDO0lBQ3pCLENBQUM7SUFFTyxxQkFBcUIsQ0FBQyxlQUE0QjtRQUN4RCxJQUFJLGVBQWUsRUFBRTtZQUNuQixPQUFPO2dCQUNMLFNBQVMsRUFBRSxlQUFlLENBQUMsUUFBUTthQUNwQyxDQUFDO1NBQ0g7YUFBTTtZQUNMLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO0lBQ0gsQ0FBQztJQUVPLGtCQUFrQixDQUFDLEtBQW9CO1FBQzdDLElBQUksS0FBSyxDQUFDLE9BQU8sS0FBSyxTQUFTLElBQUksS0FBSyxDQUFDLE9BQU8sS0FBSyxPQUFPLENBQUMsUUFBUSxFQUFFO1lBQ3JFLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBRUQsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDM0MsT0FBTyxFQUFFLENBQUMsdUJBQXVCLEVBQUUsMEJBQTBCLENBQUM7WUFDOUQsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1NBQ2pCLENBQUMsQ0FBQyxDQUFDO1FBRUosT0FBTztZQUNMLElBQUksRUFBRSxLQUFLLENBQUMsT0FBTztTQUNwQixDQUFDO0lBQ0osQ0FBQztJQUVPLGlCQUFpQixDQUFDLEtBQW9CO1FBQzVDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLHlCQUF5QixFQUFFO1lBQzVDLE1BQU0sSUFBSSxLQUFLLENBQUMsa0RBQWtELEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztTQUN6RjtRQUNELElBQUksS0FBSyxDQUFDLFdBQVcsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsK0JBQStCLElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQyw2QkFBNkIsQ0FBQyxFQUFFO1lBQy9ILE1BQU0sSUFBSSxLQUFLLENBQUMsa0hBQWtILENBQUMsQ0FBQztTQUNySTtJQUNILENBQUM7O0FBcm1CSCw0QkFzbUJDOzs7QUF1QkQ7Ozs7Ozs7Ozs7OztHQVlHO0FBQ0gsU0FBUyxrQkFBa0IsQ0FBQyxHQUFXO0lBQ3JDLE9BQU8sU0FBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsU0FBRSxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztBQUMxQyxDQUFDO0FBRUQsU0FBZ0IsZ0JBQWdCLENBQUMsSUFBZ0IsRUFBRSxLQUFvQjtJQUNyRSxxQkFBcUI7SUFDckIsTUFBTSxRQUFRLEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRWhFLElBQUksUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1FBQzFDLE1BQU0sSUFBSSxLQUFLLENBQUMsaUZBQWlGLENBQUMsQ0FBQztLQUNwRztJQUVELElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLEtBQUssQ0FBQyxLQUFLLENBQUMsT0FBTyxLQUFLLGlCQUFPLENBQUMsVUFBVSxDQUFDLEVBQUU7UUFDM0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxpRkFBaUYsQ0FBQyxDQUFDO0tBQ3BHO0lBRUQsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssS0FBSyxDQUFDLEtBQUssQ0FBQyxPQUFPLEtBQUssaUJBQU8sQ0FBQyxVQUFVLENBQUMsRUFBRTtRQUMzRCxNQUFNLElBQUksS0FBSyxDQUFDLGlGQUFpRixDQUFDLENBQUM7S0FDcEc7SUFFRCwwREFBMEQ7SUFDMUQsSUFBSSxJQUFJLENBQUMsVUFBVSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsRUFBRTtRQUN4RCxNQUFNLElBQUksS0FBSyxDQUFDLGlDQUFpQyxLQUFLLENBQUMsT0FBUSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7S0FDekU7QUFDSCxDQUFDO0FBcEJELDRDQW9CQztBQUVELFNBQVMsaUJBQWlCLENBQUksTUFBUztJQUNyQyxNQUFNLFlBQVksR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsS0FBSyxTQUFTLENBQUMsQ0FBQztJQUMzRSxPQUFPLFlBQVksQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUM7QUFDM0MsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNsb3Vkd2F0Y2ggZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3Vkd2F0Y2gnO1xuaW1wb3J0IHsgSVByb2ZpbGluZ0dyb3VwLCBQcm9maWxpbmdHcm91cCwgQ29tcHV0ZVBsYXRmb3JtIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVndXJ1cHJvZmlsZXInO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMga21zIGZyb20gJ0Bhd3MtY2RrL2F3cy1rbXMnO1xuaW1wb3J0ICogYXMgbG9ncyBmcm9tICdAYXdzLWNkay9hd3MtbG9ncyc7XG5pbXBvcnQgKiBhcyBzcXMgZnJvbSAnQGF3cy1jZGsvYXdzLXNxcyc7XG5pbXBvcnQgeyBBbm5vdGF0aW9ucywgQ2ZuUmVzb3VyY2UsIER1cmF0aW9uLCBGbiwgTGF6eSwgTmFtZXMsIFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENvZGUsIENvZGVDb25maWcgfSBmcm9tICcuL2NvZGUnO1xuaW1wb3J0IHsgRXZlbnRJbnZva2VDb25maWdPcHRpb25zIH0gZnJvbSAnLi9ldmVudC1pbnZva2UtY29uZmlnJztcbmltcG9ydCB7IElFdmVudFNvdXJjZSB9IGZyb20gJy4vZXZlbnQtc291cmNlJztcbmltcG9ydCB7IEZpbGVTeXN0ZW0gfSBmcm9tICcuL2ZpbGVzeXN0ZW0nO1xuaW1wb3J0IHsgRnVuY3Rpb25BdHRyaWJ1dGVzLCBGdW5jdGlvbkJhc2UsIElGdW5jdGlvbiB9IGZyb20gJy4vZnVuY3Rpb24tYmFzZSc7XG5pbXBvcnQgeyBjYWxjdWxhdGVGdW5jdGlvbkhhc2gsIHRyaW1Gcm9tU3RhcnQgfSBmcm9tICcuL2Z1bmN0aW9uLWhhc2gnO1xuaW1wb3J0IHsgSGFuZGxlciB9IGZyb20gJy4vaGFuZGxlcic7XG5pbXBvcnQgeyBWZXJzaW9uLCBWZXJzaW9uT3B0aW9ucyB9IGZyb20gJy4vbGFtYmRhLXZlcnNpb24nO1xuaW1wb3J0IHsgQ2ZuRnVuY3Rpb24gfSBmcm9tICcuL2xhbWJkYS5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSUxheWVyVmVyc2lvbiB9IGZyb20gJy4vbGF5ZXJzJztcbmltcG9ydCB7IExvZ1JldGVudGlvblJldHJ5T3B0aW9ucyB9IGZyb20gJy4vbG9nLXJldGVudGlvbic7XG5pbXBvcnQgeyBSdW50aW1lIH0gZnJvbSAnLi9ydW50aW1lJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGVudW0gVHJhY2luZyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBBQ1RJVkUgPSAnQWN0aXZlJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgUEFTU19USFJPVUdIID0gJ1Bhc3NUaHJvdWdoJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBESVNBQkxFRCA9ICdEaXNhYmxlZCdcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEZ1bmN0aW9uT3B0aW9ucyBleHRlbmRzIEV2ZW50SW52b2tlQ29uZmlnT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0aW1lb3V0PzogRHVyYXRpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW52aXJvbm1lbnQ/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGZ1bmN0aW9uTmFtZT86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtZW1vcnlTaXplPzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGluaXRpYWxQb2xpY3k/OiBpYW0uUG9saWN5U3RhdGVtZW50W107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdnBjPzogZWMyLklWcGM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdnBjU3VibmV0cz86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXA/OiBlYzIuSVNlY3VyaXR5R3JvdXA7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cHM/OiBlYzIuSVNlY3VyaXR5R3JvdXBbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFsbG93QWxsT3V0Ym91bmQ/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVhZExldHRlclF1ZXVlRW5hYmxlZD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVhZExldHRlclF1ZXVlPzogc3FzLklRdWV1ZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdHJhY2luZz86IFRyYWNpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHByb2ZpbGluZz86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwcm9maWxpbmdHcm91cD86IElQcm9maWxpbmdHcm91cDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbGF5ZXJzPzogSUxheWVyVmVyc2lvbltdO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJlc2VydmVkQ29uY3VycmVudEV4ZWN1dGlvbnM/OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBldmVudHM/OiBJRXZlbnRTb3VyY2VbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxvZ1JldGVudGlvbj86IGxvZ3MuUmV0ZW50aW9uRGF5cztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbG9nUmV0ZW50aW9uUm9sZT86IGlhbS5JUm9sZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBsb2dSZXRlbnRpb25SZXRyeU9wdGlvbnM/OiBMb2dSZXRlbnRpb25SZXRyeU9wdGlvbnM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGN1cnJlbnRWZXJzaW9uT3B0aW9ucz86IFZlcnNpb25PcHRpb25zO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZmlsZXN5c3RlbT86IEZpbGVTeXN0ZW07XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYWxsb3dQdWJsaWNTdWJuZXQ/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbnZpcm9ubWVudEVuY3J5cHRpb24/OiBrbXMuSUtleTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBGdW5jdGlvblByb3BzIGV4dGVuZHMgRnVuY3Rpb25PcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcnVudGltZTogUnVudGltZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjb2RlOiBDb2RlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaGFuZGxlcjogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgRnVuY3Rpb24gZXh0ZW5kcyBGdW5jdGlvbkJhc2Uge1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGdldCBjdXJyZW50VmVyc2lvbigpOiBWZXJzaW9uIHtcbiAgICBpZiAodGhpcy5fY3VycmVudFZlcnNpb24pIHtcbiAgICAgIHJldHVybiB0aGlzLl9jdXJyZW50VmVyc2lvbjtcbiAgICB9XG5cbiAgICB0aGlzLl9jdXJyZW50VmVyc2lvbiA9IG5ldyBWZXJzaW9uKHRoaXMsICdDdXJyZW50VmVyc2lvbicsIHtcbiAgICAgIGxhbWJkYTogdGhpcyxcbiAgICAgIC4uLnRoaXMuY3VycmVudFZlcnNpb25PcHRpb25zLFxuICAgIH0pO1xuXG4gICAgLy8gb3ZlcnJpZGUgdGhlIHZlcnNpb24ncyBsb2dpY2FsIElEIHdpdGggYSBsYXp5IHN0cmluZyB3aGljaCBpbmNsdWRlcyB0aGVcbiAgICAvLyBoYXNoIG9mIHRoZSBmdW5jdGlvbiBpdHNlbGYsIHNvIGEgbmV3IHZlcnNpb24gcmVzb3VyY2UgaXMgY3JlYXRlZCB3aGVuXG4gICAgLy8gdGhlIGZ1bmN0aW9uIGNvbmZpZ3VyYXRpb24gY2hhbmdlcy5cbiAgICBjb25zdCBjZm4gPSB0aGlzLl9jdXJyZW50VmVyc2lvbi5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5SZXNvdXJjZTtcbiAgICBjb25zdCBvcmlnaW5hbExvZ2ljYWxJZCA9IHRoaXMuc3RhY2sucmVzb2x2ZShjZm4ubG9naWNhbElkKSBhcyBzdHJpbmc7XG5cbiAgICBjZm4ub3ZlcnJpZGVMb2dpY2FsSWQoTGF6eS51bmNhY2hlZFN0cmluZyh7XG4gICAgICBwcm9kdWNlOiAoKSA9PiB7XG4gICAgICAgIGNvbnN0IGhhc2ggPSBjYWxjdWxhdGVGdW5jdGlvbkhhc2godGhpcyk7XG4gICAgICAgIGNvbnN0IGxvZ2ljYWxJZCA9IHRyaW1Gcm9tU3RhcnQob3JpZ2luYWxMb2dpY2FsSWQsIDI1NSAtIDMyKTtcbiAgICAgICAgcmV0dXJuIGAke2xvZ2ljYWxJZH0ke2hhc2h9YDtcbiAgICAgIH0sXG4gICAgfSkpO1xuXG4gICAgcmV0dXJuIHRoaXMuX2N1cnJlbnRWZXJzaW9uO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBmcm9tRnVuY3Rpb25Bcm4oc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgZnVuY3Rpb25Bcm46IHN0cmluZyk6IElGdW5jdGlvbiB7XG4gICAgcmV0dXJuIEZ1bmN0aW9uLmZyb21GdW5jdGlvbkF0dHJpYnV0ZXMoc2NvcGUsIGlkLCB7IGZ1bmN0aW9uQXJuIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21GdW5jdGlvbkF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IEZ1bmN0aW9uQXR0cmlidXRlcyk6IElGdW5jdGlvbiB7XG4gICAgY29uc3QgZnVuY3Rpb25Bcm4gPSBhdHRycy5mdW5jdGlvbkFybjtcbiAgICBjb25zdCBmdW5jdGlvbk5hbWUgPSBleHRyYWN0TmFtZUZyb21Bcm4oYXR0cnMuZnVuY3Rpb25Bcm4pO1xuICAgIGNvbnN0IHJvbGUgPSBhdHRycy5yb2xlO1xuXG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgRnVuY3Rpb25CYXNlIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBmdW5jdGlvbk5hbWUgPSBmdW5jdGlvbk5hbWU7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZnVuY3Rpb25Bcm4gPSBmdW5jdGlvbkFybjtcbiAgICAgIHB1YmxpYyByZWFkb25seSBncmFudFByaW5jaXBhbDogaWFtLklQcmluY2lwYWw7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgcm9sZSA9IHJvbGU7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgcGVybWlzc2lvbnNOb2RlID0gdGhpcy5ub2RlO1xuXG4gICAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgY2FuQ3JlYXRlUGVybWlzc2lvbnMgPSBhdHRycy5zYW1lRW52aXJvbm1lbnQgPz8gdGhpcy5faXNTdGFja0FjY291bnQoKTtcblxuICAgICAgY29uc3RydWN0b3IoczogQ29uc3RydWN0LCBpOiBzdHJpbmcpIHtcbiAgICAgICAgc3VwZXIocywgaSk7XG5cbiAgICAgICAgdGhpcy5ncmFudFByaW5jaXBhbCA9IHJvbGUgfHwgbmV3IGlhbS5Vbmtub3duUHJpbmNpcGFsKHsgcmVzb3VyY2U6IHRoaXMgfSk7XG5cbiAgICAgICAgaWYgKGF0dHJzLnNlY3VyaXR5R3JvdXApIHtcbiAgICAgICAgICB0aGlzLl9jb25uZWN0aW9ucyA9IG5ldyBlYzIuQ29ubmVjdGlvbnMoe1xuICAgICAgICAgICAgc2VjdXJpdHlHcm91cHM6IFthdHRycy5zZWN1cml0eUdyb3VwXSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfSBlbHNlIGlmIChhdHRycy5zZWN1cml0eUdyb3VwSWQpIHtcbiAgICAgICAgICB0aGlzLl9jb25uZWN0aW9ucyA9IG5ldyBlYzIuQ29ubmVjdGlvbnMoe1xuICAgICAgICAgICAgc2VjdXJpdHlHcm91cHM6IFtlYzIuU2VjdXJpdHlHcm91cC5mcm9tU2VjdXJpdHlHcm91cElkKHNjb3BlLCAnU2VjdXJpdHlHcm91cCcsIGF0dHJzLnNlY3VyaXR5R3JvdXBJZCldLFxuICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIG1ldHJpY0FsbChtZXRyaWNOYW1lOiBzdHJpbmcsIHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgIHJldHVybiBuZXcgY2xvdWR3YXRjaC5NZXRyaWMoe1xuICAgICAgbmFtZXNwYWNlOiAnQVdTL0xhbWJkYScsXG4gICAgICBtZXRyaWNOYW1lLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIG1ldHJpY0FsbEVycm9ycyhwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWNBbGwoJ0Vycm9ycycsIHsgc3RhdGlzdGljOiAnc3VtJywgLi4ucHJvcHMgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBtZXRyaWNBbGxEdXJhdGlvbihwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWNBbGwoJ0R1cmF0aW9uJywgcHJvcHMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBtZXRyaWNBbGxJbnZvY2F0aW9ucyhwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWNBbGwoJ0ludm9jYXRpb25zJywgeyBzdGF0aXN0aWM6ICdzdW0nLCAuLi5wcm9wcyB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIG1ldHJpY0FsbFRocm90dGxlcyhwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWNBbGwoJ1Rocm90dGxlcycsIHsgc3RhdGlzdGljOiAnc3VtJywgLi4ucHJvcHMgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgbWV0cmljQWxsQ29uY3VycmVudEV4ZWN1dGlvbnMocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgLy8gTWluaS1GQVE6IHdoeSBtYXg/IFRoaXMgbWV0cmljIGlzIGEgZ2F1Z2UgdGhhdCBpcyBlbWl0dGVkIGV2ZXJ5XG4gICAgLy8gbWludXRlLCBzbyBlaXRoZXIgbWF4IG9yIGF2ZyBvciBhIHBlcmNlbnRpbGUgbWFrZSBzZW5zZSAoYnV0IHN1bVxuICAgIC8vIGRvZXNuJ3QpLiBNYXggaXMgbW9yZSBzZW5zaXRpdmUgdG8gc3Bpa3kgbG9hZCBjaGFuZ2VzIHdoaWNoIGlzXG4gICAgLy8gcHJvYmFibHkgd2hhdCB5b3UncmUgaW50ZXJlc3RlZCBpbiBpZiB5b3UncmUgbG9va2luZyBhdCB0aGlzIG1ldHJpY1xuICAgIC8vIChMb2FkIHNwaWtlcyBtYXkgbGVhZCB0byBjb25jdXJyZW50IGV4ZWN1dGlvbiBlcnJvcnMgdGhhdCB3b3VsZFxuICAgIC8vIG90aGVyd2lzZSBub3QgYmUgdmlzaWJsZSBpbiB0aGUgYXZnKVxuICAgIHJldHVybiB0aGlzLm1ldHJpY0FsbCgnQ29uY3VycmVudEV4ZWN1dGlvbnMnLCB7IHN0YXRpc3RpYzogJ21heCcsIC4uLnByb3BzIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgbWV0cmljQWxsVW5yZXNlcnZlZENvbmN1cnJlbnRFeGVjdXRpb25zKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpY0FsbCgnVW5yZXNlcnZlZENvbmN1cnJlbnRFeGVjdXRpb25zJywgeyBzdGF0aXN0aWM6ICdtYXgnLCAuLi5wcm9wcyB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgZnVuY3Rpb25OYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGZ1bmN0aW9uQXJuOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBydW50aW1lOiBSdW50aW1lO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWw6IGlhbS5JUHJpbmNpcGFsO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBkZWFkTGV0dGVyUXVldWU/OiBzcXMuSVF1ZXVlO1xuXG4gIHB1YmxpYyByZWFkb25seSBwZXJtaXNzaW9uc05vZGUgPSB0aGlzLm5vZGU7XG5cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IGNhbkNyZWF0ZVBlcm1pc3Npb25zID0gdHJ1ZTtcblxuICBwcml2YXRlIHJlYWRvbmx5IGxheWVyczogSUxheWVyVmVyc2lvbltdID0gW107XG5cbiAgcHJpdmF0ZSBfbG9nR3JvdXA/OiBsb2dzLklMb2dHcm91cDtcblxuICAvKipcbiAgICogRW52aXJvbm1lbnQgdmFyaWFibGVzIGZvciB0aGlzIGZ1bmN0aW9uXG4gICAqL1xuICBwcml2YXRlIGVudmlyb25tZW50OiB7IFtrZXk6IHN0cmluZ106IEVudmlyb25tZW50Q29uZmlnIH0gPSB7fTtcblxuICBwcml2YXRlIHJlYWRvbmx5IGN1cnJlbnRWZXJzaW9uT3B0aW9ucz86IFZlcnNpb25PcHRpb25zO1xuICBwcml2YXRlIF9jdXJyZW50VmVyc2lvbj86IFZlcnNpb247XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEZ1bmN0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMuZnVuY3Rpb25OYW1lLFxuICAgIH0pO1xuXG4gICAgY29uc3QgbWFuYWdlZFBvbGljaWVzID0gbmV3IEFycmF5PGlhbS5JTWFuYWdlZFBvbGljeT4oKTtcblxuICAgIC8vIHRoZSBhcm4gaXMgaW4gdGhlIGZvcm0gb2YgLSBhcm46YXdzOmlhbTo6YXdzOnBvbGljeS9zZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlXG4gICAgbWFuYWdlZFBvbGljaWVzLnB1c2goaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlJykpO1xuXG4gICAgaWYgKHByb3BzLnZwYykge1xuICAgICAgLy8gUG9saWN5IHRoYXQgd2lsbCBoYXZlIEVOSSBjcmVhdGlvbiBwZXJtaXNzaW9uc1xuICAgICAgbWFuYWdlZFBvbGljaWVzLnB1c2goaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQVdTTGFtYmRhVlBDQWNjZXNzRXhlY3V0aW9uUm9sZScpKTtcbiAgICB9XG5cbiAgICB0aGlzLnJvbGUgPSBwcm9wcy5yb2xlIHx8IG5ldyBpYW0uUm9sZSh0aGlzLCAnU2VydmljZVJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIG1hbmFnZWRQb2xpY2llcyxcbiAgICB9KTtcbiAgICB0aGlzLmdyYW50UHJpbmNpcGFsID0gdGhpcy5yb2xlO1xuXG4gICAgLy8gYWRkIGFkZGl0b25hbCBtYW5hZ2VkIHBvbGljaWVzIHdoZW4gbmVjZXNzYXJ5XG4gICAgaWYgKHByb3BzLmZpbGVzeXN0ZW0pIHtcbiAgICAgIGNvbnN0IGNvbmZpZyA9IHByb3BzLmZpbGVzeXN0ZW0uY29uZmlnO1xuICAgICAgaWYgKGNvbmZpZy5wb2xpY2llcykge1xuICAgICAgICBjb25maWcucG9saWNpZXMuZm9yRWFjaChwID0+IHtcbiAgICAgICAgICB0aGlzLnJvbGU/LmFkZFRvUG9saWN5KHApO1xuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IHN0YXRlbWVudCBvZiAocHJvcHMuaW5pdGlhbFBvbGljeSB8fCBbXSkpIHtcbiAgICAgIHRoaXMucm9sZS5hZGRUb1BvbGljeShzdGF0ZW1lbnQpO1xuICAgIH1cblxuICAgIGNvbnN0IGNvZGUgPSBwcm9wcy5jb2RlLmJpbmQodGhpcyk7XG4gICAgdmVyaWZ5Q29kZUNvbmZpZyhjb2RlLCBwcm9wcyk7XG5cbiAgICBsZXQgcHJvZmlsaW5nR3JvdXBFbnZpcm9ubWVudFZhcmlhYmxlczogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfSA9IHt9O1xuICAgIGlmIChwcm9wcy5wcm9maWxpbmdHcm91cCAmJiBwcm9wcy5wcm9maWxpbmcgIT09IGZhbHNlKSB7XG4gICAgICB0aGlzLnZhbGlkYXRlUHJvZmlsaW5nKHByb3BzKTtcbiAgICAgIHByb3BzLnByb2ZpbGluZ0dyb3VwLmdyYW50UHVibGlzaCh0aGlzLnJvbGUpO1xuICAgICAgcHJvZmlsaW5nR3JvdXBFbnZpcm9ubWVudFZhcmlhYmxlcyA9IHtcbiAgICAgICAgQVdTX0NPREVHVVJVX1BST0ZJTEVSX0dST1VQX0FSTjogU3RhY2sub2Yoc2NvcGUpLmZvcm1hdEFybih7XG4gICAgICAgICAgc2VydmljZTogJ2NvZGVndXJ1LXByb2ZpbGVyJyxcbiAgICAgICAgICByZXNvdXJjZTogJ3Byb2ZpbGluZ0dyb3VwJyxcbiAgICAgICAgICByZXNvdXJjZU5hbWU6IHByb3BzLnByb2ZpbGluZ0dyb3VwLnByb2ZpbGluZ0dyb3VwTmFtZSxcbiAgICAgICAgfSksXG4gICAgICAgIEFXU19DT0RFR1VSVV9QUk9GSUxFUl9FTkFCTEVEOiAnVFJVRScsXG4gICAgICB9O1xuICAgIH0gZWxzZSBpZiAocHJvcHMucHJvZmlsaW5nKSB7XG4gICAgICB0aGlzLnZhbGlkYXRlUHJvZmlsaW5nKHByb3BzKTtcbiAgICAgIGNvbnN0IHByb2ZpbGluZ0dyb3VwID0gbmV3IFByb2ZpbGluZ0dyb3VwKHRoaXMsICdQcm9maWxpbmdHcm91cCcsIHtcbiAgICAgICAgY29tcHV0ZVBsYXRmb3JtOiBDb21wdXRlUGxhdGZvcm0uQVdTX0xBTUJEQSxcbiAgICAgIH0pO1xuICAgICAgcHJvZmlsaW5nR3JvdXAuZ3JhbnRQdWJsaXNoKHRoaXMucm9sZSk7XG4gICAgICBwcm9maWxpbmdHcm91cEVudmlyb25tZW50VmFyaWFibGVzID0ge1xuICAgICAgICBBV1NfQ09ERUdVUlVfUFJPRklMRVJfR1JPVVBfQVJOOiBwcm9maWxpbmdHcm91cC5wcm9maWxpbmdHcm91cEFybixcbiAgICAgICAgQVdTX0NPREVHVVJVX1BST0ZJTEVSX0VOQUJMRUQ6ICdUUlVFJyxcbiAgICAgIH07XG4gICAgfVxuXG4gICAgY29uc3QgZW52ID0geyAuLi5wcm9maWxpbmdHcm91cEVudmlyb25tZW50VmFyaWFibGVzLCAuLi5wcm9wcy5lbnZpcm9ubWVudCB9O1xuICAgIGZvciAoY29uc3QgW2tleSwgdmFsdWVdIG9mIE9iamVjdC5lbnRyaWVzKGVudikpIHtcbiAgICAgIHRoaXMuYWRkRW52aXJvbm1lbnQoa2V5LCB2YWx1ZSk7XG4gICAgfVxuXG4gICAgdGhpcy5kZWFkTGV0dGVyUXVldWUgPSB0aGlzLmJ1aWxkRGVhZExldHRlclF1ZXVlKHByb3BzKTtcblxuICAgIGxldCBmaWxlU3lzdGVtQ29uZmlnczogQ2ZuRnVuY3Rpb24uRmlsZVN5c3RlbUNvbmZpZ1Byb3BlcnR5W10gfCB1bmRlZmluZWQgPSB1bmRlZmluZWQ7XG4gICAgaWYgKHByb3BzLmZpbGVzeXN0ZW0pIHtcbiAgICAgIGZpbGVTeXN0ZW1Db25maWdzID0gW3tcbiAgICAgICAgYXJuOiBwcm9wcy5maWxlc3lzdGVtLmNvbmZpZy5hcm4sXG4gICAgICAgIGxvY2FsTW91bnRQYXRoOiBwcm9wcy5maWxlc3lzdGVtLmNvbmZpZy5sb2NhbE1vdW50UGF0aCxcbiAgICAgIH1dO1xuICAgIH1cblxuICAgIGNvbnN0IHJlc291cmNlOiBDZm5GdW5jdGlvbiA9IG5ldyBDZm5GdW5jdGlvbih0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBmdW5jdGlvbk5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgICAgZGVzY3JpcHRpb246IHByb3BzLmRlc2NyaXB0aW9uLFxuICAgICAgY29kZToge1xuICAgICAgICBzM0J1Y2tldDogY29kZS5zM0xvY2F0aW9uICYmIGNvZGUuczNMb2NhdGlvbi5idWNrZXROYW1lLFxuICAgICAgICBzM0tleTogY29kZS5zM0xvY2F0aW9uICYmIGNvZGUuczNMb2NhdGlvbi5vYmplY3RLZXksXG4gICAgICAgIHMzT2JqZWN0VmVyc2lvbjogY29kZS5zM0xvY2F0aW9uICYmIGNvZGUuczNMb2NhdGlvbi5vYmplY3RWZXJzaW9uLFxuICAgICAgICB6aXBGaWxlOiBjb2RlLmlubGluZUNvZGUsXG4gICAgICAgIGltYWdlVXJpOiBjb2RlLmltYWdlPy5pbWFnZVVyaSxcbiAgICAgIH0sXG4gICAgICBsYXllcnM6IExhenkubGlzdCh7IHByb2R1Y2U6ICgpID0+IHRoaXMubGF5ZXJzLm1hcChsYXllciA9PiBsYXllci5sYXllclZlcnNpb25Bcm4pIH0sIHsgb21pdEVtcHR5OiB0cnVlIH0pLFxuICAgICAgaGFuZGxlcjogcHJvcHMuaGFuZGxlciA9PT0gSGFuZGxlci5GUk9NX0lNQUdFID8gdW5kZWZpbmVkIDogcHJvcHMuaGFuZGxlcixcbiAgICAgIHRpbWVvdXQ6IHByb3BzLnRpbWVvdXQgJiYgcHJvcHMudGltZW91dC50b1NlY29uZHMoKSxcbiAgICAgIHBhY2thZ2VUeXBlOiBwcm9wcy5ydW50aW1lID09PSBSdW50aW1lLkZST01fSU1BR0UgPyAnSW1hZ2UnIDogdW5kZWZpbmVkLFxuICAgICAgcnVudGltZTogcHJvcHMucnVudGltZSA9PT0gUnVudGltZS5GUk9NX0lNQUdFID8gdW5kZWZpbmVkIDogcHJvcHMucnVudGltZS5uYW1lLFxuICAgICAgcm9sZTogdGhpcy5yb2xlLnJvbGVBcm4sXG4gICAgICAvLyBVbmNhY2hlZCBiZWNhdXNlIGNhbGxpbmcgJ19jaGVja0VkZ2VDb21wYXRpYmlsaXR5Jywgd2hpY2ggZ2V0cyBjYWxsZWQgaW4gdGhlIHJlc29sdmUgb2YgYW5vdGhlclxuICAgICAgLy8gVG9rZW4sIGFjdHVhbGx5ICptb2RpZmllcyogdGhlICdlbnZpcm9ubWVudCcgbWFwLlxuICAgICAgZW52aXJvbm1lbnQ6IExhenkudW5jYWNoZWRBbnkoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnJlbmRlckVudmlyb25tZW50KCkgfSksXG4gICAgICBtZW1vcnlTaXplOiBwcm9wcy5tZW1vcnlTaXplLFxuICAgICAgdnBjQ29uZmlnOiB0aGlzLmNvbmZpZ3VyZVZwYyhwcm9wcyksXG4gICAgICBkZWFkTGV0dGVyQ29uZmlnOiB0aGlzLmJ1aWxkRGVhZExldHRlckNvbmZpZyh0aGlzLmRlYWRMZXR0ZXJRdWV1ZSksXG4gICAgICB0cmFjaW5nQ29uZmlnOiB0aGlzLmJ1aWxkVHJhY2luZ0NvbmZpZyhwcm9wcyksXG4gICAgICByZXNlcnZlZENvbmN1cnJlbnRFeGVjdXRpb25zOiBwcm9wcy5yZXNlcnZlZENvbmN1cnJlbnRFeGVjdXRpb25zLFxuICAgICAgaW1hZ2VDb25maWc6IHVuZGVmaW5lZElmTm9LZXlzKHtcbiAgICAgICAgY29tbWFuZDogY29kZS5pbWFnZT8uY21kLFxuICAgICAgICBlbnRyeVBvaW50OiBjb2RlLmltYWdlPy5lbnRyeXBvaW50LFxuICAgICAgfSksXG4gICAgICBrbXNLZXlBcm46IHByb3BzLmVudmlyb25tZW50RW5jcnlwdGlvbj8ua2V5QXJuLFxuICAgICAgZmlsZVN5c3RlbUNvbmZpZ3MsXG4gICAgfSk7XG5cbiAgICByZXNvdXJjZS5ub2RlLmFkZERlcGVuZGVuY3kodGhpcy5yb2xlKTtcblxuICAgIHRoaXMuZnVuY3Rpb25OYW1lID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUocmVzb3VyY2UucmVmKTtcbiAgICB0aGlzLmZ1bmN0aW9uQXJuID0gdGhpcy5nZXRSZXNvdXJjZUFybkF0dHJpYnV0ZShyZXNvdXJjZS5hdHRyQXJuLCB7XG4gICAgICBzZXJ2aWNlOiAnbGFtYmRhJyxcbiAgICAgIHJlc291cmNlOiAnZnVuY3Rpb24nLFxuICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgIHNlcDogJzonLFxuICAgIH0pO1xuXG4gICAgdGhpcy5ydW50aW1lID0gcHJvcHMucnVudGltZTtcblxuICAgIGlmIChwcm9wcy5sYXllcnMpIHtcbiAgICAgIHRoaXMuYWRkTGF5ZXJzKC4uLnByb3BzLmxheWVycyk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBldmVudCBvZiBwcm9wcy5ldmVudHMgfHwgW10pIHtcbiAgICAgIHRoaXMuYWRkRXZlbnRTb3VyY2UoZXZlbnQpO1xuICAgIH1cblxuICAgIC8vIExvZyByZXRlbnRpb25cbiAgICBpZiAocHJvcHMubG9nUmV0ZW50aW9uKSB7XG4gICAgICBjb25zdCBsb2dSZXRlbnRpb24gPSBuZXcgbG9ncy5Mb2dSZXRlbnRpb24odGhpcywgJ0xvZ1JldGVudGlvbicsIHtcbiAgICAgICAgbG9nR3JvdXBOYW1lOiBgL2F3cy9sYW1iZGEvJHt0aGlzLmZ1bmN0aW9uTmFtZX1gLFxuICAgICAgICByZXRlbnRpb246IHByb3BzLmxvZ1JldGVudGlvbixcbiAgICAgICAgcm9sZTogcHJvcHMubG9nUmV0ZW50aW9uUm9sZSxcbiAgICAgICAgbG9nUmV0ZW50aW9uUmV0cnlPcHRpb25zOiBwcm9wcy5sb2dSZXRlbnRpb25SZXRyeU9wdGlvbnMgYXMgbG9ncy5Mb2dSZXRlbnRpb25SZXRyeU9wdGlvbnMsXG4gICAgICB9KTtcbiAgICAgIHRoaXMuX2xvZ0dyb3VwID0gbG9ncy5Mb2dHcm91cC5mcm9tTG9nR3JvdXBBcm4odGhpcywgJ0xvZ0dyb3VwJywgbG9nUmV0ZW50aW9uLmxvZ0dyb3VwQXJuKTtcbiAgICB9XG5cbiAgICBwcm9wcy5jb2RlLmJpbmRUb1Jlc291cmNlKHJlc291cmNlKTtcblxuICAgIC8vIEV2ZW50IEludm9rZSBDb25maWdcbiAgICBpZiAocHJvcHMub25GYWlsdXJlIHx8IHByb3BzLm9uU3VjY2VzcyB8fCBwcm9wcy5tYXhFdmVudEFnZSB8fCBwcm9wcy5yZXRyeUF0dGVtcHRzICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHRoaXMuY29uZmlndXJlQXN5bmNJbnZva2Uoe1xuICAgICAgICBvbkZhaWx1cmU6IHByb3BzLm9uRmFpbHVyZSxcbiAgICAgICAgb25TdWNjZXNzOiBwcm9wcy5vblN1Y2Nlc3MsXG4gICAgICAgIG1heEV2ZW50QWdlOiBwcm9wcy5tYXhFdmVudEFnZSxcbiAgICAgICAgcmV0cnlBdHRlbXB0czogcHJvcHMucmV0cnlBdHRlbXB0cyxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHRoaXMuY3VycmVudFZlcnNpb25PcHRpb25zID0gcHJvcHMuY3VycmVudFZlcnNpb25PcHRpb25zO1xuXG4gICAgaWYgKHByb3BzLmZpbGVzeXN0ZW0pIHtcbiAgICAgIGNvbnN0IGNvbmZpZyA9IHByb3BzLmZpbGVzeXN0ZW0uY29uZmlnO1xuICAgICAgaWYgKGNvbmZpZy5kZXBlbmRlbmN5KSB7XG4gICAgICAgIHRoaXMubm9kZS5hZGREZXBlbmRlbmN5KC4uLmNvbmZpZy5kZXBlbmRlbmN5KTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhZGRFbnZpcm9ubWVudChrZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZywgb3B0aW9ucz86IEVudmlyb25tZW50T3B0aW9ucyk6IHRoaXMge1xuICAgIHRoaXMuZW52aXJvbm1lbnRba2V5XSA9IHsgdmFsdWUsIC4uLm9wdGlvbnMgfTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkTGF5ZXJzKC4uLmxheWVyczogSUxheWVyVmVyc2lvbltdKTogdm9pZCB7XG4gICAgZm9yIChjb25zdCBsYXllciBvZiBsYXllcnMpIHtcbiAgICAgIGlmICh0aGlzLmxheWVycy5sZW5ndGggPT09IDUpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdVbmFibGUgdG8gYWRkIGxheWVyOiB0aGlzIGxhbWJkYSBmdW5jdGlvbiBhbHJlYWR5IHVzZXMgNSBsYXllcnMuJyk7XG4gICAgICB9XG4gICAgICBpZiAobGF5ZXIuY29tcGF0aWJsZVJ1bnRpbWVzICYmICFsYXllci5jb21wYXRpYmxlUnVudGltZXMuZmluZChydW50aW1lID0+IHJ1bnRpbWUucnVudGltZUVxdWFscyh0aGlzLnJ1bnRpbWUpKSkge1xuICAgICAgICBjb25zdCBydW50aW1lcyA9IGxheWVyLmNvbXBhdGlibGVSdW50aW1lcy5tYXAocnVudGltZSA9PiBydW50aW1lLm5hbWUpLmpvaW4oJywgJyk7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgVGhpcyBsYW1iZGEgZnVuY3Rpb24gdXNlcyBhIHJ1bnRpbWUgdGhhdCBpcyBpbmNvbXBhdGlibGUgd2l0aCB0aGlzIGxheWVyICgke3RoaXMucnVudGltZS5uYW1lfSBpcyBub3QgaW4gWyR7cnVudGltZXN9XSlgKTtcbiAgICAgIH1cbiAgICAgIHRoaXMubGF5ZXJzLnB1c2gobGF5ZXIpO1xuICAgIH1cbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhZGRWZXJzaW9uKFxuICAgIG5hbWU6IHN0cmluZyxcbiAgICBjb2RlU2hhMjU2Pzogc3RyaW5nLFxuICAgIGRlc2NyaXB0aW9uPzogc3RyaW5nLFxuICAgIHByb3Zpc2lvbmVkRXhlY3V0aW9ucz86IG51bWJlcixcbiAgICBhc3luY0ludm9rZUNvbmZpZzogRXZlbnRJbnZva2VDb25maWdPcHRpb25zID0ge30pOiBWZXJzaW9uIHtcblxuICAgIHJldHVybiBuZXcgVmVyc2lvbih0aGlzLCAnVmVyc2lvbicgKyBuYW1lLCB7XG4gICAgICBsYW1iZGE6IHRoaXMsXG4gICAgICBjb2RlU2hhMjU2LFxuICAgICAgZGVzY3JpcHRpb24sXG4gICAgICBwcm92aXNpb25lZENvbmN1cnJlbnRFeGVjdXRpb25zOiBwcm92aXNpb25lZEV4ZWN1dGlvbnMsXG4gICAgICAuLi5hc3luY0ludm9rZUNvbmZpZyxcbiAgICB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBnZXQgbG9nR3JvdXAoKTogbG9ncy5JTG9nR3JvdXAge1xuICAgIGlmICghdGhpcy5fbG9nR3JvdXApIHtcbiAgICAgIGNvbnN0IGxvZ1JldGVudGlvbiA9IG5ldyBsb2dzLkxvZ1JldGVudGlvbih0aGlzLCAnTG9nUmV0ZW50aW9uJywge1xuICAgICAgICBsb2dHcm91cE5hbWU6IGAvYXdzL2xhbWJkYS8ke3RoaXMuZnVuY3Rpb25OYW1lfWAsXG4gICAgICAgIHJldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLklORklOSVRFLFxuICAgICAgfSk7XG4gICAgICB0aGlzLl9sb2dHcm91cCA9IGxvZ3MuTG9nR3JvdXAuZnJvbUxvZ0dyb3VwQXJuKHRoaXMsIGAke3RoaXMubm9kZS5pZH0tTG9nR3JvdXBgLCBsb2dSZXRlbnRpb24ubG9nR3JvdXBBcm4pO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5fbG9nR3JvdXA7XG4gIH1cblxuICAvKiogQGludGVybmFsICovXG4gIHB1YmxpYyBfY2hlY2tFZGdlQ29tcGF0aWJpbGl0eSgpOiB2b2lkIHtcbiAgICAvLyBDaGVjayBlbnYgdmFyc1xuICAgIGNvbnN0IGVudkVudHJpZXMgPSBPYmplY3QuZW50cmllcyh0aGlzLmVudmlyb25tZW50KTtcbiAgICBmb3IgKGNvbnN0IFtrZXksIGNvbmZpZ10gb2YgZW52RW50cmllcykge1xuICAgICAgaWYgKGNvbmZpZy5yZW1vdmVJbkVkZ2UpIHtcbiAgICAgICAgZGVsZXRlIHRoaXMuZW52aXJvbm1lbnRba2V5XTtcbiAgICAgICAgQW5ub3RhdGlvbnMub2YodGhpcykuYWRkSW5mbyhgUmVtb3ZlZCAke2tleX0gZW52aXJvbm1lbnQgdmFyaWFibGUgZm9yIExhbWJkYUBFZGdlIGNvbXBhdGliaWxpdHlgKTtcbiAgICAgIH1cbiAgICB9XG4gICAgY29uc3QgZW52S2V5cyA9IE9iamVjdC5rZXlzKHRoaXMuZW52aXJvbm1lbnQpO1xuICAgIGlmIChlbnZLZXlzLmxlbmd0aCAhPT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBUaGUgZnVuY3Rpb24gJHt0aGlzLm5vZGUucGF0aH0gY29udGFpbnMgZW52aXJvbm1lbnQgdmFyaWFibGVzIFske2VudktleXN9XSBhbmQgaXMgbm90IGNvbXBhdGlibGUgd2l0aCBMYW1iZGFARWRnZS4gXFxcbkVudmlyb25tZW50IHZhcmlhYmxlcyBjYW4gYmUgbWFya2VkIGZvciByZW1vdmFsIHdoZW4gdXNlZCBpbiBMYW1iZGFARWRnZSBieSBzZXR0aW5nIHRoZSBcXCdyZW1vdmVJbkVkZ2VcXCcgcHJvcGVydHkgaW4gdGhlIFxcJ2FkZEVudmlyb25tZW50KClcXCcgQVBJLmApO1xuICAgIH1cblxuICAgIHJldHVybjtcbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyRW52aXJvbm1lbnQoKSB7XG4gICAgaWYgKCF0aGlzLmVudmlyb25tZW50IHx8IE9iamVjdC5rZXlzKHRoaXMuZW52aXJvbm1lbnQpLmxlbmd0aCA9PT0gMCkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG5cbiAgICBjb25zdCB2YXJpYWJsZXM6IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH0gPSB7fTtcbiAgICAvLyBTb3J0IGVudmlyb25tZW50IHNvIHRoZSBoYXNoIG9mIHRoZSBmdW5jdGlvbiB1c2VkIHRvIGNyZWF0ZVxuICAgIC8vIGBjdXJyZW50VmVyc2lvbmAgaXMgbm90IGFmZmVjdGVkIGJ5IGtleSBvcmRlciAodGhpcyBpcyBob3cgbGFtYmRhIGRvZXNcbiAgICAvLyBpdCkuIEZvciBiYWNrd2FyZHMgY29tcGF0aWJpbGl0eSB3ZSBkbyBub3Qgc29ydCBlbnZpcm9ubWVudCB2YXJpYWJsZXMgaW4gY2FzZVxuICAgIC8vIF9jdXJyZW50VmVyc2lvbiBpcyBub3QgZGVmaW5lZC4gT3RoZXJ3aXNlLCB0aGlzIHdvdWxkIGhhdmUgaW52YWxpZGF0ZWRcbiAgICAvLyB0aGUgdGVtcGxhdGUsIGFuZCBmb3IgZXhhbXBsZSwgbWF5IGNhdXNlIHVubmVlZGVkIHVwZGF0ZXMgZm9yIG5lc3RlZFxuICAgIC8vIHN0YWNrcy5cbiAgICBjb25zdCBrZXlzID0gdGhpcy5fY3VycmVudFZlcnNpb25cbiAgICAgID8gT2JqZWN0LmtleXModGhpcy5lbnZpcm9ubWVudCkuc29ydCgpXG4gICAgICA6IE9iamVjdC5rZXlzKHRoaXMuZW52aXJvbm1lbnQpO1xuXG4gICAgZm9yIChjb25zdCBrZXkgb2Yga2V5cykge1xuICAgICAgdmFyaWFibGVzW2tleV0gPSB0aGlzLmVudmlyb25tZW50W2tleV0udmFsdWU7XG4gICAgfVxuXG4gICAgcmV0dXJuIHsgdmFyaWFibGVzIH07XG4gIH1cblxuICAvKipcbiAgICogSWYgY29uZmlndXJlZCwgc2V0IHVwIHRoZSBWUEMtcmVsYXRlZCBwcm9wZXJ0aWVzXG4gICAqXG4gICAqIFJldHVybnMgdGhlIFZwY0NvbmZpZyB0aGF0IHNob3VsZCBiZSBhZGRlZCB0byB0aGVcbiAgICogTGFtYmRhIGNyZWF0aW9uIHByb3BlcnRpZXMuXG4gICAqL1xuICBwcml2YXRlIGNvbmZpZ3VyZVZwYyhwcm9wczogRnVuY3Rpb25Qcm9wcyk6IENmbkZ1bmN0aW9uLlZwY0NvbmZpZ1Byb3BlcnR5IHwgdW5kZWZpbmVkIHtcbiAgICBpZiAoKHByb3BzLnNlY3VyaXR5R3JvdXAgfHwgcHJvcHMuYWxsb3dBbGxPdXRib3VuZCAhPT0gdW5kZWZpbmVkKSAmJiAhcHJvcHMudnBjKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBjb25maWd1cmUgXFwnc2VjdXJpdHlHcm91cFxcJyBvciBcXCdhbGxvd0FsbE91dGJvdW5kXFwnIHdpdGhvdXQgY29uZmlndXJpbmcgYSBWUEMnKTtcbiAgICB9XG5cbiAgICBpZiAoIXByb3BzLnZwYykgeyByZXR1cm4gdW5kZWZpbmVkOyB9XG5cbiAgICBpZiAocHJvcHMuc2VjdXJpdHlHcm91cCAmJiBwcm9wcy5hbGxvd0FsbE91dGJvdW5kICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ29uZmlndXJlIFxcJ2FsbG93QWxsT3V0Ym91bmRcXCcgZGlyZWN0bHkgb24gdGhlIHN1cHBsaWVkIFNlY3VyaXR5R3JvdXAuJyk7XG4gICAgfVxuXG4gICAgbGV0IHNlY3VyaXR5R3JvdXBzOiBlYzIuSVNlY3VyaXR5R3JvdXBbXTtcblxuICAgIGlmIChwcm9wcy5zZWN1cml0eUdyb3VwICYmIHByb3BzLnNlY3VyaXR5R3JvdXBzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ09ubHkgb25lIG9mIHRoZSBmdW5jdGlvbiBwcm9wcywgc2VjdXJpdHlHcm91cCBvciBzZWN1cml0eUdyb3VwcywgaXMgYWxsb3dlZCcpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5zZWN1cml0eUdyb3Vwcykge1xuICAgICAgc2VjdXJpdHlHcm91cHMgPSBwcm9wcy5zZWN1cml0eUdyb3VwcztcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3Qgc2VjdXJpdHlHcm91cCA9IHByb3BzLnNlY3VyaXR5R3JvdXAgfHwgbmV3IGVjMi5TZWN1cml0eUdyb3VwKHRoaXMsICdTZWN1cml0eUdyb3VwJywge1xuICAgICAgICB2cGM6IHByb3BzLnZwYyxcbiAgICAgICAgZGVzY3JpcHRpb246ICdBdXRvbWF0aWMgc2VjdXJpdHkgZ3JvdXAgZm9yIExhbWJkYSBGdW5jdGlvbiAnICsgTmFtZXMudW5pcXVlSWQodGhpcyksXG4gICAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IHByb3BzLmFsbG93QWxsT3V0Ym91bmQsXG4gICAgICB9KTtcbiAgICAgIHNlY3VyaXR5R3JvdXBzID0gW3NlY3VyaXR5R3JvdXBdO1xuICAgIH1cblxuICAgIHRoaXMuX2Nvbm5lY3Rpb25zID0gbmV3IGVjMi5Db25uZWN0aW9ucyh7IHNlY3VyaXR5R3JvdXBzIH0pO1xuXG4gICAgaWYgKHByb3BzLmZpbGVzeXN0ZW0pIHtcbiAgICAgIGlmIChwcm9wcy5maWxlc3lzdGVtLmNvbmZpZy5jb25uZWN0aW9ucykge1xuICAgICAgICBwcm9wcy5maWxlc3lzdGVtLmNvbmZpZy5jb25uZWN0aW9ucy5hbGxvd0RlZmF1bHRQb3J0RnJvbSh0aGlzKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBjb25zdCBhbGxvd1B1YmxpY1N1Ym5ldCA9IHByb3BzLmFsbG93UHVibGljU3VibmV0ID8/IGZhbHNlO1xuICAgIGNvbnN0IHsgc3VibmV0SWRzIH0gPSBwcm9wcy52cGMuc2VsZWN0U3VibmV0cyhwcm9wcy52cGNTdWJuZXRzKTtcbiAgICBjb25zdCBwdWJsaWNTdWJuZXRJZHMgPSBuZXcgU2V0KHByb3BzLnZwYy5wdWJsaWNTdWJuZXRzLm1hcChzID0+IHMuc3VibmV0SWQpKTtcbiAgICBmb3IgKGNvbnN0IHN1Ym5ldElkIG9mIHN1Ym5ldElkcykge1xuICAgICAgaWYgKHB1YmxpY1N1Ym5ldElkcy5oYXMoc3VibmV0SWQpICYmICFhbGxvd1B1YmxpY1N1Ym5ldCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0xhbWJkYSBGdW5jdGlvbnMgaW4gYSBwdWJsaWMgc3VibmV0IGNhbiBOT1QgYWNjZXNzIHRoZSBpbnRlcm5ldC4gJyArXG4gICAgICAgICAgJ0lmIHlvdSBhcmUgYXdhcmUgb2YgdGhpcyBsaW1pdGF0aW9uIGFuZCB3b3VsZCBzdGlsbCBsaWtlIHRvIHBsYWNlIHRoZSBmdW5jdGlvbiBpbnQgYSBwdWJsaWMgc3VibmV0LCBzZXQgYGFsbG93UHVibGljU3VibmV0YCB0byB0cnVlJyk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gTGlzdCBjYW4ndCBiZSBlbXB0eSBoZXJlLCBpZiB3ZSBnb3QgdGhpcyBmYXIgeW91IGludGVuZGVkIHRvIHB1dCB5b3VyIExhbWJkYVxuICAgIC8vIGluIHN1Ym5ldHMuIFdlJ3JlIGdvaW5nIHRvIGd1YXJhbnRlZSB0aGF0IHdlIGdldCB0aGUgbmljZSBlcnJvciBtZXNzYWdlIGJ5XG4gICAgLy8gbWFraW5nIFZwY05ldHdvcmsgZG8gdGhlIHNlbGVjdGlvbiBhZ2Fpbi5cblxuICAgIHJldHVybiB7XG4gICAgICBzdWJuZXRJZHMsXG4gICAgICBzZWN1cml0eUdyb3VwSWRzOiBzZWN1cml0eUdyb3Vwcy5tYXAoc2cgPT4gc2cuc2VjdXJpdHlHcm91cElkKSxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBidWlsZERlYWRMZXR0ZXJRdWV1ZShwcm9wczogRnVuY3Rpb25Qcm9wcykge1xuICAgIGlmIChwcm9wcy5kZWFkTGV0dGVyUXVldWUgJiYgcHJvcHMuZGVhZExldHRlclF1ZXVlRW5hYmxlZCA9PT0gZmFsc2UpIHtcbiAgICAgIHRocm93IEVycm9yKCdkZWFkTGV0dGVyUXVldWUgZGVmaW5lZCBidXQgZGVhZExldHRlclF1ZXVlRW5hYmxlZCBleHBsaWNpdGx5IHNldCB0byBmYWxzZScpO1xuICAgIH1cblxuICAgIGlmICghcHJvcHMuZGVhZExldHRlclF1ZXVlICYmICFwcm9wcy5kZWFkTGV0dGVyUXVldWVFbmFibGVkKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cblxuICAgIGNvbnN0IGRlYWRMZXR0ZXJRdWV1ZSA9IHByb3BzLmRlYWRMZXR0ZXJRdWV1ZSB8fCBuZXcgc3FzLlF1ZXVlKHRoaXMsICdEZWFkTGV0dGVyUXVldWUnLCB7XG4gICAgICByZXRlbnRpb25QZXJpb2Q6IER1cmF0aW9uLmRheXMoMTQpLFxuICAgIH0pO1xuXG4gICAgdGhpcy5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogWydzcXM6U2VuZE1lc3NhZ2UnXSxcbiAgICAgIHJlc291cmNlczogW2RlYWRMZXR0ZXJRdWV1ZS5xdWV1ZUFybl0sXG4gICAgfSkpO1xuXG4gICAgcmV0dXJuIGRlYWRMZXR0ZXJRdWV1ZTtcbiAgfVxuXG4gIHByaXZhdGUgYnVpbGREZWFkTGV0dGVyQ29uZmlnKGRlYWRMZXR0ZXJRdWV1ZT86IHNxcy5JUXVldWUpIHtcbiAgICBpZiAoZGVhZExldHRlclF1ZXVlKSB7XG4gICAgICByZXR1cm4ge1xuICAgICAgICB0YXJnZXRBcm46IGRlYWRMZXR0ZXJRdWV1ZS5xdWV1ZUFybixcbiAgICAgIH07XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBidWlsZFRyYWNpbmdDb25maWcocHJvcHM6IEZ1bmN0aW9uUHJvcHMpIHtcbiAgICBpZiAocHJvcHMudHJhY2luZyA9PT0gdW5kZWZpbmVkIHx8IHByb3BzLnRyYWNpbmcgPT09IFRyYWNpbmcuRElTQUJMRUQpIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuXG4gICAgdGhpcy5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogWyd4cmF5OlB1dFRyYWNlU2VnbWVudHMnLCAneHJheTpQdXRUZWxlbWV0cnlSZWNvcmRzJ10sXG4gICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgIH0pKTtcblxuICAgIHJldHVybiB7XG4gICAgICBtb2RlOiBwcm9wcy50cmFjaW5nLFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIHZhbGlkYXRlUHJvZmlsaW5nKHByb3BzOiBGdW5jdGlvblByb3BzKSB7XG4gICAgaWYgKCFwcm9wcy5ydW50aW1lLnN1cHBvcnRzQ29kZUd1cnVQcm9maWxpbmcpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQ29kZUd1cnUgcHJvZmlsaW5nIGlzIG5vdCBzdXBwb3J0ZWQgYnkgcnVudGltZSAke3Byb3BzLnJ1bnRpbWUubmFtZX1gKTtcbiAgICB9XG4gICAgaWYgKHByb3BzLmVudmlyb25tZW50ICYmIChwcm9wcy5lbnZpcm9ubWVudC5BV1NfQ09ERUdVUlVfUFJPRklMRVJfR1JPVVBfQVJOIHx8IHByb3BzLmVudmlyb25tZW50LkFXU19DT0RFR1VSVV9QUk9GSUxFUl9FTkFCTEVEKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBV1NfQ09ERUdVUlVfUFJPRklMRVJfR1JPVVBfQVJOIGFuZCBBV1NfQ09ERUdVUlVfUFJPRklMRVJfRU5BQkxFRCBtdXN0IG5vdCBiZSBzZXQgd2hlbiBwcm9maWxpbmcgb3B0aW9ucyBlbmFibGVkJyk7XG4gICAgfVxuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgRW52aXJvbm1lbnRPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZW1vdmVJbkVkZ2U/OiBib29sZWFuXG59XG5cbi8qKlxuICogQ29uZmlndXJhdGlvbiBmb3IgYW4gZW52aXJvbm1lbnQgdmFyaWFibGVcbiAqL1xuaW50ZXJmYWNlIEVudmlyb25tZW50Q29uZmlnIGV4dGVuZHMgRW52aXJvbm1lbnRPcHRpb25zIHtcbiAgcmVhZG9ubHkgdmFsdWU6IHN0cmluZztcbn1cblxuLyoqXG4gKiBHaXZlbiBhbiBvcGFxdWUgKHRva2VuKSBBUk4sIHJldHVybnMgYSBDbG91ZEZvcm1hdGlvbiBleHByZXNzaW9uIHRoYXQgZXh0cmFjdHMgdGhlIGZ1bmN0aW9uXG4gKiBuYW1lIGZyb20gdGhlIEFSTi5cbiAqXG4gKiBGdW5jdGlvbiBBUk5zIGxvb2sgbGlrZSB0aGlzOlxuICpcbiAqICAgYXJuOmF3czpsYW1iZGE6cmVnaW9uOmFjY291bnQtaWQ6ZnVuY3Rpb246ZnVuY3Rpb24tbmFtZVxuICpcbiAqIC4ud2hpY2ggbWVhbnMgdGhhdCBpbiBvcmRlciB0byBleHRyYWN0IHRoZSBgZnVuY3Rpb24tbmFtZWAgY29tcG9uZW50IGZyb20gdGhlIEFSTiwgd2UgY2FuXG4gKiBzcGxpdCB0aGUgQVJOIHVzaW5nIFwiOlwiIGFuZCBzZWxlY3QgdGhlIGNvbXBvbmVudCBpbiBpbmRleCA2LlxuICpcbiAqIEByZXR1cm5zIGBGblNlbGVjdCg2LCBGblNwbGl0KCc6JywgYXJuKSlgXG4gKi9cbmZ1bmN0aW9uIGV4dHJhY3ROYW1lRnJvbUFybihhcm46IHN0cmluZykge1xuICByZXR1cm4gRm4uc2VsZWN0KDYsIEZuLnNwbGl0KCc6JywgYXJuKSk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiB2ZXJpZnlDb2RlQ29uZmlnKGNvZGU6IENvZGVDb25maWcsIHByb3BzOiBGdW5jdGlvblByb3BzKSB7XG4gIC8vIG11dHVhbGx5IGV4Y2x1c2l2ZVxuICBjb25zdCBjb2RlVHlwZSA9IFtjb2RlLmlubGluZUNvZGUsIGNvZGUuczNMb2NhdGlvbiwgY29kZS5pbWFnZV07XG5cbiAgaWYgKGNvZGVUeXBlLmZpbHRlcih4ID0+ICEheCkubGVuZ3RoICE9PSAxKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdsYW1iZGEuQ29kZSBtdXN0IHNwZWNpZnkgZXhhY3RseSBvbmUgb2Y6IFwiaW5saW5lQ29kZVwiLCBcInMzTG9jYXRpb25cIiwgb3IgXCJpbWFnZVwiJyk7XG4gIH1cblxuICBpZiAoISFjb2RlLmltYWdlID09PSAocHJvcHMuaGFuZGxlciAhPT0gSGFuZGxlci5GUk9NX0lNQUdFKSkge1xuICAgIHRocm93IG5ldyBFcnJvcignaGFuZGxlciBtdXN0IGJlIGBIYW5kbGVyLkZST01fSU1BR0VgIHdoZW4gdXNpbmcgaW1hZ2UgYXNzZXQgZm9yIExhbWJkYSBmdW5jdGlvbicpO1xuICB9XG5cbiAgaWYgKCEhY29kZS5pbWFnZSA9PT0gKHByb3BzLnJ1bnRpbWUgIT09IFJ1bnRpbWUuRlJPTV9JTUFHRSkpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ3J1bnRpbWUgbXVzdCBiZSBgUnVudGltZS5GUk9NX0lNQUdFYCB3aGVuIHVzaW5nIGltYWdlIGFzc2V0IGZvciBMYW1iZGEgZnVuY3Rpb24nKTtcbiAgfVxuXG4gIC8vIGlmIHRoaXMgaXMgaW5saW5lIGNvZGUsIGNoZWNrIHRoYXQgdGhlIHJ1bnRpbWUgc3VwcG9ydHNcbiAgaWYgKGNvZGUuaW5saW5lQ29kZSAmJiAhcHJvcHMucnVudGltZS5zdXBwb3J0c0lubGluZUNvZGUpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoYElubGluZSBzb3VyY2Ugbm90IGFsbG93ZWQgZm9yICR7cHJvcHMucnVudGltZSEubmFtZX1gKTtcbiAgfVxufVxuXG5mdW5jdGlvbiB1bmRlZmluZWRJZk5vS2V5czxBPihzdHJ1Y3Q6IEEpOiBBIHwgdW5kZWZpbmVkIHtcbiAgY29uc3QgYWxsVW5kZWZpbmVkID0gT2JqZWN0LnZhbHVlcyhzdHJ1Y3QpLmV2ZXJ5KHZhbCA9PiB2YWwgPT09IHVuZGVmaW5lZCk7XG4gIHJldHVybiBhbGxVbmRlZmluZWQgPyB1bmRlZmluZWQgOiBzdHJ1Y3Q7XG59XG4iXX0=