import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import { IProfilingGroup } from '@aws-cdk/aws-codeguruprofiler';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import * as logs from '@aws-cdk/aws-logs';
import * as sqs from '@aws-cdk/aws-sqs';
import { Duration } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { Architecture } from './architecture';
import { Code, CodeConfig } from './code';
import { ICodeSigningConfig } from './code-signing-config';
import { EventInvokeConfigOptions } from './event-invoke-config';
import { IEventSource } from './event-source';
import { FileSystem } from './filesystem';
import { FunctionAttributes, FunctionBase, IFunction } from './function-base';
import { LambdaInsightsVersion } from './lambda-insights';
import { Version, VersionOptions } from './lambda-version';
import { ILayerVersion } from './layers';
import { Runtime } from './runtime';
import { LogRetentionRetryOptions } from './log-retention';
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html).
 *
 * @stability stable
 */
export declare enum Tracing {
    /**
     * Lambda will respect any tracing header it receives from an upstream service.
     *
     * If no tracing header is received, Lambda will call X-Ray for a tracing decision.
     *
     * @stability stable
     */
    ACTIVE = "Active",
    /**
     * Lambda will only trace the request from an upstream service if it contains a tracing header with "sampled=1".
     *
     * @stability stable
     */
    PASS_THROUGH = "PassThrough",
    /**
     * Lambda will not trace any request.
     *
     * @stability stable
     */
    DISABLED = "Disabled"
}
/**
 * Non runtime options.
 *
 * @stability stable
 */
export interface FunctionOptions extends EventInvokeConfigOptions {
    /**
     * A description of the function.
     *
     * @default - No description.
     * @stability stable
     */
    readonly description?: string;
    /**
     * The function execution time (in seconds) after which Lambda terminates the function.
     *
     * Because the execution time affects cost, set this value
     * based on the function's expected execution time.
     *
     * @default Duration.seconds(3)
     * @stability stable
     */
    readonly timeout?: Duration;
    /**
     * Key-value pairs that Lambda caches and makes available for your Lambda functions.
     *
     * Use environment variables to apply configuration changes, such
     * as test and production environment configurations, without changing your
     * Lambda function source code.
     *
     * @default - No environment variables.
     * @stability stable
     */
    readonly environment?: {
        [key: string]: string;
    };
    /**
     * A name for the function.
     *
     * @default - AWS CloudFormation generates a unique physical ID and uses that
     * ID for the function's name. For more information, see Name Type.
     * @stability stable
     */
    readonly functionName?: string;
    /**
     * The amount of memory, in MB, that is allocated to your Lambda function.
     *
     * Lambda uses this value to proportionally allocate the amount of CPU
     * power. For more information, see Resource Model in the AWS Lambda
     * Developer Guide.
     *
     * @default 128
     * @stability stable
     */
    readonly memorySize?: number;
    /**
     * Initial policy statements to add to the created Lambda Role.
     *
     * You can call `addToRolePolicy` to the created lambda to add statements post creation.
     *
     * @default - No policy statements are added to the created Lambda role.
     * @stability stable
     */
    readonly initialPolicy?: iam.PolicyStatement[];
    /**
     * Lambda execution role.
     *
     * This is the role that will be assumed by the function upon execution.
     * It controls the permissions that the function will have. The Role must
     * be assumable by the 'lambda.amazonaws.com' service principal.
     *
     * The default Role automatically has permissions granted for Lambda execution. If you
     * provide a Role, you must add the relevant AWS managed policies yourself.
     *
     * The relevant managed policies are "service-role/AWSLambdaBasicExecutionRole" and
     * "service-role/AWSLambdaVPCAccessExecutionRole".
     *
     * @default - A unique role will be generated for this lambda function.
     * Both supplied and generated roles can always be changed by calling `addToRolePolicy`.
     * @stability stable
     */
    readonly role?: iam.IRole;
    /**
     * VPC network to place Lambda network interfaces.
     *
     * Specify this if the Lambda function needs to access resources in a VPC.
     *
     * @default - Function is not placed within a VPC.
     * @stability stable
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Where to place the network interfaces within the VPC.
     *
     * Only used if 'vpc' is supplied. Note: internet access for Lambdas
     * requires a NAT gateway, so picking Public subnets is not allowed.
     *
     * @default - the Vpc default strategy if not specified
     * @stability stable
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * (deprecated) What security group to associate with the Lambda's network interfaces. This property is being deprecated, consider using securityGroups instead.
     *
     * Only used if 'vpc' is supplied.
     *
     * Use securityGroups property instead.
     * Function constructor will throw an error if both are specified.
     *
     * @default - If the function is placed within a VPC and a security group is
     * not specified, either by this or securityGroups prop, a dedicated security
     * group will be created for this function.
     * @deprecated - This property is deprecated, use securityGroups instead
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * The list of security groups to associate with the Lambda's network interfaces.
     *
     * Only used if 'vpc' is supplied.
     *
     * @default - If the function is placed within a VPC and a security group is
     * not specified, either by this or securityGroup prop, a dedicated security
     * group will be created for this function.
     * @stability stable
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * Whether to allow the Lambda to send all network traffic.
     *
     * If set to false, you must individually add traffic rules to allow the
     * Lambda to connect to network targets.
     *
     * @default true
     * @stability stable
     */
    readonly allowAllOutbound?: boolean;
    /**
     * Enabled DLQ.
     *
     * If `deadLetterQueue` is undefined,
     * an SQS queue with default options will be defined for your Function.
     *
     * @default - false unless `deadLetterQueue` is set, which implies DLQ is enabled.
     * @stability stable
     */
    readonly deadLetterQueueEnabled?: boolean;
    /**
     * The SQS queue to use if DLQ is enabled.
     *
     * @default - SQS queue with 14 day retention period if `deadLetterQueueEnabled` is `true`
     * @stability stable
     */
    readonly deadLetterQueue?: sqs.IQueue;
    /**
     * Enable AWS X-Ray Tracing for Lambda Function.
     *
     * @default Tracing.Disabled
     * @stability stable
     */
    readonly tracing?: Tracing;
    /**
     * Enable profiling.
     *
     * @default - No profiling.
     * @see https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html
     * @stability stable
     */
    readonly profiling?: boolean;
    /**
     * Profiling Group.
     *
     * @default - A new profiling group will be created if `profiling` is set.
     * @see https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html
     * @stability stable
     */
    readonly profilingGroup?: IProfilingGroup;
    /**
     * Specify the version of CloudWatch Lambda insights to use for monitoring.
     *
     * @default - No Lambda Insights
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-Getting-Started-docker.html
     * @stability stable
     */
    readonly insightsVersion?: LambdaInsightsVersion;
    /**
     * A list of layers to add to the function's execution environment.
     *
     * You can configure your Lambda function to pull in
     * additional code during initialization in the form of layers. Layers are packages of libraries or other dependencies
     * that can be used by multiple functions.
     *
     * @default - No layers.
     * @stability stable
     */
    readonly layers?: ILayerVersion[];
    /**
     * The maximum of concurrent executions you want to reserve for the function.
     *
     * @default - No specific limit - account limit.
     * @see https://docs.aws.amazon.com/lambda/latest/dg/concurrent-executions.html
     * @stability stable
     */
    readonly reservedConcurrentExecutions?: number;
    /**
     * Event sources for this function.
     *
     * You can also add event sources using `addEventSource`.
     *
     * @default - No event sources.
     * @stability stable
     */
    readonly events?: IEventSource[];
    /**
     * The number of days log events are kept in CloudWatch Logs.
     *
     * When updating
     * this property, unsetting it doesn't remove the log retention policy. To
     * remove the retention policy, set the value to `INFINITE`.
     *
     * @default logs.RetentionDays.INFINITE
     * @stability stable
     */
    readonly logRetention?: logs.RetentionDays;
    /**
     * The IAM role for the Lambda function associated with the custom resource that sets the retention policy.
     *
     * @default - A new role is created.
     * @stability stable
     */
    readonly logRetentionRole?: iam.IRole;
    /**
     * When log retention is specified, a custom resource attempts to create the CloudWatch log group.
     *
     * These options control the retry policy when interacting with CloudWatch APIs.
     *
     * @default - Default AWS SDK retry options.
     * @stability stable
     */
    readonly logRetentionRetryOptions?: LogRetentionRetryOptions;
    /**
     * Options for the `lambda.Version` resource automatically created by the `fn.currentVersion` method.
     *
     * @default - default options as described in `VersionOptions`
     * @stability stable
     */
    readonly currentVersionOptions?: VersionOptions;
    /**
     * The filesystem configuration for the lambda function.
     *
     * @default - will not mount any filesystem
     * @stability stable
     */
    readonly filesystem?: FileSystem;
    /**
     * Lambda Functions in a public subnet can NOT access the internet.
     *
     * Use this property to acknowledge this limitation and still place the function in a public subnet.
     *
     * @default false
     * @see https://stackoverflow.com/questions/52992085/why-cant-an-aws-lambda-function-inside-a-public-subnet-in-a-vpc-connect-to-the/52994841#52994841
     * @stability stable
     */
    readonly allowPublicSubnet?: boolean;
    /**
     * The AWS KMS key that's used to encrypt your function's environment variables.
     *
     * @default - AWS Lambda creates and uses an AWS managed customer master key (CMK).
     * @stability stable
     */
    readonly environmentEncryption?: kms.IKey;
    /**
     * Code signing config associated with this function.
     *
     * @default - Not Sign the Code
     * @stability stable
     */
    readonly codeSigningConfig?: ICodeSigningConfig;
    /**
     * (deprecated) DEPRECATED.
     *
     * @default [Architecture.X86_64]
     * @deprecated use `architecture`
     */
    readonly architectures?: Architecture[];
    /**
     * The system architectures compatible with this lambda function.
     *
     * @default Architecture.X86_64
     * @stability stable
     */
    readonly architecture?: Architecture;
}
/**
 * @stability stable
 */
export interface FunctionProps extends FunctionOptions {
    /**
     * The runtime environment for the Lambda function that you are uploading.
     *
     * For valid values, see the Runtime property in the AWS Lambda Developer
     * Guide.
     *
     * Use `Runtime.FROM_IMAGE` when when defining a function from a Docker image.
     *
     * @stability stable
     */
    readonly runtime: Runtime;
    /**
     * The source code of your Lambda function.
     *
     * You can point to a file in an
     * Amazon Simple Storage Service (Amazon S3) bucket or specify your source
     * code as inline text.
     *
     * @stability stable
     */
    readonly code: Code;
    /**
     * The name of the method within your code that Lambda calls to execute your function.
     *
     * The format includes the file name. It can also include
     * namespaces and other qualifiers, depending on the runtime.
     * For more information, see https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-features.html#gettingstarted-features-programmingmodel.
     *
     * Use `Handler.FROM_IMAGE` when defining a function from a Docker image.
     *
     * NOTE: If you specify your source code as inline text by specifying the
     * ZipFile property within the Code property, specify index.function_name as
     * the handler.
     *
     * @stability stable
     */
    readonly handler: string;
}
/**
 * Deploys a file from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 *
 * @stability stable
 */
export declare class Function extends FunctionBase {
    /**
     * Returns a `lambda.Version` which represents the current version of this Lambda function. A new version will be created every time the function's configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.Function`.
     *
     * @stability stable
     */
    get currentVersion(): Version;
    /** @internal */
    static _VER_PROPS: {
        [key: string]: boolean;
    };
    /**
     * Record whether specific properties in the `AWS::Lambda::Function` resource should also be associated to the Version resource.
     *
     * See 'currentVersion' section in the module README for more details.
     *
     * @param propertyName The property to classify.
     * @param locked whether the property should be associated to the version or not.
     * @stability stable
     */
    static classifyVersionProperty(propertyName: string, locked: boolean): void;
    /**
     * Import a lambda function into the CDK using its ARN.
     *
     * @stability stable
     */
    static fromFunctionArn(scope: Construct, id: string, functionArn: string): IFunction;
    /**
     * Creates a Lambda function object which represents a function not defined within this stack.
     *
     * @param scope The parent construct.
     * @param id The name of the lambda construct.
     * @param attrs the attributes of the function to import.
     * @stability stable
     */
    static fromFunctionAttributes(scope: Construct, id: string, attrs: FunctionAttributes): IFunction;
    /**
     * Return the given named metric for this Lambda.
     *
     * @stability stable
     */
    static metricAll(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of Errors executing all Lambdas.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllErrors(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the Duration executing all Lambdas.
     *
     * @default average over 5 minutes
     * @stability stable
     */
    static metricAllDuration(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of invocations of all Lambdas.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllInvocations(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of throttled invocations of all Lambdas.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllThrottles(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of concurrent executions across all Lambdas.
     *
     * @default max over 5 minutes
     * @stability stable
     */
    static metricAllConcurrentExecutions(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas.
     *
     * @default max over 5 minutes
     * @stability stable
     */
    static metricAllUnreservedConcurrentExecutions(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Name of this function.
     *
     * @stability stable
     */
    readonly functionName: string;
    /**
     * ARN of this function.
     *
     * @stability stable
     */
    readonly functionArn: string;
    /**
     * Execution role associated with this function.
     *
     * @stability stable
     */
    readonly role?: iam.IRole;
    /**
     * The runtime configured for this lambda.
     *
     * @stability stable
     */
    readonly runtime: Runtime;
    /**
     * The principal this Lambda Function is running as.
     *
     * @stability stable
     */
    readonly grantPrincipal: iam.IPrincipal;
    /**
     * The DLQ associated with this Lambda Function (this is an optional attribute).
     *
     * @stability stable
     */
    readonly deadLetterQueue?: sqs.IQueue;
    /**
     * The construct node where permissions are attached.
     *
     * @stability stable
     */
    readonly permissionsNode: import("@aws-cdk/core").ConstructNode;
    /**
     * Whether the addPermission() call adds any permissions.
     *
     * True for new Lambdas, false for version $LATEST and imported Lambdas
     * from different accounts.
     *
     * @stability stable
     */
    protected readonly canCreatePermissions = true;
    private readonly layers;
    private _logGroup?;
    /**
     * Environment variables for this function
     */
    private environment;
    private readonly currentVersionOptions?;
    private _currentVersion?;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: FunctionProps);
    /**
     * Adds an environment variable to this Lambda function.
     *
     * If this is a ref to a Lambda function, this operation results in a no-op.
     *
     * @param key The environment variable key.
     * @param value The environment variable's value.
     * @param options Environment variable options.
     * @stability stable
     */
    addEnvironment(key: string, value: string, options?: EnvironmentOptions): this;
    /**
     * Adds one or more Lambda Layers to this Lambda function.
     *
     * @param layers the layers to be added.
     * @stability stable
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayers(...layers: ILayerVersion[]): void;
    /**
     * (deprecated) Add a new version for this Lambda.
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want to
     * deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version.
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda source code, or omit to skip validation.
     * @param description A description for this version.
     * @param provisionedExecutions A provisioned concurrency configuration for a function's version.
     * @param asyncInvokeConfig configuration for this version when it is invoked asynchronously.
     * @returns A new Version object.
     * @deprecated This method will create an AWS::Lambda::Version resource which
     * snapshots the AWS Lambda function *at the time of its creation* and it
     * won't get updated when the function changes. Instead, use
     * `this.currentVersion` to obtain a reference to a version resource that gets
     * automatically recreated when the function configuration (or code) changes.
     */
    addVersion(name: string, codeSha256?: string, description?: string, provisionedExecutions?: number, asyncInvokeConfig?: EventInvokeConfigOptions): Version;
    /**
     * The LogGroup where the Lambda function's logs are made available.
     *
     * If either `logRetention` is set or this property is called, a CloudFormation custom resource is added to the stack that
     * pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the correct log retention
     * period (never expire, by default).
     *
     * Further, if the log group already exists and the `logRetention` is not set, the custom resource will reset the log retention
     * to never expire even if it was configured with a different value.
     *
     * @stability stable
     */
    get logGroup(): logs.ILogGroup;
    /** @internal */
    _checkEdgeCompatibility(): void;
    /**
     * Configured lambda insights on the function if specified. This is acheived by adding an imported layer which is added to the
     * list of lambda layers on synthesis.
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-extension-versions.html
     */
    private configureLambdaInsights;
    private renderEnvironment;
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    private configureVpc;
    private buildDeadLetterQueue;
    private buildDeadLetterConfig;
    private buildTracingConfig;
    private validateProfiling;
}
/**
 * Environment variables options.
 *
 * @stability stable
 */
export interface EnvironmentOptions {
    /**
     * When used in Lambda@Edge via edgeArn() API, these environment variables will be removed.
     *
     * If not set, an error will be thrown.
     *
     * @default false - using the function in Lambda
     * @see https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-requirements-limits.html#lambda-requirements-lambda-function-configuration
     * @stability stable
     * @Edge will throw
     */
    readonly removeInEdge?: boolean;
}
export declare function verifyCodeConfig(code: CodeConfig, props: FunctionProps): void;
