# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['thecollector', 'thecollector.archive']

package_data = \
{'': ['*']}

install_requires = \
['msgpack', 'simplejson']

setup_kwargs = {
    'name': 'thecollector',
    'version': '0.8.4',
    'description': 'A library to store robot data in a msgpack format',
    'long_description': '[![image](https://raw.githubusercontent.com/MomsFriendlyRobotCompany/the-collector/master/pics/header.jpg)](https://github.com/MomsFriendlyRobotCompany/the-collector)\n\n# The Collector\n\n[![Actions Status](https://github.com/MultipedRobotics/dh/workflows/CheckPackage/badge.svg)](https://github.com/MultipedRobotics/pyservos/actions)\n![GitHub](https://img.shields.io/github/license/multipedrobotics/pyservos)\n[![Latest Version](https://img.shields.io/pypi/v/the-collector.svg)](https://pypi.python.org/pypi/the-collector/)\n<!-- [![License](https://img.shields.io/pypi/l/the-collector.svg)](https://pypi.python.org/pypi/the-collector/) -->\n[![image](https://img.shields.io/pypi/pyversions/the-collector.svg)](https://pypi.python.org/pypi/the-collector)\n[![image](https://img.shields.io/pypi/format/the-collector.svg)](https://pypi.python.org/pypi/the-collector)\n\n**This is still under heavy development**\n\nThe idea behind this a container that can store data and time tag the\ndata when it is captured. The main structure is a dict which has keys\nfor each data series stored.\n\nThis was written for a class I taught on robotics. It is meant to be simple and\nteach the students some things. There are probably better solutions out there,\nbut I like this. :smirk:\n\nAdditionally, there is nothing magically about what this does:\n\n- It provides a generic interface to using `pickle`, `json`, or `msgpack` as\nthe protocol for saving data to disk\n- It also allows you to convert between them if needed\n- Bag files can be read using the original protocol, thus data is never lost\nif this library goes away\n- Designed to be simple and straight forward\n\n## Setup\n\n### Install\n\nThe suggested way to install this is via the `pip` command as follows:\n\n    pip install the_collector\n    pip install the_collector[numpy]\n\nIf you install `numpy`, then you get access to working with numpy arrays\nusing the functions: `array_pack()` and `array_pack()`. These really don\'t\nsave you much.\n\n# Usage\n\n## BagIt\n\nBag stores data in memory until the buffer size limit is reached then it dumps\nthe data to a file.\n\n```python\n#!/usr/bin/env python3\nfrom __future__ import print_function\nfrom the_collector import BagIt\nfrom the_collector import Json, MsgPack, Pickle\nimport json\n\n\nd = {\'a\': 1, \'b\': 2}\n\nbag = BagIt(Json)      # ascii, cross platform\n# bag = BagIt(Pickle)  # binary, python only but handles python objects well\n# bag = BagIt(MsgPack) # binary, may have to write special packer for classes\n\nfor i in range(10):\n    bag.push(\'test\', d) # push value d to key \'test\'\n    bag.push(\'bob\', d)\n    bag.push(\'tom\', (\'a\', i,))\n\n# timestamp adds a timestamp automatically to the bag file. Thus, you won\'t\n# over write bob.json.bag each time you run this program because the filename\n# is bob-2019-04-20-15:35:25.6543.json.bag\nfname = bag.write(\'bob\', timestamp=False)\n\nprint(">> created:", fname)\n\ndata = bag.read(fname) # data is just a dictionary\nprint(data)\n```\n\nNow, since there is nothing super special `the_collector` does with packing\ndata, you can always read the bag files using the original libraries:\n\n```python\nwith open(fname, \'rb\') as fd:\n    data = json.load(fd)\n\nfor key, val in data.items():\n    print("{}[{}]".format(key, len(val)))\n    for v in val:\n        print("{}".format(v), end=\' \')\n        print(\' \')\n```\n\n## Circular Buffer\n\nThis is perhaps a little out of place, but I use circular buffers with handling data\nso I stuck it in this library.\n\n```python\nfrom the_collector import CircularBuffer\n\ncb = CircularBuffer(60)  # can only hold 60 items before it copies over data\n\n# Let\'s push way more than 60 things\nfor i in range(200):\n    cb.push(i)\n\nprint(cb.get_all())  # print everything\nprint(\'get cb[7]\', cb[7])\nprint(\'get cb[0]\', cb[0])\nprint(\'get last\', cb.get_last())\n```\n\n## Data Tuple\n\nUse a generic `namedtuple` for data storage. It will automatically insert a\ntimestamp when created. This is useful for tagging data with a timestamp and\nnot having to remember to do it yourself.\n\n**WARNING:** `json` and `msgpack` have issues with `tuples`, so this is best\nused with `pickle` or you have to accept `json` will turn it into a list and\n`msgpack` will turn it into a regular `tuple`. Either way, you still keep the\ntimestamp.\n\n```python\nfrom the_collector import Data\n\nd = Data((1,2,3,))  # timestamp generate when made\n\nprint("Data[{}]: {}".format(d.timestamp, d.data))\nprint("Namedtuple output:", d)\n```\n\n# Todo\n\n- add github actions\n- look at enabling `BytesIO` for testing/working so you don\'t litter filing system\nwith test bag files\n- support PEP517 and replace `setup.py` with `pyprogram.toml`\n- move tests from `nose` to `pytest`\n- see if there is value in `matplotlib` integration or just add an iterator capability\n    - Example: `plt.plot(mydata[\'bob\'])` and will plot 2D or 3D data\n\n# Change Log\n\nDate        | Version| Notes\n------------|--------|----------------------------------\n2019-07-06  | 0.8.2  | add generic data container\n2019-04-28  | 0.8.0  | can store data using `json`, `pickle`, or `msgpack`\n2018-07-25  | 0.7.0  | added `msgpack` messages and a way to do custom messages\n2018-07-14  | 0.6.0  | changed interface to support buffered writing to disk\n2018-07-09  | 0.5.0  | moved away from `json` and now using `msgpack`\n2017-11-23  | 0.4.0  | fixes, documentation, unit tests\n2017-10-04  | 0.0.1  | init\n\n# The MIT License (MIT)\n\nCopyright (c) 2017 Kevin J. Walchko\n\nPermission is hereby granted, free of charge, to any person obtaining a\ncopy of this software and associated documentation files (the\n"Software"), to deal in the Software without restriction, including\nwithout limitation the rights to use, copy, modify, merge, publish,\ndistribute, sublicense, and/or sell copies of the Software, and to\npermit persons to whom the Software is furnished to do so, subject to\nthe following conditions:\n\nThe above copyright notice and this permission notice shall be included\nin all copies or substantial portions of the Software.\n\nTHE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,\nEXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF\nMERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.\nIN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY\nCLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,\nTORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE\nSOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.\n',
    'author': 'walchko',
    'author_email': 'walchko@users.noreply.github.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://pypi.org/project/the-collector/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6',
}


setup(**setup_kwargs)
