"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualGatewayListener = void 0;
const cdk = require("@aws-cdk/core");
const utils_1 = require("./private/utils");
const shared_interfaces_1 = require("./shared-interfaces");
/**
 * (experimental) Represents the properties needed to define listeners for a VirtualGateway.
 *
 * @experimental
 */
class VirtualGatewayListener {
    /**
     * (experimental) Returns an HTTP Listener for a VirtualGateway.
     *
     * @experimental
     */
    static http(options = {}) {
        return new HttpGatewayListener(options);
    }
    /**
     * (experimental) Returns an HTTP2 Listener for a VirtualGateway.
     *
     * @experimental
     */
    static http2(options = {}) {
        return new Http2GatewayListener(options);
    }
    /**
     * (experimental) Returns a GRPC Listener for a VirtualGateway.
     *
     * @experimental
     */
    static grpc(options = {}) {
        return new GrpcGatewayListener(options);
    }
}
exports.VirtualGatewayListener = VirtualGatewayListener;
/**
 * Represents the properties needed to define an HTTP Listener for a VirtualGateway
 */
class HttpGatewayListener extends VirtualGatewayListener {
    constructor(options = {}) {
        super();
        /**
         * Protocol the listener implements
         */
        this.protocol = shared_interfaces_1.Protocol.HTTP;
        this.port = options.port ? options.port : 8080;
        this.healthCheck = options.healthCheck;
    }
    /**
     * Called when the GatewayListener type is initialized. Can be used to enforce
     * mutual exclusivity
     */
    bind(_scope) {
        return {
            listener: {
                portMapping: {
                    port: this.port,
                    protocol: this.protocol,
                },
                healthCheck: this.healthCheck ? renderHealthCheck(this.healthCheck, this.protocol, this.port) : undefined,
            },
        };
    }
}
/**
* Represents the properties needed to define an HTTP2 Listener for a VirtualGateway
*/
class Http2GatewayListener extends HttpGatewayListener {
    constructor(options = {}) {
        super(options);
        this.protocol = shared_interfaces_1.Protocol.HTTP2;
    }
}
/**
 * Represents the properties needed to define a GRPC Listener for Virtual Gateway
 */
class GrpcGatewayListener extends VirtualGatewayListener {
    constructor(options = {}) {
        super();
        /**
         * Protocol the listener implements
         */
        this.protocol = shared_interfaces_1.Protocol.GRPC;
        this.port = options.port ? options.port : 8080;
        this.healthCheck = options.healthCheck;
    }
    /**
     * Called when the GatewayListener type is initialized. Can be used to enforce
     * mutual exclusivity
     */
    bind(_scope) {
        return {
            listener: {
                portMapping: {
                    port: this.port,
                    protocol: shared_interfaces_1.Protocol.GRPC,
                },
                healthCheck: this.healthCheck ? renderHealthCheck(this.healthCheck, this.protocol, this.port) : undefined,
            },
        };
    }
}
function renderHealthCheck(hc, listenerProtocol, listenerPort) {
    if (hc.protocol === shared_interfaces_1.Protocol.TCP) {
        throw new Error('TCP health checks are not permitted for gateway listeners');
    }
    if (hc.protocol === shared_interfaces_1.Protocol.GRPC && hc.path) {
        throw new Error('The path property cannot be set with Protocol.GRPC');
    }
    const protocol = hc.protocol ? hc.protocol : listenerProtocol;
    const healthCheck = {
        healthyThreshold: hc.healthyThreshold || 2,
        intervalMillis: (hc.interval || cdk.Duration.seconds(5)).toMilliseconds(),
        path: hc.path || ((protocol === shared_interfaces_1.Protocol.HTTP || protocol === shared_interfaces_1.Protocol.HTTP2) ? '/' : undefined),
        port: hc.port || listenerPort,
        protocol: hc.protocol || listenerProtocol,
        timeoutMillis: (hc.timeout || cdk.Duration.seconds(2)).toMilliseconds(),
        unhealthyThreshold: hc.unhealthyThreshold || 2,
    };
    utils_1.validateHealthChecks(healthCheck);
    return healthCheck;
}
//# sourceMappingURL=data:application/json;base64,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