"""A2C training utilities with NARX-style features.

This module provides helper classes and functions used to train A2C agents with
NARX (Nonlinear AutoRegressive with eXogenous inputs) representations.
"""

import numpy as np
import torch
from torch import nn
from torch.nn import functional as F
from torch.utils.tensorboard import SummaryWriter


def clip_grad_norm_(module, max_grad_norm):
    """Clip gradients to prevent exploding gradients.

    Args:
        module (torch.optim.Optimizer): Optimizer whose parameter gradients will
            be clipped.
        max_grad_norm (float): Maximum gradient norm.
    """
    nn.utils.clip_grad_norm_(
        [p for g in module.param_groups for p in g["params"]], max_grad_norm
    )


def mish(input):
    """Apply the Mish activation function."""
    return input * torch.tanh(F.softplus(input))


class Mish(nn.Module):
    """PyTorch module implementing the Mish activation."""

    def __init__(self):
        """Initialize Mish activation module."""
        super().__init__()

    def forward(self, input):
        """Forward pass."""
        return mish(input)


def t(x):
    """Convert input to a float PyTorch tensor."""
    x = np.array(x) if not isinstance(x, np.ndarray) else x
    return torch.from_numpy(x).float()


class Actor(nn.Module):
    """Actor network for an actor-critic algorithm.

    Args:
        state_dim (int): State dimension.
        n_actions (int): Action dimension.
        activation (torch.nn.Module): Activation class (e.g., ``nn.Tanh``).
    """

    def __init__(self, state_dim, n_actions, activation=nn.Tanh):
        """Initialize the actor network."""
        super().__init__()
        self.n_actions = n_actions
        self.model = nn.Sequential(
            nn.Linear(state_dim, 64),
            activation(),
            nn.Linear(64, 64),
            activation(),
            nn.Linear(64, n_actions),
        )

        logstds_param = nn.Parameter(torch.full((n_actions,), 0.1))
        self.register_parameter("logstds", logstds_param)

    def forward(self, X):
        """Compute an action distribution for a batch of states."""
        means = self.model(X)
        stds = torch.clamp(self.logstds.exp(), 1e-3, 50)

        return torch.distributions.Normal(means, stds)


class Critic(nn.Module):
    """Critic network for an actor-critic algorithm.

    Args:
        state_dim (int): State dimension.
        activation (torch.nn.Module): Activation class (e.g., ``nn.Tanh``).
    """

    def __init__(self, state_dim, activation=nn.Tanh):
        """Build critic network layers.

        Args:
            state_dim: State dimension.
            activation: Activation module class.
        """
        super().__init__()
        self.model = nn.Sequential(
            nn.Linear(state_dim + state_dim, 64),
            activation(),
            nn.Linear(64, 64),
            activation(),
            nn.Linear(64, 1),
        )

    def forward(self, X):
        """Estimate state value for input features."""
        return self.model(X)


def discounted_rewards(rewards, dones, gamma):
    """Compute discounted returns for a sequence of rewards."""
    ret = 0
    discounted = []
    for reward, done in zip(rewards[::-1], dones[::-1]):
        ret = reward + ret * gamma * (1 - done)
        discounted.append(ret)

    return discounted[::-1]


def process_memory_narx(memory, gamma=0.99, discount_rewards=True):
    """Convert collected transitions into tensors suitable for training.

    The function also builds an augmented critic input that concatenates the
    current state with a lagged state (previous step) to mimic a NARX-style
    representation.

    Args:
        memory (list[tuple]): Tuples ``(action, reward, state, next_state, done)``.
        gamma (float): Discount factor. Defaults to ``0.99``.
        discount_rewards (bool): If True, uses discounted returns. Defaults to True.

    Returns:
        tuple: ``(actions, rewards, states, next_states, dones, critic_states)``.
    """
    actions = []
    states = []
    next_states = []
    rewards = []
    dones = []
    critic_states = []  # Инициализация для хранения состояний и предыдущих действий

    # Используем None или 0 как заполнитель для предыдущего действия первого состояния
    prev_state = np.zeros(memory[0][2].shape)
    prev_next_state = np.zeros(memory[0][2].shape)
    for action, reward, state, next_state, done in memory:
        actions.append(action)
        rewards.append(reward)
        states.append(state)
        next_states.append(np.concatenate((next_state.flatten(), prev_next_state)))
        dones.append(done)
        # Добавляем текущее состояние и предыдущее действие в hist_values
        critic_states.append(np.concatenate((state.flatten(), prev_state)))
        prev_state = (
            state.flatten()
        )  # Обновляем предыдущее действие для следующей итерации
        prev_next_state = next_state.flatten()  #
    if discount_rewards:
        rewards = discounted_rewards(rewards, dones, gamma)

    actions = t(actions).view(-1, 1)
    states = t(states)
    next_states = t(next_states)
    rewards = t(rewards).view(-1, 1)
    dones = t(dones).view(-1, 1)
    critic_states = t(critic_states)  # Преобразование списка в тензор

    return actions, rewards, states, next_states, dones, critic_states


class A2CLearner:
    """Learner implementing Advantage Actor-Critic (A2C) updates."""

    def __init__(
        self,
        actor,
        critic,
        gamma=0.9,
        entropy_beta=0.01,
        actor_lr=4e-4,
        critic_lr=4e-3,
        max_grad_norm=0.5,
    ):
        """Initialize learner with optimizers and hyperparameters.

        Args:
            actor: Policy network.
            critic: Value network.
            gamma: Discount factor.
            entropy_beta: Entropy regularization weight.
            actor_lr: Learning rate for actor.
            critic_lr: Learning rate for critic.
            max_grad_norm: Gradient clipping norm.
        """
        self.gamma = gamma
        self.max_grad_norm = max_grad_norm
        self.actor = actor
        self.critic = critic
        self.entropy_beta = entropy_beta
        self.actor_optim = torch.optim.Adam(actor.parameters(), lr=actor_lr)
        self.critic_optim = torch.optim.Adam(critic.parameters(), lr=critic_lr)
        self.writer = SummaryWriter()

    def learn(self, memory, steps, discount_rewards=True):
        """Update actor/critic using a batch of collected transitions.

        Args:
            memory (list): Collected transitions.
            steps (int): Global step index used for logging.
            discount_rewards (bool): If True, uses discounted returns as TD target.
        """
        (
            actions,
            rewards,
            states,
            next_states,
            dones,
            critic_states,
        ) = process_memory_narx(memory, self.gamma, discount_rewards)

        if discount_rewards:
            td_target = rewards
        else:
            td_target = rewards + self.gamma * self.critic(next_states) * (1 - dones)
        value = self.critic(critic_states)
        advantage = td_target - value

        # actor
        norm_dists = self.actor(states)
        logs_probs = norm_dists.log_prob(actions)
        entropy = norm_dists.entropy().mean()

        actor_loss = (
            -logs_probs * advantage.detach()
        ).mean() - entropy * self.entropy_beta
        self.actor_optim.zero_grad()
        actor_loss.backward()

        clip_grad_norm_(self.actor_optim, self.max_grad_norm)
        self.writer.add_histogram(
            "gradients/actor",
            torch.cat([p.grad.view(-1) for p in self.actor.parameters()]),
            global_step=steps,
        )
        self.writer.add_histogram(
            "parameters/actor",
            torch.cat([p.data.view(-1) for p in self.actor.parameters()]),
            global_step=steps,
        )
        self.actor_optim.step()

        # critic
        critic_loss = F.mse_loss(td_target, value)
        self.critic_optim.zero_grad()
        critic_loss.backward()
        clip_grad_norm_(self.critic_optim, self.max_grad_norm)
        self.writer.add_histogram(
            "gradients/critic",
            torch.cat([p.grad.view(-1) for p in self.critic.parameters()]),
            global_step=steps,
        )
        self.writer.add_histogram(
            "parameters/critic",
            torch.cat([p.data.view(-1) for p in self.critic.parameters()]),
            global_step=steps,
        )
        self.critic_optim.step()

        # reports
        self.writer.add_scalar(
            "losses/log_probs", -logs_probs.mean(), global_step=steps
        )
        self.writer.add_scalar("losses/entropy", entropy, global_step=steps)
        self.writer.add_scalar(
            "losses/entropy_beta", self.entropy_beta, global_step=steps
        )
        self.writer.add_scalar("losses/actor", actor_loss, global_step=steps)
        self.writer.add_scalar("losses/advantage", advantage.mean(), global_step=steps)
        self.writer.add_scalar("losses/critic", critic_loss, global_step=steps)


class Runner:
    """Environment interaction loop used to collect training data."""

    def __init__(self, env, actor, writer):
        """Create runner for data collection.

        Args:
            env: Environment instance.
            actor: Policy network used to select actions.
            writer: TensorBoard writer for logging rewards.
        """
        self.env = env
        self.actor = actor
        self.state = None
        self.done = True
        self.steps = 0
        self.episode_reward = 0
        self.episode_rewards = []
        # Initialize previous action as zeros; adjust the size based on your action space
        self.prev_action = np.zeros(self.env.action_space.shape)
        self.writer = writer

    @staticmethod
    def _flatten_observation(observation):
        """Flatten environment observations to shape ``(n,)``."""
        return np.asarray(observation, dtype=np.float32).reshape(-1)

    def reset(self):
        """Reset environment and episode state."""
        self.episode_reward = 0
        self.done = False
        self.state, info = self.env.reset()
        self.state = self._flatten_observation(self.state)
        # Reset previous action at the start of each episode
        self.prev_action = np.zeros(self.env.action_space.shape)

    def run(self, max_steps, memory=None) -> list:
        """Run the environment for a fixed number of steps and collect transitions.

        Args:
            max_steps (int): Number of environment steps to execute.
            memory (list, optional): Existing list to append transitions to.

        Returns:
            list: Collected transitions.
        """
        if not memory:
            memory = []

        for i in range(max_steps):
            if self.done:
                self.reset()

            dists = self.actor(
                torch.tensor(self.state, dtype=torch.float32).unsqueeze(0)
            )
            actions = dists.sample().detach().numpy()
            actions_clipped = np.clip(
                actions, self.env.action_space.low, self.env.action_space.high
            )

            next_state, reward, terminated, truncated, info = self.env.step(
                actions_clipped[0]
            )
            self.done = terminated or truncated
            next_state = self._flatten_observation(next_state)

            # Here, instead of just the state, we store the state concatenated with the previous action
            memory.append((actions, reward, self.state, next_state, self.done))

            self.prev_action = actions_clipped[0]  # Update the previous action
            self.state = next_state
            self.steps += 1
            self.episode_reward += reward

            if self.done:
                self.episode_rewards.append(self.episode_reward)
                # Assuming writer is defined and configured globally
                self.writer.add_scalar(
                    "episode_reward", self.episode_reward, global_step=self.steps
                )

        return memory
