import os

import matlab.engine
import numpy as np

from tensoraerospace.aerospacemodel.base import ModelBase


class LongitudinalF16(ModelBase):
    r"""F-16 aircraft ✈ in isolated longitudinal channel.

    Action space:
        stab_act: elevator [rad]

    State space:
        alpha: angle of attack [rad]
        wz: pitch angular velocity [rad/s]
        stab: elevator position [rad]
        dstab: elevator angular velocity [rad/s]

    Example usage:
        >>> model = LongitudinalF16(initial_state)
        >>> x_t = model.run_step([ [0], ])

    Args:
        x0: Initial state.
        t0: (Optional) Initial time.
        dt: (Optional) Discretization step.
    """

    def __init__(self, x0, selected_state_output=None, t0=0, dt: float = 0.01):
        super(LongitudinalF16, self).__init__(x0, selected_state_output, t0, dt)
        self.matlab_files_path = os.path.join(os.path.dirname(__file__), "matlab_code")
        self.eng = matlab.engine.start_matlab()  # Start Matlab instance
        self.eng.addpath(self.matlab_files_path)
        self.list_state = ["alpha", "wz", "stab", "dstab"]
        self.control_list = [
            "stab",
        ]
        self.action_space_length = len(self.control_list)
        self.param = self.eng.airplane_parameters()  # Get control object parameters
        self.x_history = [x0]
        self._initialize_selected_state_index(
            self.selected_state_output, self.list_state
        )

    def get_param(self):
        """Get control object parameters.

        Returns:
            Control object parameters.
        """
        return self.param

    def set_param(self, new_param):
        """Set new control object parameters.

        Args:
           new_param: Control object parameters.
        """
        self.param = new_param

    def run_step(self, u: matlab.double):
        """Calculate control object state.

        Control signal format:

        >>> stab_act = 0
        >>> [
        >>>    [stab_act],
        >>> ]

        Args:
            u: Control signal.

        Returns:
            Control object state.

        Usage example:

        >>> from tensoraerospace.aerospacemodel.f16.nonlinear.longitudinal import initial_state, LongitudinalF16
        >>> model = LongitudinalF16(initial_state)
        >>> x_t = model.run_step([ [0], ])
        """
        if not isinstance(u, matlab.double):
            u = matlab.double(u)
        if len(list(u)) != self.action_space_length:
            raise Exception(
                "Размерность управляющего вектора задана неверно."
                + f" Текущее значение {len(list(u))}, не соответсвует {self.action_space_length}"
            )
        x_t = self.eng.step(
            self.x_history[-1], self.dt, u, self.t0, self.time_step, self.param
        )
        self.x_history.append(x_t)
        self.u_history.append(u)
        self.time_step += 1
        if self.selected_state_output:
            return np.array(x_t[self.selected_state_index])
        return np.array(x_t)
