function mx = GetMx(alpha,beta,fi,drn,del,dnos,Wx,Wy,V,l)
% --------------------------------------------------------------
%       Вычисление коэффициента момента крена mx.
% --------------------------------------------------------------
% mx = GetMx(alpha,beta,fi,drn,del,Wx,Wy,V)  вычисляет коэффициент
%    момента крена в зависимости от величин:
% alpha - угол атаки, рад.
% beta - угол скольжения, рад.
% fi - отклонение стабилизатора, рад.
% drn - отклонение руля направления, рад.
% del - отклонение элеронов (дифференциальное), рад.
% dnos - отклонение носков крыла, рад.
% V - воздушная скорость, м/с
% Wx - угловая скорость крена, рад/с
% Wy - угловая скорость рыскания, рад/с
%
% Зависимости заданы в виде таблиц.
% Таблицы интерполяции хранятся в оперативной памяти для ускорения
% численного расчёта.
% --------------------------------------------------------------
persistent init_mdl
persistent alpha1 alpha2 beta1 fi2 mx1 mx_nos1 mxdel20 mxdel20_nos
persistent mxdrn30 dmxbt1 mxwy1 dmxwy_nos1 mxwx1 dmxwx_nos1
persistent interpmx interpmx_nos interpdmxbt interpmxwx1 interpmxwx2 interpmxwy1 interpmxwy2 interpmxdel interpmxdel_nos interpmxdrn


% Инициализация
if isempty(init_mdl)

init_mdl = 1;

alpha1 = deg2rad([-20 -15 -10 -5 0 5 10 15 20 25 30 35 40 45 50 55 60 70 80 90]);
alpha2 = deg2rad([-20 -15 -10 -5 0 5 10 15 20 25 30 35 40 45]);
beta1 = deg2rad([-30 -25 -20 -15 -10 -8 -6 -4 -2 0 2 4 6 8 10 15 20 25 30]);
fi2 = deg2rad([-25 0 25]);

% Коэффициент момента крена от углов атаки и скольжения.
% mx = f(alpha, beta, fi)

% fi = -25 grad
mx1(:,:,1) = [-0.006	0.0065	0.0133	0.0217	0.0268	0.0238	0.0219	0.0179	0.0121	0	-0.0096	-0.0167	-0.021	-0.0239	-0.0245	-0.0196	-0.0107	-0.0039	-0.0118;
    -0.0048	0.0059	0.0178	0.0242	0.0187	0.0157	0.013	0.0106	0.0061	0	-0.0059	-0.0101	-0.0146	-0.0162	-0.0189	-0.0245	-0.0179	-0.006	0.0048;
    -0.0033	0.0095	0.0173	0.0184	0.0128	0.01	0.0088	0.0056	0.0027	0	-0.0047	-0.0077	-0.0118	-0.0136	-0.0158	-0.022	-0.014	-0.006	0.0069;
    0.0298	0.0245	0.0233	0.0211	0.0178	0.0144	0.0113	0.0072	0.003	0	-0.0039	-0.0081	-0.0123	-0.0149	-0.0188	-0.0221	-0.0241	-0.0253	-0.0209;
    0.0276	0.0285	0.0262	0.0225	0.0189	0.0151	0.0112	0.0075	0.0035	0	-0.0035	-0.0075	-0.0114	-0.0151	-0.0187	-0.0223	-0.026	-0.0283	-0.0274;
    0.039	0.0337	0.039	0.0282	0.024	0.0195	0.0142	0.0096	0.0049	0	-0.0047	-0.0094	-0.0138	-0.0188	-0.023	-0.0292	-0.0339	-0.0349	-0.0398;
    0.0562	0.0558	0.054	0.0455	0.0346	0.0285	0.0218	0.0147	0.0067	0	-0.0068	-0.0143	-0.0219	-0.0282	-0.0343	-0.0447	-0.0531	-0.0546	-0.055;
    0.0737	0.067	0.0629	0.0568	0.0439	0.0361	0.0272	0.0185	0.0091	0	-0.0087	-0.0183	-0.0286	-0.0367	-0.0433	-0.0568	-0.0626	-0.0672	-0.0736;
    0.0761	0.0708	0.0654	0.0551	0.0454	0.0377	0.0284	0.0185	0.0093	0	-0.0101	-0.018	-0.0293	-0.0369	-0.0448	-0.0542	-0.0642	-0.0694	-0.0743;
    0.091	0.0713	0.0627	0.0513	0.0397	0.0331	0.0261	0.0175	0.008	0	-0.0089	-0.0174	-0.0263	-0.0347	-0.0411	-0.0525	-0.0637	-0.0724	-0.0797;
    0.0743	0.0429	0.0101	0.011	0.0025	0.0152	0.018	0.0126	0.0091	0	-0.0066	-0.0124	-0.016	-0.0194	-0.0225	-0.0308	-0.035	-0.0628	-0.0943;
    0.0704	0.053	0.0453	0.0184	0.0067	-0.002	-0.0017	0.0028	-0.0011	0	0.0018	0.0009	-0.0003	0.003	0.01	-0.0017	-0.0281	-0.0358	-0.0533;
    0.0665	0.0605	0.0353	0.0132	0.0077	0.0092	0.0156	0.0096	0.0048	0	-0.0077	-0.0117	-0.0123	-0.015	-0.013	-0.018	-0.0403	-0.0656	-0.0716;
    0.0788	0.0563	0.0344	0.0234	0.015	0.014	0.0091	0.0089	0.0037	0	-0.0052	-0.0082	-0.0124	-0.0135	-0.0178	-0.0274	-0.037	-0.0579	-0.0804;
    0.0605	0.0568	0.0469	0.034	0.0169	0.0146	0.0129	0.0089	0.0055	0	-0.0022	-0.0065	-0.009	-0.017	-0.02	-0.0371	-0.05	-0.0599	-0.0636;
    0.0453	0.0323	0.0257	0.014	-0.0003	0.0024	0.0042	0.0025	0.0025	0	-0.0064	-0.013	-0.0176	-0.028	-0.0173	-0.0316	-0.0433	-0.0499	-0.0629;
    0.061	0.0413	0.0336	0.023	0.0137	0.0122	0.0106	0.0064	0.0048	0	-0.0026	-0.0049	-0.0095	-0.0132	-0.0141	-0.0234	-0.034	-0.0417	-0.0614;
    0.0713	0.0603	0.0501	0.0191	0.0221	0.019	0.0124	0.0097	0.0057	0	-0.0066	-0.0102	-0.0143	-0.0153	-0.0172	-0.0292	-0.0466	-0.0568	-0.0678;
    0.0614	0.0507	0.0405	0.0309	0.0202	0.0167	0.0167	0.0078	0.0067	0	-0.0039	-0.0075	-0.0124	-0.0156	-0.019	-0.0297	-0.0393	-0.0495	-0.0602;
    0.0601	0.046	0.0363	0.0253	0.0213	0.0183	0.0147	0.0091	0.0056	0	-0.0006	-0.0012	-0.0086	-0.0152	-0.0191	-0.0231	-0.0341	-0.0438	-0.0579];

% fi = 0 grad
mx1(:,:,2) = [-0.0153	-0.0028	0.0091	0.0188	0.0234	0.0173	0.0106	0.009	0.0041	0	-0.0031	-0.0064	-0.0084	-0.0128	-0.0171	-0.012	-0.0022	0.0097	0.0225;
    -0.0132	-0.0028	0.0077	0.0145	0.0104	0.0084	0.006	0.0039	0.0025	0	-0.0029	-0.005	-0.008	-0.0086	-0.0116	-0.016	-0.009	0.005	0.0119;
    -0.0102	-0.0013	0.0094	0.0134	0.0107	0.0102	0.0081	0.006	0.0011	0	-0.0004	-0.0048	-0.0071	-0.0091	-0.0102	-0.0135	-0.0094	0.0013	0.0108;
    0.0087	0.0153	0.0186	0.0194	0.0183	0.0156	0.0125	0.0088	0.0043	0	-0.0038	-0.0087	-0.0126	-0.0158	-0.0186	-0.0199	-0.0189	-0.0157	-0.0095;
    0.0157	0.019	0.0199	0.0207	0.0185	0.0153	0.011	0.0071	0.0033	0	-0.003	-0.0067	-0.0107	-0.0147	-0.0182	-0.0204	-0.0196	-0.0187	-0.0154;
    0.0318	0.0307	0.0296	0.0272	0.0219	0.018	0.0132	0.0089	0.0043	0	-0.0037	-0.0081	-0.0126	-0.0173	-0.0209	-0.0263	-0.0288	-0.0299	-0.031;
    0.051	0.051	0.0496	0.0422	0.0328	0.0271	0.0207	0.0139	0.0056	0	-0.0065	-0.0137	-0.0207	-0.0266	-0.0322	-0.0418	-0.0486	-0.0501	-0.0501;
    0.0732	0.0679	0.0638	0.0574	0.0433	0.0357	0.0274	0.0187	0.009	0	-0.0088	-0.0188	-0.0284	-0.0369	-0.043	-0.0567	-0.0624	-0.0663	-0.0714;
    0.0895	0.0815	0.0692	0.0579	0.0453	0.0354	0.027	0.0171	0.0076	0	-0.0085	-0.0177	-0.0271	-0.0365	-0.044	-0.0569	-0.0682	-0.0804	-0.0884;
    0.0884	0.0785	0.0665	0.0536	0.04	0.0326	0.0254	0.0181	0.0081	0	-0.0082	-0.0165	-0.0258	-0.033	-0.0403	-0.0538	-0.0669	-0.0788	-0.0882;
    0.082	0.0505	0.0234	0.0143	0.0064	0.0189	0.0196	0.0133	0.0071	0	-0.0057	-0.0118	-0.0165	-0.0205	-0.0248	-0.0327	-0.0418	-0.0687	-0.1003;
    0.079	0.061	0.039	0.0095	0.0037	0.0029	0.015	0.0143	0.0097	0	0.0016	0.0003	-0.0018	-0.0017	0.0011	-0.0034	-0.0326	-0.0547	-0.0726;
    0.0721	0.0573	0.0302	0.0087	0.005	0.0104	0.0174	0.0124	0.0062	0	-0.0075	-0.0108	-0.0131	-0.0145	-0.0148	-0.0185	-0.0399	-0.0671	-0.0815;
    0.0744	0.0576	0.0331	0.0248	0.017	0.0179	0.0163	0.0191	0.0115	0	-0.0042	-0.018	-0.0148	-0.0156	-0.0206	-0.0285	-0.0364	-0.0608	-0.0778;
    0.0534	0.0411	0.0262	0.0238	0.0147	0.0144	0.013	0.0091	0.0056	0	-0.0051	-0.0123	-0.0152	-0.0212	-0.0222	-0.0313	-0.0337	-0.0486	-0.0609;
    0.0587	0.0422	0.032	0.0261	0.0176	0.0151	0.0117	0.0065	0.0045	0	-0.004	-0.0081	-0.0133	-0.0187	-0.0201	-0.0286	-0.0346	-0.0488	-0.0613;
    0.065	0.0481	0.0387	0.0301	0.0229	0.0192	0.0155	0.0094	0.0063	0	-0.0029	-0.0055	-0.0111	-0.0163	-0.0188	-0.026	-0.0346	-0.044	-0.0609;
    0.0663	0.0538	0.0422	0.0307	0.0245	0.022	0.016	0.0128	0.0073	0	-0.005	-0.0069	-0.012	-0.0165	-0.022	-0.0282	-0.0397	-0.0513	-0.0638;
    0.0683	0.0554	0.043	0.0325	0.0208	0.0149	0.0126	0.0036	0.0045	0	-0.0045	-0.0086	-0.0134	-0.0159	-0.0186	-0.0303	-0.0408	-0.0532	-0.0661;
    0.0701	0.0534	0.041	0.0293	0.0205	0.0188	0.0163	0.011	0.0066	0	0	-0.0001	-0.0067	-0.0124	-0.0154	-0.0242	-0.0359	-0.0483	-0.065];

% fi = 25 grad
mx1(:,:,3) = [-0.01380	-0.00900	0.01060	0.02270	0.02480	0.01450	0.01120	0.00500	0.00310	0.00000	-0.00330	-0.00810	-0.00770	-0.01600	-0.01860	-0.01670	-0.00440	-0.00740	-0.02560;
    -0.00610	0.00330	0.01400	0.02090	0.01570	0.01050	0.00660	0.00600	0.00270	0.00000	-0.00240	-0.00490	-0.00750	-0.00900	-0.01340	-0.01830	-0.01230	-0.00170	0.00810;
    0.00000	0.00740	0.01310	0.01510	0.01390	0.01080	0.00880	0.00340	0.00080	0.00000	-0.00060	-0.00510	-0.00760	-0.00960	-0.01300	-0.01400	-0.01230	-0.00650	0.00080;
    0.01710	0.01960	0.01860	0.02040	0.01810	0.01420	0.01110	0.00810	0.00390	0.00000	-0.00350	-0.00710	-0.01090	-0.01410	-0.01750	-0.02030	-0.01840	-0.01940	-0.01740;
    0.02670	0.02610	0.02450	0.02150	0.01880	0.01470	0.01050	0.00580	0.00260	0.00000	-0.00290	-0.00650	-0.01080	-0.01520	-0.01880	-0.02150	-0.02470	-0.02630	-0.02690;
    0.04270	0.03760	0.03550	0.02850	0.02200	0.01800	0.01380	0.00990	0.00650	0.00000	-0.00610	-0.01110	-0.01430	-0.01860	-0.02280	-0.02930	-0.03580	-0.03780	-0.04250;
    0.06220	0.05960	0.05510	0.04540	0.03310	0.02660	0.02080	0.01460	0.00740	0.00000	-0.00670	-0.01580	-0.02210	-0.02710	-0.03320	-0.04410	-0.05340	-0.05730	-0.05970;
    0.07760	0.06960	0.06230	0.05440	0.04350	0.03720	0.03030	0.02130	0.01120	0.00000	-0.01100	-0.02190	-0.03030	-0.03790	-0.04540	-0.05600	-0.06370	-0.07080	-0.07830;
    0.08300	0.07940	0.06940	0.05580	0.04270	0.03320	0.02430	0.01720	0.00790	0.00000	-0.01020	-0.02020	-0.02150	-0.02940	-0.03940	-0.05200	-0.06530	-0.07530	-0.07840;
    0.08920	0.07600	0.06350	0.05240	0.03060	0.02140	0.01740	0.01360	0.00610	0.00000	-0.00770	-0.01420	-0.02020	-0.02210	-0.02770	-0.04930	-0.06050	-0.07320	-0.08640;
    0.07910	0.04520	0.01940	0.00410	-0.00460	0.01120	0.01090	0.00610	0.00310	0.00000	-0.00380	-0.00720	-0.01070	-0.01280	-0.01560	-0.02410	-0.03910	-0.06490	-0.09890;
    0.07510	0.05630	0.03480	0.00710	-0.00300	-0.00770	-0.00020	0.00850	0.00160	0.00000	-0.00040	-0.00060	0.00050	0.00290	0.00860	-0.00130	-0.02910	-0.05030	-0.06910;
    0.06730	0.05830	0.02970	0.00500	-0.00020	0.00310	0.01060	0.00530	0.00550	0.00000	-0.00540	-0.00770	-0.00990	-0.00580	-0.00680	-0.01200	-0.03670	-0.06540	-0.07410;
    0.07780	0.06250	0.04110	0.03260	0.01870	0.01630	0.01410	0.01650	0.01150	0.00000	-0.00210	-0.00790	-0.01050	-0.01340	-0.01490	-0.02880	-0.03750	-0.05880	-0.07380;
    0.06190	0.05190	0.03930	0.03260	0.01920	0.01170	0.01510	0.01030	0.00620	0.00000	-0.00470	-0.01150	-0.01510	-0.02300	-0.02580	-0.03920	-0.04590	-0.05850	-0.06850;
    0.04760	0.03360	0.02580	0.01490	0.00160	0.00450	0.00660	0.00460	0.00350	0.00000	-0.00780	-0.01570	-0.02150	-0.02440	-0.02320	-0.03650	-0.04740	-0.05520	-0.06920;
    0.06110	0.04280	0.03210	0.02630	0.02190	0.01650	0.01610	0.01020	0.00710	0.00000	-0.00420	-0.00810	-0.01420	-0.01900	-0.02110	-0.02550	-0.03130	-0.04200	-0.06030;
    0.06540	0.05020	0.03580	0.02240	0.01850	0.01750	0.01300	0.01120	0.00640	0.00000	-0.00640	-0.00970	-0.01460	-0.01810	-0.02090	-0.02900	-0.04240	-0.05680	-0.07200;
    0.06380	0.05060	0.03800	0.02870	0.01790	0.01380	0.01340	0.00500	0.00520	0.00000	-0.00280	-0.00520	-0.01010	-0.01470	-0.01940	-0.03020	-0.03950	-0.05210	-0.06530;
    0.06070	0.04860	0.04070	0.03050	0.02110	0.01800	0.01650	0.01160	0.00700	0.00000	-0.00080	-0.00170	-0.01980	-0.01300	-0.01500	-0.02440	-0.03460	-0.04250	-0.05460];

% Коэффициент момента крена от углов атаки и скольжения, носки выпущены.
% mx_nos = f(alpha2, beta)
mx_nos1 = [-0.0205	-0.017	-0.0076	0.0047	0.015	0.0134	0.0008	0.0013	0.0027	0	-0.0012	-0.0031	-0.0054	-0.0051	-0.0054	0.0036	0.0159	0.0256	0.0297;
    -0.006	-0.0042	-0.0007	0.0033	0.0006	-0.0002	0.0022	0.0039	0.0019	0	-0.0015	-0.003	-0.0039	-0.0028	-0.0003	-0.0024	0.0023	0.0052	0.0072;
    -0.0081	-0.0061	-0.0001	0.0018	0.0034	0.0022	0.0016	0.0006	0	0	-0.0003	-0.0008	-0.0011	-0.002	-0.0026	-0.0007	0.0007	0.0071	0.0093;
    0.0106	0.0102	0.0104	0.0103	0.0093	0.0073	0.0052	0.003	0.0012	0	-0.001	-0.0027	-0.0044	-0.0065	-0.0087	-0.01	-0.0103	-0.0102	-0.0106;
    0.0238	0.0232	0.0224	0.024	0.0168	0.0134	0.0098	0.006	0.0029	0	-0.0027	-0.0058	-0.0094	-0.0134	-0.0164	-0.02	-0.022	-0.0228	-0.0234;
    0.039	0.0361	0.0353	0.0315	0.0248	0.0202	0.0149	0.01	0.0049	0	-0.0049	-0.01	-0.0149	-0.0206	-0.025	-0.0317	-0.0355	-0.0363	-0.0392;
    0.0485	0.0463	0.043	0.0347	0.0263	0.0213	0.0155	0.01	0.0046	0	-0.0048	-0.0115	-0.0175	-0.0227	-0.0278	-0.036	-0.0443	-0.0476	-0.0498;
    0.0462	0.0462	0.045	0.042	0.0297	0.0241	0.0172	0.0113	0.0052	0	-0.0056	-0.0123	-0.0187	-0.0248	-0.0304	-0.0369	-0.0399	-0.0411	-0.0411;
    0.048	0.0335	0.029	0.0209	0.0158	0.0141	0.0095	0.0058	0.0005	0	-0.006	-0.0117	-0.0175	-0.0183	-0.0203	-0.025	-0.0335	-0.0378	-0.0521;
    0.0731	0.0573	0.0371	0.0221	0.0233	0.0203	0.0175	0.012	0.0061	0	-0.0058	-0.0128	-0.0183	-0.0186	-0.026	-0.0248	-0.0398	-0.061	-0.0758;
    0.0752	0.0632	0.0428	0.0235	0.0106	0.0133	0.0138	0.0094	0.0075	0	-0.0063	-0.0095	-0.011	-0.0101	-0.0151	-0.028	-0.0473	-0.0677	-0.0797;
    0.0528	0.0479	0.0422	0.019	0.0078	0.0069	0.0117	0.007	0.0022	0	0.0014	-0.0057	-0.0076	-0.0077	-0.0091	-0.0203	-0.0435	-0.0492	-0.0541;
    0.0555	0.0435	0.0339	0.0173	0.0094	0.0156	0.0193	0.011	0.011	0	-0.0074	-0.0126	-0.0194	-0.0213	-0.0253	-0.0332	-0.0498	-0.0594	-0.0714;
    0.05	0.0493	0.0351	0.0306	0.0179	0.0158	0.0128	0.0077	0.0019	0	-0.0118	-0.0124	-0.015	-0.0173	-0.0214	-0.0341	-0.0386	-0.0528	-0.0535];

% Коэффициент момента крена при максимальном отклонении элеронов (20 град)
% mxdel20 = f(alpha1,beta)
mxdel20 = [-0.0499   -0.0325   -0.0184   -0.0113   -0.0023   -0.0059   -0.0125   -0.0116   -0.0170   -0.0211   -0.0242   -0.0271   -0.0331   -0.0352   -0.0392  -0.0302   -0.0231   -0.0090    0.0084;
   -0.0476   -0.0346   -0.0215   -0.0132   -0.0180   -0.0211   -0.0246   -0.0248   -0.0284   -0.0311   -0.0320   -0.0341   -0.0366   -0.0349   -0.0386  -0.0434   -0.0351   -0.0220   -0.0090;
   -0.0452   -0.0339   -0.0272   -0.0245   -0.0250   -0.0259   -0.0267   -0.0292   -0.0337   -0.0325   -0.0327   -0.0349   -0.0371   -0.0401   -0.0433  -0.0438   -0.0411   -0.0344   -0.0231;
   -0.0344   -0.0303   -0.0258   -0.0230   -0.0242   -0.0270   -0.0301   -0.0334   -0.0368   -0.0402   -0.0440   -0.0480   -0.0511   -0.0541   -0.0555  -0.0567   -0.0539   -0.0494   -0.0453;
   -0.0404   -0.0372   -0.0327   -0.0302   -0.0323   -0.0342   -0.0373   -0.0414   -0.0451   -0.0482   -0.0510   -0.0536   -0.0570   -0.0595   -0.0624  -0.0645   -0.0620   -0.0575   -0.0543;
   -0.0251   -0.0256   -0.0241   -0.0252   -0.0297   -0.0334   -0.0378   -0.0425   -0.0472   -0.0517   -0.0554   -0.0586   -0.0618   -0.0653   -0.0685  -0.0730   -0.0741   -0.0726   -0.0731;
   -0.0031   -0.0026   -0.0027   -0.0091   -0.0185   -0.0235   -0.0301   -0.0366   -0.0437   -0.0501   -0.0557   -0.0608   -0.0665   -0.0717   -0.0771  -0.0865   -0.0929   -0.0930   -0.0925;
    0.0158    0.0145    0.0121    0.0063   -0.0068   -0.0135   -0.0214   -0.0313   -0.0401   -0.0492   -0.0576   -0.0656   -0.0729   -0.0780   -0.0855  -0.0986   -0.1044   -0.1068   -0.1081;
    0.0074    0.0045    0.0038    0.0063    0.0026   -0.0053   -0.0137   -0.0228   -0.0322   -0.0416   -0.0515   -0.0606   -0.0689   -0.0784   -0.0863  -0.0900   -0.0875   -0.0882   -0.0911;
    0.0296    0.0258    0.0237    0.0157    0.0046   -0.0025   -0.0105   -0.0202   -0.0287   -0.0374   -0.0454   -0.0536   -0.0617   -0.0686   -0.0763  -0.0874   -0.0954   -0.0975   -0.1013;
    0.0404    0.0081   -0.0149   -0.0074   -0.0196   -0.0105   -0.0122   -0.0193   -0.0244   -0.0306   -0.0362   -0.0429   -0.0456   -0.0493   -0.0501  -0.0623   -0.0548   -0.0778   -0.1101;
    0.0309    0.0126    0.0020   -0.0246   -0.0223   -0.0246   -0.0172   -0.0215   -0.0275   -0.0358   -0.0354   -0.0373   -0.0361   -0.0343   -0.0324  -0.0301   -0.0567   -0.0673   -0.0856;
    0.0419    0.0253    0.0041   -0.0183   -0.0154   -0.0061   -0.0014   -0.0057   -0.0117   -0.0195   -0.0276   -0.0310   -0.0347   -0.0346   -0.0339  -0.0310   -0.0534   -0.0746   -0.0912;
    0.0579    0.0418    0.0181    0.0110    0.0035    0.0019         0   -0.0010   -0.0018   -0.0116   -0.0170   -0.0198   -0.0243   -0.0236   -0.0311  -0.0386   -0.0457   -0.0694   -0.0855;
    0.0415    0.0304    0.0210    0.0194    0.0072    0.0061    0.0046    0.0007   -0.0017   -0.0069   -0.0129   -0.0218   -0.0249   -0.0302   -0.0297  -0.0419   -0.0435   -0.0529   -0.0640;
    0.0467    0.0291    0.0239    0.0180    0.0083    0.0054    0.0013   -0.0026   -0.0048   -0.0100   -0.0143   -0.0204   -0.0237   -0.0283   -0.0296  -0.0393   -0.0452   -0.0504   -0.0680;
    0.0490    0.0323    0.0267    0.0182    0.0089    0.0046   -0.0005   -0.0043   -0.0070   -0.0119   -0.0155   -0.0193   -0.0235   -0.0277   -0.0296  -0.0389   -0.0474   -0.0530   -0.0697;
    0.0405    0.0396    0.0286    0.0184    0.0070    0.0070   -0.0035   -0.0060   -0.0067   -0.0088   -0.0146   -0.0156   -0.0207   -0.0250   -0.0270  -0.0384   -0.0486   -0.0596   -0.0685;
    0.0616    0.0483    0.0387    0.0293    0.0161    0.0127    0.0091    0.0048    0.0027   -0.0023   -0.0069   -0.0136   -0.0140   -0.0185   -0.0225  -0.0357   -0.0451   -0.0547   -0.0680;
    0.0706    0.0522    0.0400    0.0317    0.0182    0.0143    0.0105    0.0076    0.0037    0.0012   -0.0035   -0.0062   -0.0100   -0.0133   -0.0170  -0.0305   -0.0388   -0.0510   -0.0694];


% --//--//--//--//--//--//--//--//--, носки выпущены.
% mxdel20_nos = f(alpha2,beta)
mxdel20_nos = [-0.0522   -0.0388   -0.0295   -0.0190   -0.0133   -0.0214   -0.0230   -0.0214   -0.0213   -0.0219   -0.0217   -0.0242   -0.0274   -0.0289   -0.0304  -0.0245   -0.0137   -0.0038    0.0101;
   -0.0462   -0.0450   -0.0440   -0.0419   -0.0373   -0.0351   -0.0328   -0.0283   -0.0284   -0.0307   -0.0324   -0.0328   -0.0339   -0.0348   -0.0357  -0.0313   -0.0284   -0.0272   -0.0253;
   -0.0487   -0.0476   -0.0407   -0.0409   -0.0382   -0.0361   -0.0375   -0.0380   -0.0391   -0.0399   -0.0403   -0.0406   -0.0412   -0.0439   -0.0446  -0.0427   -0.0425   -0.0347   -0.0336;
   -0.0419   -0.0447   -0.0428   -0.0407   -0.0446   -0.0458   -0.0469   -0.0493   -0.0508   -0.0524   -0.0533   -0.0537   -0.0550   -0.0566   -0.0578  -0.0614   -0.0591   -0.0579   -0.0603;
   -0.0295   -0.0292   -0.0307   -0.0313   -0.0354   -0.0387   -0.0410   -0.0450   -0.0486   -0.0512   -0.0541   -0.0568   -0.0599   -0.0623   -0.0651  -0.0696   -0.0699   -0.0707   -0.0704;
   -0.0165   -0.0188   -0.0174   -0.0204   -0.0271   -0.0316   -0.0364   -0.0414   -0.0474   -0.0527   -0.0574   -0.0618   -0.0661   -0.0705   -0.0750  -0.0819   -0.0840   -0.0841   -0.0858;
    0.0048    0.0017   -0.0026   -0.0198   -0.0179   -0.0221   -0.0277   -0.0329   -0.0389   -0.0432   -0.0479   -0.0529   -0.0576   -0.0625   -0.0674  -0.0759   -0.0830   -0.0872   -0.0901;
   -0.0069   -0.0068   -0.0063   -0.0089   -0.0156   -0.0195   -0.0265   -0.0321   -0.0378   -0.0447   -0.0489   -0.0526   -0.0584   -0.0643   -0.0699  -0.0761   -0.0795   -0.0787   -0.0783;
    0.0156    0.0048    0.0053    0.0037   -0.0065   -0.0075   -0.0100   -0.0148   -0.0190   -0.0229   -0.0282   -0.0345   -0.0369   -0.0405   -0.0415  -0.0516   -0.0527   -0.0521   -0.0634;
    0.0308    0.0244    0.0078   -0.0102   -0.0021   -0.0006   -0.0086   -0.0144   -0.0196   -0.0261   -0.0306   -0.0369   -0.0417   -0.0441   -0.0464  -0.0386   -0.0485   -0.0657   -0.0820;
    0.0380    0.0302    0.0149    0.0016   -0.0080   -0.0039   -0.0111   -0.0148   -0.0212   -0.0238   -0.0333   -0.0372   -0.0376   -0.0354   -0.0357  -0.0458   -0.0589   -0.0742   -0.0817;
    0.0232    0.0189    0.0168   -0.0049   -0.0121   -0.0153   -0.0107   -0.0166   -0.0238   -0.0263   -0.0301   -0.0318   -0.0357   -0.0356   -0.0354  -0.0429   -0.0644   -0.0665   -0.0707;
    0.0436    0.0345    0.0217    0.0087    0.0033    0.0121    0.0088    0.0090    0.0071   -0.0080   -0.0097   -0.0210   -0.0288   -0.0347   -0.0351  -0.0405   -0.0534   -0.0665   -0.0754;
    0.0499    0.0450    0.0350    0.0263    0.0128    0.0097    0.0089    0.0058    0.0002   -0.0059   -0.0096   -0.0168   -0.0172   -0.0232   -0.0233  -0.0378   -0.0460   -0.0560   -0.0599];

% Коэффициент момента крена при максимальном отклонении РН (-30 градусов)
% mxdrn30 = f(alpha1,beta)
mxdrn30 = [-0.0100    0.0057    0.0178    0.0291    0.0365    0.0364    0.0336    0.0316    0.0251    0.0216    0.0159    0.0154    0.0142    0.0100    0.0005   0.0079    0.0191    0.0311    0.0465;
   -0.0062    0.0064    0.0192    0.0249    0.0258    0.0263    0.0271    0.0243    0.0213    0.0192    0.0168    0.0149    0.0121    0.0109    0.0066   0.0076    0.0150    0.0275    0.0440;
   -0.0054    0.0058    0.0172    0.0212    0.0240    0.0255    0.0268    0.0246    0.0205    0.0187    0.0172    0.0131    0.0113    0.0098    0.0047   0.0076    0.0115    0.0230    0.0343;
    0.0260    0.0316    0.0342    0.0330    0.0310    0.0311    0.0289    0.0252    0.0204    0.0153    0.0111    0.0072    0.0031   -0.0010   -0.0047  -0.0068   -0.0079   -0.0050    0.0006;
    0.0291    0.0328    0.0338    0.0329    0.0293    0.0298    0.0261    0.0220    0.0181    0.0145    0.0111    0.0078    0.0035   -0.0010   -0.0054  -0.0087   -0.0093   -0.0080   -0.0044;
    0.0410    0.0430    0.0430    0.0394    0.0330    0.0314    0.0271    0.0230    0.0183    0.0138    0.0097    0.0056    0.0008   -0.0045   -0.0087  -0.0152   -0.0183   -0.0183   -0.0205;
    0.0638    0.0638    0.0624    0.0550    0.0440    0.0399    0.0341    0.0278    0.0207    0.0135    0.0071    0.0004   -0.0071   -0.0138   -0.0201  -0.0309   -0.0383   -0.0397   -0.0397;
    0.0820    0.0770    0.0730    0.0653    0.0518    0.0481    0.0410    0.0328    0.0227    0.0134    0.0046   -0.0045   -0.0143   -0.0239   -0.0320  -0.0454   -0.0530   -0.0570   -0.0620;
    0.1090    0.0930    0.0810    0.0710    0.0532    0.0476    0.0414    0.0315    0.0227    0.0139    0.0058   -0.0030   -0.0120   -0.0231   -0.0324  -0.0501   -0.0598   -0.0719   -0.0799;
    0.0930    0.0836    0.0716    0.0609    0.0447    0.0425    0.0367    0.0307    0.0228    0.0145    0.0049   -0.0032   -0.0118   -0.0212   -0.0302  -0.0466   -0.0571   -0.0690   -0.0784;
    0.0820    0.0505    0.0236    0.0170    0.0047    0.0242    0.0271    0.0246    0.0215    0.0128    0.0082    0.0012   -0.0052   -0.0092   -0.0165  -0.0291   -0.0358   -0.0625   -0.0939;
    0.0640    0.0550    0.0330    0.0033   -0.0018   -0.0037    0.0099    0.0121    0.0076    0.0012    0.0007         0   -0.0010   -0.0015   -0.0033  -0.0085   -0.0379   -0.0600   -0.0690;
    0.0584    0.0577    0.0360    0.0088    0.0047    0.0092    0.0143    0.0140    0.0129    0.0030   -0.0006   -0.0053   -0.0073   -0.0088   -0.0110  -0.0143   -0.0426   -0.0641   -0.0648;
    0.0825    0.0635    0.0405    0.0319    0.0229    0.0200    0.0229    0.0236    0.0139    0.0013    0.0017   -0.0056   -0.0091   -0.0109   -0.0174  -0.0263   -0.0349   -0.0578   -0.0771;
    0.0536    0.0446    0.0302    0.0250    0.0164    0.0162    0.0156    0.0124    0.0087    0.0033   -0.0035   -0.0074   -0.0137   -0.0143   -0.0195  -0.0279   -0.0332   -0.0479   -0.0570;
    0.0580    0.0430    0.0325    0.0260    0.0161    0.0143    0.0120    0.0081    0.0064    0.0014   -0.0039   -0.0069   -0.0138   -0.0161   -0.0194  -0.0281   -0.0347   -0.0454   -0.0600;
    0.0600    0.0448    0.0350    0.0270    0.0182    0.0157    0.0130    0.0077    0.0048   -0.0012   -0.0055   -0.0078   -0.0140   -0.0172   -0.0202  -0.0290   -0.0375   -0.0470   -0.0624;
    0.0656    0.0550    0.0440    0.0330    0.0229    0.0206    0.0162    0.0112    0.0039   -0.0005   -0.0023   -0.0077   -0.0125   -0.0165   -0.0209  -0.0308   -0.0420   -0.0529   -0.0636;
    0.0680    0.0570    0.0450    0.0341    0.0219    0.0183    0.0116    0.0118    0.0063   -0.0005   -0.0045   -0.0065   -0.0108   -0.0162   -0.0173  -0.0292   -0.0401   -0.0522   -0.0634;
    0.0660    0.0532    0.0390    0.0269    0.0174    0.0156    0.0102    0.0089    0.0069    0.0008   -0.0030   -0.0051   -0.0074   -0.0132   -0.0156  -0.0247   -0.0368   -0.0513   -0.0638];

% Какая-то поправка на производную mx по углу скольжения
% dmxbt = f(alpha1)
dmxbt1 = [0 0 0 0 0 0 0 0.0007 0.0005 0.0003 0 0 0 0 0 0 0 0 0 0]';

% Производная mx по угловой скорости рыскания
% mxwy = f(alpha1)
mxwy1 = [-0.1550   -0.1550   -0.1550   -0.2010   -0.0024    0.0880    0.2050    0.2200    0.3190    0.4370    0.6800    0.1000    0.4470   -0.3300   -0.0680   0.1180    0.0802    0.0529    0.0868   -0.0183]';
mxwy1 = -mxwy1;

% Приращение mxwy при выпуске носков
% dmxwy_nos = f(alpha2)
dmxwy_nos1 = [0.0290    0.0290    0.0290    0.1750    0.0665    0.0360    0.0070    0.0660    0.2010    0.0060   -0.0680   -0.5370   -0.7870   -0.3940]';
dmxwy_nos1 = -dmxwy_nos1;

% Производная mx по угловой скорости крена
% mxwx = f(alpha1)
mxwx1 = [-0.3660   -0.3660   -0.3660   -0.3770   -0.3450   -0.4340   -0.4080   -0.3880   -0.3290   -0.2940   -0.2300   -0.2100   -0.1200    -0.1000   -0.1000  -0.1200   -0.1400   -0.1000   -0.1500   -0.2000]';

% Приращение mxwx при выпуске носков
% dmxwx_nos = f(alpha2)
dmxwx_nos1 = [0.0060    0.0060    0.0060    0.0180   -0.1000    0.0200    0.0580    0.0870    0.0270   -0.0560   -0.0820    0.3620    0.1940    0.0970]';

interpmx = csapi({alpha1, beta1, fi2}, mx1);
interpmx_nos = csaps({alpha2, beta1}, mx_nos1, 1.0 - 1.0e-5);
interpdmxbt = pchip(alpha1, dmxbt1);
interpmxwx1 = pchip(alpha1, mxwx1);
interpmxwx2 = pchip(alpha2, dmxwx_nos1);
interpmxwy1 = pchip(alpha1, mxwy1);
interpmxwy2 = pchip(alpha2, dmxwy_nos1);
interpmxdel = csaps({alpha1, beta1}, mxdel20, 1.0 - 1.0e-5);
interpmxdel_nos = csaps({alpha2, beta1}, mxdel20_nos, 1.0 - 1.0e-5);
interpmxdrn = csaps({alpha1, beta1}, mxdrn30, 1.0 - 1.0e-5);
end

ntraj = size(alpha, 2);
mx = zeros(1, ntraj);
mx0 = zeros(1, ntraj);
mx_nos = zeros(1, ntraj);
dmxbt = zeros(1, ntraj);
mxwx = zeros(1, ntraj);
mxwy = zeros(1, ntraj);
mxdel = zeros(1, ntraj);
mxdel_nos = zeros(1, ntraj);
mxdrn = zeros(1, ntraj);
for i = 1 : ntraj
    mx(i) = fnval(interpmx, {(alpha(i)), (beta(i)), (fi(i))});
    mx0(i) = fnval(interpmx, {(alpha(i)), (beta(i)), (0)});
    mx_nos(i) = fnval(interpmx_nos, {(alpha(i)), (beta(i))});
    dmxbt(i) = fnval(interpdmxbt, alpha(i));
    mxwx(i) = fnval(interpmxwx1, alpha(i)) + fnval(interpmxwx2, alpha(i)).*(dnos / deg2rad(25));
    mxwy(i) = fnval(interpmxwy1, alpha(i)) + fnval(interpmxwy2, alpha(i)).*(dnos / deg2rad(25));
    mxdel(i) = fnval(interpmxdel, {(alpha(i)), (beta(i))});
    mxdel_nos(i) = fnval(interpmxdel_nos, {(alpha(i)), (beta(i))});
    mxdrn(i) = fnval(interpmxdrn, {(alpha(i)), (beta(i))});
end

% Cборка
dmx_nos = mx_nos - mx0;
dmxdel = mxdel - mx0;
dmxdel_nos = mxdel_nos - mx_nos - dmxdel;
dmxdrn = mxdrn - mx0;

mx = mx + dmx_nos.*(dnos / deg2rad(25)) + ...
          (dmxdel + dmxdel_nos.*(dnos / deg2rad(25))).*(del / deg2rad(20)) + ...
          dmxdrn.*(drn / deg2rad(-30)) + ...
          mxwx.*((Wx.*l)./(2 * V)) + mxwy.*((Wy.*l)./(2 * V)) + dmxbt.*beta;

