# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyuploadcare',
 'pyuploadcare.api',
 'pyuploadcare.dj',
 'pyuploadcare.resources',
 'pyuploadcare.transformations',
 'pyuploadcare.ucare_cli',
 'pyuploadcare.ucare_cli.commands']

package_data = \
{'': ['*'], 'pyuploadcare.dj': ['static/uploadcare/*']}

install_requires = \
['httpx>=0.18.2,<0.19.0',
 'pydantic[email]>=1.8.2,<2.0.0',
 'python-dateutil>=2.8.2,<3.0.0',
 'pytz>=2021.1,<2022.0',
 'typing-extensions>=3.10.0,<4.0.0']

extras_require = \
{'django': ['Django>=1.11']}

entry_points = \
{'console_scripts': ['ucare = pyuploadcare.ucare_cli.main:main']}

setup_kwargs = {
    'name': 'pyuploadcare',
    'version': '3.0.0',
    'description': 'Python library for Uploadcare.com',
    'long_description': '.. image:: https://ucarecdn.com/2f4864b7-ed0e-4411-965b-8148623aa680/-/inline/yes/uploadcare-logo-mark.svg\n   :target: https://uploadcare.com/?utm_source=github&utm_campaign=pyuploadcare\n   :height: 64 px\n   :width: 64 px\n   :align: left\n\n=============================================\nPyUploadcare: a Python library for Uploadcare\n=============================================\n\n.. image:: https://badge.fury.io/py/pyuploadcare.svg\n   :target: https://badge.fury.io/py/pyuploadcare\n.. image:: https://github.com/uploadcare/pyuploadcare/actions/workflows/test.yml/badge.svg\n   :target: https://github.com/uploadcare/pyuploadcare/actions/workflows/test.yml\n   :alt: Build Status\n.. image:: https://readthedocs.org/projects/pyuploadcare/badge/?version=latest\n   :target: https://readthedocs.org/projects/pyuploadcare/?badge=latest\n   :alt: Documentation Status\n.. image:: https://coveralls.io/repos/github/uploadcare/pyuploadcare/badge.svg?branch=master\n   :target: https://coveralls.io/github/uploadcare/pyuploadcare?branch=master\n   :alt: Coverage\n.. image:: https://img.shields.io/badge/tech-stack-0690fa.svg?style=flat\n   :target: https://stackshare.io/uploadcare/stacks/\n   :alt: Uploadcare tech stack\n\n\nSimple file uploads for the web are of most importance\nfor us at `Uploadcare`_. Today, everyone is used to the routine of\nallowing users to upload their pics or attach resumes. The routine\ncovers it all: installing image processing libraries, adjusting permissions,\nensuring servers never go down, and enabling CDN.\n\nThis library consists of the `Uploadcare`_ API interface and a couple\nof Django goodies.\n\nSimple as that, Uploadcare ``ImageField`` can be added to an\nexisting Django project in just a couple of `simple steps`_.\nThis will enable your users to see the upload progress, pick files\nfrom Google Drive or Instagram, and edit a form while files are\nbeing uploaded asynchronously.\n\n.. code-block:: python\n\n    from django import forms\n    from django.db import models\n\n    from pyuploadcare.dj.models import ImageField\n    from pyuploadcare.dj.forms import FileWidget\n\n\n    class Candidate(models.Model):\n        photo = ImageField(blank=True, manual_crop="")\n\n\n    # optional. provide advanced widget options: https://uploadcare.com/docs/uploads/widget/config/#options\n    class CandidateForm(forms.Form):\n        photo = ImageField(widget=FileWidget(attrs={\n            \'data-cdn-base\': \'https://cdn.super-candidates.com\',\n            \'data-image-shrink\': \'1024x1024\',\n        }))\n\n.. image:: https://ucarecdn.com/dbb4021e-b20e-40fa-907b-3da0a4f8ed70/-/resize/800/manual_crop.png\n\nFeatures\n========\n\n- Python wrapper for Uploadcare `REST`_ and `Upload`_ APIs.\n- Django widget with useful manual crop and multi-upload.\n- *ucare* console utility.\n\nRequirements\n============\n\n``pyuploadcare`` requires Python 3.6, 3.7, 3.8, 3.9.\n\nTo use ``pyuploadcare`` with Python 2.7 please install ``pyuploadcare < 3.0``.\n\nIf you\'re using ``pyuploadcare`` with Django, check ``tox.ini`` or\n``.github/workflows`` for supported Python-Django combinations.\n\nInstallation\n============\n\nIn order to install ``pyuploadcare``, simply run:\n\n.. code-block:: console\n\n    $ pip install pyuploadcare\n\nor, if you prefer it the old way:\n\n.. code-block:: console\n\n    $ easy_install pyuploadcare\n\nTo use in Django project install with extra dependencies::\n\n.. code-block:: console\n\n    $ pip install pyuploadcare[django]\n\nUsage\n=====\n\nYou can find an example project `here`_.\n\nCore API\n--------\n\nInitialization\n^^^^^^^^^^^^^^\n\nYou can use pyuploadcare in any Python project. You need to pass\nyour project keys to ``Uploadcare`` client::\n\n    from pyuploadcare import Uploadcare\n    uploadcare = Uploadcare(public_key=\'<your public key>\', secret_key=\'<your private key>\')\n\n\nUploading files\n^^^^^^^^^^^^^^^\n\nUpload single file. ``File.upload`` method can accept file object or URL. Depending of file object size\ndirect or multipart upload method will be chosen::\n\n    with open(\'file.txt\') as file_object:\n        ucare_file: File = uploadcare.upload(file_object)\n\nUpload file from url::\n\n    ucare_file: File = uploadcare.upload("https://github.githubassets.com/images/modules/logos_page/Octocat.png")\n\nUpload multiple files. Direct upload method is used::\n\n    file1 = open(\'file1.txt\')\n    file2 = open(\'file2.txt\')\n    ucare_files: List[File] = uploadcare.upload_files([file1, file2])\n\nSend single file via multipart upload::\n\n    with open(\'file.txt\') as file_object:\n        ucare_file: File = uploadcare.upload(file_object)\n\n``Uploadcare.upload`` method accepts optional callback function to track uploading progress.\nExample of using callback function for printing progress::\n\n    >>> def print_progress(info: UploadProgress):\n    ...     print(f\'{info.done}/{info.total} B\')\n\n    >>> # multipart upload is used\n    >>> with open(\'big_file.jpg\', \'rb\') as fh:\n    ...    uploadcare.upload(fh, callback=print_progress)\n    0/11000000 B\n    5242880/11000000 B\n    10485760/11000000 B\n    11000000/11000000 B\n\n    >>> # upload from url is used\n    >>> uploadcare.upload("https://github.githubassets.com/images/modules/logos_page/Octocat.png", callback=print_progress)\n    32590/32590 B\n\n    >>> # direct upload is used. Callback is called just once after successful upload\n    >>> with open(\'small_file.jpg\', \'rb\') as fh:\n    ...     uploadcare.upload(fh, callback=print_progress)\n    56780/56780 B\n\n\nManaging files\n^^^^^^^^^^^^^^\n\nGet list of files::\n\n    files: FileList = uploadcare.list_files(stored=True, limit=10)\n    for file in files:\n        print(file.info)\n\nGet existing file::\n\n    file: File = uploadcare.file("740e1b8c-1ad8-4324-b7ec-112c79d8eac2")\n    print(file.info)\n\nStore single file::\n\n    file: File = uploadcare.file("740e1b8c-1ad8-4324-b7ec-112c79d8eac2")\n    file.store()\n\nStore multiple files::\n\n    files = [\n        \'6c5e9526-b0fe-4739-8975-72e8d5ee6342\',\n        \'a771f854-c2cb-408a-8c36-71af77811f3b\'\n    ]\n    uploadcare.store_files(files)\n\nDelete single file::\n\n    file: File = uploadcare.file("740e1b8c-1ad8-4324-b7ec-112c79d8eac2")\n    file.delete()\n\nDelete multiple files::\n\n    files = [\n        \'6c5e9526-b0fe-4739-8975-72e8d5ee6342\',\n        \'a771f854-c2cb-408a-8c36-71af77811f3b\'\n    ]\n    uploadcare.delete_files(files)\n\nCopy file to the local storage::\n\n    file: File = uploadcare.file("740e1b8c-1ad8-4324-b7ec-112c79d8eac2")\n    copied_file: File = file.create_local_copy(store=True)\n\nCopy file to the remote storage::\n\n    file = uploadcare.file("740e1b8c-1ad8-4324-b7ec-112c79d8eac2")\n    copied_file: File = file.create_remote_copy(target=\'mytarget\', make_public=True)\n\n\nFile groups\n^^^^^^^^^^^\n\nCreate file group::\n\n    file_1: File = uploadcare.file(\'6c5e9526-b0fe-4739-8975-72e8d5ee6342\')\n    file_2: File = uploadcare.file(\'a771f854-c2cb-408a-8c36-71af77811f3b\')\n    file_group: FileGroup = uploadcare.create_file_group([file_1, file_2])\n\nGet file group::\n\n    file_group: FileGroup = uploadcare.file_group(\'0513dda0-582f-447d-846f-096e5df9e2bb~2\')\n    print(file_group.info())\n\nStores all group\'s files::\n\n    file_group: FileGroup = uploadcare.file_group(\'0513dda0-582f-447d-846f-096e5df9e2bb~2\')\n    file_group.store()\n\nList file groups::\n\n    file_groups: List[FileGroup] = uploadcare.list_file_groups(limit=10)\n    for file_group in file_groups:\n        print(file_group.info)\n\n\nVideo conversion\n^^^^^^^^^^^^^^^^\n\nAfter each video file upload you obtain a file identifier in UUID format. Then you can use this file identifier to convert your video in multiple ways::\n\n    file = uploadcare.file(\'740e1b8c-1ad8-4324-b7ec-112c79d8eac2\')\n    transformation = (\n        VideoTransformation()\n            .format(Format.mp4)\n            .size(width=640, height=480, resize_mode=ResizeMode.add_padding)\n            .quality(Quality.lighter)\n            .cut(start_time=\'2:30.535\', length=\'2:20.0\')\n            .thumbs(10)\n    )\n    converted_file: File = file.convert(transformation)\n\nor you can use API directly to convert single or multiple files::\n\n    transformation = VideoTransformation().format(VideoFormat.webm).thumbs(2)\n    paths: List[str] = [\n        transformation.path("740e1b8c-1ad8-4324-b7ec-112c79d8eac2"),\n    ]\n\n    response = uploadcare.video_convert_api.convert(paths)\n    video_convert_info = response.result[0]\n    converted_file = uploadcare.file(video_convert_info.uuid)\n\n    video_convert_status = uploadcare.video_convert_api.status(video_convert_info.token)\n\n\nDocument Conversion\n^^^^^^^^^^^^^^^^^^^\n\nAfter each document file upload you obtain a file identifier in UUID format. Then you can use this file identifier to convert your document to a new format::\n\n    file = uploadcare.file(\'0e1cac48-1296-417f-9e7f-9bf13e330dcf\')\n    transformation = DocumentTransformation().format(DocumentFormat.pdf)\n    converted_file: File = file.convert(transformation)\n\nor create an image of a particular page (if using image format)::\n\n    file = uploadcare.file(\'5dddafa0-a742-4a51-ac40-ae491201ff97\')\n    transformation = DocumentTransformation().format(DocumentFormat.png).page(1)\n    converted_file: File = file.convert(transformation)\n\nor you can use API directly to convert single or multiple files::\n\n    transformation = DocumentTransformation().format(DocumentFormat.pdf)\n\n    paths: List[str] = [\n        transformation.path("0e1cac48-1296-417f-9e7f-9bf13e330dcf"),\n    ]\n\n    response = uploadcare.document_convert_api.convert([path])\n    document_convert_info = response.result[0]\n    converted_file = uploadcare.file(document_convert_info.uuid)\n\n    document_convert_status = uploadcare.document_convert_api.status(document_convert_info.token)\n\n\nWebhooks\n^^^^^^^^\n\nCreate webhook::\n\n    webhook: Webhook = uploadcare.create_webhook("https://path/to/webhook")\n\nList webhooks::\n\n    webhooks: List[Webhook] = list(uploadcare.list_webhooks(limit=10))\n\nUpdate webhook::\n\n    webhook: Webhook = uploadcare.update_webhook(webhook_id, is_active=False)\n\nDelete webhook::\n\n    uploadcare.delete_webhook(webhook_id)\n\n\nProjects\n^^^^^^^^\n\nGet project info::\n\n    project_info: ProjectInfo = uploadcare.get_project_info()\n\n\nImage transformations\n^^^^^^^^^^^^^^^^^^^^^\n\nUploadcare allows to apply image transformations to files. ``File.cdn_url`` attribute returns CDN url::\n\n    >>> file_ = File(\'a771f854-c2cb-408a-8c36-71af77811f3b\')\n    >>> file_.cdn_url\n    https://ucarecdn.com/a771f854-c2cb-408a-8c36-71af77811f3b/\n\nYou can set default effects by string::\n\n    >>> file_.set_effects(\'effect/flip/-/effect/mirror/\')\n    >>> file_.cdn_url\n    https://ucarecdn.com/a771f854-c2cb-408a-8c36-71af77811f3b/-/effect/flip/-/effect/mirror/\n\nor by image transformation builder::\n\n    >>> file_.set_effects(ImageTransformation().grayscale().flip())\n    >>> file_.cdn_url\n    https://ucarecdn.com/a771f854-c2cb-408a-8c36-71af77811f3b/-/grayscale/-/flip/\n\n\nSecure delivery\n^^^^^^^^^^^^^^^\n\nYou can use your own custom domain and CDN provider for deliver files with authenticated URLs (see `original documentation`_).\n\nGenerate secure for file::\n\n    from pyuploadcare import Uploadcare\n    from pyuploadcare.secure_url import AkamaiSecureUrlBuilder\n\n    secure_url_bulder = AkamaiSecureUrlBuilder("your cdn>", "<your secret for token generation>")\n\n    uploadcare = Uploadcare(\n        public_key=\'<your public key>\',\n        secret_key=\'<your private key>\',\n        secure_url_builder=secure_url_bulder,\n    )\n\n    secure_url = uploadcare.generate_secure_url(\'52da3bfc-7cd8-4861-8b05-126fef7a6994\')\n\nGenerate secure for file with transformations::\n\n    secure_url = uploadcare.generate_secure_url(\n        \'52da3bfc-7cd8-4861-8b05-126fef7a6994/-/resize/640x/other/transformations/\'\n    )\n\n\nTesting\n=======\n\nBesides the `Github Actions`_ we use tox. In order to run tests just:\n\n.. code-block:: console\n\n    $ pip install tox\n    $ tox\n\n\nSecurity issues\n===============\n\nIf you think you ran into something in Uploadcare libraries which might have\nsecurity implications, please hit us up at `bugbounty@uploadcare.com`_\nor Hackerone.\n\nWe\'ll contact you personally in a short time to fix an issue through co-op and\nprior to any public disclosure.\n\n\nFeedback\n========\n\nIssues and PRs are welcome. You can provide your feedback or drop us a support\nrequest at `hello@uploadcare.com`_.\n\n.. _Uploadcare: https://uploadcare.com/?utm_source=github&utm_campaign=pyuploadcare\n.. _simple steps: https://pyuploadcare.readthedocs.org/en/latest/quickstart.html\n.. _Github Actions: https://github.com/uploadcare/pyuploadcare/actions\n.. _REST: https://uploadcare.com/api-refs/rest-api/?utm_source=github&utm_campaign=pyuploadcare\n.. _Upload: https://uploadcare.com/api-refs/upload-api/?utm_source=github&utm_campaign=pyuploadcare\n.. _original documentation: https://uploadcare.com/docs/security/secure-delivery/?utm_source=github&utm_campaign=pyuploadcare\n.. _here: https://github.com/uploadcare/pyuploadcare-example\n.. _bugbounty@uploadcare.com: mailto:bugbounty@uploadcare.com\n.. _hello@uploadcare.com: mailto:hello@uploadcare.com\n',
    'author': 'Uploadcare Inc',
    'author_email': 'hello@uploadcare.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/uploadcare/pyuploadcare',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.6.2,<4.0.0',
}


setup(**setup_kwargs)
