# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['aiohttp_csrf']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp-session>=2,<3', 'aiohttp>=3.6.2,<3.8', 'blake3>=0.1.8,<0.2.0']

setup_kwargs = {
    'name': 'aiohttp-csrf',
    'version': '0.1.0',
    'description': 'CSRF protection for aiohttp-server',
    'long_description': 'aiohttp_csrf\n=============\n\nThe library provides csrf (xsrf) protection for [aiohttp.web](https://docs.aiohttp.org/en/latest/web.html).\n\n**note:** The package [aiohttp-csrf-fixed](https://pypi.org/project/aiohttp-csrf-fixed) is aiohttp_csrf 0.0.2 +\n[this commit](https://github.com/oplik0/aiohttp-csrf/commit/b1bd9207f43a2abf30e32e72ecdb10983a251823). The maintainer\ndidn\'t submit a PR so I just saw it by chance. I haven\'t had  time to closely examine it but I think it\'s just removing\nthe HTTP security error that happens if no CSRF is provided. Why do that? An HTTP error is good because it tells the\nclient what happened and lets you handle it by middleware.\n\n**Breaking Change:** New in 0.1.0 is Blake3 hashes are used by default. This means you must pass `secret_phrase` to\n`aiohttp_csrf.storage.SessionStorage`\n\n![image](https://img.shields.io/travis/wikibusiness/aiohttp-csrf.svg%0A%20:target:%20https://travis-ci.org/wikibusiness/aiohttp-csrf)\n\nBasic usage\n-----------\n\nThe library allows you to implement csrf (xsrf) protection for requests\n\nBasic usage example:\n\n``` {.sourceCode .python}\nimport aiohttp_csrf\nfrom aiohttp import web\n\nFORM_FIELD_NAME = \'_csrf_token\'\nCOOKIE_NAME = \'csrf_token\'\n\n\ndef make_app():\n    csrf_policy = aiohttp_csrf.policy.FormPolicy(FORM_FIELD_NAME)\n\n    csrf_storage = aiohttp_csrf.storage.CookieStorage(COOKIE_NAME)\n\n    app = web.Application()\n\n    aiohttp_csrf.setup(app, policy=csrf_policy, storage=csrf_storage)\n\n    app.middlewares.append(aiohttp_csrf.csrf_middleware)\n\n    async def handler_get_form_with_token(request):\n        token = await aiohttp_csrf.generate_token(request)\n\n\n        body = \'\'\'\n            <html>\n                <head><title>Form with csrf protection</title></head>\n                <body>\n                    <form method="POST" action="/">\n                        <input type="hidden" name="{field_name}" value="{token}" />\n                        <input type="text" name="name" />\n                        <input type="submit" value="Say hello">\n                    </form>\n                </body>\n            </html>\n        \'\'\'  # noqa\n\n        body = body.format(field_name=FORM_FIELD_NAME, token=token)\n\n        return web.Response(\n            body=body.encode(\'utf-8\'),\n            content_type=\'text/html\',\n        )\n\n    async def handler_post_check(request):\n        post = await request.post()\n\n        body = \'Hello, {name}\'.format(name=post[\'name\'])\n\n        return web.Response(\n            body=body.encode(\'utf-8\'),\n            content_type=\'text/html\',\n        )\n\n    app.router.add_route(\n        \'GET\',\n        \'/\',\n        handler_get_form_with_token,\n    )\n\n    app.router.add_route(\n        \'POST\',\n        \'/\',\n        handler_post_check,\n    )\n\n    return app\n\n\nweb.run_app(make_app())\n```\n\n### Initialize\n\nFirst of all, you need to initialize `aiohttp_csrf` in your application:\n\n``` {.sourceCode .python}\napp = web.Application()\n\ncsrf_policy = aiohttp_csrf.policy.FormPolicy(FORM_FIELD_NAME)\n\ncsrf_storage = aiohttp_csrf.storage.CookieStorage(COOKIE_NAME)\n\naiohttp_csrf.setup(app, policy=csrf_policy, storage=csrf_storage)\n```\n\n### Middleware and decorators\n\nAfter initialize you can use `@aiohttp_csrf.csrf_protect` for handlers, that you want to protect. Or you can\ninitialize `aiohttp_csrf.csrf_middleware` and do not disturb about using\ndecorator ([full middleware example here](demo/middleware.py)):\n\n``` {.sourceCode .python}\n...\napp.middlewares.append(aiohttp_csrf.csrf_middleware)\n...\n```\n\nIn this case all your handlers will be protected.\n\n**Note:** we strongly recommend to use `aiohttp_csrf.csrf_middleware` and `@aiohttp_csrf.csrf_exempt` instead of\nmanually managing with `@aiohttp_csrf.csrf_protect`. But if you prefer to use `@aiohttp_csrf.csrf_protect`, don\'t forget\nto use `@aiohttp_csrf.csrf_protect` for both methods: GET and\nPOST ([manual protection example](demo/manual_protection.py))\n\nIf you want to use middleware, but need handlers without protection, you can use `@aiohttp_csrf.csrf_exempt`. Mark you\nhandler with this decorator and this handler will not check the token:\n\n``` {.sourceCode .python}\n@aiohttp_csrf.csrf_exempt\nasync def handler_post_not_check(request):\n    ...\n```\n\n### Generate token\n\nFor generate token you need to call `aiohttp_csrf.generate_token` in your handler:\n\n``` {.sourceCode .python}\n@aiohttp_csrf.csrf_protect\nasync def handler_get(request):\n    token = await aiohttp_csrf.generate_token(request)\n    ...\n```\n\nAdvanced usage\n--------------\n\n### Policies\n\nYou can use different policies for check tokens. Library provides 3 types of policy:\n\n- **FormPolicy**. This policy will search token in the body of your POST request (Usually use for forms) or as a GET\n  variable of the same name. You need to specify name of field that will be checked.\n- **HeaderPolicy**. This policy will search token in headers of your POST request (Usually use for AJAX requests). You\n  need to specify name of header that will be checked.\n- **FormAndHeaderPolicy**. This policy combines behavior of **FormPolicy** and **HeaderPolicy**.\n\nYou can implement your custom policies if needed. But make sure that your custom policy\nimplements `aiohttp_csrf.policy.AbstractPolicy` interface.\n\n### Storages\n\nYou can use different types of storages for storing token. Library provides 2 types of storage:\n\n- **CookieStorage**. Your token will be stored in cookie variable. You need to specify cookie name.\n- **SessionStorage**. Your token will be stored in session. You need to specify session variable name.\n\n**Important:** If you want to use session storage, you need setup aiohttp\\_session in your\napplication ([session storage example](demo/session_storage.py#L22))\n\nYou can implement your custom storages if needed. But make sure that your custom storage\nimplements `aiohttp_csrf.storage.AbstractStorage` interface.\n\n### Token generators\n\nYou can use different token generator in your application. By default storages\nusing `aiohttp_csrf.token_generator.SimpleTokenGenerator`\n\nBut if you need more secure token generator - you can use `aiohttp_csrf.token_generator.HashedTokenGenerator`\n\nAnd you can implement your custom token generators if needed. But make sure that your custom token generator\nimplements `aiohttp_csrf.token_generator.AbstractTokenGenerator` interface.\n\n### Invalid token behavior\n\nBy default, if token is invalid, `aiohttp_csrf` will raise `aiohttp.web.HTTPForbidden` exception.\n\nYou have ability to specify your custom error handler. It can be:\n\n- **callable instance. Input parameter - aiohttp request.**\n\n``` {.sourceCode .python}\ndef custom_error_handler(request):\n    # do something\n    return aiohttp.web.Response(status=403)\n\n# or\n\nasync def custom_async_error_handler(request):\n    # await do something\n    return aiohttp.web.Response(status=403)\n```\n\nIt will be called instead of protected handler.\n\n- **sub class of Exception**. In this case this Exception will be raised.\n\n``` {.sourceCode .python}\nclass CustomException(Exception):\n    pass\n```\n\nYou can specify custom error handler globally, when initialize `aiohttp_csrf` in your application:\n\n``` {.sourceCode .python}\n...\nclass CustomException(Exception):\n    pass\n\n...\naiohttp_csrf.setup(app, policy=csrf_policy, storage=csrf_storage, error_renderer=CustomException)\n...\n```\n\nIn this case custom error handler will be applied to all protected handlers.\n\nOr you can specify custom error handler locally, for specific handler:\n\n``` {.sourceCode .python}\n...\nclass CustomException(Exception):\n    pass\n\n...\n@aiohttp_csrf.csrf_protect(error_renderer=CustomException)\ndef handler_with_custom_csrf_error(request):\n    ...\n```\n\nIn this case custom error handler will be applied to this handler only. For all other handlers will be applied global\nerror handler.\n',
    'author': 'TensorTom',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8.3,<4',
}


setup(**setup_kwargs)
