from typing import Any, Callable, Dict, List, Optional, Sequence, Set, Tuple, Type, TypeVar, Union
from uuid import UUID

import torch

Shape = Tuple[int, ...]

class EinsumSpec(object):
    input_ints: List[List[int]]
    output_ints: List[int]
    int_sizes: List[int]
    def __init__(
        self,
        input_ints: List[List[int]],
        output_ints: List[int],
        int_sizes: Dict[int, int],
    ): ...
    def flops(self) -> int: ...
    def optimize_dp(
        self,
        check_outer: bool,
        mem_limit: Optional[int] = None,
        hash_limit: Optional[int] = None,
    ) -> List[List[int]]: ...
    def optimize(
        self,
        check_outer: bool,
        mem_limit: Optional[int] = None,
        hash_limit: Optional[int] = None,
    ) -> List[List[int]]: ...
    def normalize(self) -> "EinsumSpec": ...
    def validate(self) -> bool: ...
    def shapes(self) -> Tuple[List[List[int]], List[int]]: ...
    def to_einsum_string(self) -> str: ...
    @staticmethod
    def string_to_ints(string: str) -> Tuple[List[List[int]], List[int]]: ...

def optimize_einsum_spec_cached(
    spec: EinsumSpec,
    check_outer: bool,
    mem_limit: Optional[int] = None,
    hash_limit: Optional[int] = None,
) -> Optional[List[List[int]]]: ...

class RearrangeSpec(object):
    input_ints: List[List[int]]
    output_ints: List[List[int]]
    int_sizes: List[Optional[int]]
    def __init__(
        self,
        input_ints: List[List[int]],
        output_ints: List[List[int]],
        int_sizes: List[Optional[int]],
    ): ...
    def is_identity(self) -> bool: ...
    def is_permute(self) -> bool: ...
    def shapes(self) -> Tuple[List[int], List[int]]: ...
    def is_valid(self) -> bool: ...
    def to_einops_string(self) -> str: ...
    @staticmethod
    def from_einops_string(string: str) -> RearrangeSpec: ...
    def to_einops_string_and_letter_sizes(
        self,
    ) -> Tuple[str, List[Tuple[str, int]]]: ...
    def apply(self, tensor: torch.Tensor) -> torch.Tensor: ...
    @staticmethod
    def fuse(inner: RearrangeSpec, outer: RearrangeSpec) -> RearrangeSpec: ...
    def canonicalize(self, special_case_ones: bool = True) -> RearrangeSpec: ...
    def fill_empty_ints(self) -> RearrangeSpec: ...
    def conform_to_input_shape(self, shape: Shape, coerce: bool) -> RearrangeSpec: ...
    @staticmethod
    def ident(rank: int) -> RearrangeSpec: ...
    def to_py_rearrange_spec(self, shape: Shape) -> Any: ...
    @staticmethod
    def flatten(rank: int) -> RearrangeSpec: ...
    @staticmethod
    def unflatten(shape: Sequence[int]) -> RearrangeSpec: ...
    @staticmethod
    def unflatten_axis(ndim: int, axis: int, shape: Sequence[int]) -> RearrangeSpec: ...
    @staticmethod
    def check_rank(rank: int) -> None: ...
    @staticmethod
    def expand_at_axes(
        orig_ndim: int, axes: Union[int, Sequence[int]], counts: Optional[Union[int, Sequence[int]]] = None
    ) -> RearrangeSpec:
        """
        if counts is None, this does unsqueeze (aka, uses counts = [1, ..., 1])
        """
    @staticmethod
    def unsqueeze(orig_ndim: int, axes: Union[int, Sequence[int]]) -> RearrangeSpec:
        """
        alias for expand_at_axes with counts=None
        """
    @staticmethod
    def prepend_batch_shape(shape: Sequence[int], old_rank: int) -> RearrangeSpec:
        """meant to replace many use cases of python `repeat_at_axes` (now 'expand_at_axes' in rust)"""
        ...

Axis = Union[None, int, Sequence[int]]

# TODO: below could be subclasses of circuit if we wanted. This could be nice in various ways...

IrreducibleNode = Union[ArrayConstant, Symbol]
"""
These nodes are uneffected by rewrites, and satisfy
AlgebraicRewrite(Replace(X, IrreducibleNode->Y)) == Replace(AlgebraicRewrite(IrreducibleNode), IrreducibleNode->Y)
except for hashmap iteration order or other unfortunate nondeterminism
"""

Leaf = Union[ArrayConstant, Symbol, ScalarConstant]
LeafConstant = Union[ArrayConstant, ScalarConstant]
Var = Union[StoredCumulantVar, DiscreteVar]

class Circuit:
    @property
    def shape(self) -> Shape: ...
    @property
    def is_constant(self) -> bool: ...
    @property
    def is_explicitly_computable(self) -> bool: ...
    @property
    def can_be_sampled(self) -> bool: ...
    @property
    def name(self) -> str: ...
    @property
    def hash(self) -> bytes: ...
    @property
    def hash_base16(self) -> str: ...
    def children(self) -> List[Circuit]: ...
    def self_flops(self) -> int: ...
    def total_flops(self) -> int: ...
    def max_non_input_size(self) -> int: ...
    def compiler_print(self, bijection=False, shape_only_necessary=False) -> None: ...
    def print_stats(self) -> None: ...
    def compiler_repr(self, bijection=False, shape_only_necessary=False) -> str: ...
    def numel(self) -> int: ...
    def rank(self) -> int: ...
    def to_py(self) -> Any: ...
    def child_axis_map(self) -> List[List[Optional[int]]]: ...
    def evaluate(self, device_dtype: TorchDeviceDtypeOp = TorchDeviceDtypeOp()) -> torch.Tensor: ...
    def map_children(self, fn: Callable[[Circuit], Circuit]) -> Circuit: ...
    def map_children_enumerate(self, fn: Callable[[int, Circuit], Circuit]) -> Circuit: ...
    def total_arrayconstant_size(self) -> int: ...
    def get_compatible_device_dtype(self) -> TorchDeviceDtype: ...
    def rename(self, name: Optional[str]) -> Circuit: ...
    def visit(self, f: Callable[[Circuit], None]): ...
    def add(self, other: Circuit, name: Optional[str] = None) -> Add: ...
    def sub(self, other: Circuit, name: Optional[str] = None) -> Add: ...
    def mul(self, other: Circuit, name: Optional[str] = None) -> Einsum: ...
    def mul_scalar(self, scalar: float, name: Optional[str] = None, scalar_name: Optional[str] = None) -> Einsum: ...
    def reduce(self, op_name: str, axis: Axis = None, name: Optional[str] = None) -> Circuit: ...
    def sum(self, axis: Axis = None, name: Optional[str] = None) -> Einsum: ...
    def mean(self, axis: Axis = None, name: Optional[str] = None) -> Einsum: ...
    def max(self, axis: Axis = None, name: Optional[str] = None) -> Circuit: ...
    def min(self, axis: Axis = None, name: Optional[str] = None) -> Circuit: ...
    def index(self, index: Sequence[TorchAxisIndex], name: Optional[str] = None) -> Index: ...
    def expand_at_axes(
        self,
        axes: Union[int, Sequence[int]],
        counts: Optional[Union[int, Sequence[int]]] = None,
        name: Optional[str] = None,
    ) -> Rearrange:
        """
        if counts is None, this does unsqueeze (aka, uses counts = [1, ..., 1])
        """
    def unsqueeze(self, axes: Union[int, Sequence[int]], name: Optional[str] = None) -> Rearrange:
        """
        adds 1 dims, also alias for expand_at_axes with counts=None
        """
    def flatten(self, name: Optional[str] = None) -> Rearrange: ...
    def unflatten(self, shape: Shape, name: Optional[str] = None) -> Rearrange: ...
    def unflatten_axis(self, axis: int, shape: Shape, name: Optional[str] = None) -> Rearrange: ...
    def rearrange(self, spec: RearrangeSpec, name: Optional[str] = None) -> Rearrange:
        """
        Alias for Rearrange(node, spec, name)
        """
    def rearrange_str(self, string: str, name: Optional[str] = None) -> Rearrange:
        """
        Alias for Rearrange.from_einops_string(node, string, name)
        """
    def as_einsum(self) -> Optional[Einsum]: ...
    def as_array_constant(self) -> Optional[ArrayConstant]: ...
    def as_symbol(self) -> Optional[Symbol]: ...
    def as_scalar_constant(self) -> Optional[ScalarConstant]: ...
    def as_add(self) -> Optional[Add]: ...
    def as_rearrange(self) -> Optional[Rearrange]: ...
    def as_index(self) -> Optional[Index]: ...
    def as_general_function(self) -> Optional[GeneralFunction]: ...
    def as_concat(self) -> Optional[Concat]: ...
    def as_scatter(self) -> Optional[Scatter]: ...
    def as_module_node(self) -> Optional[ModuleNode]: ...
    def as_auto_tag(self) -> Optional[AutoTag]: ...
    def as_discrete_var(self) -> Optional[DiscreteVar]: ...
    def as_stored_cumulant_var(self) -> Optional[StoredCumulantVar]: ...
    def as_cumulant(self) -> Optional[Cumulant]: ...
    def as_einsum_unwrap(self) -> Einsum: ...
    def as_array_constant_unwrap(self) -> ArrayConstant: ...
    def as_symbol_unwrap(self) -> Symbol: ...
    def as_scalar_constant_unwrap(self) -> ScalarConstant: ...
    def as_add_unwrap(self) -> Add: ...
    def as_rearrange_unwrap(self) -> Rearrange: ...
    def as_index_unwrap(self) -> Index: ...
    def as_general_function_unwrap(self) -> GeneralFunction: ...
    def as_concat_unwrap(self) -> Concat: ...
    def as_scatter_unwrap(self) -> Scatter: ...
    def as_module_node_unwrap(self) -> ModuleNode: ...
    def as_auto_tag_unwrap(self) -> AutoTag: ...
    def as_discrete_var_unwrap(self) -> DiscreteVar: ...
    def as_stored_cumulant_var_unwrap(self) -> StoredCumulantVar: ...
    def as_cumulant_unwrap(self) -> Cumulant: ...
    def is_einsum(self) -> bool: ...
    def is_array_constant(self) -> bool: ...
    def is_symbol(self) -> bool: ...
    def is_scalar_constant(self) -> bool: ...
    def is_add(self) -> bool: ...
    def is_rearrange(self) -> bool: ...
    def is_index(self) -> bool: ...
    def is_general_function(self) -> bool: ...
    def is_concat(self) -> bool: ...
    def is_scatter(self) -> bool: ...
    def is_module_node(self) -> bool: ...
    def is_auto_tag(self) -> bool: ...
    def is_discrete_var(self) -> bool: ...
    def is_stored_cumulant_var(self) -> bool: ...
    def is_cumulant(self) -> bool: ...
    def is_irreducible_node(self) -> bool: ...
    def is_leaf(self) -> bool: ...
    def is_leaf_constant(self) -> bool: ...
    def is_var(self) -> bool: ...
    def into_irreducible_node(self) -> Optional[IrreducibleNode]: ...
    def into_leaf(self) -> Optional[Leaf]: ...
    def into_leaf_constant(self) -> Optional[LeafConstant]: ...
    def into_var(self) -> Optional[Var]: ...

class ArrayConstant(Circuit):
    def __init__(self, value: torch.Tensor, name: Optional[str] = "ArrayConstant") -> None: ...
    @property
    def uuid(self) -> UUID: ...
    @property
    def value(self) -> torch.Tensor: ...
    @staticmethod
    def randn(
        *shape: int,
        name: Optional[str] = None,
        device_dtype: TorchDeviceDtypeOp = TorchDeviceDtypeOp(),
        seed: Optional[int] = None,
    ) -> "ArrayConstant": ...
    @staticmethod
    def new_named_axes(
        value: torch.Tensor, name: Optional[str], named_axes: List[Optional[str]]
    ) -> "ArrayConstant": ...
    @staticmethod
    def save_rrfs() -> str: ...  # string is base16 key
    @staticmethod
    def from_hash(name: Optional[str], hash_base16: str) -> "ArrayConstant": ...
    @staticmethod
    def from_hash_prefix(
        name: Optional[str], hash_base16: str, cache: Optional[TensorCacheRrfs] = None
    ) -> "ArrayConstant": ...
    def tensor_hash_base16(self) -> str: ...

class Symbol(Circuit):
    def __init__(self, shape: Shape, uuid: UUID, name: Optional[str] = "Symbol") -> None: ...
    @property
    def uuid(self) -> UUID: ...
    @staticmethod
    def new_with_random_uuid(shape: Shape, name: Optional[str] = "Symbol") -> "Symbol": ...
    @staticmethod
    def new_with_none_uuid(shape: Shape, name: Optional[str] = "Symbol") -> "Symbol": ...

class ScalarConstant(Circuit):
    def __init__(self, value: float, shape: Shape = (), name: Optional[str] = "ScalarConstant") -> None: ...
    @property
    def value(self) -> float: ...
    def is_zero(self) -> bool: ...
    def is_one(self) -> bool: ...
    def evolve_shape(self, shape: Shape) -> ScalarConstant: ...

class Einsum(Circuit):
    def __init__(
        self,
        *args: Tuple[Circuit, Tuple[int, ...]],
        out_axes: Tuple[int, ...],
        name: Optional[str] = None,
    ) -> None: ...
    @property
    def args(self) -> List[Tuple[Circuit, Tuple[int, ...]]]: ...
    @property
    def out_axes(self) -> Tuple[int, ...]: ...
    def all_input_circuits(self) -> List[Circuit]: ...
    def all_input_axes(self) -> List[Tuple[int, ...]]: ...
    def next_axis(self) -> int: ...
    @staticmethod
    def from_einsum_string(string: str, *nodes: Circuit, name: Optional[str] = None) -> Einsum: ...
    @staticmethod
    def from_fancy_string(string: str, *nodes: Circuit, name: Optional[str] = None) -> Einsum: ...
    @staticmethod
    def from_spec(spec: EinsumSpec, *circuits: Circuit, name: Optional[str] = None) -> Einsum: ...
    @staticmethod
    def new_diag(node: Circuit, ints: List[int], name: Optional[str] = None) -> Einsum: ...
    @staticmethod
    def new_trace(node: Circuit, ints: List[int], name: Optional[str] = None) -> Einsum: ...
    @staticmethod
    def elementwise_broadcasted(*nodes: Circuit, name: Optional[str] = None) -> Einsum: ...
    def evolve(
        self,
        args: Optional[Sequence[Tuple[Circuit, Tuple[int, ...]]]] = None,
        out_axes: Optional[Tuple[int, ...]] = None,
    ) -> Einsum: ...
    def normalize_ints(self) -> Einsum: ...

class Add(Circuit):
    def __init__(self, *args: Circuit, name: Optional[str] = None) -> None: ...
    @property
    def nodes(self) -> List[Circuit]: ...
    def has_broadcast(self) -> bool: ...
    def nodes_and_rank_differences(self) -> List[Tuple[Circuit, int]]: ...
    def to_counts(self) -> Dict[Circuit, int]: ...
    # @staticmethod
    # def from_weighted_list(
    #     nodes_and_weights: Sequence[Tuple[Circuit, float]], use_1_weights: bool = False, name: Optional[str] = None
    # ) -> Add: ...
    @staticmethod
    def from_weighted_nodes(
        *args: Tuple[Circuit, float], use_1_weights: bool = False, name: Optional[str] = None
    ) -> Add: ...
    @staticmethod
    def minus(positive: Circuit, negative: Circuit, name: Optional[str] = None) -> Add: ...

class Rearrange(Circuit):
    def __init__(self, node: Circuit, spec: RearrangeSpec, name: Optional[str] = None) -> None: ...
    @staticmethod
    def from_einops_string(node: Circuit, string: str, name: Optional[str] = None) -> Rearrange: ...
    @property
    def node(self) -> Circuit: ...
    @property
    def spec(self) -> RearrangeSpec: ...
    @staticmethod
    def reshape(node: Circuit, shape: Shape) -> Rearrange: ...

TorchAxisIndex = Union[int, slice, torch.Tensor]

class Index(Circuit):
    def __init__(
        self,
        node: Circuit,
        index: Sequence[TorchAxisIndex],
        name: Optional[str] = None,
    ) -> None: ...
    @property
    def node(self) -> Circuit: ...
    @property
    def idx(self) -> List[TorchAxisIndex]: ...

class Scatter(Circuit):
    def __init__(
        self,
        node: Circuit,
        index: Sequence[TorchAxisIndex],
        shape: Tuple[int, ...],
        name: Optional[str] = None,
    ) -> None: ...
    @property
    def node(self) -> Circuit: ...
    @property
    def idx(self) -> List[TorchAxisIndex]: ...
    def is_identity(self) -> bool: ...

ConvPaddingShorthand = Union[int, List[int], List[Tuple[int, int]]]

class Conv(Circuit):
    def __init__(
        self,
        input: Circuit,
        filter: Circuit,
        stride: List[int],
        padding: ConvPaddingShorthand = 0,
        name: Optional[str] = None,
    ) -> None: ...
    @property
    def input(self) -> Circuit: ...
    @property
    def filter(self) -> Circuit: ...
    @property
    def strides(self) -> List[int]: ...
    @property
    def padding(self) -> List[Tuple[int, int]]: ...
    def is_identity(self) -> bool: ...

class GeneralFunctionSpec(object):
    function: Callable
    get_shape: Callable
    get_jacobian: Optional[Callable]
    num_non_batchable_output_dims: int
    input_batchability: List[bool]
    name: str
    def __init__(
        self,
        function: Callable,
        get_shape: Callable,
        get_jacobian: Optional[Callable],
        num_non_batchable_output_dims: int,
        input_batchability: List[bool],
        name: str,
    ) -> None: ...
    def is_batchable(self) -> bool: ...

class PyGFSpecShapeGetter(object):
    num_non_batchable: int
    removed_from_end: int
    def __init__(self, num_non_batchable: int, removed_from_end: int) -> None: ...
    def __call__(self, args: Sequence[Shape]) -> Optional[Shape]: ...

class GeneralFunction(Circuit):
    def __init__(
        self,
        *args: Circuit,
        spec: GeneralFunctionSpec,
        name: Optional[str] = None,
    ) -> None: ...
    @property
    def nodes(self) -> List[Circuit]: ...
    @property
    def spec(self) -> GeneralFunctionSpec: ...
    @staticmethod
    def new_by_name(*args: Circuit, spec_name: str, name: Optional[str] = None) -> GeneralFunction: ...
    @staticmethod
    def new_by_name_op(*args: Circuit, spec_name: str, name: Optional[str] = None) -> Optional[GeneralFunction]: ...

class Concat(Circuit):
    def __init__(self, *args: Circuit, axis: int, name: Optional[str] = None) -> None: ...
    @property
    def nodes(self) -> List[Circuit]: ...
    @property
    def axis(self) -> int: ...
    def get_sizes_at_axis(self) -> List[int]: ...
    @staticmethod
    def stack(*args: Circuit, axis: int, name: Optional[str] = None) -> Concat: ...

class ModuleNode(Circuit):
    def __init__(self, spec: ModuleNodeSpec, name: Optional[str], **kwargs: Circuit) -> None: ...
    @staticmethod
    def new_flat(nodes: Sequence[Circuit], spec: ModuleNodeSpec, name: Optional[str] = None) -> "ModuleNode": ...
    @property
    def spec(self) -> ModuleNodeSpec: ...
    @property
    def nodes(self) -> List[Circuit]: ...
    def expand(self) -> Circuit: ...

class ModuleNodeSpec(object):
    def __init__(
        self, spec_circuit: Circuit, arg_specs: Sequence[ModuleNodeArgSpec], name: Optional[str] = None
    ) -> None: ...
    @property
    def spec_circuit(self) -> Circuit: ...
    @property
    def input_specs(self) -> List[ModuleNodeArgSpec]: ...
    @staticmethod
    def new_extract(
        circuit: Circuit, input_specs: Sequence[Tuple[Circuit, ModuleNodeArgSpec]], name: Optional[str] = None
    ) -> ModuleNodeSpec: ...
    @staticmethod
    def new_auto(circuit: Circuit, name: Optional[str] = None) -> ModuleNodeSpec: ...

class ModuleNodeArgSpec(object):
    def __init__(self, symbol: Symbol, batchable: bool = False, expandable: bool = False) -> None: ...
    @property
    def symbol(self) -> Symbol: ...
    @property
    def batchable(self) -> bool: ...
    @property
    def expandable(self) -> bool: ...

# variables

class AutoTag(Circuit):
    def __init__(self, node: Circuit, uuid: UUID, name: Optional[str] = None) -> None: ...
    @property
    def node(self) -> Circuit: ...
    @property
    def uuid(self) -> UUID: ...
    @staticmethod
    def new_with_random_uuid(node: Circuit, name: Optional[str] = None) -> AutoTag: ...

class DiscreteVar(Circuit):
    def __init__(self, values: Circuit, probs_and_group: Optional[Circuit] = None, name: Optional[str] = None) -> None:
        """
        None probs_and_group create a new uniform group
        """
    @property
    def values(self) -> Circuit: ...
    @property
    def probs_and_group(self) -> Circuit: ...
    @staticmethod
    def new_uniform(values: Circuit, name: Optional[str] = None) -> DiscreteVar: ...
    @staticmethod
    def uniform_probs_and_group(size: int, name: Optional[str] = None) -> AutoTag: ...

class StoredCumulantVar(Circuit):
    def __init__(self, cumulants: Dict[int, Circuit], uuid: Optional[UUID], name: Optional[str] = None) -> None: ...
    @property
    def cumulants(self) -> Dict[int, Circuit]: ...
    @property
    def uuid(self) -> UUID: ...
    @staticmethod
    def new_mv(
        mean: Circuit,
        variance: Circuit,
        higher_cumulants: Dict[int, Circuit],
        uuid: Optional[UUID],
        name: Optional[str] = None,
    ) -> StoredCumulantVar: ...

class Cumulant(Circuit):
    def __init__(self, *args: Circuit, name: Optional[str] = None) -> None: ...
    @property
    def nodes(self) -> List[Circuit]: ...
    def equivalent_explicitly_computable_circuit(self) -> Circuit: ...
    @property
    def cumulant_number(self) -> int: ...
    @staticmethod
    def new_canon_rearrange(*args: Circuit, name: Optional[str] = None) -> Cumulant: ...

class TorchDeviceDtype(object):
    def __init__(self, device: str, dtype: str) -> None: ...
    @property
    def device(self) -> str: ...
    @property
    def dtype(self) -> str: ...
    @staticmethod
    def from_tensor(tensor: torch.Tensor) -> TorchDeviceDtype: ...

class TorchDeviceDtypeOp(object):
    def __init__(self, device: Optional[str] = None, dtype: Optional[str] = None) -> None: ...
    @staticmethod
    def default() -> TorchDeviceDtypeOp: ...
    @property
    def device(self) -> Optional[str]: ...
    @property
    def dtype(self) -> Optional[str]: ...

class OptimizationSettings(object):
    verbose: int
    max_memory: int
    max_single_tensor_memory: int
    max_memory_fallback: Optional[int]
    scheduling_num_mem_chunks: int
    distribute_min_size: Optional[int]
    scheduling_naive: bool
    scheduling_timeout: int
    scheduling_simplify: bool
    scheduling_simplify_lossy: bool
    adjust_numerical_scale: bool
    numerical_scale_min: float
    numerical_scale_max: float
    capture_and_print: bool
    create_tests: bool
    log_simplifications: bool
    log_slow_einsums: bool
    save_circ: bool
    def __init__(
        # warning: the pyi defaults are not exactly accurate. real values are in circuit_optimizer.rs.
        # those values should be the same as these, but I don't feel like guaranteeing that cuz lazy.
        self,
        verbose=0,
        max_memory=9_000_000_000,
        max_single_tensor_memory=9_000_000_000,
        max_memory_fallback=None,
        scheduling_num_mem_chunks=200,
        distribute_min_size=None,
        scheduling_naive=False,
        scheduling_timeout=5_000,
        scheduling_simplify=True,
        scheduling_simplify_lossy=False,
        adjust_numerical_scale=False,
        numerical_scale_min=1e-8,
        numerical_scale_max=1e8,
        capture_and_print=False,
        create_tests=False,
        log_simplifications=False,
        log_slow_einsums=False,
        save_circ=False,
    ) -> None: ...

class OptimizationContext(object):
    settings: OptimizationSettings
    cache: None  # there is a cache, but python cant see it
    def stringify_logs(self) -> str: ...
    @staticmethod
    def new_settings(settings: OptimizationSettings) -> OptimizationContext: ...
    @staticmethod
    def new_settings_circuit(
        settings: OptimizationSettings,
        circuit: Circuit,
    ) -> OptimizationContext: ...

# to generate below exception stubs, `cargo run print_exception_stubs`
class BatchError(ValueError): ...
class BatchNumberDoesntDivideError(BatchError): ...
class BatchRequiresMultipleAxesError(BatchError): ...
class BatchAxisOriginatesTooHighError(BatchError): ...
class ConstructError(ValueError): ...
class ConstructNotSupportedYetError(ConstructError): ...
class ConstructConvStridePaddingNotFullError(ConstructError): ...
class ConstructConvInputWrongShapeError(ConstructError): ...
class ConstructConvFilterWrongShapeError(ConstructError): ...
class ConstructConvFilterMustBeOddLengthError(ConstructError): ...
class ConstructConvStrideMustDivideError(ConstructError): ...
class ConstructConvGroupsMustDivideError(ConstructError): ...
class ConstructConvInputFilterDifferentNumInputChannelsError(ConstructError): ...
class ConstructDiscreteVarNoSamplesDimError(ConstructError): ...
class ConstructDiscreteVarWrongSamplesDimError(ConstructError): ...
class ConstructDiscreteVarProbsMustBe1dError(ConstructError): ...
class ConstructStoredCumulantVarNeedsMeanVarianceError(ConstructError): ...
class ConstructStoredCumulantVarInvalidCumulantNumberError(ConstructError): ...
class ConstructStoredCumulantVarCumulantWrongShapeError(ConstructError): ...
class ConstructEinsumLenShapeDifferentFromAxesError(ConstructError): ...
class ConstructEinsumShapeDifferentError(ConstructError): ...
class ConstructEinsumOutputNotSubsetError(ConstructError): ...
class ConstructRearrangeWrongInputShapeError(ConstructError): ...
class ConstructGeneralFunctionWrongInputShapeError(ConstructError): ...
class ConstructConcatZeroNodesError(ConstructError): ...
class ConstructConcatShapeDifferentError(ConstructError): ...
class ConstructAxisOutOfBoundsError(ConstructError): ...
class ConstructScatterShapeWrongError(ConstructError): ...
class ConstructScatterIndexTypeUnimplementedError(ConstructError): ...
class ConstructUnknownGeneralFunctionError(ConstructError): ...
class ConstructModuleNodeWrongNumberChildrenError(ConstructError): ...
class ConstructModuleNodeUnknownArgumentError(ConstructError): ...
class ConstructNamedAxisAboveRankError(ConstructError): ...
class ConstructNoEquivalentExplicitlyComputableError(ConstructError): ...
class ConstructUnflattenButNDimNot1Error(ConstructError): ...
class ExpandError(ValueError): ...
class ExpandWrongNumChildrenError(ExpandError): ...
class ExpandBatchingRankTooLowError(ExpandError): ...
class ExpandFixedIndexError(ExpandError): ...
class ExpandConcatAxisError(ExpandError): ...
class ExpandInconsistentBatchesError(ExpandError): ...
class ExpandNodeUnhandledVariantError(ExpandError): ...
class MiscInputError(ValueError): ...
class MiscInputNotBroadcastableError(MiscInputError): ...
class MiscInputItemOutOfBoundsError(MiscInputError): ...
class MiscInputChildrenMultipleDtypesError(MiscInputError): ...
class MiscInputChildrenMultipleDevicesError(MiscInputError): ...
class ParseError(ValueError): ...
class ParseNoRegexMatchError(ParseError): ...
class ParseNumberFailError(ParseError): ...
class ParseFailError(ParseError): ...
class ParseInvalidUuidError(ParseError): ...
class ParseWrongNumberChildrenError(ParseError): ...
class ParseInvalidIndentationError(ParseError): ...
class ParseInvalidVariantError(ParseError): ...
class ParseInvalidSerialNumberError(ParseError): ...
class ParseExtraUnneededStringError(ParseError): ...
class ParseShapeNeededError(ParseError): ...
class ParseGotMultipleCircuitsError(ParseError): ...
class ParseCircuitRefNotFoundError(ParseError): ...
class ParseModuleNotFoundError(ParseError): ...
class ParseEinsumStringInvalidError(ParseError): ...
class RearrangeSpecError(ValueError): ...
class RearrangeSpecInputNotConformableError(RearrangeSpecError): ...
class RearrangeSpecHasWildcardSizesError(RearrangeSpecError): ...
class RearrangeSpecNotConvertableError(RearrangeSpecError): ...
class RearrangeSpecTooManyWildcardSizesError(RearrangeSpecError): ...
class RearrangeSpecAxesAndCountsDontMatchError(RearrangeSpecError): ...
class RearrangeSpecNotFusableError(RearrangeSpecError): ...
class RearrangeSpecRepeatedAxesError(RearrangeSpecError): ...
class RearrangeSpecIntsNotUniqueError(RearrangeSpecError): ...
class RearrangeSpecInpNotInOutError(RearrangeSpecError): ...
class RearrangeSpecIntNotInSizesError(RearrangeSpecError): ...
class RearrangeSpecIntOnlyInOutputWithoutSizeError(RearrangeSpecError): ...
class RearrangeSpecInputAxisHasMultipleWildCardsError(RearrangeSpecError): ...
class RearrangeSpecLenShapeTooLargeError(RearrangeSpecError): ...
class RearrangeSpecAxesToCombineNotSubsetError(RearrangeSpecError): ...
class RearrangeSpecCantUnflattenScalarError(RearrangeSpecError): ...
class PermError(ValueError): ...
class PermIntsNotUniqueError(PermError): ...
class PermNotContiguousIntsLessThanCountError(PermError): ...
class RearrangeParseError(ValueError): ...
class RearrangeParseArrowIssueError(RearrangeParseError): ...
class RearrangeParseFailedToMatchRegexError(RearrangeParseError): ...
class RearrangeParseTooManyAxesError(RearrangeParseError): ...
class SchedulingError(ValueError): ...
class SchedulingLeafSymbolsError(SchedulingError): ...
class SchedulingOOMError(ValueError): ...
class SchedulingOOMManyError(SchedulingOOMError): ...
class SchedulingOOMSingleError(SchedulingOOMError): ...
class TensorEvalError(ValueError): ...
class TensorEvalNotExplicitlyComputableError(TensorEvalError): ...
class TensorEvalDeviceDtypeErrorError(TensorEvalError): ...
class SampleError(ValueError): ...
class SampleUnhandledVarErrorError(SampleError): ...
class SampleGroupWithIncorrectNdimError(SampleError): ...
class SampleDifferentNumSubsetsThanGroupsError(SampleError): ...
class IndexError(ValueError): ...
class IndexTensorNDimTooLargeError(IndexError): ...
class IndexIndexOutOfBoundsError(IndexError): ...
class IndexIndexRankTooHighError(IndexError): ...
class NestEinsumsError(ValueError): ...
class NestEinsumsMultipleRestError(NestEinsumsError): ...
class NestEinsumsMatcherMatchedMultipleError(NestEinsumsError): ...
class NestEinsumsMatcherMatchedNoneError(NestEinsumsError): ...
class NestEinsumsTraversalMatchedNothingError(NestEinsumsError): ...

class Schedule(object):
    @property
    def device_dtype(self) -> TorchDeviceDtype: ...
    @property
    def constants(self) -> Dict[bytes, torch.Tensor]: ...
    @property
    def instructions(self) -> List[Union[bytes, Circuit]]: ...
    def replace_tensors(self, map: Dict[bytes, torch.Tensor]) -> Schedule: ...
    def map_tensors(self, f: Callable[[bytes, torch.Tensor], torch.Tensor]) -> Schedule: ...
    def get_stats(self) -> ScheduleStats: ...
    def evaluate(self, settings: OptimizationSettings = OptimizationSettings()) -> torch.Tensor: ...
    def evaluate_many(self, settings: OptimizationSettings = OptimizationSettings()) -> List[torch.Tensor]: ...
    def serialize(self) -> str: ...
    @staticmethod
    def deserialize(string: str, device: str, tensor_cache: Optional[TensorCacheRrfs]) -> Schedule: ...
    def evaluate_remote(self, url: str) -> torch.Tensor: ...
    def evaluate_remote_many(self, url: str) -> List[torch.Tensor]: ...
    def tosend(self) -> ScheduleToSend: ...

class ScheduleStats(object):
    @property
    def max_mem(self) -> int: ...
    @property
    def constant_mem(self) -> int: ...
    @property
    def max_circuit_set(self) -> Set[Circuit]: ...

class ScheduleToSend(object):
    """this is just used to detach schedule from thread before sending to remote worker"""

    def evaluate_remote(self, url: str, device: str) -> torch.Tensor: ...
    def evaluate_remote_many(self, url: str, device: str) -> List[torch.Tensor]: ...

class Regex:
    @property
    def pattern(self) -> str: ...
    @property
    def escape_dot(self) -> bool: ...
    def __init__(self, pattern: str, escape_dot: bool = True) -> None: ...
    def call(self, s: str) -> bool: ...

MatcherIn = Union[
    bool,
    str,
    set[str],
    Type[Circuit],
    set[Type[Circuit]],
    Circuit,
    set[Circuit],
    Regex,
    Matcher,
    Callable[[Circuit], bool],
]

class Matcher(object):
    """
    Matchers just return a yes/no of whether the input circuit matches.
    They can be matchers to names, or any property of circuits (see methods)
    """

    def __init__(self, *inps: MatcherIn) -> None:
        """Does 'any' of the inputs if there are multiple"""
    def __call__(self, circuit: Circuit) -> bool: ...
    def call(self, circuit: Circuit) -> bool: ...
    def debug_print_to_str(self) -> str: ...
    def get_first(self, circuit: Circuit) -> Optional[Circuit]: ...
    @staticmethod
    def true_matcher() -> Matcher: ...
    @staticmethod
    def false_matcher() -> Matcher: ...
    @staticmethod
    def regex(pattern: str, escape_dot: bool = True) -> Matcher: ...
    @staticmethod
    def types(*types: Type[Circuit]) -> Matcher: ...
    @staticmethod
    def circuits(*circuits: Circuit) -> Matcher: ...
    @staticmethod
    def all(*matchers: MatcherIn) -> Matcher: ...
    @staticmethod
    def any(*matchers: MatcherIn) -> Matcher: ...
    def mk_not(self) -> Matcher: ...
    def mk_and(self, *others: MatcherIn) -> Matcher: ...
    def mk_or(self, *others: MatcherIn) -> Matcher: ...
    def __invert__(self) -> Matcher: ...
    def __and__(self, other: MatcherIn) -> Matcher: ...
    def __or__(self, other: MatcherIn) -> Matcher: ...
    def __rand__(self, other: MatcherIn) -> Matcher: ...
    def __ror__(self, other: MatcherIn) -> Matcher: ...
    def to_iterative_matcher(self) -> IterativeMatcher: ...
    def filter(
        self,
        term_if_matches: bool = False,
        start_depth: Optional[int] = None,
        end_depth: Optional[int] = None,
        term_early_at: MatcherIn = False,
    ) -> IterativeMatcher:
        """end_depth is exclusive"""
    def filter_sl(
        self,
        term_if_matches: bool = False,
        depth_slice: slice = slice(None),
        term_early_at: MatcherIn = False,
    ) -> IterativeMatcher: ...
    def chain(
        self,
        *rest: IterativeMatcherIn,
    ) -> IterativeMatcher:
        """
        For finding paths that satisfy a succession of matchers.

        A node is considered matched if `items` has a chain of length 1
        containing a matcher that matches the node.

        On a single iteration, this returns a new matcher as follows:
          - we want to construct the chains that will become the `items` of the new matcher
          - for each chain in `items`, see whether the first matcher in the chain matches the node
            - if so, it is considered satisfied
              - the tail of this chain will now itself be one of the new chains
              - Note that the same node is allowed to satisfy it, so we run
                again with the same node and the tail of this chain
            - whether or not it matches, it returns a new matcher, and we construct a new chain
              with this new matcher as the head, and the old tail as the tail

        Generally you will probably use `mk_chain` or `chain` to construct an instance with a single chain, say
        (A, B, C); but during iteration (e.g. in a Getter) new matchers will be
        constructed with multiple chains, e.g. after A matches a node the new matcher will have `items`
        ((A', B, C), (B, C)). It's possible B will also match this same node, in
        which case the new matcher will have `items`
        ((A', B, C), (B, C), (B', C), (C,)). If this is then applied to a node
        that matches C, that node will be matched, since `items` contains the
        chain (C,).
        """
    def chain_many(
        self,
        *rest: Sequence[IterativeMatcherIn],
    ) -> IterativeMatcher:
        """
        For finding paths that satisfy a succession of matchers.

        A node is considered matched if `items` has a chain of length 1
        containing a matcher that matches the node.

        On a single iteration, this returns a new matcher as follows:
          - we want to construct the chains that will become the `items` of the new matcher
          - for each chain in `items`, see whether the first matcher in the chain matches the node
            - if so, it is considered satisfied
              - the tail of this chain will now itself be one of the new chains
              - Note that the same node is allowed to satisfy it, so we run
                again with the same node and the tail of this chain
            - whether or not it matches, it returns a new matcher, and we construct a new chain
              with this new matcher as the head, and the old tail as the tail

        Generally you will probably use `mk_chain` or `chain` to construct an instance with a single chain, say
        (A, B, C); but during iteration (e.g. in a Getter) new matchers will be
        constructed with multiple chains, e.g. after A matches a node the new matcher will have `items`
        ((A', B, C), (B, C)). It's possible B will also match this same node, in
        which case the new matcher will have `items`
        ((A', B, C), (B, C), (B', C), (C,)). If this is then applied to a node
        that matches C, that node will be matched, since `items` contains the
        chain (C,).
        """
    def get(self, circuit: Circuit, fancy_validate: bool = False) -> Set[Circuit]: ...
    def get_unique_op(self, circuit: Circuit, fancy_validate: bool = False) -> Optional[Circuit]: ...
    def get_unique(self, circuit: Circuit, fancy_validate: bool = False) -> Circuit: ...
    def validate(self, circuit: Circuit) -> None: ...
    def getter(self, default_fancy_validate: bool = False) -> BoundGetter: ...
    def update(
        self,
        circuit: Circuit,
        transform: TransformIn,
        transform_along_modified_path: TransformIn = Transform.ident(),
        cache_transform: bool = True,
        cache_transform_along_modified_path: bool = True,
        cache_update: bool = True,
        fancy_validate: bool = False,
    ) -> Circuit: ...
    def updater(
        self,
        transform: TransformIn,
        transform_along_modified_path: TransformIn = Transform.ident(),
        cache_transform: bool = True,
        cache_transform_along_modified_path: bool = True,
        cache_update: bool = True,
        default_fancy_validate: bool = False,
    ) -> BoundUpdater:
        """
        transform_along_modified_path runs on modified parent nodes, not including nodes which are matched
        """

IterativeMatcherIn = Union[MatcherIn, IterativeMatcher]

class IterativeMatcher:
    """
    Returns IterateMatchResults when called (see docs for that). Useful for iteratively searching
    through a circuit for some matching nodes.
    """

    def __init__(self, *inps: IterativeMatcherIn) -> None:
        """Does 'any' of the inputs if there arejo multiple"""
    def match_iterate(self, circuit: Circuit) -> IterateMatchResults: ...
    def __call__(self, circuit: Circuit) -> IterateMatchResults:
        """alias for match_iterate"""
    def debug_print_to_str(self) -> str: ...
    @staticmethod
    def noop_traversal() -> IterativeMatcher: ...
    def filter(
        self,
        term_if_matches: bool = False,
        start_depth: Optional[int] = None,
        end_depth: Optional[int] = None,
        term_early_at: MatcherIn = False,
    ) -> IterativeMatcher:
        """Helper with some basic rules you may want to use to control your node matching iterations.

        term_if_matches : bool
            if true, stops once it has found a match
        start_depth : Optional[int]
            depth at which to start matching
        end_depth : Optional[int]
            stops at this depth (e.g., exclusively match before this depth)
        term_early_at : NodeMatcher
            specifies nodes at which to stop iteration"""
    def filter_sl(
        self,
        term_if_matches: bool = False,
        depth_slice: slice = slice(None),
        term_early_at: MatcherIn = False,
    ) -> IterativeMatcher:
        """Helper with some basic rules you may want to use to control your node matching iterations.

        term_if_matches : bool
            if true, stops once it has found a match
        depth_slice : slice
            depth range at which to start and stop matching
            (you might find SLICER useful for this)
        term_early_at : NodeMatcher
            specifies nodes at which to stop iteration"""
    def chain(
        self,
        *rest: IterativeMatcherIn,
    ) -> IterativeMatcher:
        """
        For finding paths that satisfy a succession of matchers.

        A node is considered matched if `items` has a chain of length 1
        containing a matcher that matches the node.

        On a single iteration, this returns a new matcher as follows:
          - we want to construct the chains that will become the `items` of the new matcher
          - for each chain in `items`, see whether the first matcher in the chain matches the node
            - if so, it is considered satisfied
              - the tail of this chain will now itself be one of the new chains
              - Note that the same node is allowed to satisfy it, so we run
                again with the same node and the tail of this chain
            - whether or not it matches, it returns a new matcher, and we construct a new chain
              with this new matcher as the head, and the old tail as the tail

        Generally you will probably use `mk_chain` or `chain` to construct an instance with a single chain, say
        (A, B, C); but during iteration (e.g. in a Getter) new matchers will be
        constructed with multiple chains, e.g. after A matches a node the new matcher will have `items`
        ((A', B, C), (B, C)). It's possible B will also match this same node, in
        which case the new matcher will have `items`
        ((A', B, C), (B, C), (B', C), (C,)). If this is then applied to a node
        that matches C, that node will be matched, since `items` contains the
        chain (C,).
        """
    def chain_many(
        self,
        *rest: Sequence[IterativeMatcherIn],
    ) -> IterativeMatcher:
        """
        For finding paths that satisfy a succession of matchers.

        A node is considered matched if `items` has a chain of length 1
        containing a matcher that matches the node.

        On a single iteration, this returns a new matcher as follows:
          - we want to construct the chains that will become the `items` of the new matcher
          - for each chain in `items`, see whether the first matcher in the chain matches the node
            - if so, it is considered satisfied
              - the tail of this chain will now itself be one of the new chains
              - Note that the same node is allowed to satisfy it, so we run
                again with the same node and the tail of this chain
            - whether or not it matches, it returns a new matcher, and we construct a new chain
              with this new matcher as the head, and the old tail as the tail

        Generally you will probably use `mk_chain` or `chain` to construct an instance with a single chain, say
        (A, B, C); but during iteration (e.g. in a Getter) new matchers will be
        constructed with multiple chains, e.g. after A matches a node the new matcher will have `items`
        ((A', B, C), (B, C)). It's possible B will also match this same node, in
        which case the new matcher will have `items`
        ((A', B, C), (B, C), (B', C), (C,)). If this is then applied to a node
        that matches C, that node will be matched, since `items` contains the
        chain (C,).
        """
    @staticmethod
    def any(*matchers: IterativeMatcherIn) -> IterativeMatcher: ...
    @staticmethod
    def mk_chain(first: IterativeMatcherIn, *matchers: IterativeMatcherIn) -> IterativeMatcher:
        """
        For finding paths that satisfy a succession of matchers.

        A node is considered matched if `items` has a chain of length 1
        containing a matcher that matches the node.

        On a single iteration, this returns a new matcher as follows:
          - we want to construct the chains that will become the `items` of the new matcher
          - for each chain in `items`, see whether the first matcher in the chain matches the node
            - if so, it is considered satisfied
              - the tail of this chain will now itself be one of the new chains
              - Note that the same node is allowed to satisfy it, so we run
                again with the same node and the tail of this chain
            - whether or not it matches, it returns a new matcher, and we construct a new chain
              with this new matcher as the head, and the old tail as the tail

        Generally you will probably use `mk_chain` or `chain` to construct an instance with a single chain, say
        (A, B, C); but during iteration (e.g. in a Getter) new matchers will be
        constructed with multiple chains, e.g. after A matches a node the new matcher will have `items`
        ((A', B, C), (B, C)). It's possible B will also match this same node, in
        which case the new matcher will have `items`
        ((A', B, C), (B, C), (B', C), (C,)). If this is then applied to a node
        that matches C, that node will be matched, since `items` contains the
        chain (C,).
        """
    @staticmethod
    def mk_chain_many(*matchers: Sequence[IterativeMatcherIn]) -> IterativeMatcher:
        """each chain must be non-empty

        For finding paths that satisfy a succession of matchers.

        A node is considered matched if `items` has a chain of length 1
        containing a matcher that matches the node.

        On a single iteration, this returns a new matcher as follows:
          - we want to construct the chains that will become the `items` of the new matcher
          - for each chain in `items`, see whether the first matcher in the chain matches the node
            - if so, it is considered satisfied
              - the tail of this chain will now itself be one of the new chains
              - Note that the same node is allowed to satisfy it, so we run
                again with the same node and the tail of this chain
            - whether or not it matches, it returns a new matcher, and we construct a new chain
              with this new matcher as the head, and the old tail as the tail

        Generally you will probably use `mk_chain` or `chain` to construct an instance with a single chain, say
        (A, B, C); but during iteration (e.g. in a Getter) new matchers will be
        constructed with multiple chains, e.g. after A matches a node the new matcher will have `items`
        ((A', B, C), (B, C)). It's possible B will also match this same node, in
        which case the new matcher will have `items`
        ((A', B, C), (B, C), (B', C), (C,)). If this is then applied to a node
        that matches C, that node will be matched, since `items` contains the
        chain (C,).
        """
    @staticmethod
    def mk_func(f: Callable[[Circuit], IterateMatchResults]) -> IterativeMatcher: ...
    def mk_or(self, *others: IterativeMatcherIn) -> IterativeMatcher: ...
    def __or__(self, other: IterativeMatcherIn) -> IterativeMatcher: ...
    def __ror__(self, other: IterativeMatcherIn) -> IterativeMatcher: ...
    def get(self, circuit: Circuit, fancy_validate: bool = False) -> Set[Circuit]: ...
    def get_unique_op(self, circuit: Circuit, fancy_validate: bool = False) -> Optional[Circuit]: ...
    def get_unique(self, circuit: Circuit, fancy_validate: bool = False) -> Circuit: ...
    def validate(self, circuit: Circuit) -> None: ...
    def getter(self, default_fancy_validate: bool = False) -> BoundGetter: ...
    def update(
        self,
        circuit: Circuit,
        transform: TransformIn,
        transform_along_modified_path: TransformIn = Transform.ident(),
        cache_transform: bool = True,
        cache_transform_along_modified_path: bool = True,
        cache_update: bool = True,
        fancy_validate: bool = False,
    ) -> Circuit:
        """
        transform_along_modified_path runs on modified parent nodes, not including nodes which are matched
        """
    def updater(
        self,
        transform: TransformIn,
        transform_along_modified_path: TransformIn = Transform.ident(),
        cache_transform: bool = True,
        cache_transform_along_modified_path: bool = True,
        cache_update: bool = True,
        default_fancy_validate: bool = False,
    ) -> BoundUpdater:
        """
        transform_along_modified_path runs on modified parent nodes, not including nodes which are matched
        """

class IterateMatchResults:
    """updated: new matcher to be applied to children (if None, use the same matcher)
    finished: is this matcher done (interpret as apply False matcher to all children)
    found: did we find this node?"""

    updated: Optional[IterativeMatcher]
    finished: bool
    found: bool
    def __init__(
        self, updated: Optional[IterativeMatcher] = None, finished: bool = False, found: bool = False
    ) -> None: ...
    def to_tup(self) -> Tuple[Optional[IterativeMatcher], bool, bool]: ...
    def none_if_finshed(self, IterativeMatcher) -> Tuple[Optional[IterativeMatcher], bool]: ...

TransformIn = Union[Callable[[Circuit], Circuit], Transform]

class Transform:
    def __init__(self, inp: TransformIn) -> None: ...
    def run(self, circuit: Circuit) -> Circuit: ...
    def __call__(self, circuit: Circuit) -> Circuit:
        """Alias for run"""
    def debug_print_to_str(self) -> str: ...
    @staticmethod
    def ident() -> Transform: ...
    def updater(
        self,
        transform_along_modified_path: TransformIn = Transform.ident(),
        cache_transform: bool = True,
        cache_transform_along_modified_path: bool = True,
        cache_update: bool = True,
    ) -> Updater:
        """
        transform_along_modified_path runs on modified parent nodes, not including nodes which are matched
        """

class Updater:
    default_fancy_validate: bool
    def __init__(
        self,
        transform: TransformIn,
        transform_along_modified_path: TransformIn = Transform.ident(),
        cache_transform: bool = True,
        cache_transform_along_modified_path: bool = True,
        cache_update: bool = True,
        default_fancy_validate: bool = False,
    ) -> None:
        """
        transform_along_modified_path runs on modified parent nodes, not including nodes which are matched
        """
    @property
    def transform(self) -> Transform: ...
    @property
    def transform_along_modified_path(self) -> Transform: ...
    @property
    def cache_transform(self) -> bool: ...
    @property
    def cache_transform_along_modified_path(self) -> bool: ...
    @property
    def cache_update(self) -> bool: ...
    def update(
        self, circuit: Circuit, matcher: IterativeMatcherIn, fancy_validate: Optional[bool] = None
    ) -> Circuit: ...
    def __call__(self, circuit: Circuit, matcher: IterativeMatcherIn, fancy_validate: Optional[bool] = None) -> Circuit:
        """Alias for update"""
    def bind(self, matcher: IterativeMatcherIn) -> BoundUpdater: ...

class BoundUpdater:
    updater: Updater
    @property
    def matcher(self) -> IterativeMatcher: ...
    @matcher.setter
    def matcher(self, matcher: IterativeMatcherIn) -> None: ...
    def __init__(
        self,
        updater: Updater,
        matcher: IterativeMatcherIn,
    ) -> None: ...
    def update(self, circuit: Circuit, fancy_validate: Optional[bool] = None) -> Circuit: ...
    def __call__(self, circuit: Circuit, fancy_validate: Optional[bool] = None) -> Circuit:
        """Alias for update"""

class Getter:
    default_fancy_validate: bool
    """Fancy validation checks that all of the matchers matched something. For
    instance, each name/type/regex should match something. See tests for more details."""
    def __init__(self, default_fancy_validate: bool = False) -> None: ...
    def get(
        self, circuit: Circuit, matcher: IterativeMatcherIn, fancy_validate: Optional[bool] = None
    ) -> set[Circuit]: ...
    def __call__(
        self, circuit: Circuit, matcher: IterativeMatcherIn, fancy_validate: Optional[bool] = None
    ) -> set[Circuit]:
        """Alias for get"""
    def get_unique_op(
        self, circuit: Circuit, matcher: IterativeMatcherIn, fancy_validate: Optional[bool] = None
    ) -> Optional[Circuit]: ...
    def get_unique(
        self, circuit: Circuit, matcher: IterativeMatcherIn, fancy_validate: Optional[bool] = None
    ) -> Circuit: ...
    def validate(self, circuit: Circuit, matcher: IterativeMatcherIn) -> None: ...
    def bind(self, matcher: IterativeMatcherIn) -> BoundGetter: ...

class BoundGetter:
    getter: Getter
    @property
    def matcher(self) -> IterativeMatcher: ...
    @matcher.setter
    def matcher(self, matcher: IterativeMatcherIn) -> None: ...
    def __init__(self, getter: Getter, matcher: IterativeMatcherIn) -> None: ...
    def get(self, circuit: Circuit, fancy_validate: Optional[bool] = None) -> set[Circuit]: ...
    def __call__(self, circuit: Circuit, fancy_validate: Optional[bool] = None) -> set[Circuit]:
        """Alias for get"""
    def get_unique_op(self, circuit: Circuit, fancy_validate: Optional[bool] = None) -> Optional[Circuit]: ...
    def get_unique(self, circuit: Circuit, fancy_validate: Optional[bool] = None) -> Circuit: ...
    def validate(self, circuit: Circuit) -> None: ...

class Batcher:
    ban_multiple_matches_on_node: bool
    default_fancy_validate: bool
    @property
    def replacements(self) -> List[Transform]: ...
    @property
    def matchers(self) -> List[IterativeMatcher]: ...
    @property
    def suffix(self) -> Optional[str]: ...
    def __init__(
        self,
        *batchers: Tuple[IterativeMatcherIn, TransformIn],
        ban_multiple_matches_on_node: bool = False,
        default_fancy_validate: bool = False,
        suffix: Optional[str] = None,
    ) -> None: ...
    def batch(self, circuit: Circuit, fancy_validate: Optional[bool] = None) -> Circuit: ...
    def __call__(self, circuit: Circuit, fancy_validate: Optional[bool] = None) -> Circuit:
        """Alias for batch"""

class TensorCacheRrfs(object):
    def __init__(self, cutoff: int, small_capacityint, large_capacityint, device: str) -> None: ...
    def get_tensor(self, prefix: str) -> torch.Tensor: ...
    def get_tensor_if_cached(self, prefix: str) -> Optional[torch.Tensor]: ...

def add_collapse_scalar_inputs(add: Add) -> Optional[Add]: ...
def add_deduplicate(add: Add) -> Optional[Add]: ...
def add_flatten_once(add: Add) -> Optional[Add]: ...
def remove_add_few_input(add: Add) -> Optional[Add]: ...
def add_pull_removable_axes(add: Add, remove_non_common_axes: bool) -> Optional[Circuit]: ...
def einsum_flatten_once(einsum: Einsum) -> Optional[Einsum]: ...
def einsum_elim_identity(einsum: Einsum) -> Optional[Circuit]: ...
def index_merge_scalar(index: Index) -> Optional[Circuit]: ...
def index_elim_identity(index: Index) -> Optional[Circuit]: ...
def index_fuse(index: Index) -> Optional[Index]: ...
def rearrange_fuse(node: Rearrange) -> Optional[Rearrange]: ...
def rearrange_merge_scalar(rearrange: Rearrange) -> Optional[Circuit]: ...
def rearrange_elim_identity(rearrange: Rearrange) -> Optional[Circuit]: ...
def concat_elim_identity(concat: Concat) -> Optional[Circuit]: ...
def concat_merge_uniform(concat: Concat) -> Optional[Concat]: ...
def concat_pull_removable_axes(concat: Concat) -> Optional[Circuit]: ...
def generalfunction_pull_removable_axes(node: GeneralFunction) -> Optional[Circuit]: ...
def generalfunction_merge_inverses(node: GeneralFunction) -> Optional[Circuit]: ...
def generalfunction_special_case_simplification(node: GeneralFunction) -> Optional[Circuit]: ...
def generalfunction_evaluate_simple(node: GeneralFunction) -> Optional[Circuit]: ...
def einsum_pull_removable_axes(node: Einsum) -> Optional[Circuit]: ...
def scatter_pull_removable_axes(node: Scatter) -> Optional[Circuit]: ...
def add_make_broadcasts_explicit(node: Add) -> Optional[Add]: ...
def distribute(node: Einsum, operand_idx: int, do_broadcasts: bool) -> Optional[Add]: ...
def distribute_all(node: Einsum) -> Optional[Add]: ...
def einsum_of_permute_merge(node: Einsum) -> Optional[Einsum]: ...
def permute_of_einsum_merge(node: Rearrange) -> Optional[Einsum]: ...
def einsum_elim_zero(node: Einsum) -> Optional[ScalarConstant]: ...
def einsum_merge_scalars(node: Einsum) -> Optional[Einsum]: ...
def push_down_index(node: Index) -> Optional[Circuit]: ...
def deep_push_down_index(node: Circuit, min_size: Optional[int] = None) -> Circuit: ...
def index_split_axes(node: Index, top_axes: Set[int]) -> Optional[Index]: ...
def add_elim_zeros(node: Add) -> Optional[Add]: ...
def compiler_simp(node: Circuit) -> Circuit: ...
def basic_simp(node: Circuit) -> Circuit: ...
def compiler_simp_step(node: Circuit) -> Optional[Circuit]: ...
def compiler_simp_until_same(node: Circuit) -> Circuit: ...
def strip_names(node: Circuit) -> Circuit: ...
def deep_canonicalize(node: Circuit) -> Circuit: ...
def canonicalize_node(node: Circuit) -> Circuit: ...
def deep_maybe_distribute(node: Circuit, settings: OptimizationSettings) -> Circuit: ...
def maybe_distribute(node: Einsum, settings: OptimizationSettings) -> Optional[Circuit]: ...
def einsum_nest_optimize(node: Einsum) -> Optional[Einsum]: ...
def deep_optimize_einsums(node: Circuit) -> Circuit: ...
def einsum_permute_to_rearrange(node: Einsum) -> Optional[Circuit]: ...
def einsum_nest_path(node: Einsum, path: List[List[int]]) -> Einsum: ...
def scatter_elim_identity(node: Scatter) -> Optional[Circuit]: ...
def einsum_pull_scatter(node: Einsum) -> Optional[Circuit]: ...
def add_pull_scatter(node: Add) -> Optional[Circuit]: ...
def index_einsum_to_scatter(node: Index) -> Optional[Circuit]: ...
def optimize_circuit(circuit: Circuit, settings: OptimizationSettings) -> Circuit: ...
def cast_circuit(circuit: Circuit, device_dtype: TorchDeviceDtype) -> Circuit: ...
def scatter_to_concat(scatter: Scatter) -> Circuit: ...
def count_nodes(circuit: Circuit) -> int: ...
def scheduled_evaluate(circuit: Circuit, settings: OptimizationSettings) -> torch.Tensor: ...
def optimize_and_evaluate(circuit: Circuit, settings: OptimizationSettings) -> torch.Tensor: ...
def optimize_and_evaluate_many(circuits: List[Circuit], settings: OptimizationSettings) -> List[torch.Tensor]: ...
def optimize_to_schedule(circuit: Circuit, settings: OptimizationSettings) -> Schedule: ...
def optimize_to_schedule_many(circuits: List[Circuit], settings: OptimizationSettings) -> Schedule: ...
def flat_concat(circuits: List[Circuit]) -> Concat: ...
def flat_concat_back(circuits: List[Circuit]) -> Tuple[Concat, List[Circuit]]: ...
def deep_heuristic_nest_adds(circuit: Circuit) -> Circuit: ...
def generalfunction_pull_concat(circuit: GeneralFunction) -> Optional[Concat]: ...
def concat_fuse(circuit: Concat) -> Optional[Concat]: ...
def concat_drop_size_zero(circuit: Concat) -> Optional[Concat]: ...
def index_concat_drop_unreached(circuit: Index) -> Optional[Index]: ...
def einsum_pull_concat(circuit: Einsum) -> Optional[Circuit]: ...
def add_pull_concat(circuit: Add) -> Optional[Circuit]: ...
def split_to_concat(circuit: Circuit, axis: int, sections: List[int]) -> Optional[Circuit]: ...
def deep_pull_concat_messy(circuit: Circuit, min_size: Optional[int]) -> Circuit: ...
def deep_pull_concat(circuit: Circuit, min_size: Optional[int] = None) -> Circuit: ...
def batch_to_concat(circuit: Circuit, axis: int, num_batches: int) -> Circuit: ...
def batch_einsum(einsum: Einsum, settings: OptimizationSettings) -> Circuit: ...
def set_named_axes(circuit: Circuit, named_axes: Dict[int, str]) -> Circuit: ...
def propagate_named_axes(circuit: Circuit, named_axes: Dict[int, str], do_merge: bool) -> Circuit: ...
def toposort_circuit(circuit: Circuit) -> List[Circuit]: ...
def einsum_push_down_trace(circuit: Einsum) -> Optional[Einsum]: ...
def einsum_concat_to_add(circuit: Einsum) -> Optional[Einsum]: ...
def add_pull_diags(circuit: Add) -> Optional[Circuit]: ...
def concat_repeat_to_rearrange(circuit: Concat) -> Optional[Concat]: ...
def add_outer_product_broadcasts_on_top(circuit: Add) -> Optional[Add]: ...
def extract_add(add: Add, sub: Add) -> Optional[Add]: ...
def add_fuse_scalar_multiples(add: Add) -> Optional[Add]: ...
def concat_to_scatter(concat: Concat) -> Optional[Scatter]: ...
def replace_all_randn_seeded(circuit: Circuit) -> Circuit: ...
def opt_eval_each_subcircuit_until_fail(circuit: Circuit, settings: OptimizationSettings) -> None: ...

# circuit manipulation functions
def deep_map(circuit: Circuit, fn: Callable[[Circuit], Circuit]) -> Circuit: ...
def deep_map_preorder(circuit: Circuit, fn: Callable[[Circuit], Circuit]) -> Circuit: ...
def filter_nodes(circuit: Circuit, fn: Callable[[Circuit], bool]) -> List[Circuit]: ...
def replace_nodes(circuit: Circuit, map: Dict[Circuit, Circuit]) -> Circuit: ...
def path_get(circuit: Circuit, path: List[int]) -> Circuit: ...
def einsum_elim_removable_axes_weak(circuit: Einsum) -> Optional[Circuit]: ...
def add_elim_removable_axes_weak(circuit: Add) -> Optional[Circuit]: ...
def update_nodes(
    circuit: Circuit, matcher: Callable[[Circuit], bool], mapper: Callable[[Circuit], Circuit]
) -> Circuit: ...
def update_path(circuit: Circuit, path: List[int], updater: Callable[[Circuit], Circuit]) -> Circuit: ...
def expand_node(circuit: Circuit, inputs: List[Circuit]) -> Circuit: ...
def save_tensor_rrfs(tensor: torch.Tensor) -> str: ...  # string is base16 key
def tensor_from_hash(hash: str) -> torch.Tensor: ...
def inline_all_modules(circuit: Circuit) -> Circuit: ...
def replace_expand_bottom_up(circuit: Circuit, f: Callable[[Circuit], Optional[Circuit]]) -> Circuit: ...
def replace_expand_bottom_up_dict(circuit: Circuit, dict: Dict[Circuit, Circuit]) -> Circuit: ...
def parse_compiler_repr_bijection(
    string: str,
    module_spec_map: Dict[str, ModuleNodeSpec] = {},
    reference_circuits: Dict[str, Circuit] = {},
    tensors_as_random: bool = False,
    tensors_as_random_device_dtype=TorchDeviceDtypeOp(None, None),
    tensor_cache: Optional[TensorCacheRrfs] = None,
) -> Circuit: ...
def circuit_is_leaf(circuit: Circuit) -> bool: ...
def circuit_is_irreducible_node(circuit: Circuit) -> bool: ...
def circuit_is_leaf_constant(circuit: Circuit) -> bool: ...
def circuit_is_var(circuit: Circuit) -> bool: ...
def circuit_server_serve(url: str, device: str, tensor_cache: Optional[TensorCacheRrfs]): ...
def visit_circuit(circuit: Circuit, f: Callable[[Circuit], None]): ...
def print_circuit_type_check(x: Type[Circuit]) -> Type[Circuit]: ...
def hash_tensor(x: torch.Tensor) -> bytes: ...
def default_var_matcher() -> IterativeMatcher: ...

# TODO: better doc + examples for sampling stuff!

class SampleSpec:
    def sample_var(self, d: DiscreteVar) -> Circuit: ...
    def get_empirical_weights(self) -> Circuit: ...
    def get_overall_weight(self) -> Circuit: ...
    def get_sample_shape(self) -> Circuit: ...
    def get_transform(self) -> Transform: ...
    def get_sampler(
        self,
        var_matcher: IterativeMatcherIn = default_var_matcher(),
        default_fancy_validate: bool = False,
        suffix: Optional[str] = "sample",
    ) -> Batcher:
        """
        get a sampler: a Batcher configured to sample from vars. It will sample
        from all vars that are matched by var_matcher and batch over them
        together.

        Note: sampling the child of a cumulant doesn't work - use
        SampleTransform for estimation. As such, the var_matcher should term
        early at cumulants.

        Note that even if this doesn't match any vars it will always add batch dims.
        """

class RandomSampleSpec(SampleSpec):
    probs_and_group_evaluation_settings: OptimizationSettings
    def __init__(
        self,
        shape: Shape = (),
        replace: bool = True,
        probs_and_group_evaluation_settings: OptimizationSettings = OptimizationSettings(),
    ): ...
    @property
    def shape(self) -> Shape: ...
    @property
    def replace(self) -> bool: ...
    def indices_cache(self) -> Dict[bytes, torch.Tensor]: ...

class RunDiscreteVarAllSpec:
    def __init__(
        self,
        groups: Sequence[Circuit],
        subsets: Optional[Sequence[slice]] = None,
    ): ...
    @property
    def groups(self) -> List[Circuit]: ...
    @property
    def subsets(self) -> List[slice]: ...
    @staticmethod
    def create_full_from_circuits(*circuits: Circuit) -> RunDiscreteVarAllSpec: ...

SampleSpecIn = RandomSampleSpec | RunDiscreteVarAllSpec

class SampleTransform:
    def __init__(
        self,
        sample_spec: SampleSpecIn,
        var_matcher: IterativeMatcherIn = default_var_matcher(),
        cumulant_matcher: IterativeMatcherIn = Cumulant,
        suffix: Optional[str] = "sample",
    ):
        """
        See docs for `estimate` for details on `cumulant_matcher`.
        """
    @property
    def sampler(self) -> Batcher: ...
    @property
    def cumulant_matcher(self) -> IterativeMatcher: ...
    @property
    def sample_spec(self) -> SampleSpec: ...
    def estimate(self, circuit: Circuit) -> Circuit:
        """
        Recursively estimate all cumulants matched by cumulant_number.
        This doesn't require that the circuit is constant.
        If your matcher allows for recursive matching, then it will estimate nested cumulants.
        The default matcher estimates all nested cumulants.

        Note the the cumulant estimation process involves creating nested
        cumulants and then recusively estimating these, so the traversal will
        hit cumulants which weren't originally in the circuit - keep this in mind when passing
        a cumulant_matcher.
        """
    def sample(self, circuit: Circuit) -> Circuit:
        """
        Runs sample on circuit. See docs for get_sampler on SampleSpec.
        """
    def estimate_and_sample(self, circuit: Circuit) -> Circuit:
        """
        first run estimate, then run sample
        """
    def __call__(self, c: Circuit) -> Circuit:
        """
        alias for estimate_and_sample
        """

def factored_cumulant_expectation_rewrite(circuit: Cumulant) -> Circuit: ...

IntOrMatcher = Union[int, MatcherIn]

class NestEinsumsRest:
    flat: bool
    def __init__(self, flat: bool = False) -> None: ...

NestEinsumsSpecMultiple = Union[NestEinsumsRest, Sequence[NestEinsumsSpec]]  # type: ignore
NestEinsumsSpecSub = Union[IntOrMatcher, NestEinsumsSpecMultiple]  # type: ignore

class NestEinsumsSpecInfo:
    spec: NestEinsumsSpecMultiple
    name: Optional[str]
    out_axes_perm: Optional[Sequence[int]]
    def __init__(
        self,
        spec: NestEinsumsSpecMultiple,
        name: Optional[str] = None,
        out_axes_perm: Optional[Sequence[int]] = None,
    ) -> None: ...

NestEinsumsSpec = Union[NestEinsumsSpecSub, NestEinsumsSpecInfo]  # type: ignore

class NestEinsumsIdxsInfo:
    idxs: NestEinsumsIdxsItem
    name: Optional[str]
    out_axes: Optional[Sequence[int]]
    out_axes_perm: Optional[Sequence[int]]
    def __init__(
        self,
        idxs: NestEinsumsIdxsItem,
        name: Optional[str] = None,
        out_axes: Optional[Sequence[int]] = None,
        out_axes_perm: Optional[Sequence[int]] = None,
    ) -> None: ...

NestEinsumsIdxsItem = Union[int, Sequence[NestEinsumsIdxsInfo]]

def einsum_flatten(einsum: Einsum, traversal: Optional[IterativeMatcher] = None) -> Einsum: ...
def nest_einsums(einsum: Einsum, spec: NestEinsumsSpecSub, traversal: Optional[IterativeMatcher] = None) -> Einsum: ...
def nest_flat_einsum_strict(einsum: Einsum, item: NestEinsumsIdxsItem) -> Einsum:
    """
    be warned: if you pass invalid out_axes, this function will panic!
    """

def nest_einsums_strict(
    einsum: Einsum, spec: NestEinsumsIdxsItem, traversal: Optional[IterativeMatcher] = None
) -> Einsum:
    """
    be warned: if you pass invalid out_axes, this function will panic!
    """
