"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildAction = exports.CodeBuildActionType = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const __1 = require("..");
const action_1 = require("../action");
const source_action_1 = require("../codecommit/source-action");
/**
 * The type of the CodeBuild action that determines its CodePipeline Category -
 * Build, or Test.
 * The default is Build.
 */
var CodeBuildActionType;
(function (CodeBuildActionType) {
    /**
     * The action will have the Build Category.
     * This is the default.
     */
    CodeBuildActionType[CodeBuildActionType["BUILD"] = 0] = "BUILD";
    /**
     * The action will have the Test Category.
     */
    CodeBuildActionType[CodeBuildActionType["TEST"] = 1] = "TEST";
})(CodeBuildActionType = exports.CodeBuildActionType || (exports.CodeBuildActionType = {}));
/**
 * CodePipeline build action that uses AWS CodeBuild.
 */
class CodeBuildAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: props.type === CodeBuildActionType.TEST
                ? codepipeline.ActionCategory.TEST
                : codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 1, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.input, ...props.extraInputs || []],
            resource: props.project,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_codepipeline_actions_CodeBuildActionProps(props);
        this.props = props;
    }
    /**
     * Reference a CodePipeline variable defined by the CodeBuild project this action points to.
     * Variables in CodeBuild actions are defined using the 'exported-variables' subsection of the 'env'
     * section of the buildspec.
     *
     * @param variableName the name of the variable to reference.
     *   A variable by this name must be present in the 'exported-variables' section of the buildspec
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-syntax
     */
    variable(variableName) {
        return this.variableExpression(variableName);
    }
    bound(scope, _stage, options) {
        var _b;
        // check for a cross-account action if there are any outputs
        if ((this.actionProperties.outputs || []).length > 0) {
            const pipelineStack = cdk.Stack.of(scope);
            const projectStack = cdk.Stack.of(this.props.project);
            if (pipelineStack.account !== projectStack.account) {
                throw new Error('A cross-account CodeBuild action cannot have outputs. ' +
                    'This is a known CodeBuild limitation. ' +
                    'See https://github.com/aws/aws-cdk/issues/4169 for details');
            }
        }
        // grant the Pipeline role the required permissions to this Project
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.props.project.projectArn],
            actions: [
                `codebuild:${this.props.executeBatchBuild ? 'BatchGetBuildBatches' : 'BatchGetBuilds'}`,
                `codebuild:${this.props.executeBatchBuild ? 'StartBuildBatch' : 'StartBuild'}`,
                `codebuild:${this.props.executeBatchBuild ? 'StopBuildBatch' : 'StopBuild'}`,
            ],
        }));
        // allow the Project access to the Pipeline's artifact Bucket
        // but only if the project is not imported
        // (ie., has a role) - otherwise, the IAM library throws an error
        if (this.props.project.role) {
            if ((this.actionProperties.outputs || []).length > 0) {
                options.bucket.grantReadWrite(this.props.project);
            }
            else {
                options.bucket.grantRead(this.props.project);
            }
        }
        if (this.props.project instanceof codebuild.Project) {
            this.props.project.bindToCodePipeline(scope, {
                artifactBucket: options.bucket,
            });
        }
        for (const inputArtifact of this.actionProperties.inputs || []) {
            // if any of the inputs come from the CodeStarConnectionsSourceAction
            // with codeBuildCloneOutput=true,
            // grant the Project's Role to use the connection
            const connectionArn = inputArtifact.getMetadata(__1.CodeStarConnectionsSourceAction._CONNECTION_ARN_PROPERTY);
            if (connectionArn) {
                this.props.project.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['codestar-connections:UseConnection'],
                    resources: [connectionArn],
                }));
            }
            // if any of the inputs come from the CodeCommitSourceAction
            // with codeBuildCloneOutput=true,
            // grant the Project's Role git pull access to the repository
            const codecommitRepositoryArn = inputArtifact.getMetadata(source_action_1.CodeCommitSourceAction._FULL_CLONE_ARN_PROPERTY);
            if (codecommitRepositoryArn) {
                this.props.project.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['codecommit:GitPull'],
                    resources: [codecommitRepositoryArn],
                }));
            }
        }
        const configuration = {
            ProjectName: this.props.project.projectName,
            EnvironmentVariables: this.props.environmentVariables &&
                cdk.Stack.of(scope).toJsonString(codebuild.Project.serializeEnvVariables(this.props.environmentVariables, (_b = this.props.checkSecretsInPlainTextEnvVariables) !== null && _b !== void 0 ? _b : true, this.props.project)),
        };
        if ((this.actionProperties.inputs || []).length > 1) {
            // lazy, because the Artifact name might be generated lazily
            configuration.PrimarySource = cdk.Lazy.string({ produce: () => this.props.input.artifactName });
        }
        if (this.props.executeBatchBuild) {
            configuration.BatchEnabled = 'true';
            this.props.project.enableBatchBuilds();
            if (this.props.combineBatchBuildArtifacts) {
                configuration.CombineArtifacts = 'true';
            }
        }
        return {
            configuration,
        };
    }
}
exports.CodeBuildAction = CodeBuildAction;
_a = JSII_RTTI_SYMBOL_1;
CodeBuildAction[_a] = { fqn: "@aws-cdk/aws-codepipeline-actions.CodeBuildAction", version: "1.152.0" };
//# sourceMappingURL=data:application/json;base64,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