# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['deserialize', 'deserialize.decorators']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'deserialize',
    'version': '2.0.1',
    'description': 'A library to make deserialization easy.',
    'long_description': '![Code scanning - action](https://github.com/dalemyers/deserialize/workflows/Code%20scanning%20-%20action/badge.svg) [![PyPI version](https://badge.fury.io/py/deserialize.svg)](https://badge.fury.io/py/deserialize) ![Azure DevOps builds](https://img.shields.io/azure-devops/build/dalemyers/Github/3)\n\n# deserialize\n\nA library to make deserialization easy. To get started, just run `pip install deserialize`\n\n### How it used to be\n\nWithout the library, if you want to convert:\n\n```json\n{\n    "a": 1,\n    "b": 2\n}\n```\n\ninto a dedicated class, you had to do something like this:\n\n```python\nclass MyThing:\n\n    def __init__(self, a, b):\n        self.a = a\n        self.b = b\n\n    @staticmethod\n    def from_json(json_data):\n        a_value = json_data.get("a")\n        b_value = json_data.get("b")\n\n        if a_value is None:\n            raise Exception("\'a\' was None")\n        elif b_value is None:\n            raise Exception("\'b\' was None")\n        elif type(a_value) != int:\n            raise Exception("\'a\' was not an int")\n        elif type(b_value) != int:\n            raise Exception("\'b\' was not an int")\n\n        return MyThing(a_value, b_value)\n\nmy_instance = MyThing.from_json(json_data)\n```\n\n### How it is now\n\nWith `deserialize` all you need to do is this:\n\n```python\nimport deserialize\n\nclass MyThing:\n    a: int\n    b: int\n\nmy_instance = deserialize.deserialize(MyThing, json_data)\n```\n\nThat\'s it. It will pull out all the data and set it for you type checking and even checking for null values.\n\nIf you want null values to be allowed though, that\'s easy too:\n\n```python\nfrom typing import Optional\n\nclass MyThing:\n    a: Optional[int]\n    b: Optional[int]\n```\n\nNow `None` is a valid value for these.\n\nTypes can be nested as deep as you like. For example, this is perfectly valid:\n\n```python\nclass Actor:\n    name: str\n    age: int\n\nclass Episode:\n    title: str\n    identifier: st\n    actors: List[Actor]\n\nclass Season:\n    episodes: List[Episode]\n    completed: bool\n\nclass TVShow:\n    seasons: List[Season]\n    creator: str\n```\n\n## Advanced Usage\n\n### Custom Keys\n\nIt may be that you want to name your properties in your object something different to what is in the data. This can be for readability reasons, or because you have to (such as if your data item is named `__class__`). This can be handled too. Simply use the `key` annotation as follows:\n\n```python\n@deserialize.key("identifier", "id")\nclass MyClass:\n    value: int\n    identifier: str\n```\n\nThis will now assign the data with the key `id` to the field `identifier`. You can have multiple annotations to override multiple keys.\n\n### Auto Snake\n\nData will often come in with the keys either camelCased or PascalCased. Since Python uses snake_case as standard for members, this means that custom keys are often used to do the conversion. To make this easier, you can add the `auto_snake` decorator and it will do this conversion for you where it can.\n\n```python\n@deserialize.auto_snake()\nclass MyClass:\n    some_integer: int\n    some_string: str\n```\n\nNow you can pass this data and it will automatically parse:\n\n```python\n{\n    "SomeInteger": 3,\n    "SomeString": "Hello"\n}\n```\n\nNote that all fields need to be snake cased if you use this decorator.\n\n### Unhandled Fields\n\nUsually, if you don\'t specify the field in your definition, but it does exist in the data, you just want to ignore it. Sometimes however, you want to know if there is extra data. In this case, when calling `deserialize(...)` you can set `throw_on_unhandled=True` and it will raise an exception if any fields in the data are unhandled.\n\nAdditionally, sometimes you want this, but know of a particular field that can be ignored. You can mark these as allowed to be unhandled with the decorator `@allow_unhandled("key_name")`.\n\n### Ignored Keys\n\nYou may want some properties in your object that aren\'t loaded from disk, but instead created some other way. To do this, use the `ignore` decorator. Here\'s an example:\n\n```python\n@deserialize.ignore("identifier")\nclass MyClass:\n    value: int\n    identifier: str\n```\n\nWhen deserializing, the library will now ignore the `identifier` property.\n\n### Parsers\n\nSometimes you\'ll want something in your object in a format that the data isn\'t in. For example, if you get the data:\n\n```python\n{\n    "successful": True,\n    "timestamp": 1543770752\n}\n```\n\nYou may want that to be represented as:\n\n```python\nclass Result:\n    successful: bool\n    timestamp: datetime.datetime\n```\n\nBy default, it will fail on this deserialization as the value in the data is not a timestamp. To correct this, use the `parser` decorator to tell it a function to use to parse the data. E.g.\n\n```python\n@deserialize.parser("timestamp", datetime.datetime.fromtimestamp)\nclass Result:\n    successful: bool\n    timestamp: datetime.datetime\n```\n\nThis will now detect when handling the data for the _key_ `timestamp` and run it through the parser function supplied before assigning it to your new class instance.\n\nThe parser is run _before_ type checking is done. This means that if you had something like `Optional[datetime.datetime]`, you should ensure your parser can handle the value being `None`. Your parser will obviously need to return the type that you have declared on the property in order to work.\n\n\n### Subclassing\n\nSubclassing is supported. If you have a type `Shape` for example, which has a subclass `Rectangle`, any properties on `Shape` are supported if you try and decode some data into a `rectangle object.\n\n### Raw Storage\n\nIt can sometimes be useful to keep a reference to the raw data that was used to construct an object. To do this, simply set the `raw_storage_mode` paramater to `RawStorageMode.ROOT` or `RawStorageMode.ALL`. This will store the data in a parameter named `__deserialize_raw__` on the root object, or on all objects in the tree respectively.\n\n### Defaults\n\nSome data will come to you with fields missing. In these cases, a default is often known. To do this, simply decorate your class like this:\n\n```python\n@deserialize.default("value", 0)\nclass IntResult:\n    successful: bool\n    value: int\n```\n\nIf you pass in data like `{"successful": True}` this will deserialize to a default value of `0` for `value`. Note, that this would not deserialize since `value` is not optional: `{"successful": True, "value": None}`.\n\n### Post-processing\n\nNot everything can be set on your data straight away. Some things need to be figured out afterwards. For this you need to do some post-processing. The easiest way to do this is through the `@constructed` decorator. This decorator takes a function which will be called whenever a new instance is constructed with that instance as an argument. Here\'s an example which converts polar coordinates from using degrees to radians:\n\n```python\ndata = {\n    "angle": 180.0,\n    "magnitude": 42.0\n}\n\ndef convert_to_radians(instance):\n    instance.angle = instance.angle * math.pi / 180\n\n@deserialize.constructed(convert_to_radians)\nclass PolarCoordinate:\n    angle: float\n    magnitude: float\n\npc = deserialize.deserialize(PolarCoordinate, data)\n\nprint(pc.angle, pc.magnitude)\n\n>>> 3.141592653589793 42.0\n```\n\n\n### Downcasting\n\nData often comes in the form of having the type as a field in the data. This can be difficult to parse. For example:\n\n```python\ndata = [\n    {\n        "data_type": "foo",\n        "foo_prop": "Hello World",\n    },\n    {\n        "data_type": "bar",\n        "bar_prop": "Goodbye World",\n    }\n]\n```\n\nSince the fields differ between the two, there\'s no good way of parsing this data. You could use optional fields on some base class, try multiple deserializations until you find the right one, or do the deserialization based on a mapping you build of the `data_type` field. None of those solutions are elegant though, and all have issues if the types are nested. Instead, you can use the `downcast_field` and `downcast_identifier` decorators.\n\n`downcast_field` is specified on a base class and gives the name of the field that contains the type information. `downcast_identifier` takes in a base class and an identifier (which should be one of the possible values of the `downcast_field` from the base class). Internally, when a class with a downcast field is detected, the field will be extacted, and a subclass with a matching identifier will be searched for. If no such class exists, an `UndefinedDowncastException` will be thrown.\n\nHere\'s an example which would handle the above data:\n\n```python\n@deserialize.downcast_field("data_type")\nclass MyBase:\n    type_name: str\n\n\n@deserialize.downcast_identifier(MyBase, "foo")\nclass Foo(MyBase):\n    foo_prop: str\n\n\n@deserialize.downcast_identifier(MyBase, "bar")\nclass Bar(MyBase):\n    bar_prop: str\n\n\nresult = deserialize.deserialize(List[MyBase], data)\n```\n\nHere, `result[0]` will be an instance of `Foo` and `result[1]` will be an instance of `Bar`.\n\nIf you can\'t describe all of your types, you can use `@deserialize.allow_downcast_fallback` on your base class and any unknowns will be left as dictionaries.\n',
    'author': 'Dale Myers',
    'author_email': 'dale@myers.io',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/dalemyers/deserialize',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7.2,<4.0.0',
}


setup(**setup_kwargs)
