"""
    sphinxcontrib.mat_documenters
    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    Extend autodoc directives to matlabdomain.

    :copyright: Copyright 2014-2024 by the sphinxcontrib-matlabdomain team, see AUTHORS.
    :license: BSD, see LICENSE for details.
"""

from .mat_types import (  # noqa: E401
    MatModule,
    MatObject,
    MatFunction,
    MatClass,
    MatProperty,
    MatMethod,
    MatScript,
    MatException,
    MatModuleAnalyzer,
    MatApplication,
    entities_table,
    entities_name_map,
    strip_package_prefix,
    try_get_module_entity_or_default,
)

import re
import traceback
import inspect

from docutils.statemachine import ViewList

import sphinx.util
from sphinx.locale import _
from sphinx.pycode import PycodeError
from sphinx.ext.autodoc import (
    py_ext_sig_re as mat_ext_sig_re,
    identity,
    Options,
    ALL,
    INSTANCEATTR,
    members_option,
    exclude_members_option,
    EMPTY,
    SUPPRESS,
    annotation_option,
    bool_option,
    Documenter as PyDocumenter,
    ModuleDocumenter as PyModuleDocumenter,
    FunctionDocumenter as PyFunctionDocumenter,
    ClassDocumenter as PyClassDocumenter,
    ExceptionDocumenter as PyExceptionDocumenter,
    DataDocumenter as PyDataDocumenter,
    MethodDocumenter as PyMethodDocumenter,
)

mat_ext_sig_re = re.compile(
    r"""^ ([+@]?[+@\w.]+::)?            # explicit module name
          ([+@]?[+@\w.]+\.)?            # module and/or class name(s)
          ([+@]?\w+)  \s*               # thing name
          (?: \((.*)\)                  # optional: arguments
           (?:\s* -> \s* (.*))?         #           return annotation
          )? $                          # and nothing more
          """,
    re.VERBOSE,
)


# TODO: check MRO's for all classes, attributes and methods!!!


logger = sphinx.util.logging.getLogger("matlab-domain")


class MatlabDocumenter(PyDocumenter):
    """
    Base class for documenters of MATLAB objects.
    """

    domain = "mat"

    def parse_name(self):
        """
        From: sphinx/ext/autodoc/__init__.py
        Determine what module to import and what attribute to document.

        Returns True and sets *self.modname*, *self.objpath*, *self.fullname*,
        *self.args* and *self.retann* if parsing and resolving was successful.
        """
        # first, parse the definition -- auto directives for classes and
        # functions can contain a signature which is then used instead of
        # an autogenerated one
        try:
            explicit_modname, path, base, args, retann = mat_ext_sig_re.match(
                self.name
            ).groups()
        except AttributeError:
            logger.warning(
                "[sphinxcontrib-matlabdomain] Invalid signature for mat:auto%s (%r)",
                self.objtype,
                self.name,
            )
            return False

        # support explicit module and class name separation via ::
        if explicit_modname is not None:
            modname = explicit_modname[:-2]
            parents = path and path.rstrip(".").split(".") or []
        else:
            modname = None
            parents = []

        self.modname, self.objpath = self.resolve_name(modname, parents, path, base)

        if not self.modname:
            return False

        self.args = args
        self.retann = retann
        self.fullname = (self.modname or "") + (
            self.objpath and "." + ".".join(self.objpath) or ""
        )
        self.fullname = self.fullname.lstrip(".")

        return True

    def import_object(self):
        """Import the object given by *self.modname* and *self.objpath* and set
        it as *self.object*.

        Returns True if successful, False if an error occurred.
        """
        try:
            msg = f"[sphinxcontrib-matlabdomain] MatlabDocumenter.import_object {self.modname=}, {self.objpath=}, {self.fullname=}."
            logger.debug(msg)
            if len(self.objpath) > 1:
                lookup_name = ".".join([self.modname, self.objpath[0]])
                lookup_name = lookup_name.lstrip(".")
                obj = entities_table[lookup_name]
                self.object = self.get_attr(obj, self.objpath[1])
            else:
                lookup_name = self.fullname.lstrip(".")
                self.object = try_get_module_entity_or_default(lookup_name)
            return True
        # this used to only catch SyntaxError, ImportError and AttributeError,
        # but importing modules with side effects can raise all kinds of errors
        except Exception:
            if self.objpath:
                errmsg = (
                    "[sphinxcontrib-matlabdomain] Failed to import %s %r from module %r"
                    % (self.objtype, ".".join(self.objpath), self.modname)
                )
            else:
                errmsg = "[sphinxcontrib-matlabdomain] Failed to import %s %r" % (
                    self.objtype,
                    self.fullname,
                )
            errmsg += (
                "; the following exception was raised:\n%s" % traceback.format_exc()
            )
            logger.warning(errmsg)
            self.env.note_reread()
            return False

    def add_content(self, more_content, no_docstring=False):
        """Add content from docstrings, attribute documentation and user."""
        sourcename = "docstring of %s" % self.fullname

        # add content from docstrings
        if not no_docstring:
            docstrings = self.get_doc()
            if not docstrings:
                # append at least a dummy docstring, so that the event
                # autodoc-process-docstring is fired and can add some
                # content if desired
                docstrings.append([])
            if self.env.config.matlab_auto_link:
                docstrings = self.auto_link(docstrings)
            for i, line in enumerate(self.process_doc(docstrings)):
                self.add_line(line, sourcename, i)

        # add additional content (e.g. from document), if present
        if more_content:
            for line, src in zip(more_content.data, more_content.items):
                self.add_line(line, src[0], src[1])

    def class_object(self):
        # the associated MatClass object (for class, property and method documenters)
        return None

    def auto_link_basic(self, docstrings):
        return self.auto_link_see_also(docstrings)

    def auto_link_see_also(self, docstrings):
        # autolink known names in See also
        see_also_re = re.compile(r"(See also:?\s*)(\b.*\b)(.*)", re.IGNORECASE)
        see_also_cond_re = re.compile(r"(\s*)(\b.*\b)(.*)")
        class_re = re.compile(r"(.*)\.([^\.]+)")
        is_see_also_line = False
        for i in range(len(docstrings)):
            for j in range(len(docstrings[i])):
                line = docstrings[i][j]
                if line:  # non-blank line
                    if is_see_also_line:
                        # find name
                        match = see_also_cond_re.search(line)
                        if match is not None:
                            entries_str = match.group(2)  # the entries
                    elif match := see_also_re.search(line):
                        is_see_also_line = True  # line begins with "See also"
                        entries_str = match.group(2)  # the entries
                elif is_see_also_line:  # blank line following see also section
                    is_see_also_line = False  # end see also section

                if is_see_also_line and entries_str:
                    # split on ,
                    entries = re.split(r"\s*,\s*", entries_str)
                    for k in range(len(entries)):
                        if entries[k].endswith("`"):
                            continue

                        # search in entities_table (for matching class or function name)
                        if (
                            self.env.config.matlab_keep_package_prefix
                            and entries[k] in entities_table
                        ):
                            o = entities_table[entries[k]]
                        elif (
                            not self.env.config.matlab_keep_package_prefix
                            and entries[k] in entities_name_map
                        ):
                            o = entities_table[entities_name_map[entries[k]]]
                        else:
                            o = None
                        if o:
                            if isinstance(o, dict):
                                if "class" in o:
                                    o = o["class"]
                                elif "func" in o:
                                    o = o["func"]
                            role = o.ref_role()
                            if role in ["class", "func"]:
                                entries[k] = f":{role}:`{entries[k]}`"
                                continue

                        # if we have an associated class, search properties and methods
                        cls = self.class_object()
                        if cls:
                            name = entries[k].rstrip("()")
                            if name in cls.methods:
                                entries[
                                    k
                                ] = f":meth:`{name}() <{cls.fullname(self.env)}.{name}>`"
                                continue
                            elif name in cls.properties:
                                entries[
                                    k
                                ] = f":attr:`{name} <{cls.fullname(self.env)}.{name}>`"
                                continue

                        # see if it is a fully qualified property or method name we recognize
                        match2 = class_re.search(entries[k])
                        if match2:
                            m1 = match2.group(1)
                            m2 = match2.group(2)
                            if (
                                self.env.config.matlab_keep_package_prefix
                                and m1 in entities_table
                            ):
                                cls = entities_table[entries[k]]
                            elif (
                                not self.env.config.matlab_keep_package_prefix
                                and m1 in entities_name_map
                            ):
                                cls = entities_table[entities_name_map[m1]]
                            else:
                                cls = None
                            if isinstance(cls, dict):
                                cls = cls["class"]
                            if cls and cls.ref_role() == "class":
                                name = m2.rstrip("()")
                                if name in cls.methods:
                                    entries[k] = f":meth:`{entries[k]}`"
                                    continue
                                elif name in cls.properties:
                                    entries[k] = f":attr:`{entries[k]}`"
                                    continue

                        # not yet handled
                        entries[k] = f"``{entries[k]}``"

                    docstrings[i][j] = (
                        match.group(1) + ", ".join(entries) + match.group(3)
                    )
        return docstrings

    def detect_literal_block(self, line, no_link_state):
        # skip over literal blocks (i.e. line ending with ::, blank line, indented line)
        # returns not_in_literal_block = True if the line is not in a literal block
        not_in_literal_block = False
        if line.endswith("::"):
            no_link_state = -1  # 1st sign of literal block
        elif not line:  # blank line
            if no_link_state == -1:  # if 1st sign already detected
                no_link_state = -2  # 2nd sign of literal block
            elif no_link_state == 1:  # if in literal block
                no_link_state = -2  # need to look again for indented line
        elif no_link_state == -2:  # following 1st 2 signs
            if line.startswith("  "):  # indented line, literal block
                no_link_state = 1  # no linking!
            else:  # not indented line, literal block has ended
                no_link_state = 0  # restart linking!
                not_in_literal_block = True  # including in this line
        elif no_link_state != 1:  # not in a literal block, go ahead and link
            not_in_literal_block = True

        return not_in_literal_block, no_link_state

    def auto_link_all(self, docstrings):
        # auto-link known classes and functions everywhere
        for n, o in entities_table.items():
            if isinstance(o, dict):
                if "class" in o:
                    o = o["class"]
                elif "func" in o:
                    o = o["func"]
            role = o.ref_role()
            if role in ["class", "func"]:
                nn = n.replace("+", "")  # remove + from name
                # negative look-behind for ` . + < @ * <non-breaking space>
                look_behind = r"(?<!(`|\.|\+|<|@|\*| ))\b"
                # negative look-ahead for ` * or <non-breaking space> or
                # " Properties:" or " Methods:" or .<alphanum>
                look_ahead = r"\b(?!(`|\*| |\sProperties:|\sMethods:|\.\w))"
                look_ahead2 = r"\b(?!(`|\*| |\sProperties:|\sMethods:))"
                # entity_name is NOT followed by .<property_or_method>
                pat = look_behind + nn.replace(".", r"\.") + look_ahead
                p = re.compile(pat)
                if role == "class":
                    # entity_name IS followed by .<property_or_method>
                    pat2 = (
                        look_behind + nn.replace(".", r"\.") + r"\.(\w+)" + look_ahead2
                    )
                    p2 = re.compile(pat2)
                no_link_state = 0  # normal mode (no literal block detected)
                for i in range(len(docstrings)):
                    for j in range(len(docstrings[i])):
                        not_in_literal_block, no_link_state = self.detect_literal_block(
                            docstrings[i][j], no_link_state
                        )
                        if not_in_literal_block:
                            docstrings[i][j] = p.sub(
                                f":{role}:`{nn}`", docstrings[i][j]
                            )
                            if role == "class":
                                if match := p2.search(docstrings[i][j]):
                                    # if match.group(1) is a property
                                    #   -> :attr:`{nn}.{match.group(1)}`
                                    for nnn, ooo in o.properties.items():
                                        if match.group(2) == nnn:
                                            docstrings[i][j] = p2.sub(
                                                f":attr:`{nn}.{nnn}`", docstrings[i][j]
                                            )
                                            break
                                    # if match.group(1) is a method
                                    #   -> :meth:`{nn}.{match.group(1)}`
                                    for nnn, ooo in o.methods.items():
                                        if match.group(2) == nnn:
                                            docstrings[i][j] = p2.sub(
                                                f":meth:`{nn}.{nnn}`", docstrings[i][j]
                                            )
                                            break

        return docstrings

    def auto_link(self, docstrings):
        # basic auto-linking
        if self.env.config.matlab_auto_link:  # "basic" or "all" (i.e. not None)
            docstrings = self.auto_link_basic(docstrings)

        # auto-link everywhere
        if self.env.config.matlab_auto_link == "all":
            docstrings = self.auto_link_all(docstrings)

        return docstrings

    def auto_link_methods(self, class_obj, docstrings):
        for n, o in class_obj.methods.items():
            # negative look-behind for ` . < @ * <non-breaking space>, then <name>()
            pat = r"(?<!(`|\.|<|@|\*| ))\b" + n + r"\(\)(?! )"
            p = re.compile(pat)
            no_link_state = 0  # normal mode (no literal block detected)
            for i in range(len(docstrings)):
                for j in range(len(docstrings[i])):
                    not_in_literal_block, no_link_state = self.detect_literal_block(
                        docstrings[i][j], no_link_state
                    )
                    if not_in_literal_block:
                        docstrings[i][j] = p.sub(
                            f":meth:`{n}() <{class_obj.fullname(self.env)}.{n}>`",
                            docstrings[i][j],
                        )

        return docstrings

    def get_object_members(self, want_all):
        """Return `(members_check_module, members)` where `members` is a
        list of `(membername, member)` pairs of the members of *self.object*.

        If *want_all* is True, return all members.  Else, only return those
        members given by *self.options.members* (which may also be none).
        """
        analyzed_member_names = set()
        if self.analyzer:
            attr_docs = self.analyzer.find_attr_docs()
            namespace = ".".join(self.objpath)
            for item in attr_docs.items():
                if item[0][0] == namespace:
                    analyzed_member_names.add(item[0][1])
        if not want_all:
            if not self.options.members:
                return False, []
            # specific members given
            members = []
            for mname in self.options.members:
                try:
                    members.append((mname, self.get_attr(self.object, mname)))
                except AttributeError:
                    if mname not in analyzed_member_names:
                        logger.warning(
                            "[sphinxcontrib-matlabdomain] missing attribute %s in object %s",
                            mname,
                            self.fullname,
                        )
        elif self.options.inherited_members:
            # safe_getmembers() uses dir() which pulls in members from all
            # base classes
            members = inspect.get_members(self.object, attr_getter=self.get_attr)
        else:
            # __dict__ contains only the members directly defined in
            # the class (but get them via getattr anyway, to e.g. get
            # unbound method objects instead of function objects);
            # using keys() because apparently there are objects for which
            # __dict__ changes while getting attributes
            try:
                obj_dict = self.get_attr(self.object, "__dict__")
            except AttributeError:
                members = []
            else:
                members = [
                    (mname, self.get_attr(self.object, mname, None))
                    for mname in list(obj_dict.keys())
                ]
        membernames = set(m[0] for m in members)
        # add instance attributes from the analyzer
        for aname in analyzed_member_names:
            if aname not in membernames and (want_all or aname in self.options.members):
                members.append((aname, INSTANCEATTR))
        return False, sorted(members)

    def filter_members(self, members, want_all):
        """Filter the given member list.

        Members are skipped if

        - they are special methods (except if given explicitly or the
          special-members option is set)
        - they are private (except if given explicitly or the private-members
          option is set)
        - they are protected (except if given explicitly or the protected-members
          option is set)
        - they are hidden (except if given explicitly or the hidden-members
          option is set)
        - they are friend (except if given explicitly or the friend-members
          option is set)

        - they are undocumented (except if the undoc-members option is set)

        The user can override the skipping decision by connecting to the
        ``autodoc-skip-member`` event.
        """

        def member_is_special(member):
            # TODO implement special matlab methods: disp, subsref, etc.
            return False

        def member_is_private(member):
            attrs = self.get_attr(member, "attrs", None)
            if attrs:
                access = attrs.get("Access", None)
                get_access = attrs.get("GetAccess", None)
                if access:
                    if access == "private":
                        return True
                elif get_access:
                    if get_access == "private":
                        return True
                return False
            else:
                return False

        def member_is_protected(member):
            attrs = self.get_attr(member, "attrs", None)
            if attrs:
                access = attrs.get("Access", None)
                get_access = attrs.get("GetAccess", None)
                if access:
                    if access == "protected":
                        return True
                elif get_access:
                    if get_access == "protected":
                        return True
                return False
            else:
                return False

        def member_is_hidden(member):
            attrs = self.get_attr(member, "attrs", None)
            if attrs:
                hidden = attrs.get("Hidden", None)
                # It is either None or True
                if hidden:
                    return True
                return False
            else:
                return False

        def member_is_friend(member):
            attrs = self.get_attr(member, "attrs", None)
            if attrs:
                access = attrs.get("Access", None)
                if access:
                    # Only friend meta classes define access lists
                    if isinstance(access, list):
                        return True
                    elif access:
                        # This is a friend meta class
                        return access[0] == "?"
                return False
            else:
                return False

        def member_is_friend_of(member, friends):
            attrs = self.get_attr(member, "attrs", None)
            if attrs:
                access = attrs.get("Access", None)
                if not isinstance(access, list):
                    access = [access]
                for has_access in access:
                    if has_access in friends:
                        return True
                else:
                    return False
            else:
                return False

        ret = []

        # search for members in source code too
        namespace = ".".join(self.objpath)  # will be empty for modules

        if self.analyzer:
            attr_docs = self.analyzer.find_attr_docs()
        else:
            attr_docs = {}

        # process members and determine which to skip
        for membername, member in members:
            # if isattr is True, the member is documented as an attribute
            isattr = False

            doc = self.get_attr(member, "__doc__", None)
            # if the member __doc__ is the same as self's __doc__, it's just
            # inherited and therefore not the member's doc
            cls = self.get_attr(member, "__class__", None)
            if cls:
                cls_doc = self.get_attr(cls, "__doc__", None)
                if cls_doc == doc:
                    doc = None
            has_doc = bool(doc)

            keep = False
            if want_all and member_is_special(member):
                # special methods
                if self.options.special_members is ALL:
                    keep = has_doc or self.options.undoc_members
                elif (
                    self.options.special_members
                    and self.options.special_members is not ALL
                    and membername in self.options.special_members
                ):
                    keep = has_doc or self.options.undoc_members
            elif want_all and member_is_private(member):
                # ignore private members
                if self.options.private_members is ALL:
                    keep = has_doc or self.options.undoc_members
                elif (
                    self.options.private_members
                    and self.options.private_members is not ALL
                    and membername in self.options.private_members
                ):
                    keep = has_doc or self.options.undoc_members
            elif want_all and member_is_protected(member):
                # ignore protected members
                if self.options.protected_members is ALL:
                    keep = has_doc or self.options.undoc_members
                elif (
                    self.options.protected_members
                    and self.options.protected_members is not ALL
                    and membername in self.options.protected_members
                ):
                    keep = has_doc or self.options.undoc_members
            elif want_all and member_is_hidden(member):
                # ignore hidden members
                if self.options.hidden_members is ALL:
                    keep = has_doc or self.options.undoc_members
                elif (
                    self.options.hidden_members
                    and self.options.hidden_members is not ALL
                    and membername in self.options.hidden_members
                ):
                    keep = has_doc or self.options.undoc_members
            elif want_all and member_is_friend(member):
                # ignore friend members
                if self.options.friend_members is ALL:
                    keep = has_doc or self.options.undoc_members
                elif (
                    self.options.friend_members
                    and self.options.friend_members is not ALL
                    and member_is_friend_of(member, self.options.friend_members)
                ):
                    keep = has_doc or self.options.undoc_members
            elif (namespace, membername) in attr_docs:
                # keep documented attributes
                keep = True
                isattr = True
            else:
                # ignore undocumented members if :undoc-members: is not given
                keep = has_doc or self.options.undoc_members

            # give the user a chance to decide whether this member
            # should be skipped
            if self.env.app:
                # let extensions preprocess docstrings
                skip_user = self.env.app.emit_firstresult(
                    "autodoc-skip-member",
                    self.objtype,
                    membername,
                    member,
                    not keep,
                    self.options,
                )
                if skip_user is not None:
                    keep = not skip_user

            if keep:
                ret.append((membername, member, isattr))

        return ret

    def document_members(self, all_members=False):
        """Generate reST for member documentation.

        If *all_members* is True, do all members, else those given by
        *self.options.members*.
        """
        # set current namespace for finding members
        self.env.temp_data["autodoc:module"] = self.modname
        if self.objpath:
            self.env.temp_data["autodoc:class"] = self.objpath[0]

        want_all = (
            all_members or self.options.inherited_members or self.options.members is ALL
        )
        # find out which members are documentable
        members_check_module, members = self.get_object_members(want_all)

        # remove members given by exclude-members
        if self.options.exclude_members:
            members = [
                (membername, member)
                for (membername, member) in members
                if membername not in self.options.exclude_members
            ]

        # document non-skipped members
        memberdocumenters = []
        for mname, member, isattr in self.filter_members(members, want_all):
            classes = []
            for name, cls in self.documenters.items():
                if name.startswith("mat:"):
                    if cls.can_document_member(member, mname, isattr, self):
                        classes.append(cls)
            if not classes:
                # don't know how to document this member
                continue
            # prefer the documenter with the highest priority
            classes.sort(key=lambda cls: cls.priority)
            # give explicitly separated module name, so that members
            # of inner classes can be documented
            full_mname = self.modname + "::" + ".".join(self.objpath + [mname])
            documenter = classes[-1](self.directive, full_mname, self.indent)
            memberdocumenters.append((documenter, isattr))
        member_order = self.options.member_order or self.env.config.autodoc_member_order
        if member_order == "groupwise":
            # sort by group; relies on stable sort to keep items in the
            # same group sorted alphabetically
            memberdocumenters.sort(key=lambda e: e[0].member_order)
        elif member_order == "bysource" and self.analyzer:
            # sort by source order, by virtue of the module analyzer
            tagorder = self.analyzer.tagorder

            def keyfunc(entry):
                fullname = entry[0].name.split("::")[1]
                return tagorder.get(fullname, len(tagorder))

            memberdocumenters.sort(key=keyfunc)

        for documenter, isattr in memberdocumenters:
            documenter.generate(
                all_members=True,
                real_modname=self.real_modname,
                check_module=members_check_module and not isattr,
            )

        # reset current objects
        self.env.temp_data["autodoc:module"] = None
        self.env.temp_data["autodoc:class"] = None

    def generate(
        self,
        more_content=None,
        real_modname=None,
        check_module=False,
        all_members=False,
    ):
        """Generate reST for the object given by *self.name*, and possibly for
        its members.

        If *more_content* is given, include that content. If *real_modname* is
        given, use that module name to find attribute docs. If *check_module* is
        True, only generate if the object is defined in the module name it is
        imported from. If *all_members* is True, document all members.
        """
        if not self.parse_name():
            # need a module to import
            logger.warning(
                "[sphinxcontrib-matlabdomain] don't know which module to import for autodocumenting "
                '%r (try placing a "module" or "currentmodule" directive '
                "in the document, or giving an explicit module name)",
                self.name,
            )
            return

        # now, import the module and get object to document
        if not self.import_object():
            return

        # If there is no real module defined, figure out which to use.
        # The real module is used in the module analyzer to look up the module
        # where the attribute documentation would actually be found in.
        # This is used for situations where you have a module that collects the
        # functions and classes of internal submodules.
        self.real_modname = real_modname or self.get_real_modname()

        # try to also get a source code analyzer for attribute docs
        try:
            self.analyzer = MatModuleAnalyzer.for_module(self.real_modname)
            # parse right now, to get PycodeErrors on parsing (results will
            # be cached anyway)
            self.analyzer.find_attr_docs()
        except PycodeError as err:
            self.env.app.debug(
                "[sphinxcontrib-matlabdomain] module analyzer failed: %s", err
            )
            # no source file -- e.g. for builtin and C modules
            self.analyzer = None
            # at least add the module.__file__ as a dependency
            if hasattr(self.module, "__file__") and self.module.__file__:
                self.directive.record_dependencies.add(self.module.__file__)
        else:
            self.directive.record_dependencies.add(self.analyzer.srcname)

        # check __module__ of object (for members not given explicitly)
        if check_module:
            if not self.check_module():
                return

        # make sure that the result starts with an empty line.  This is
        # necessary for some situations where another directive preprocesses
        # reST and no starting newline is present
        self.add_line("", "<autodoc>")

        # format the object's signature, if any
        sig = self.format_signature()

        # generate the directive header and options, if applicable
        self.add_directive_header(sig)
        self.add_line("", "<autodoc>")

        # e.g. the module directive doesn't have content
        self.indent += self.content_indent

        # add all content (from docstrings, attribute docs etc.)
        self.add_content(more_content)

        # document members, if possible
        self.document_members(all_members)


class MatModuleDocumenter(MatlabDocumenter, PyModuleDocumenter):
    def parse_name(self):
        ret = MatlabDocumenter.parse_name(self)
        if self.args or self.retann:
            logger.warning(
                "[sphinxcontrib-matlabdomain] signature arguments or return annotation given for automodule %s",
                self.fullname,
            )
        return ret

    def add_directive_header(self, sig):
        MatlabDocumenter.add_directive_header(self, sig)

        # add some module-specific options
        if self.options.synopsis:
            self.add_line("   :synopsis: " + self.options.synopsis, "<autodoc>")
        if self.options.platform:
            self.add_line("   :platform: " + self.options.platform, "<autodoc>")
        if self.options.deprecated:
            self.add_line("   :deprecated:", "<autodoc>")

    def get_object_members(self, want_all):
        if want_all:
            if not hasattr(self.object, "__all__"):
                # for implicit module members, check __module__ to avoid
                # documenting imported objects
                return True, self.object.safe_getmembers()
            else:
                memberlist = [name for name, obj in self.object.__all__]
        else:
            memberlist = self.options.members or []
        ret = []
        for mname in memberlist:
            try:
                attr = self.get_attr(self.object, mname, None)
                if attr:
                    ret.append((mname, attr))
                else:
                    raise AttributeError
            except AttributeError:
                logger.warning(
                    "[sphinxcontrib-matlabdomain] missing attribute mentioned"
                    " in :members: or __all__: module %s, attribute %s",
                    sphinx.util.inspect.safe_getattr(self.object, "__name__", "???"),
                    mname,
                )
        return False, ret


class MatModuleLevelDocumenter(MatlabDocumenter):
    """
    Specialized Documenter subclass for objects on module level (functions,
    classes, data/constants).

    From: sphinx/ext/autodoc/__init__.py
    Resolve the module and name of the object to document given by the
    arguments and the current module/class.

    Must return a pair of the module name and a chain of attributes; for
    example, it would return ``('zipfile', ['ZipFile', 'open'])`` for the
    ``zipfile.ZipFile.open`` method.
    """

    def resolve_name(self, modname, parents, path, base):
        if modname is None:
            if path:
                modname = path.rstrip(".")
            else:
                # if documenting a toplevel object without explicit module,
                # it can be contained in another auto directive ...
                modname = self.env.temp_data.get("autodoc:module")
                # ... or in the scope of a module directive
                if not modname:
                    modname = self.env.temp_data.get("mat:module")
                # ... else, it stays None, which means invalid
        return modname, parents + [base]


class MatClassLevelDocumenter(MatlabDocumenter):
    """
    Specialized Documenter subclass for objects on class level (methods,
    attributes).
    """

    def resolve_name(self, modname, parents, path, base):
        if modname is None:
            if path:
                mod_cls = path.rstrip(".")
            else:
                mod_cls = None
                # if documenting a class-level object without path,
                # there must be a current class, either from a parent
                # auto directive ...
                mod_cls = self.env.temp_data.get("autodoc:class")
                # ... or from a class directive
                if mod_cls is None:
                    mod_cls = self.env.temp_data.get("mat:class")
                # ... if still None, there's no way to know
                if mod_cls is None:
                    return None, []
            modname, _, cls = mod_cls.rpartition(".")
            parents = [cls]
            # if the module name is still missing, get it like above
            if not modname:
                modname = self.env.temp_data.get("autodoc:module")
            if not modname:
                modname = self.env.temp_data.get("mat:module")
            # ... else, it stays None, which means invalid
        return modname, parents + [base]


class MatDocstringSignatureMixin(object):
    """
    Mixin for FunctionDocumenter and MethodDocumenter to provide the
    feature of reading the signature from the docstring.
    """

    def _find_signature(self, encoding=None):
        docstrings = MatlabDocumenter.get_doc(self)
        if len(docstrings) != 1:
            return
        doclines = docstrings[0]
        setattr(self, "__new_doclines", doclines)
        if not doclines:
            return
        # match first line of docstring against signature RE
        match = mat_ext_sig_re.match(doclines[0])
        if not match:
            return
        exmod, path, base, args, retann = match.groups()
        # the base name must match ours
        if not self.objpath or base != self.objpath[-1]:
            return
        # re-prepare docstring to ignore indentation after signature
        docstrings = MatlabDocumenter.get_doc(self)
        doclines = docstrings[0]
        # ok, now jump over remaining empty lines and set the remaining
        # lines as the new doclines
        i = 1
        while i < len(doclines) and not doclines[i].strip():
            i += 1
        setattr(self, "__new_doclines", doclines[i:])
        return args, retann

    def get_doc(self, encoding=None):
        lines = getattr(self, "__new_doclines", None)
        if lines is not None:
            return [lines]
        return MatlabDocumenter.get_doc(self)

    def format_signature(self):
        if self.args is None and self.env.config.autodoc_docstring_signature:
            # only act if a signature is not explicitly given already, and if
            # the feature is enabled
            result = self._find_signature()
            if result is not None:
                self.args, self.retann = result
        return MatlabDocumenter.format_signature(self)


class MatFunctionDocumenter(MatDocstringSignatureMixin, MatModuleLevelDocumenter):
    """
    Specialized Documenter subclass for functions.
    """

    objtype = "function"
    member_order = 30

    @classmethod
    def can_document_member(cls, member, membername, isattr, parent):
        return isinstance(member, MatFunction)

    def format_args(self):
        if self.object.args:
            return "(" + ", ".join(self.object.args) + ")"

    def document_members(self, all_members=False):
        pass


def make_baseclass_links(env, obj):
    """Returns list of base class links"""
    obj_bases = obj.__bases__
    links = []
    if len(obj_bases):
        base_classes = obj_bases.items()
        for base_class_name, entity in base_classes:
            if not entity:
                links.append(":class:`%s`" % base_class_name)
            else:
                links.append(entity.link(env))

    return links


class MatClassDocumenter(MatModuleLevelDocumenter):
    """
    Specialized Documenter subclass for classes.
    """

    objtype = "class"
    member_order = 20
    option_spec = {
        "members": members_option,
        "undoc-members": bool_option,
        "noindex": bool_option,
        "inherited-members": bool_option,
        "show-inheritance": bool_option,
        "member-order": identity,
        "exclude-members": exclude_members_option,
        "special-members": members_option,
        "private-members": members_option,
        "protected-members": members_option,
        "hidden-members": members_option,
        "friend-members": members_option,
    }

    @classmethod
    def can_document_member(cls, member, membername, isattr, parent):
        return isinstance(member, MatClass)

    def import_object(self):
        ret = MatModuleLevelDocumenter.import_object(self)
        # if the class is documented under another name, document it
        # as data/attribute
        if ret:
            if hasattr(self.object, "__name__"):
                self.doc_as_attr = self.objpath[-1] != self.object.__name__
            else:
                self.doc_as_attr = True
        return ret

    def format_args(self):
        """Format arguments

        We use the method named the same as the class for arguments, but we only
        renders the arguments if `matlab_class_signature` is True and the method
        exist.
        """
        ctor = self.get_attr(self.object, self.object.name, None)

        if ctor is None or not isinstance(ctor, MatMethod):
            return None
        if ctor and not self.env.config.matlab_class_signature:
            return None
        if ctor.args:
            return "(" + ", ".join(ctor.args) + ")"

    def format_signature(self):
        if self.doc_as_attr:
            return ""

        # get constructor method signature from docstring
        if self.env.config.autodoc_docstring_signature:
            # only act if the feature is enabled
            init_doc = MatMethodDocumenter(self.directive, self.object.name)
            init_doc.object = self.get_attr(self.object, self.object.name, None)
            init_doc.objpath = [self.object.name]
            result = init_doc._find_signature()
            if result is not None:
                # use args only for Class signature
                return "(%s)" % result[0]

        return MatModuleLevelDocumenter.format_signature(self)

    def add_directive_header(self, sig):
        if self.doc_as_attr:
            self.directivetype = "attribute"
        MatlabDocumenter.add_directive_header(self, sig)

        # add inheritance info, if wanted
        if not self.doc_as_attr and self.options.show_inheritance:
            self.add_line("", "<autodoc>")
            base_class_links = make_baseclass_links(self.env, self.object)
            if base_class_links:
                self.add_line(
                    _("   Bases: %s") % ", ".join(base_class_links), "<autodoc>"
                )

    def get_doc(self, encoding=None):
        content = self.env.config.autoclass_content

        docstrings = []
        attrdocstring = self.get_attr(self.object, "__doc__", None)
        if attrdocstring:
            docstrings.append(attrdocstring)

        # for classes, what the "docstring" is can be controlled via a
        # config value; the default is only the class docstring
        if content in ("both", "init"):
            # get __init__ method document from __init__.__doc__
            if self.env.config.autodoc_docstring_signature:
                # only act if the feature is enabled
                init_doc = MatMethodDocumenter(self.directive, self.object.name)
                init_doc.object = self.get_attr(self.object, self.object.name, None)
                init_doc.objpath = [self.object.name]
                init_doc._find_signature()  # this effects to get_doc() result
                initdocstring = "\n".join(
                    ["\n".join(line) for line in init_doc.get_doc()]
                )
            else:
                initdocstring = self.get_attr(
                    self.get_attr(self.object, self.object.name, None), "__doc__"
                )
            # for new-style classes, no __init__ means default __init__
            if initdocstring == object.__init__.__doc__:
                initdocstring = None
            if initdocstring:
                if content == "init":
                    docstrings = [initdocstring]
                else:
                    docstrings.append(initdocstring)
        doc = []
        for docstring in docstrings:
            doc.append(sphinx.util.docstrings.prepare_docstring(docstring))
        return doc

    def add_content(self, more_content, no_docstring=False):
        if self.doc_as_attr:
            classname = sphinx.util.inspect.safe_getattr(self.object, "__name__", None)
            if classname:
                content = ViewList([_("alias of :class:`%s`") % classname], source="")
                MatModuleLevelDocumenter.add_content(self, content, no_docstring=True)
        else:
            MatModuleLevelDocumenter.add_content(self, more_content)

    def class_object(self):
        # the associated MatClass object
        return self.object

    def auto_link_basic(self, docstrings):
        docstrings = MatlabDocumenter.auto_link_basic(self, docstrings)
        return self.auto_link_class_members(docstrings)

    def auto_link_class_members(self, docstrings):
        # auto link property and method names in class docstring
        prop_re = re.compile(r"(.* Properties:)", re.IGNORECASE)
        meth_re = re.compile(r"(.* Methods:)", re.IGNORECASE)
        is_prop_line = False
        is_meth_line = False
        for i in range(len(docstrings)):
            for j in range(len(docstrings[i])):
                line = docstrings[i][j]
                if line:  # non-blank line
                    if prop_re.search(line):  # line ends with "Properties:"
                        is_prop_line = True
                        is_meth_line = False
                    elif meth_re.search(line):  # line ends with "Methods:"
                        is_prop_line = False
                        is_meth_line = True
                    elif is_prop_line:
                        # auto-link first word to corresponding property, if it exists
                        docstrings[i][j] = self.link_member("attr", line)
                    elif is_meth_line:
                        # auto-link first word to corresponding method, if it exists
                        docstrings[i][j] = self.link_member("meth", line)
                elif is_prop_line:  # blank line following properties section
                    is_prop_line = False  # end properties section
                elif is_meth_line:  # blank line following methods section
                    is_meth_line = False  # end methods section

        return docstrings

    def link_member(self, type, line):
        if type == "meth":
            parens = "()"
        else:
            parens = ""
        p = re.compile(r"((\*\s*)?(\b\w*\b))(?=\s*-)")
        if match := p.search(line):
            name = match.group(3)
            line = p.sub(
                f"* :{type}:`{name}{parens} <{self.object.fullname(self.env)}.{name}>`",
                line,
                1,
            )
        return line

    def auto_link_all(self, docstrings):
        docstrings = self.auto_link_methods(self.object, docstrings)
        return MatlabDocumenter.auto_link_all(self, docstrings)

    def document_members(self, all_members=False):
        if self.doc_as_attr:
            return

        want_all = (
            all_members or self.options.inherited_members or self.options.members is ALL
        )
        # find out which members are documentable
        members_check_module, members = self.get_object_members(want_all)

        # use filtered members to check for empty sections
        filtered_members = [
            (membername, member)
            for (membername, member, isattr) in self.filter_members(members, want_all)
        ]

        # create list of properties
        prop_names = [
            membername
            for (membername, member) in filtered_members
            if isinstance(member, MatProperty)
        ]
        # create list of constructors
        if self.env.config.autoclass_content in ("both", "init"):
            # skip constructor section, since its docstring has already been used for the class
            cons_names = []
        else:
            cons_names = [
                membername
                for (membername, member) in filtered_members
                if isinstance(member, MatMethod) and member.name == member.cls.name
            ]
        # create list of non-constructor methods
        meth_names = [
            membername
            for (membername, member) in filtered_members
            if isinstance(member, MatMethod) and member.name != member.cls.name
        ]
        # create list of other members
        other_names = [
            membername
            for (membername, member) in filtered_members
            if not isinstance(member, MatMethod) and not isinstance(member, MatProperty)
            # exclude parent modules with names matching members (as in Myclass.Myclass)
            and not (hasattr(member, "module") and member.name == member.module)
        ]
        # create list of members that are not properties
        non_properties = [
            membername
            for (membername, member) in members
            if not isinstance(member, MatProperty)
        ]
        # create list of members that are not constructors
        non_constructors = [
            membername
            for (membername, member) in members
            if not isinstance(member, MatMethod) or member.name != member.cls.name
        ]
        # create list of members that are not non-constructor methods
        non_methods = [
            membername
            for (membername, member) in members
            if not isinstance(member, MatMethod) or member.name == member.cls.name
        ]
        # create list of members that are not non-constructor methods
        non_other = [
            membername
            for (membername, member) in members
            if isinstance(member, MatMethod) or isinstance(member, MatProperty)
        ]

        # container
        if cons_names or prop_names or meth_names or other_names:
            self.add_line("", "<autodoc>")
            self.add_line(".. container:: members", "<autodoc>")
            self.add_line("", "<autodoc>")
            self.indent += "   "

        # constructor
        if cons_names:
            self.document_member_section(
                "Constructor Summary", non_constructors, all_members
            )

        # properties
        if prop_names:
            self.document_member_section(
                "Property Summary", non_properties, all_members
            )

        # methods
        if meth_names:
            self.document_member_section("Method Summary", non_methods, all_members)

        # other
        if other_names:
            self.document_member_section("Other", non_other, all_members)

    def document_member_section(self, heading, new_excludes, all_members):
        # save up original indent and exclude_members
        indent = self.indent
        if self.options.exclude_members:
            exclude_members = self.options.exclude_members
        else:
            exclude_members = []

        # output heading and section content
        self.add_line(heading, "<autodoc>")
        self.indent += "   "
        self.add_line(".. ", "<autodoc>")  # a comment, to force a <dd> in the HTML
        if exclude_members is EMPTY:
            self.options.exclude_members = set(new_excludes)
        else:
            self.options.exclude_members = set(list(exclude_members) + new_excludes)
        MatModuleLevelDocumenter.document_members(self, all_members)

        # restore original indent and exclude_members
        self.indent = indent
        self.options.exclude_members = exclude_members


class MatExceptionDocumenter(MatlabDocumenter, PyExceptionDocumenter):
    @classmethod
    def can_document_member(cls, member, membername, isattr, parent):
        return isinstance(member, MatException)


class MatDataDocumenter(MatModuleLevelDocumenter, PyDataDocumenter):
    @classmethod
    def can_document_member(cls, member, membername, isattr, parent):
        return isinstance(member, MatScript)


class MatMethodDocumenter(MatDocstringSignatureMixin, MatClassLevelDocumenter):
    """
    Specialized Documenter subclass for methods (normal, static and class).
    """

    objtype = "method"
    member_order = 50
    priority = 1  # must be more than FunctionDocumenter

    @classmethod
    def can_document_member(cls, member, membername, isattr, parent):
        return isinstance(member, MatMethod)

    def import_object(self):
        ret = MatClassLevelDocumenter.import_object(self)
        if self.object.attrs.get("Static"):
            self.directivetype = "staticmethod"
            # document class and static members before ordinary ones
            self.member_order = self.member_order - 1
        else:
            self.directivetype = "method"
        return ret

    def format_args(self):
        """Format argument

        We omit `obj` and `self` from the output if they are the first argument
        unless it's a class constructor.

        Rendering -> "(arg1, arg2, arg3)"

        """
        is_ctor = self.object.cls.name == self.object.name

        if self.object.args:
            if self.object.args[0] in ("obj", "self") and not is_ctor:
                return "(" + ", ".join(self.object.args[1:]) + ")"
            else:
                return "(" + ", ".join(self.object.args) + ")"

    def document_members(self, all_members=False):
        pass

    def class_object(self):
        # the associated MatClass object
        return self.object.cls

    def auto_link_self(self, docstrings):
        name = self.object.name
        # negative look-behind for ` or . or < or @ or * or <non-breaking space>
        # and negative look-ahead for * or <non-breaking space> or .<alphanum>
        p = re.compile(r"(?<!(`|\.|<|@|\*| ))\b" + name + r"\b(?!\*| |\.\w)")
        no_link_state = 0  # normal mode (no literal block detected)
        for i in range(len(docstrings)):
            for j in range(len(docstrings[i])):
                not_in_literal_block, no_link_state = self.detect_literal_block(
                    docstrings[i][j], no_link_state
                )
                if not_in_literal_block and docstrings[i][j]:  # also not blank line
                    if match := p.search(docstrings[i][j]):
                        docstrings[i][j] = p.sub(
                            f":meth:`{name}() <{self.class_object().fullname(self.env)}.{name}>`",
                            docstrings[i][j],
                        )
        return docstrings

    def auto_link_all(self, docstrings):
        docstrings = self.auto_link_methods(self.object.cls, docstrings)
        docstrings = self.auto_link_self(docstrings)
        return MatlabDocumenter.auto_link_all(self, docstrings)


class MatAttributeDocumenter(MatClassLevelDocumenter):
    """
    Specialized Documenter subclass for attributes.
    """

    objtype = "attribute"
    member_order = 60
    option_spec = dict(MatModuleLevelDocumenter.option_spec)
    option_spec["annotation"] = annotation_option

    # must be higher than the MethodDocumenter, else it will recognize
    # some non-data descriptors as methods
    priority = 10

    @classmethod
    def can_document_member(cls, member, membername, isattr, parent):
        return isinstance(member, MatProperty)

    def document_members(self, all_members=False):
        pass

    def import_object(self):
        ret = MatClassLevelDocumenter.import_object(self)
        # getset = self.object.name.split('_')

        if isinstance(self.object, MatMethod):
            self._datadescriptor = True
        else:
            # if it's not a data descriptor
            self._datadescriptor = False
        return ret

    def get_real_modname(self):
        return (
            self.get_attr(self.parent or self.object, "__module__", None)
            or self.modname
        )

    def add_directive_header(self, sig):
        MatClassLevelDocumenter.add_directive_header(self, sig)
        if not self.options.annotation:
            if not self._datadescriptor:
                obj_default = ""
                if self.env.config.matlab_show_property_default_value:
                    obj_default = self.object.default
                if obj_default is None:
                    obj_default = ""
                # Multi-line default values are truncated to the first part
                # and postfixed with an ellipsis.
                if "\n" in obj_default:
                    obj_default_parts = obj_default.split("\n")
                    obj_default = obj_default_parts[0] + " ... " + obj_default_parts[-1]

                if obj_default:
                    obj_default = " = " + obj_default

                if self.env.config.matlab_show_property_specs:
                    obj_default = self.object.specs + obj_default

                self.add_line("   :annotation: " + obj_default, "<autodoc>")
        elif self.options.annotation is SUPPRESS:
            pass
        else:
            self.add_line("   :annotation: = " + self.options.annotation, "<autodoc>")

    def add_content(self, more_content, no_docstring=False):
        # if not self._datadescriptor:
        #     # if it's not a data descriptor, its docstring is very probably the
        #     # wrong thing to display
        #     no_docstring = True
        MatClassLevelDocumenter.add_content(self, more_content, no_docstring)

    def class_object(self):
        # the associated MatClass object
        return self.object.cls

    def auto_link_self(self, docstrings):
        name = self.object.name
        # negative look-behind for ` or . or < or * or <non-breaking space>
        # and negative look-ahead for ` * or <non-breaking space>
        p = re.compile(r"(?<!(`|\.|<|\*| ))\b" + name + r"\b(?!`|\*| )")
        no_link_state = 0  # normal mode (no literal block detected)
        for i in range(len(docstrings)):
            for j in range(len(docstrings[i])):
                not_in_literal_block, no_link_state = self.detect_literal_block(
                    docstrings[i][j], no_link_state
                )
                if not_in_literal_block and docstrings[i][j]:  # also not blank line
                    if match := p.search(docstrings[i][j]):
                        docstrings[i][j] = p.sub(
                            f":attr:`{name} <{self.class_object().fullname(self.env)}.{name}>`",
                            docstrings[i][j],
                        )
        return docstrings

    def auto_link_all(self, docstrings):
        docstrings = self.auto_link_methods(self.object.cls, docstrings)
        docstrings = self.auto_link_self(docstrings)
        return MatlabDocumenter.auto_link_all(self, docstrings)


class MatScriptDocumenter(MatModuleLevelDocumenter):
    """
    Specialized Documenter subclass for scripts.
    """

    objtype = "script"

    @classmethod
    def can_document_member(cls, member, membername, isattr, parent):
        return isinstance(member, MatScript)

    def document_members(self, all_members=False):
        pass


class MatApplicationDocumenter(MatModuleLevelDocumenter):
    """
    Specialized Documenter subclass for Matlab Applications (.mlapp)
    """

    objtype = "application"

    @classmethod
    def can_document_member(cls, member, membername, isattr, parent):
        return isinstance(member, MatApplication)

    def document_members(self, all_members=False):
        pass
