import periodictable
from enum import Enum


class CalcType(Enum):
    UNDEFINED = 0
    SP = 1
    OPT = 2
    CONSTR_OPT = 3
    FREQ = 4
    TS = 5
    MEP = 6

    NMR = 10
    UVVIS = 11
    UVVIS_TDA = 12
    MO = 13
    CONF_SEARCH = 14
    CONSTR_CONF_SEARCH = 15

    OPTFREQ = 20


ATOMIC_NUMBER = {}
ATOMIC_SYMBOL = {}
LOWERCASE_ATOMIC_SYMBOLS = {}

for el in periodictable.elements:
    ATOMIC_NUMBER[el.symbol] = el.number
    ATOMIC_SYMBOL[el.number] = el.symbol
    LOWERCASE_ATOMIC_SYMBOLS[el.symbol.lower()] = el.symbol

# The first synonym will be used as detailed name in the documentation
SYN_TYPES = {
    CalcType.SP: [
        "Single-point Energy",
        "sp",
        "singlepoint",
        "singlepointenergy",
        "energy",
        "scf",
    ],
    CalcType.OPT: [
        "Geometrical Optimisation",
        "opt",
        "optimize",
        "optimise",
        "optimisation",
        "optimization",
        "geometricaloptimisation",
        "geometricaloptimization",
        "geomopt",
    ],
    CalcType.CONSTR_OPT: [
        "Constrained Optimisation",
        "optconstr",
        "constropt",
        "optmod",
        "constrainedoptimisation",
        "constrainedoptimization",
    ],
    CalcType.FREQ: [
        "Frequency Calculation",
        "freq",
        "frequency",
        "hess",
        "hessian",
        "freqcalc",
        "frequencycalculation",
        "freqcalculation",
    ],
    CalcType.TS: [
        "TS Optimisation",
        "ts",
        "tsoptimisation",
        "tsoptimization",
        "optts",
        "tsopt",
    ],
    CalcType.NMR: [
        "NMR Calculation",
        "nmr",
        "nmrprediction",
        "nmrcalculation",
        "nmrcalc",
        "calcnmr",
    ],
    CalcType.UVVIS: [
        "UV-Vis Calculation",
        "uvvis",
        "uvvisprediction",
        "uvviscalculation",
        "uvviscalc",
        "td",
        "tddft",
        "absorbance",
    ],
    CalcType.UVVIS_TDA: ["TDA UV-Vis Calculation", "tda", "tdadft"],
    CalcType.MEP: [
        "Minimum energy path",
        "mep",
        "minimumenergypath",
        "minimalenergypath",
        "neb",
        "nudgedelasticband",
        "reactionpath",
    ],
    CalcType.MO: [
        "MO Calculation",
        "mo",
        "mocalc",
        "mocalculation",
        "molecularorbitals",
        "mos",
    ],
    CalcType.OPTFREQ: ["Opt+Freq Calculation", "optfreq", "opt+freq"],
    CalcType.CONF_SEARCH: [
        "Conformational Search",
        "conformersearch",
        "confsearch",
        "conformationalsearch",
        "conformationalsampling",
        "crest",
        "conf",
    ],
    CalcType.CONSTR_CONF_SEARCH: [
        "Constrained Conformational Search",
        "constrainedconformersearch",
        "constrconfsearch",
        "constrainedconformationalsearch",
        "constrconformersearch",
        "constrconformationalsearch",
        "constrconfsampling",
    ],
}

THEORY_LEVELS = {
    "hf": ["hartreefock"],
    # Some old exotic semi-empirical methods in ORCA are missing
    "semiempirical": ["am1", "pm3", "pm6", "pm7", "mndo"],
    "xtb": ["gfn2xtb", "xtb2", "xtb1", "gfn1xtb", "gfn0xtb", "gfnff", "gfn2xtbgfnff"],
    "special": ["hf3c", "pbeh3c", "r2scan3c", "b973c"],
    "mp2": ["mp2", "rimp2"],
    "cc": [
        "ccsd",
        "ccsdt",
        "qcisd",
        "qcisdt",
        "lpnoccsd",
        "lpnoccsdt",
        "lpnoqcisd",
        "lpnoqcisdt",
        "dlpnoccsd",
        "dlpnoccsdt",
        "dlpnoqcisd",
        "dlpnoqcisdt",
    ],
    # Everything else is assumed to be DFT, because it most likely is
}

SYN_SOFTWARE = {
    "gaussian": ["g16", "gaussian16"],
    "orca": ["orca5"],
    "xtb": [],
}

SYN_SOLVENTS = {
    "acetone": ["acetone"],
    "acetonitrile": ["acetonitrile", "acn", "ch3cn", "mecn"],
    "benzene": ["benzene", "c6h6"],
    "dichloromethane": ["ch2cl2", "dcm"],
    "chloroform": ["chcl3"],
    "carbondisulfide": ["cs2"],
    "dimethylformamide": ["dmf"],
    "dimethylsulfoxide": ["dmso"],
    "diethylether": ["ether", "et2o"],
    "water": ["h2o"],
    "methanol": ["meoh"],
    "nhexane": ["hexane", "hex"],
    "tetrahydrofuran": ["thf"],
    "toluene": ["phme", "phch3"],
    "ethanol": [],
    "isoquinoline": [],
    "quinoline": [],
    "dichloroethane": ["12dichloroethane"],
    "carbontetrachloride": ["ccl4", "carbontetrachloride"],
    "chlorobenzene": ["phcl"],
    "nitromethane": ["meno2"],
    "heptane": [],
    "cyclohexane": [],
    "aniline": [],
    "argon": [],
    "krypton": [],
    "xenon": [],
    "noctanol": ["1octanol", "octanol"],
    "111trichloroethane": [],
    "112trichloroethane": [],
    "124trimethylbenzene": [],
    "12dibromoethane": [],
    "12ethanediol": [],
    "14dioxane": [],
    "1bromo2methylpropane": [],
    "1bromooctane": [],
    "1bromopentane": [],
    "1bromopropane": [],
    "1butanol": [],
    "1chlorohexane": [],
    "1chloropentane": [],
    "1chloropropane": [],
    "1decanol": [],
    "1fluorooctane": [],
    "1heptanol": [],
    "1hexanol": [],
    "1hexene": [],
    "1hexyne": [],
    "1iodobutane": [],
    "1iodohexadecane": [],
    "1iodopentane": [],
    "1iodopropane": [],
    "1nitropropane": [],
    "1nonanol": [],
    "1pentanol": [],
    "1pentene": [],
    "1propanol": [],
    "222trifluoroethanol": [],
    "224trimethylpentane": [],
    "24dimethylpentane": [],
    "24dimethylpyridine": [],
    "26dimethylpyridine": [],
    "2bromopropane": [],
    "2butanol": [],
    "2chlorobutane": [],
    "2heptanone": [],
    "2hexanone": [],
    "2methoxyethanol": [],
    "2methyl1propanol": [],
    "2methyl2propanol": [],
    "2methylpentane": [],
    "2methylpyridine": [],
    "2nitropropane": [],
    "2octanone": [],
    "2pentanone": [],
    "2propanol": [],
    "2propen1ol": [],
    "3methylpyridine": [],
    "3pentanone": [],
    "4heptanone": [],
    "4methyl2pentanone": [],
    "4methylpyridine": [],
    "5nonanone": [],
    "aceticacid": [],
    "acetophenone": [],
    "achlorotoluene": [],
    "anisole": [],
    "benzaldehyde": [],
    "benzonitrile": [],
    "benzylalcohol": [],
    "bromobenzene": [],
    "bromoethane": [],
    "bromoform": [],
    "butanal": [],
    "butanoicacid": [],
    "butanone": [],
    "butanonitrile": [],
    "butylamine": [],
    "butylethanoate": [],
    "carbondisulfide": [],
    "cis1,2dimethylcyclohexane": [],
    "cisdecalin": [],
    "cyclohexanone": [],
    "cyclopentane": [],
    "cyclopentanol": [],
    "cyclopentanone": [],
    "decalinmixture": [],
    "dibromomethane": [],
    "dibutylether": [],
    "diethylamine": [],
    "diethylsulfide": [],
    "diiodomethane": [],
    "diisopropylether": [],
    "dimethyldisulfide": [],
    "diphenylether": [],
    "dipropylamine": [],
    "e1,2dichloroethene": [],
    "e2pentene": [],
    "ethanethiol": [],
    "ethylbenzene": [],
    "ethylethanoate": [],
    "ethylmethanoate": [],
    "ethylphenylether": [],
    "fluorobenzene": [],
    "formamide": [],
    "formicacid": [],
    "hexanoicacid": [],
    "iodobenzene": [],
    "iodoethane": [],
    "iodomethane": [],
    "isopropylbenzene": [],
    "mcresol": [],
    "mesitylene": [],
    "methylbenzoate": [],
    "methylbutanoate": [],
    "methylcyclohexane": [],
    "methylethanoate": [],
    "methylmethanoate": [],
    "methylpropanoate": [],
    "mxylene": [],
    "nbutylbenzene": [],
    "ndecane": [],
    "ndodecane": [],
    "nhexadecane": [],
    "nitrobenzene": ["phno2"],
    "nitroethane": [],
    "nmethylaniline": [],
    "nmethylformamidemixture": [],
    "nndimethylacetamide": [],
    "nndimethylformamide": ["dmf"],
    "nnonane": [],
    "noctane": [],
    "npentadecane": [],
    "npentane": [],
    "nundecane": [],
    "ochlorotoluene": [],
    "ocresol": [],
    "odichlorobenzene": [],
    "onitrotoluene": [],
    "oxylene": [],
    "pentanal": [],
    "pentanoicacid": [],
    "pentylamine": [],
    "pentylethanoate": [],
    "perfluorobenzene": [],
    "pisopropyltoluene": [],
    "propanal": [],
    "propanoicacid": [],
    "propanonitrile": [],
    "propylamine": [],
    "propylethanoate": [],
    "pxylene": [],
    "pyridine": [],
    "secbutylbenzene": [],
    "tertbutylbenzene": [],
    "tetrachloroethene": [],
    "tetrahydrothiophenessdioxide": [],
    "tetralin": [],
    "thiophene": [],
    "thiophenol": [],
    "transdecalin": [],
    "tributylphosphate": [],
    "trichloroethene": [],
    "triethylamine": [],
    "xylenemixture": [],
    "z12dichloroethene": [],
}


SYN_METHODS = {
    "hfs": [],
    "xalpha": [],
    "lsd": [],
    "vwn5": [],
    "vwn3": [],
    "bnull": [],
    "bvwn": [],
    "bp": [],
    "blyp": [],
    "gp": [],
    "glyp": [],
    "pbe": [],
    "revpbe": [],
    "rpbe": [],
    "pwp": [],
    "olyp": [],
    "opbe": [],
    "xlyp": [],
    "m05": [],
    "m052x": [],
    "m06": [],
    "m062x": [],
    "m06hf": [],
    "m06l": [],
    "m08hx": [],
    "tpss": [],
    "revtpss": [],
    "scanfunc": [],
    "b1lyp": [],
    "b3lyp": [],
    "bhandhlyp": [],
    "camb3lyp": [],
    "lcblyp": [],
    "o3lyp": [],
    "x3lyp": [],
    "wb2plyp": [],
    "wb2gpplyp": [],
    "b1p": [],
    "b3p": [],
    "g1lyp": [],
    "g3lyp": [],
    "g1p": [],
    "g3p": [],
    "pbe0": ["pbe1pbe"],
    "pwp1": [],
    "mpwpw": [],
    "mpwlyp": [],
    "mpw1pw": [],
    "mpw1lyp": [],
    "mpw1pw91": [],
    "mpw1lyp": [],
    "mpw1pbe": [],
    "mpw3pbe": [],
    "pw6b95d3": [],
    "pw86pbe": [],
    "rpw86pbe": [],
    "pw910": [],
    "pw6b95": [],
    "pwlda": [],
    "pw91": [],
    "b97": [],
    "b97d": [],
    "b97d3": [],
    "wb97": [],
    "wb97x": [],
    "wb97xd": [],
    "wb97xd3": [],
    "wb97xv": [],
    "wb97xd3bj": [],
    "b98": [],
    "tpssh": [],
    "tpss0": [],
    "b2plyp": [],
    "mpw2plyp": [],
    "b2gpplyp": [],
    "b2kplyp": [],
    "b2tplyp": [],
    "pwpb95": [],
    "b3p86": [],
    "hse06": ["hse1pbe"],
    "ohse2pbe": [],
    "ohse1pbe": [],
    "pbeh1pbe": [],
    "b1b95": [],
    "apfd": [],
    "lcwhpbe": [],
    "thcthhyb": [],
    "bmk": [],
    "hissbpbe": [],
    "mn15": [],
    "mn15l": [],
    "m11": [],
    "m11l": [],
    "sogga11x": [],
    "sogga11": [],
    "n12sx": [],
    "n12": [],
    "mn12sx": [],
    "mn12l": [],
    "vsxc": [],
    "hcth407": [],
    "hcth93": [],
    "hcth147": [],
    "thcth": [],
    "hf": ["hartreefock"],
    "mndo": [],
    "am1": [],
    "pm3": [],
    "pm6": [],
    "pm7": ["pm7r6"],
    "gfn2xtb": ["xtb2", "gfn2"],
    "gfn1xtb": ["xtb1", "gfn1", "gfnxtb"],
    "gfn0xtb": ["xtb0", "gfn0"],
    "gfnff": [],
    "gfn2xtbgfnff": ["gfn2gfnff"],  # GFN2-xTB//GFN-FF
    "mp2": [],
    "rimp2": [],
    "hf3c": [],
    "pbeh3c": [],
    "r2scan3c": [],
    "b973c": [],
    "ccsd": [],
    "ccsdt": [],
    "qcisd": [],
    "qcisdt": [],
    "lpnoccsd": [],
    "lpnoccsdt": [],
    "lpnoqcisd": [],
    "lpnoqcisdt": [],
    "dlpnoccsd": [],
    "dlpnoccsdt": [],
    "dlpnoqcisd": [],
    "dlpnoqcisdt": [],
}


# Gaussian allows the combination of exchange and correlation functionals
# without using any special keyword.

EXCHANGE_FUNCTIONALS = {
    "s": "S",
    "xa": "XA",
    "b": "B",
    "pw91": "PW91",
    "mpw": "mPW",
    "g96": "G96",
    "pbe": "PBE",
    "o": "O",
    "tpss": "TPSS",
    "revtpss": "RevTPSS",
    "brx": "BRx",
    "pkzb": "PKZB",
    "wpbeh": "wPBEh",
    "pbeh": "PBEh",
}

CORRELATION_FUNCTIONALS = {
    "vwn": "VWN",
    "vwn5": "VWN5",
    "lyp": "LYP",
    "pl": "PL",
    "p86": "P86",
    "pw91": "PW91",
    "b95": "B95",
    "pbe": "PBE",
    "tpss": "TPSS",
    "revtpss": "RevTPSS",
    "kcis": "KCIS",
    "brc": "BRC",
    "pkzb": "PKZB",
    "vp86": "VP86",
    "v5lyp": "V5LYP",
}

SYN_BASIS_SETS = {
    "sto3g": [],
    "mini": [],
    "minis": [],
    "minix": [],
    "midi": [],
    "midix": [],
    "321g": [],
    "321gsp": [],
    "422gsp": [],
    "431g": [],
    "621g": [],
    "631g": [],
    "631gd": ["631g*"],
    "m631g": [],
    "m631gd": ["m631g*"],
    "631gdp": ["631g**"],
    "631g2d": [],
    "631g2dp": [],
    "631g2d2p": [],
    "631g2df": [],
    "631g2df2p": [],
    "631g2df2pd": [],
    "631+gd": ["631+g*"],
    "631+gdp": ["631+g**"],
    "631+g2d": [],
    "631+g2dp": [],
    "631+g2d2p": [],
    "631+g2df": [],
    "631+g2df2p": [],
    "631+g2df2pd": [],
    "631++gdp": ["631++g**"],
    "631++g2dp": [],
    "631++g2d2p": [],
    "631++g2df": [],
    "631++g2df2p": [],
    "631++g2df2pd": [],
    "6311gd": ["6311g*"],
    "6311gdp": ["6311g**"],
    "6311g2d": [],
    "6311g2dp": [],
    "6311g2d,2p": [],
    "6311g2df": [],
    "6311g2df2p": [],
    "6311g2df2pd": [],
    "6311g3df": [],
    "6311g3df3pd": [],
    "6311+gd": ["6311+g*"],
    "6311+gdp": ["6311+g**"],
    "6311+g2d": [],
    "6311+g2dp": [],
    "6311+g2d2p": [],
    "6311+g2df": [],
    "6311+g2df2p": [],
    "6311+g2df2pd": [],
    "6311+g3df": [],
    "6311+g3df2p": [],
    "6311+g3df3pd": [],
    "6311++gdp": ["6311++g**"],
    "6311++g2d": [],
    "6311++g2dp": [],
    "6311++g2d2p": [],
    "6311++g2df": [],
    "6311++g2df2p": [],
    "6311++g2df2pd": [],
    "6311++g3df": [],
    "6311++g3df3pd": [],
    "havt+dz": [],
    "havq+dz": [],
    "hav5+dz": [],
    "w1dz": [],
    "w1mtsmall": [],
    "sv": [],
    "svp": [],
    "tzv": [],
    "tzvp": [],
    "tzvpp": [],
    "qzvp": [],
    "qzvpp": [],
    "dkhsvp": [],
    "dkhtzvp": [],
    "dkhtzvpp": [],
    "dkhqzvp": [],
    "dkhqzvpp": [],
    "zorasvp": [],
    "zoratzvp": [],
    "zoratzvpp": [],
    "zoraqzvp": [],
    "zoraqzvpp": [],
    "def2msvp": [],
    "def2mtzvp": [],
    "def2svp": [],
    "def2tzvpf": [],
    "def2tzvp": [],
    "def2tzvpp": [],
    "def2qzvp": [],
    "def2qzvpp": [],
    "def2svpd": [],
    "def2tzvpd": [],
    "def2tzvppd": [],
    "def2qzvpd": [],
    "def2qzvppd": [],
    "def2sv": [],
    "def2svpp": [],
    "def2tzv": [],
    "def2qzv": [],
    "dkhdef2svp": [],
    "dkhdef2tzvpf": [],
    "dkhdef2tzvp": [],
    "dkhdef2tzvpp": [],
    "dkhdef2qzvp": [],
    "dkhdef2qzvpp": [],
    "zoradef2svp": [],
    "zoradef2tzvpf": [],
    "zoradef2tzvp": [],
    "zoradef2tzvpp": [],
    "zoradef2qzvp": [],
    "zoradef2qzvpp": [],
    "madef2msvp": [],
    "madef2svp": [],
    "madef2tzvpf": [],
    "madef2tzvp": [],
    "madef2tzvpp": [],
    "madef2qzvp": [],
    "madef2qzvpp": [],
    "madkhdef2svp": [],
    "madkhdef2tzvpf": [],
    "madkhdef2tzvp": [],
    "madkhdef2tzvpp": [],
    "madkhdef2qzvp": [],
    "madkhdef2qzvpp": [],
    "mazoradef2svp": [],
    "mazoradef2tzvpf": [],
    "mazoradef2tzvp": [],
    "mazoradef2tzvpp": [],
    "mazoradef2qzvpp": [],
    "anopvdz": [],
    "anopvtz": [],
    "anopvqz": [],
    "anopv5z": [],
    "anopv6z": [],
    "auganopvdz": [],
    "auganopvtz": [],
    "auganopvqz": [],
    "auganopv5z": [],
    "sauganopvdz": [],
    "sauganopvtz": [],
    "sauganopvqz": [],
    "sauganopv5z": [],
    "anorccdzp": [],
    "anorcctzp": [],
    "anorccqzp": [],
    "anorccfull": [],
    "pc0": [],
    "pc1": [],
    "pc2": [],
    "pc3": [],
    "pc4": [],
    "augpc0": [],
    "augpc1": [],
    "augpc2": [],
    "augpc3": [],
    "augpc4": [],
    "pcj0": [],
    "pcj1": [],
    "pcj2": [],
    "pcj3": [],
    "pcj4": [],
    "augpcj0": [],
    "augpcj1": [],
    "augpcj2": [],
    "augpcj3": [],
    "augpcj4": [],
    "pcseg0": [],
    "pcseg1": [],
    "pcseg2": [],
    "pcseg3": [],
    "pcseg4": [],
    "augpcseg0": [],
    "augpcseg1": [],
    "augpcseg2": [],
    "augpcseg3": [],
    "augpcseg4": [],
    "pcsseg0": [],
    "pcsseg1": [],
    "pcsseg2": [],
    "pcsseg3": [],
    "pcsseg4": [],
    "augpcsseg0": [],
    "augpcsseg1": [],
    "augpcsseg2": [],
    "augpcsseg3": [],
    "augpcsseg4": [],
    "ccpvdz": [],
    "ccpvtz": [],
    "ccpvqz": [],
    "ccpv5z": [],
    "ccpv6z": [],
    "augccpvdz": [],
    "augccpvtz": [],
    "augccpvqz": [],
    "augccpv5z": [],
    "augccpv6z": [],
    "ccpvd+dz": [],
    "ccpvt+dz": [],
    "ccpvq+dz": [],
    "ccpv5+dz": [],
    "augccpvtzj": [],
    "ccpcvdz": [],
    "ccpcvtz": [],
    "ccpcvqz": [],
    "ccpcv5z": [],
    "ccpcv6z": [],
    "augccpcvdz": [],
    "augccpcvtz": [],
    "augccpcvqz": [],
    "augccpcv5z": [],
    "augccpcv6z": [],
    "ccpwcvdz": [],
    "ccpwcvtz": [],
    "ccpwcvqz": [],
    "ccpwcv5z": [],
    "augccpwcvdz": [],
    "augccpwcvtz": [],
    "augccpwcvqz": [],
    "augccpwcv5z": [],
    "ccpvdzpp": [],
    "ccpvtzpp": [],
    "ccpvqzpp": [],
    "ccpv5zpp": [],
    "augccpvdzpp": [],
    "augccpvtzpp": [],
    "augccpvqzpp": [],
    "augccpv5zpp": [],
    "ccpcvdzpp": [],
    "ccpcvtzpp": [],
    "ccpcvqzpp": [],
    "ccpcv5zpp": [],
    "augccpcvdzpp": [],
    "augccpcvtzpp": [],
    "augccpcvqzpp": [],
    "augccpcv5zpp": [],
    "ccpwvdzpp": [],
    "ccpwvtzpp": [],
    "ccpwvqzpp": [],
    "ccpwv5zpp": [],
    "augccpwvdzpp": [],
    "augccpwvtzpp": [],
    "augccpwvqzpp": [],
    "augccpwv5zpp": [],
    "ccpvdzdk": [],
    "ccpvtzdk": [],
    "ccpvqzdk": [],
    "ccpv5zdk": [],
    "augccpvdzdk": [],
    "augccpvtzdk": [],
    "augccpvqzdk": [],
    "augccpv5zdk": [],
    "ccpwcvdzdk": [],
    "ccpwcvtzdk": [],
    "ccpwcvqzdk": [],
    "ccpwcv5zdk": [],
    "augccpvd+dz": [],
    "augccpvt+dz": [],
    "augccpvq+dz": [],
    "augccpv5+dz": [],
    "augccpv6+dz": [],
    "w1opt": [],
    "w1qz": [],
    "augccpwvdzdk": [],
    "augccpwvtzdk": [],
    "augccpwvqzdk": [],
    "augccpwv5zdk": [],
    "partridge1": [],
    "partridge2": [],
    "partridge3": [],
    "partridge4": [],
    "anosz": [],
    "lanl08": [],
    "lanl08f": [],
    "lanl2dz": [],
    "lanl2tz": [],
    "lanl2tzf": [],
    "lanl2mb": [],
    "sapporodzp2012": [],
    "sapporotzp2012": [],
    "sapporoqzp2012": [],
    "sapporodkh3dzp2012": [],
    "sapporodkh3tzp2012": [],
    "sapporodkh3qzp2012": [],
    "sarcdkhsvp": [],
    "sarcdkhtzvp": [],
    "sarcdkhtzvpp": [],
    "sarc2dkhqzv": [],
    "sarc2dkhqzvp": [],
    "sarc2zoraqzv": [],
    "sarc2zoraqzvp": [],
    "d95": [],
    "d95p": [],
    "eprii": [],
    "epriii": [],
    "igloii": [],
    "igloiii": [],
    "ugbs": [],
    "cp": [],
    "cpppp": [],
    "wachters+f": [],
    "ccpvdzf12": [],
    "ccpvtzf12": [],
    "ccpvqzf12": [],
    "ccpvdzppf12": [],
    "ccpvtzppf12": [],
    "ccpvqzppf12": [],
    "ccpcvdzf12": [],
    "ccpcvtzf12": [],
    "ccpcvqzf12": [],
    "d95v": [],
    "shc": [],
    "cep4g": [],
    "cep31g": [],
    "cep121g": [],
    "sdd": [],  # D95, also named DZ or DZP (Dunning-Hay) in the BSE up to Ar, then "Stuttgart/Dresden ECPs" (called Stuttgart RLC in the BSE) for the rest of the periodic table
    "sddall": [],
    "dgdzvp": [],
    "dgdzvp2": [],
    "gdtzvp": [],
    "cbsb7": [],
    # Auxiliary
    "def2": [],
    "def2mtzvp": [],
    "def2mtzvpp": [],
    "x2c": [],
    "sarc": [],
    "w1mtsmall": [],
    "w1dz": [],
    "w1tz": [],
    "w1qz": [],
    "w1opt": [],
}

# Not all basis sets below will be in SYN_BASIS_SETS (to complete).
# ccinput will still attempt to find the key for the indexified input string.
# Only orbital basis sets are included below.
BASIS_SET_EXCHANGE_KEY = {
    "2zapanr": "2ZaPa-NR",
    "2zapanrcv": "2ZaPa-NR-CV",
    "321g": "3-21G",
    "3zapanr": "3ZaPa-NR",
    "3zapanrcv": "3ZaPa-NR-CV",
    "431g": "4-31G",
    "4zapanr": "4ZaPa-NR",
    "4zapanrcv": "4ZaPa-NR-CV",
    "521g": "5-21G",
    "5zapanr": "5ZaPa-NR",
    "5zapanrcv": "5ZaPa-NR-CV",
    "621g": "6-21G",
    "631++g": "6-31++G",
    "631++gd": "6-31++G*",
    "631++gdp": "6-31++G**",
    "631++gdpj": "6-31++G**-J",
    "631+g": "6-31+G",
    "631+gd": "6-31+G*",
    "631+gdj": "6-31+G*-J",
    "631+gdp": "6-31+G**",
    "6311++g": "6-311++G",
    "6311++g2d2p": "6-311++G(2d,2p)",
    "6311++g3df3pd": "6-311++G(3df,3pd)",
    "6311++gd": "6-311++G*",
    "6311++gdp": "6-311++G**",
    "6311++gdpj": "6-311++G**-J",
    "6311+g": "6-311+G",
    "6311+g2dp": "6-311+G(2d,p)",
    "6311+gd": "6-311+G*",
    "6311+gdj": "6-311+G*-J",
    "6311+gdp": "6-311+G**",
    "6311g": "6-311G",
    "6311g2df2pd": "6-311G(2df,2pd)",
    "6311gdp": "6-311G(d,p)",
    "6311gj": "6-311G-J",
    "6311gd": "6-311G*",
    "6311gdp": "6-311G**",
    "6311xxgdp": "6-311xxG(d,p)",
    "631g": "6-31G",
    "631g2dfp": "6-31G(2df,p)",
    "631g3df3pd": "6-31G(3df,3pd)",
    "631gdp": "6-31G(d,p)",
    "631gblaudeau": "6-31G-Blaudeau",
    "631gj": "6-31G-J",
    "631gd": "6-31G*",
    "631gdblaudeau": "6-31G*-Blaudeau",
    "631gdp": "6-31G**",
    "6zapanr": "6ZaPa-NR",
    "7zapanr": "7ZaPa-NR",
    "acv2zj": "acv2z-J",
    "acv3zj": "acv3z-J",
    "acv4zj": "acv4z-J",
    "ahgbs5": "AHGBS-5",
    "ahgbs7": "AHGBS-7",
    "ahgbs9": "AHGBS-9",
    "ahgbsp15": "AHGBSP1-5",
    "ahgbsp17": "AHGBSP1-7",
    "ahgbsp19": "AHGBSP1-9",
    "ahgbsp25": "AHGBSP2-5",
    "ahgbsp27": "AHGBSP2-7",
    "ahgbsp29": "AHGBSP2-9",
    "ahgbsp35": "AHGBSP3-5",
    "ahgbsp37": "AHGBSP3-7",
    "ahgbsp39": "AHGBSP3-9",
    "ahlrichspvdz": "AhlrichspVDZ",
    "ahlrichstzv": "AhlrichsTZV",
    "ahlrichsvdz": "AhlrichsVDZ",
    "ahlrichsvtz": "AhlrichsVTZ",
    "anodk3": "ANO-DK3",
    "anor": "ANO-R",
    "anor0": "ANO-R0",
    "anor1": "ANO-R1",
    "anor2": "ANO-R2",
    "anor3": "ANO-R3",
    "anorcc": "ANO-RCC",
    "anorccmb": "ANO-RCC-MB",
    "anorccvdz": "ANO-RCC-VDZ",
    "anorccvdzp": "ANO-RCC-VDZP",
    "anorccvqzp": "ANO-RCC-VQZP",
    "anorccvtz": "ANO-RCC-VTZ",
    "anorccvtzp": "ANO-RCC-VTZP",
    "anovtdz": "ANO-VT-DZ",
    "anovtqz": "ANO-VT-QZ",
    "anovttz": "ANO-VT-TZ",
    "aprccpvq+dz": "apr-cc-pV(Q+d)Z",
    "atzpzora": "ATZP-ZORA",
    "augccpcv5z": "aug-cc-pCV5Z",
    "augccpcvdz": "aug-cc-pCVDZ",
    "augccpcvdzdk": "aug-cc-pCVDZ-DK",
    "augccpcvqz": "aug-cc-pCVQZ",
    "augccpcvqzdk": "aug-cc-pCVQZ-DK",
    "augccpcvtz": "aug-cc-pCVTZ",
    "augccpcvtzdk": "aug-cc-pCVTZ-DK",
    "augccpv5+dz": "aug-cc-pV(5+d)Z",
    "augccpvd+dz": "aug-cc-pV(D+d)Z",
    "augccpvq+dz": "aug-cc-pV(Q+d)Z",
    "augccpvt+dz": "aug-cc-pV(T+d)Z",
    "augccpv5z": "aug-cc-pV5Z",
    "augccpv5zdk": "aug-cc-pV5Z-DK",
    "augccpv5zpp": "aug-cc-pV5Z-PP",
    "augccpv6z": "aug-cc-pV6Z",
    "augccpv7z": "aug-cc-pV7Z",
    "augccpvdz": "aug-cc-pVDZ",
    "augccpvdzdk": "aug-cc-pVDZ-DK",
    "augccpvdzdk3": "aug-cc-pVDZ-DK3",
    "augccpvdzpp": "aug-cc-pVDZ-PP",
    "augccpvdzx2c": "aug-cc-pVDZ-X2C",
    "augccpvqz": "aug-cc-pVQZ",
    "augccpvqzdk": "aug-cc-pVQZ-DK",
    "augccpvqzdk3": "aug-cc-pVQZ-DK3",
    "augccpvqzpp": "aug-cc-pVQZ-PP",
    "augccpvqzx2c": "aug-cc-pVQZ-X2C",
    "augccpvtz": "aug-cc-pVTZ",
    "augccpvtzdk": "aug-cc-pVTZ-DK",
    "augccpvtzdk3": "aug-cc-pVTZ-DK3",
    "augccpvtzj": "aug-cc-pVTZ-J",
    "augccpvtzpp": "aug-cc-pVTZ-PP",
    "augccpvtzx2c": "aug-cc-pVTZ-X2C",
    "augccpwcv5z": "aug-cc-pwCV5Z",
    "augccpwcv5zdk": "aug-cc-pwCV5Z-DK",
    "augccpwcv5zpp": "aug-cc-pwCV5Z-PP",
    "augccpwcvdz": "aug-cc-pwCVDZ",
    "augccpwcvdzdk3": "aug-cc-pwCVDZ-DK3",
    "augccpwcvdzpp": "aug-cc-pwCVDZ-PP",
    "augccpwcvdzx2c": "aug-cc-pwCVDZ-X2C",
    "augccpwcvqz": "aug-cc-pwCVQZ",
    "augccpwcvqzdk": "aug-cc-pwCVQZ-DK",
    "augccpwcvqzdk3": "aug-cc-pwCVQZ-DK3",
    "augccpwcvqzpp": "aug-cc-pwCVQZ-PP",
    "augccpwcvqzx2c": "aug-cc-pwCVQZ-X2C",
    "augccpwcvtz": "aug-cc-pwCVTZ",
    "augccpwcvtzdk": "aug-cc-pwCVTZ-DK",
    "augccpwcvtzdk3": "aug-cc-pwCVTZ-DK3",
    "augccpwcvtzpp": "aug-cc-pwCVTZ-PP",
    "augccpwcvtzx2c": "aug-cc-pwCVTZ-X2C",
    "augccx5z": "aug-ccX-5Z",
    "augccxdz": "aug-ccX-DZ",
    "augccxqz": "aug-ccX-QZ",
    "augccxtz": "aug-ccX-TZ",
    "augmccpv5z": "aug-mcc-pV5Z",
    "augmccpv6z": "aug-mcc-pV6Z",
    "augmccpv7z": "aug-mcc-pV7Z",
    "augmccpv8z": "aug-mcc-pV8Z",
    "augmccpvqz": "aug-mcc-pVQZ",
    "augmccpvtz": "aug-mcc-pVTZ",
    "augpc0": "aug-pc-0",
    "augpc1": "aug-pc-1",
    "augpc2": "aug-pc-2",
    "augpc3": "aug-pc-3",
    "augpc4": "aug-pc-4",
    "augpch1": "aug-pcH-1",
    "augpch2": "aug-pcH-2",
    "augpch3": "aug-pcH-3",
    "augpch4": "aug-pcH-4",
    "augpcj0": "aug-pcJ-0",
    "augpcj02006": "aug-pcJ-0_2006",
    "augpcj1": "aug-pcJ-1",
    "augpcj12006": "aug-pcJ-1_2006",
    "augpcj2": "aug-pcJ-2",
    "augpcj22006": "aug-pcJ-2_2006",
    "augpcj3": "aug-pcJ-3",
    "augpcj32006": "aug-pcJ-3_2006",
    "augpcj4": "aug-pcJ-4",
    "augpcj42006": "aug-pcJ-4_2006",
    "augpcs0": "aug-pcS-0",
    "augpcs1": "aug-pcS-1",
    "augpcs2": "aug-pcS-2",
    "augpcs3": "aug-pcS-3",
    "augpcs4": "aug-pcS-4",
    "augpcseg0": "aug-pcseg-0",
    "augpcseg1": "aug-pcseg-1",
    "augpcseg2": "aug-pcseg-2",
    "augpcseg3": "aug-pcseg-3",
    "augpcseg4": "aug-pcseg-4",
    "augpcsseg0": "aug-pcSseg-0",
    "augpcsseg1": "aug-pcSseg-1",
    "augpcsseg2": "aug-pcSseg-2",
    "augpcsseg3": "aug-pcSseg-3",
    "augpcsseg4": "aug-pcSseg-4",
    "augpcx1": "aug-pcX-1",
    "augpcx2": "aug-pcX-2",
    "augpcx3": "aug-pcX-3",
    "augpcx4": "aug-pcX-4",
    "augpv7z": "aug-pV7Z",
    "binning641": "binning641",
    "binning641d": "binning641(d)",
    "binning641df": "binning641(df)",
    "binning641+": "binning641+",
    "binning641+d": "binning641+(d)",
    "binning641+df": "binning641+(df)",
    "binning962": "binning962",
    "binning962d": "binning962(d)",
    "binning962df": "binning962(df)",
    "binning962+": "binning962+",
    "binning962+d": "binning962+(d)",
    "binning962+df": "binning962+(df)",
    "ccpcv5z": "cc-pCV5Z",
    "ccpcvdz": "cc-pCVDZ",
    "ccpcvdzdk": "cc-pCVDZ-DK",
    "ccpcvdzf12": "cc-pCVDZ-F12",
    "ccpcvqz": "cc-pCVQZ",
    "ccpcvqzdk": "cc-pCVQZ-DK",
    "ccpcvqzf12": "cc-pCVQZ-F12",
    "ccpcvtz": "cc-pCVTZ",
    "ccpcvtzdk": "cc-pCVTZ-DK",
    "ccpcvtzf12": "cc-pCVTZ-F12",
    "ccpv5+dz": "cc-pV(5+d)Z",
    "ccpvd+dz": "cc-pV(D+d)Z",
    "ccpvq+dz": "cc-pV(Q+d)Z",
    "ccpvt+dz": "cc-pV(T+d)Z",
    "ccpv5z": "cc-pV5Z",
    "ccpv5zfisffw": "cc-pV5Z(fi/sf/fw)",
    "ccpv5zfisflc": "cc-pV5Z(fi/sf/lc)",
    "ccpv5zfisfsc": "cc-pV5Z(fi/sf/sc)",
    "ccpv5zptsffw": "cc-pV5Z(pt/sf/fw)",
    "ccpv5zptsflc": "cc-pV5Z(pt/sf/lc)",
    "ccpv5zptsfsc": "cc-pV5Z(pt/sf/sc)",
    "ccpv5zdk": "cc-pV5Z-DK",
    "ccpv5zf12": "cc-pV5Z-F12",
    "ccpv5zf12rev2": "cc-pV5Z-F12(rev2)",
    "ccpv5zpp": "cc-pV5Z-PP",
    "ccpv6z": "cc-pV6Z",
    "ccpv8z": "cc-pV8Z",
    "ccpv9z": "cc-pV9Z",
    "ccpvdz": "cc-pVDZ",
    "ccpvdzfisffw": "cc-pVDZ(fi/sf/fw)",
    "ccpvdzfisflc": "cc-pVDZ(fi/sf/lc)",
    "ccpvdzfisfsc": "cc-pVDZ(fi/sf/sc)",
    "ccpvdzptsffw": "cc-pVDZ(pt/sf/fw)",
    "ccpvdzptsflc": "cc-pVDZ(pt/sf/lc)",
    "ccpvdzptsfsc": "cc-pVDZ(pt/sf/sc)",
    "ccpvdzsegopt": "cc-pVDZ(seg-opt)",
    "ccpvdzdk": "cc-pVDZ-DK",
    "ccpvdzdk3": "cc-pVDZ-DK3",
    "ccpvdzf12": "cc-pVDZ-F12",
    "ccpvdzf12rev2": "cc-pVDZ-F12(rev2)",
    "ccpvdzpp": "cc-pVDZ-PP",
    "ccpvdzx2c": "cc-pVDZ-X2C",
    "ccpvqz": "cc-pVQZ",
    "ccpvqzfisffw": "cc-pVQZ(fi/sf/fw)",
    "ccpvqzfisflc": "cc-pVQZ(fi/sf/lc)",
    "ccpvqzfisfsc": "cc-pVQZ(fi/sf/sc)",
    "ccpvqzptsffw": "cc-pVQZ(pt/sf/fw)",
    "ccpvqzptsflc": "cc-pVQZ(pt/sf/lc)",
    "ccpvqzptsfsc": "cc-pVQZ(pt/sf/sc)",
    "ccpvqzsegopt": "cc-pVQZ(seg-opt)",
    "ccpvqzdk": "cc-pVQZ-DK",
    "ccpvqzdk3": "cc-pVQZ-DK3",
    "ccpvqzf12": "cc-pVQZ-F12",
    "ccpvqzf12rev2": "cc-pVQZ-F12(rev2)",
    "ccpvqzpp": "cc-pVQZ-PP",
    "ccpvqzx2c": "cc-pVQZ-X2C",
    "ccpvtz": "cc-pVTZ",
    "ccpvtzfisffw": "cc-pVTZ(fi/sf/fw)",
    "ccpvtzfisflc": "cc-pVTZ(fi/sf/lc)",
    "ccpvtzfisfsc": "cc-pVTZ(fi/sf/sc)",
    "ccpvtzptsffw": "cc-pVTZ(pt/sf/fw)",
    "ccpvtzptsflc": "cc-pVTZ(pt/sf/lc)",
    "ccpvtzptsfsc": "cc-pVTZ(pt/sf/sc)",
    "ccpvtzsegopt": "cc-pVTZ(seg-opt)",
    "ccpvtzdk": "cc-pVTZ-DK",
    "ccpvtzdk3": "cc-pVTZ-DK3",
    "ccpvtzf12": "cc-pVTZ-F12",
    "ccpvtzf12rev2": "cc-pVTZ-F12(rev2)",
    "ccpvtzpp": "cc-pVTZ-PP",
    "ccpvtzx2c": "cc-pVTZ-X2C",
    "ccpwcv5z": "cc-pwCV5Z",
    "ccpwcv5zdk": "cc-pwCV5Z-DK",
    "ccpwcv5zpp": "cc-pwCV5Z-PP",
    "ccpwcvdz": "cc-pwCVDZ",
    "ccpwcvdzdk3": "cc-pwCVDZ-DK3",
    "ccpwcvdzpp": "cc-pwCVDZ-PP",
    "ccpwcvdzx2c": "cc-pwCVDZ-X2C",
    "ccpwcvqz": "cc-pwCVQZ",
    "ccpwcvqzdk": "cc-pwCVQZ-DK",
    "ccpwcvqzdk3": "cc-pwCVQZ-DK3",
    "ccpwcvqzpp": "cc-pwCVQZ-PP",
    "ccpwcvqzx2c": "cc-pwCVQZ-X2C",
    "ccpwcvtz": "cc-pwCVTZ",
    "ccpwcvtzdk": "cc-pwCVTZ-DK",
    "ccpwcvtzdk3": "cc-pwCVTZ-DK3",
    "ccpwcvtzpp": "cc-pwCVTZ-PP",
    "ccpwcvtzx2c": "cc-pwCVTZ-X2C",
    "ccemd2": "ccemd-2",
    "ccemd3": "ccemd-3",
    "ccjpv5z": "ccJ-pV5Z",
    "ccjpvdz": "ccJ-pVDZ",
    "ccjpvqz": "ccJ-pVQZ",
    "ccjpvtz": "ccJ-pVTZ",
    "ccx5z": "ccX-5Z",
    "ccxdz": "ccX-DZ",
    "ccxqz": "ccX-QZ",
    "ccxtz": "ccX-TZ",
    "coemd2": "coemd-2",
    "coemd3": "coemd-3",
    "coemd4": "coemd-4",
    "coemdref": "coemd-ref",
    "colognedkh2": "CologneDKH2",
    "crenbl": "CRENBL",
    "crenblecp": "CRENBLECP",
    "crenbs": "CRENBS",
    "crenbsecp": "CRENBSECP",
    "daugccpv5z": "d-aug-cc-pV5Z",
    "daugccpv6z": "d-aug-cc-pV6Z",
    "daugccpvdz": "d-aug-cc-pVDZ",
    "daugccpvqz": "d-aug-cc-pVQZ",
    "daugccpvtz": "d-aug-cc-pVTZ",
    "def2ecp": "def2-ECP",
    "def2qzvp": "def2-QZVP",
    "def2qzvpd": "def2-QZVPD",
    "def2qzvpp": "def2-QZVPP",
    "def2qzvppd": "def2-QZVPPD",
    "def2svp": "def2-SV(P)",
    "def2svp": "def2-SVP",
    "def2svpd": "def2-SVPD",
    "def2tzvp": "def2-TZVP",
    "def2tzvpd": "def2-TZVPD",
    "def2tzvpp": "def2-TZVPP",
    "def2tzvppd": "def2-TZVPPD",
    "demon2kdzvpgga": "deMon2k-DZVP-GGA",
    "dfo+nrlmol": "DFO+-NRLMOL",
    "dfo1": "DFO-1",
    "dfo1bhs": "DFO-1-BHS",
    "dfo2": "DFO-2",
    "dfonrlmol": "DFO-NRLMOL",
    "dgaussdzvp": "dgauss-dzvp",
    "dgaussdzvp2": "dgauss-dzvp2",
    "dgausstzvp": "dgauss-tzvp",
    "dhfecp": "dhf-ECP",
    "dhfqzvp": "dhf-QZVP",
    "dhfqzvpp": "dhf-QZVPP",
    "dhfsvp": "dhf-SV(P)",
    "dhfsvp": "dhf-SVP",
    "dhftzvp": "dhf-TZVP",
    "dhftzvpp": "dhf-TZVPP",
    "dzdunninghay": "DZ(Dunning-Hay)",
    "dz+doublerydbergdunninghay": "DZ+DoubleRydberg(Dunning-Hay)",
    "dz+rydbergdunninghay": "DZ+Rydberg(Dunning-Hay)",
    "dzpdunninghay": "DZP(Dunning-Hay)",
    "dzp+diffusedunninghay": "DZP+Diffuse(Dunning-Hay)",
    "dzp+rydbergdunninghay": "DZP+Rydberg(Dunning-Hay)",
    "fano5z": "FANO-5Z",
    "fano6z": "FANO-6Z",
    "fanodz": "FANO-DZ",
    "fanoqz": "FANO-QZ",
    "fanotz": "FANO-TZ",
    "hgbs5": "HGBS-5",
    "hgbs7": "HGBS-7",
    "hgbs9": "HGBS-9",
    "hgbsp15": "HGBSP1-5",
    "hgbsp17": "HGBSP1-7",
    "hgbsp19": "HGBSP1-9",
    "hgbsp25": "HGBSP2-5",
    "hgbsp27": "HGBSP2-7",
    "hgbsp29": "HGBSP2-9",
    "hgbsp35": "HGBSP3-5",
    "hgbsp37": "HGBSP3-7",
    "hgbsp39": "HGBSP3-9",
    "igloii": "IGLO-II",
    "igloiii": "IGLO-III",
    "jgaussdzp": "jgauss-dzp",
    "jgaussqz2p": "jgauss-qz2p",
    "jgaussqzp": "jgauss-qzp",
    "jgausstzp1": "jgauss-tzp1",
    "jgausstzp2": "jgauss-tzp2",
    "jorge5zp": "jorge-5ZP",
    "jorge5zpdkh": "jorge-5ZP-DKH",
    "jorge6zp": "jorge-6ZP",
    "jorge6zpdkh": "jorge-6ZP-DKH",
    "jorgea5zp": "jorge-A5ZP",
    "jorgeadzp": "jorge-ADZP",
    "jorgeaqzp": "jorge-AQZP",
    "jorgeatzp": "jorge-ATZP",
    "jorgedzp": "jorge-DZP",
    "jorgedzpdkh": "jorge-DZP-DKH",
    "jorgeqzp": "jorge-QZP",
    "jorgeqzpdkh": "jorge-QZP-DKH",
    "jorgetzp": "jorge-TZP",
    "jorgetzpdkh": "jorge-TZP-DKH",
    "julccpvd+dz": "jul-cc-pV(D+d)Z",
    "julccpvq+dz": "jul-cc-pV(Q+d)Z",
    "julccpvt+dz": "jul-cc-pV(T+d)Z",
    "junccpvd+dz": "jun-cc-pV(D+d)Z",
    "junccpvq+dz": "jun-cc-pV(Q+d)Z",
    "junccpvt+dz": "jun-cc-pV(T+d)Z",
    "kogaunpolarized": "Kogaunpolarized",
    "lanl08": "LANL08",
    "lanl08d": "LANL08(d)",
    "lanl08f": "LANL08(f)",
    "lanl08+": "LANL08+",
    "lanl2dz": "LANL2DZ",
    "lanl2dzecp": "LANL2DZECP",
    "lanl2dzdp": "LANL2DZdp",
    "lanl2tz": "LANL2TZ",
    "lanl2tzf": "LANL2TZ(f)",
    "lanl2tz+": "LANL2TZ+",
    "m631g": "m6-31G",
    "m631gd": "m6-31G*",
    "maugccpvd+dz": "maug-cc-pV(D+d)Z",
    "maugccpvq+dz": "maug-cc-pV(Q+d)Z",
    "maugccpvt+dz": "maug-cc-pV(T+d)Z",
    "mayccpvq+dz": "may-cc-pV(Q+d)Z",
    "mayccpvt+dz": "may-cc-pV(T+d)Z",
    "midi": "MIDI",
    "midi": "MIDI!",
    "midix": "MIDIX",
    "mini": "MINI",
    "modifiedlanl2dz": "modified-LANL2DZ",
    "nasaamesano": "NASAAmesANO",
    "nasaamesano2": "NASAAmesANO2",
    "nasaamesccpcv5z": "NASAAmescc-pCV5Z",
    "nasaamesccpcvqz": "NASAAmescc-pCVQZ",
    "nasaamesccpcvtz": "NASAAmescc-pCVTZ",
    "nasaamesccpv5z": "NASAAmescc-pV5Z",
    "nasaamesccpvqz": "NASAAmescc-pVQZ",
    "nasaamesccpvtz": "NASAAmescc-pVTZ",
    "nlov": "NLO-V",
    "nmrdkhtz2p": "NMR-DKH(TZ2P)",
    "orp": "ORP",
    "partridgeuncontracted1": "PartridgeUncontracted1",
    "partridgeuncontracted2": "PartridgeUncontracted2",
    "partridgeuncontracted3": "PartridgeUncontracted3",
    "partridgeuncontracted4": "PartridgeUncontracted4",
    "pc0": "pc-0",
    "pc1": "pc-1",
    "pc2": "pc-2",
    "pc3": "pc-3",
    "pc4": "pc-4",
    "pcemd2": "pcemd-2",
    "pcemd3": "pcemd-3",
    "pcemd4": "pcemd-4",
    "pch1": "pcH-1",
    "pch2": "pcH-2",
    "pch3": "pcH-3",
    "pch4": "pcH-4",
    "pcj0": "pcJ-0",
    "pcj02006": "pcJ-0_2006",
    "pcj1": "pcJ-1",
    "pcj12006": "pcJ-1_2006",
    "pcj2": "pcJ-2",
    "pcj22006": "pcJ-2_2006",
    "pcj3": "pcJ-3",
    "pcj32006": "pcJ-3_2006",
    "pcj4": "pcJ-4",
    "pcj42006": "pcJ-4_2006",
    "pcs0": "pcS-0",
    "pcs1": "pcS-1",
    "pcs2": "pcS-2",
    "pcs3": "pcS-3",
    "pcs4": "pcS-4",
    "pcseg0": "pcseg-0",
    "pcseg1": "pcseg-1",
    "pcseg2": "pcseg-2",
    "pcseg3": "pcseg-3",
    "pcseg4": "pcseg-4",
    "pcsseg0": "pcSseg-0",
    "pcsseg1": "pcSseg-1",
    "pcsseg2": "pcSseg-2",
    "pcsseg3": "pcSseg-3",
    "pcsseg4": "pcSseg-4",
    "pcx1": "pcX-1",
    "pcx2": "pcX-2",
    "pcx3": "pcX-3",
    "pcx4": "pcX-4",
    "psbkjc": "pSBKJC",
    "ptmdzp": "Pt-mDZP",
    "pv6z": "pV6Z",
    "pv7z": "pV7Z",
    "roosaugmenteddoublezetaano": "RoosAugmentedDoubleZetaANO",
    "roosaugmentedtriplezetaano": "RoosAugmentedTripleZetaANO",
    "s321g": "s3-21G",
    "s321gd": "s3-21G*",
    "s631g": "s6-31G",
    "s631gd": "s6-31G*",
    "sadlejpvtz": "SadlejpVTZ",
    "sadlej+": "Sadlej+",
    "sapporodkh3dzp": "Sapporo-DKH3-DZP",
    "sapporodkh3dzp2012": "Sapporo-DKH3-DZP-2012",
    "sapporodkh3dzp2012diffuse": "Sapporo-DKH3-DZP-2012-diffuse",
    "sapporodkh3dzpdiffuse": "Sapporo-DKH3-DZP-diffuse",
    "sapporodkh3qzp": "Sapporo-DKH3-QZP",
    "sapporodkh3qzp2012": "Sapporo-DKH3-QZP-2012",
    "sapporodkh3qzp2012diffuse": "Sapporo-DKH3-QZP-2012-diffuse",
    "sapporodkh3qzpdiffuse": "Sapporo-DKH3-QZP-diffuse",
    "sapporodkh3tzp": "Sapporo-DKH3-TZP",
    "sapporodkh3tzp2012": "Sapporo-DKH3-TZP-2012",
    "sapporodkh3tzp2012diffuse": "Sapporo-DKH3-TZP-2012-diffuse",
    "sapporodkh3tzpdiffuse": "Sapporo-DKH3-TZP-diffuse",
    "sapporodzp": "Sapporo-DZP",
    "sapporodzp2012": "Sapporo-DZP-2012",
    "sapporodzp2012diffuse": "Sapporo-DZP-2012-diffuse",
    "sapporodzpdiffuse": "Sapporo-DZP-diffuse",
    "sapporoqzp": "Sapporo-QZP",
    "sapporoqzp2012": "Sapporo-QZP-2012",
    "sapporoqzp2012diffuse": "Sapporo-QZP-2012-diffuse",
    "sapporoqzpdiffuse": "Sapporo-QZP-diffuse",
    "sapporotzp": "Sapporo-TZP",
    "sapporotzp2012": "Sapporo-TZP-2012",
    "sapporotzp2012diffuse": "Sapporo-TZP-2012-diffuse",
    "sapporotzpdiffuse": "Sapporo-TZP-diffuse",
    "sarcdkh2": "SARC-DKH2",
    "sarczora": "SARC-ZORA",
    "sarc2qzvdkh2": "SARC2-QZV-DKH2",
    "sarc2qzvzora": "SARC2-QZV-ZORA",
    "sarc2qzvpdkh2": "SARC2-QZVP-DKH2",
    "sarc2qzvpzora": "SARC2-QZVP-ZORA",
    "sbkjcpolarizedp2dlfk": "SBKJCPolarized(p,2d)-LFK",
    "sbkjcecp": "SBKJC-ECP",
    "sbkjcvdz": "SBKJC-VDZ",
    "sbo4dzd3g": "SBO4-DZ(d)-3G",
    "sbo4dzdp3g": "SBO4-DZ(d,p)-3G",
    "sbo4sz3g": "SBO4-SZ-3G",
    "scaledmini": "ScaledMINI",
    "sto2g": "STO-2G",
    "sto3g": "STO-3G",
    "sto3g*": "STO-3G*",
    "sto4g": "STO-4G",
    "sto5g": "STO-5G",
    "sto6g": "STO-6G",
    "stuttgartrlc": "StuttgartRLC",
    "stuttgartrlcecp": "StuttgartRLCECP",
    "stuttgartrsc1997": "StuttgartRSC1997",
    "stuttgartrsc1997ecp": "StuttgartRSC1997ECP",
    "stuttgartrscano": "StuttgartRSCANO",
    "stuttgartrscsegmented+ecp": "StuttgartRSCSegmented+ECP",
    "svdunninghay": "SV(Dunning-Hay)",
    "sv+doublerydbergdunninghay": "SV+DoubleRydberg(Dunning-Hay)",
    "sv+rydbergdunninghay": "SV+Rydberg(Dunning-Hay)",
    "svpdunninghay": "SVP(Dunning-Hay)",
    "svp+diffusedunninghay": "SVP+Diffuse(Dunning-Hay)",
    "svp+diffuse+rydbergdunninghay": "SVP+Diffuse+Rydberg(Dunning-Hay)",
    "svp+rydbergdunninghay": "SVP+Rydberg(Dunning-Hay)",
    "tzdunninghay": "TZ(Dunning-Hay)",
    "tzpzora": "TZP-ZORA",
    "ugbs": "UGBS",
    "unccemdref": "un-ccemd-ref",
    "unpcemdref": "un-pcemd-ref",
    "wachters+f": "Wachters+f",
    "wtbs": "WTBS",
    "x2cqzvpall": "x2c-QZVPall",
    "x2cqzvpall2c": "x2c-QZVPall-2c",
    "x2cqzvpall2cs": "x2c-QZVPall-2c-s",
    "x2cqzvpalls": "x2c-QZVPall-s",
    "x2cqzvppall": "x2c-QZVPPall",
    "x2cqzvppall2c": "x2c-QZVPPall-2c",
    "x2cqzvppall2cs": "x2c-QZVPPall-2c-s",
    "x2cqzvppalls": "x2c-QZVPPall-s",
    "x2csvpall": "x2c-SV(P)all",
    "x2csvpall2c": "x2c-SV(P)all-2c",
    "x2csvpalls": "x2c-SV(P)all-s",
    "x2csvpall": "x2c-SVPall",
    "x2csvpall2c": "x2c-SVPall-2c",
    "x2csvpalls": "x2c-SVPall-s",
    "x2ctzvpall": "x2c-TZVPall",
    "x2ctzvpall2c": "x2c-TZVPall-2c",
    "x2ctzvpalls": "x2c-TZVPall-s",
    "x2ctzvppall": "x2c-TZVPPall",
    "x2ctzvppall2c": "x2c-TZVPPall-2c",
    "x2ctzvppalls": "x2c-TZVPPall-s",
}


SOFTWARE_METHODS = {
    "orca": {
        "hfs": "HFS",
        "xalpha": "XAlpha",
        "lsd": "LSD",
        "vwn5": "VWN5",
        "vwn3": "VWN3",
        "pwlda": "PWLDA",
        "bnull": "BNULL",
        "bvwn": "BVWN",
        "bp": "BP",
        "pw91": "PW91",
        "mpwpw": "mPWPW",
        "mpwlyp": "mPWLYP",
        "blyp": "BLYP",
        "gp": "GP",
        "glyp": "GLYP",
        "pbe": "PBE",
        "revpbe": "revPBE",
        "rpbe": "RPBE",
        "pwp": "PWP",
        "olyp": "OLYP",
        "opbe": "OPBE",
        "xlyp": "XLYP",
        "b97d": "B97-D",
        "b97d3": "B97-D3",
        "pw86pbe": "PW86PBE",
        "rpw86pbe": "RPW86PBE",
        "m06l": "M06L",
        "tpss": "TPSS",
        "revtpss": "revTPSS",
        "scanfunc": "SCANfunc",
        "b1lyp": "B1LYP",
        "b3lyp": "B3LYP",
        "b1p": "B1P",
        "b3p": "B3P",
        "g1lyp": "G1LYP",
        "g3lyp": "G3LYP",
        "g1p": "G1P",
        "g3p": "G3P",
        "pbe0": "PBE0",
        "pwp1": "PWP1",
        "m06": "M06",
        "m062x": "M062X",
        "mpw1pw": "mPW1PW",
        "mpw1lyp": "mPW1LYP",
        "pw910": "PW91_0",
        "o3lyp": "O3LYP",
        "x3lyp": "X3LYP",
        "pw6b95": "PW6B95",
        "b97": "B97",
        "bhandhlyp": "BHANDHLYP",
        "wb97": "wB97",
        "wb97x": "wB97X",
        "wb97xd3": "wB97X-D3",
        "wb97xv": "wB97X-V",
        "wb97xd3bj": "wB97X-D3BJ",
        "camb3lyp": "CAM-B3LYP",
        "lcblyp": "LC-BLYP",
        "tpssh": "TPSSH",
        "tpss0": "TPSS0",
        "b2plyp": "B2PLYP",
        "mpw2plyp": "mPW2PLYP",
        "b2gpplyp": "B2GP-PLYP",
        "b2kplyp": "B2K-PLYP",
        "b2tplyp": "B2T-PLYP",
        "pwpb95": "PWPB95",
        "wb2plyp": "wB2PLYP",
        "wb2gp-plyp": "wB2GP-PLYP",
        "am1": "AM1",
        "pm3": "PM3",
        "mndo": "MNDO",
        "gfn2xtb": "xtb2",
        "gfn1xtb": "xtb1",
        #'gfn0-xtb': 'xtb0', # Not available in ORCA 5.0.2
        "gfnff": "xtbff",
        "hf": "HF",
        "mp2": "MP2",
        "rimp2": "RI-MP2",
        "hf3c": "HF-3c",
        "pbeh3c": "PBEh-3c",
        "r2scan3c": "r2SCAN-3c",
        "b973c": "B97-3c",
        "ccsd": "CCSD",
        "ccsdt": "CCSD(T)",
        "qcisd": "QCISD",
        "qcisdt": "QCISD(T)",
        "lpnoccsd": "LPNO-CCSD",
        "lpnoccsdt": "LPNOCCSD(T)",
        "lpnoqcisd": "LPNO-QCISD",
        "lpnoqcisdt": "LPNO-QCISD(T)",
        "dlpnoccsd": "DLPNO-CCSD",
        "dlpnoccsdt": "DLPNO-CCSD(T)",
        "dlpnoqcisd": "DLPNO-QCISD",
        "dlpnoqcisdt": "DLPNO-QCISD(T)",
    },
    "gaussian": {
        "b97d": "B97D",
        "b97d3": "B97D3",
        "m06l": "M06L",
        "m08hx": "M08HX",
        "b1lyp": "B1LYP",
        "b3lyp": "B3LYP",
        "b3p86": "B3P86",
        "pbe0": "PBE1PBE",
        "hse06": "HSE1PBE",
        "ohse2pbe": "OHSE2PBE",
        "ohse1pbe": "OHSE1PBE",
        "pbeh1pbe": "PBEh1PBE",
        "b1b95": "B1B95",
        "m05": "M05",
        "m052x": "M052X",
        "m06": "M06",
        "m06hf": "M06HF",
        "m062x": "M062X",
        "o3lyp": "O3LYP",
        "apfd": "APFD",
        "x3lyp": "X3LYP",
        "pw6b95": "PW6B95",
        "pw6b95d3": "PW6B95D3",
        "mpw1pw91": "mPW1PW91",
        "mpw1lyp": "mPW1LYP",
        "mpw1pbe": "mPW1PBE",
        "mpw3pbe": "mPW3PBE",
        "wb97": "wB97",
        "wb97x": "wB97X",
        "wb97xd": "wB97XD",  # Gaussian says dispersion model D2, not D3
        "b98": "B98",
        "camb3lyp": "CAM-B3LYP",
        "lcwhpbe": "LC-wHPBE",
        "tpssh": "TPSSH",
        "thcthhyb": "tHCTHhyb",
        "bmk": "BMK",
        "hissbpbe": "HISSbPBE",
        "mn15": "MN15",
        "mn15l": "MN15L",
        "m11": "M11",
        "m11l": "M11L",
        "sogga11x": "SOGGA11X",
        "sogga11": "SOGGA11",
        "n12sx": "N12SX",
        "n12": "N12",
        "mn12sx": "MN12SX",
        "mn12l": "MN12L",
        "vsxc": "VSXC",
        "hcth407": "HCTH/407",
        "hcth93": "HCTH/93",
        "hcth147": "HCTH/147",
        "thcth": "tHCTH",
        "am1": "AM1",
        "pm3": "PM3",
        "pm6": "PM6",
        "pm7": "PM7",
        "hf": "HF",
    },
    "xtb": {
        "gfn2xtb": "gfn2-xtb",
        "gfn1xtb": "gfn1-xtb",
        "gfn0xtb": "gfn0-xtb",
        "gfnff": "gfn-ff",
        "gfn2xtbgfnff": "gfn2-xtb//gfn-ff",
    },
}

SOFTWARE_BASIS_SETS = {
    "orca": {
        "sto3g": "STO-3G",
        "mini": "MINI",
        "minis": "MINIS",
        "minix": "MINIX",
        "midi": "MIDI",
        "321g": "3-21G",
        "321gsp": "3-21GSP",
        "422gsp": "4-22GSP",
        "631g": "6-31G",
        "631gd": "6-31G(d)",
        "m631g": "m6-31G",
        "m631gd": "m6-31G(d)",
        "631gdp": "6-31G(d,p)",
        "631g2d": "6-31G(2d)",
        "631g2dp": "6-31G(2d,p)",
        "631g2d2p": "6-31G(2d,2p)",
        "631g2df": "6-31G(2df)",
        "631g2df2p": "6-31G(2df,2p)",
        "631g2df2pd": "6-31G(2df,2pd)",
        "631+gd": "6-31+G(d)",
        "631+gdp": "6-31+G(d,p)",
        "631+g2d": "6-31+G(2d)",
        "631+g2dp": "6-31+G(2d,p)",
        "631+g2d2p": "6-31+G(2d,2p)",
        "631+g2df": "6-31+G(2df)",
        "631+g2df2p": "6-31+G(2df,2p)",
        "631+g2df2pd": "6-31+G(2df,2pd)",
        "631++gdp": "6-31++G(d,p)",
        "631++g2dp": "6-31++G(2d,p)",
        "631++g2d2p": "6-31++G(2d,2p)",
        "631++g2df": "6-31++G(2df)",
        "631++g2df2p": "6-31++G(2df,2p)",
        "631++g2df2pd": "6-31++G(2df,2pd)",
        "6311gd": "6-311G(d)",
        "6311gdp": "6-311G(d,p)",
        "6311g2d": "6-311G(2d)",
        "6311g2dp": "6-311G(2d,p)",
        "6311g2d2p": "6-311G(2d,2p)",
        "6311g2df": "6-311G(2df)",
        "6311g2df2p": "6-311G(2df,2p)",
        "6311g2df2pd": "6-311G(2df,2pd)",
        "6311g3df": "6-311G(3df)",
        "6311g3df3pd": "6-311G(3df,3pd)",
        "6311+gd": "6-311+G(d)",
        "6311+gdp": "6-311+G(d,p)",
        "6311+g2d": "6-311+G(2d)",
        "6311+g2dp": "6-311+G(2d,p)",
        "6311+g2d2p": "6-311+G(2d,2p)",
        "6311+g2df": "6-311+G(2df)",
        "6311+g2df2p": "6-311+G(2df,2p)",
        "6311+g2df2pd": "6-311+G(2df,2pd)",
        "6311+g3df": "6-311+G(3df)",
        "6311+g3df2p": "6-311+G(3df,2p)",
        "6311+g3df3pd": "6-311+G(3df,3pd)",
        "6311++gdp": "6-311++G(d,p)",
        "6311++g2d": "6-311++G(2d)",
        "6311++g2dp": "6-311++G(2d,p)",
        "6311++g2d2p": "6-311++G(2d,2p)",
        "6311++g2df": "6-311++G(2df)",
        "6311++g2df2p": "6-311++G(2df,2p)",
        "6311++g2df2pd": "6-311++G(2df,2pd)",
        "6311++g3df": "6-311++G(3df)",
        "6311++g3df3pd": "6-311++G(3df,3pd)",
        "havt+dz": "haV(t+d)z",
        "havq+dz": "haV(q+d)z",
        "hav5+dz": "haV(5+d)z",
        "w1dz": "W1-DZ",
        "w1mtsmall": "w1-mtsmall",
        "w1dz": "W1-TZ",
        "sv": "SV",
        "svp": "SVP",
        "tzv": "TZV",
        "tzvp": "TZVP",
        "tzvpp": "TZVPP",
        "qzvp": "QZVP",
        "qzvpp": "QZVPP",
        "dkhsvp": "DKH-SVP",
        "dkhtzvp": "DKH-TZVP",
        "dkhtzv-pp": "DKH-TZV-PP",
        "dkhqzvp": "DKH-QZVP",
        "dkhqzv-pp": "DKH-QZV-PP",
        "zorasvp": "ZORA-SVP",
        "zoratzvp": "ZORA-TZVP",
        "zoratzvpp": "ZORA-TZV-PP",
        "zoraqzvp": "ZORA-QZVP",
        "zoraqzvpp": "ZORA-QZV-PP",
        "def2msvp": "Def2-mSVP",
        "def2mtzvp": "Def2-mTZVP",
        "def2svp": "Def2-SVP",
        "def2tzvpf": "Def2-TZVP(-f)",
        "def2tzvp": "Def2-TZVP",
        "def2tzvpp": "Def2-TZVPP",
        "def2qzvp": "Def2-QZVP",
        "def2qzvpp": "Def2-QZVPP",
        "def2svpd": "Def2-SVPD",
        "def2tzvpd": "Def2-TZVPD",
        "def2tzvppd": "Def2-TZVPPD",
        "def2qzvpd": "Def2-QZVPD",
        "def2qzvppd": "Def2-QZVPPD",
        "dkhdef2svp": "DKH-Def2-SVP",
        "dkhdef2tzvpf": "DKH-Def2-TZVP(-f)",
        "dkhdef2tzvp": "DKH-Def2-TZVP",
        "dkhdef2tzvpp": "DKH-Def2-TZVPP",
        "dkhdef2qzvp": "DKH-Def2-QZVP",
        "dkhdef2qzvpp": "DKH-Def2-QZVPP",
        "zoradef2svp": "ZORA-Def2-SVP",
        "zoradef2tzvpf": "ZORA-Def2-TZVP(-f)",
        "zoradef2tzvp": "ZORA-Def2-TZVP",
        "zoradef2tzvpp": "ZORA-Def2-TZVPP",
        "zoradef2qzvp": "ZORA-Def2-QZVP",
        "zoradef2qzvpp": "ZORA-Def2-QZVPP",
        "madef2msvp": "ma-Def2-mSVP",
        "madef2svp": "ma-Def2-SVP",
        "madef2tzvpf": "ma-Def2-TZVP(-f)",
        "madef2tzvp": "ma-Def2-TZVP",
        "madef2tzvpp": "ma-Def2-TZVPP",
        "madef2qzvp": "ma-Def2-QZVP",
        "madef2qzvpp": "ma-Def2-QZVPP",
        "madkhdef2svp": "ma-DKH-Def2-SVP",
        "madkhdef2tzvpf": "ma-DKH-Def2-TZVP(-f)",
        "madkhdef2tzvp": "ma-DKH-Def2-TZVP",
        "madkhdef2tzvpp": "ma-DKH-Def2-TZVPP",
        "madkhdef2qzvp": "ma-DKH-def2-QZVP",
        "madkhdef2qzvpp": "ma-DKH-def2-QZVPP",
        "mazoradef2svp": "ma-ZORA-def2-SVP",
        "mazoradef2tzvpf": "ma-ZORA-Def2-TZVP(-f)",
        "mazoradef2tzvp": "ma-ZORA-Def2-TZVP",
        "mazoradef2tzvpp": "ma-ZORA-Def2-TZVPP",
        "mazoradef2qzvpp": "ma-ZORA-Def2-QZVPP",
        "anopvdz": "ANO-pVDZ",
        "anopvtz": "ANO-pVTZ",
        "anopvqz": "ANO-pVQZ",
        "anopv5z": "ANO-pV5Z",
        "anopv6z": "ANO-pV6Z",
        "auganopvdz": "aug-ANO-pVDZ",
        "auganopvtz": "aug-ANO-pVTZ",
        "auganopvqz": "aug-ANO-pVQZ",
        "auganopv5z": "aug-ANO-pV5Z",
        "sauganopvdz": "saug-ANO-pVDZ",
        "sauganopvtz": "saug-ANO-pVTZ",
        "sauganopvqz": "saug-ANO-pVQZ",
        "sauganopv5z": "saug-ANO-pV5Z",
        "anorccdzp": "ANO-RCC-DZP",
        "anorcctzp": "ANO-RCC-TZP",
        "anorccqzp": "ANO-RCC-QZP",
        "anorccfull": "ANO-RCC-Full",
        "pc0": "pc-0",
        "pc1": "pc-1",
        "pc2": "pc-2",
        "pc3": "pc-3",
        "pc4": "pc-4",
        "augpc0": "aug-pc-0",
        "augpc1": "aug-pc-1",
        "augpc2": "aug-pc-2",
        "augpc3": "aug-pc-3",
        "augpc4": "aug-pc-4",
        "pcj0": "pcJ-0",
        "pcj1": "pcJ-1",
        "pcj2": "pcJ-2",
        "pcj3": "pcJ-3",
        "pcj4": "pcJ-4",
        "augpcj0": "aug-pcJ-0",
        "augpcj1": "aug-pcJ-1",
        "augpcj2": "aug-pcJ-2",
        "augpcj3": "aug-pcJ-3",
        "augpcj4": "aug-pcJ-4",
        "pcseg0": "pcseg-0",
        "pcseg1": "pcseg-1",
        "pcseg2": "pcseg-2",
        "pcseg3": "pcseg-3",
        "pcseg4": "pcseg-4",
        "augpcseg0": "aug-pcseg-0",
        "augpcseg1": "aug-pcseg-1",
        "augpcseg2": "aug-pcseg-2",
        "augpcseg3": "aug-pcseg-3",
        "augpcseg4": "aug-pcseg-4",
        "pcsseg0": "pcSseg-0",
        "pcsseg1": "pcSseg-1",
        "pcsseg2": "pcSseg-2",
        "pcsseg3": "pcSseg-3",
        "pcsseg4": "pcSseg-4",
        "augpcsseg0": "aug-pcSseg-0",
        "augpcsseg1": "aug-pcSseg-1",
        "augpcsseg2": "aug-pcSseg-2",
        "augpcsseg3": "aug-pcSseg-3",
        "augpcsseg4": "aug-pcSseg-4",
        "ccpvdz": "cc-pVDZ",
        "ccpvtz": "cc-pVTZ",
        "ccpvqz": "cc-pVQZ",
        "ccpv5z": "cc-pV5Z",
        "ccpv6z": "cc-pV6Z",
        "augccpvdz": "aug-cc-pVDZ",
        "augccpvtz": "aug-cc-pVTZ",
        "augccpvqz": "aug-cc-pVQZ",
        "augccpv5z": "aug-cc-pV5Z",
        "augccpv6z": "aug-cc-pV6Z",
        "ccpvd+dz": "cc-pVD(+d)Z",
        "ccpvt+dz": "cc-pVT(+d)Z",
        "ccpvq+dz": "cc-pVQ(+d)Z",
        "ccpv5+dz": "cc-pV5(+d)Z",
        "augccpvtzj": "aug-cc-pVTZ-J",
        "ccpcvdz": "cc-pCVDZ",
        "ccpcvtz": "cc-pCVTZ",
        "ccpcvqz": "cc-pCVQZ",
        "ccpcv5z": "cc-pCV5Z",
        "ccpcv6z": "cc-pCV6Z",
        "augccpcvdz": "aug-cc-pCVDZ",
        "augccpcvtz": "aug-cc-pCVTZ",
        "augccpcvqz": "aug-cc-pCVQZ",
        "augccpcv5z": "aug-cc-pCV5Z",
        "augccpcv6z": "aug-cc-pCV6Z",
        "ccpwcvdz": "cc-pwCVDZ",
        "ccpwcvtz": "cc-pwCVTZ",
        "ccpwcvqz": "cc-pwCVQZ",
        "ccpwcv5z": "cc-pwCV5Z",
        "augccpwcvdz": "aug-cc-pwCVDZ",
        "augccpwcvtz": "aug-cc-pwCVTZ",
        "augccpwcvqz": "aug-cc-pwCVQZ",
        "augccpwcv5z": "aug-cc-pwCV5Z",
        "ccpvdzpp": "cc-pVDZ-PP",
        "ccpvtzpp": "cc-pVTZ-PP",
        "ccpvqzpp": "cc-pVQZ-PP",
        "ccpv5zpp": "cc-pV5Z-PP",
        "augccpvdz-pp": "aug-cc-pVDZ-PP",
        "augccpvtz-pp": "aug-cc-pVTZ-PP",
        "augccpvqz-pp": "aug-cc-pVQZ-PP",
        "augccpv5z-pp": "aug-cc-pV5Z-PP",
        "ccpcvdzpp": "cc-pCVDZ-PP",
        "ccpcvtzpp": "cc-pCVTZ-PP",
        "ccpcvqzpp": "cc-pCVQZ-PP",
        "ccpcv5zpp": "cc-pCV5Z-PP",
        "augccpcvdz-pp": "aug-cc-pCVDZ-PP",
        "augccpcvtz-pp": "aug-cc-pCVTZ-PP",
        "augccpcvqz-pp": "aug-cc-pCVQZ-PP",
        "augccpcv5z-pp": "aug-cc-pCV5Z-PP",
        "ccpwvdzpp": "cc-pwVDZ-PP",
        "ccpwvtzpp": "cc-pwVTZ-PP",
        "ccpwvqzpp": "cc-pwVQZ-PP",
        "ccpwv5zpp": "cc-pwV5Z-PP",
        "augccpwvdz-pp": "aug-cc-pwVDZ-PP",
        "augccpwvtz-pp": "aug-cc-pwVTZ-PP",
        "augccpwvqz-pp": "aug-cc-pwVQZ-PP",
        "augccpwv5z-pp": "aug-cc-pwV5Z-PP",
        "ccpvdzdk": "cc-pVDZ-DK",
        "ccpvtzdk": "cc-pVTZ-DK",
        "ccpvqzdk": "cc-pVQZ-DK",
        "ccpv5zdk": "cc-pV5Z-DK",
        "augccpvdzdk": "aug-cc-pVDZ-DK",
        "augccpvtzdk": "aug-cc-pVTZ-DK",
        "augccpvqzdk": "aug-cc-pVQZ-DK",
        "augccpv5zdk": "aug-cc-pV5Z-DK",
        "ccpwcvdzdk": "cc-pwCVDZ-DK",
        "ccpwcvtzdk": "cc-pwCVTZ-DK",
        "ccpwcvqzdk": "cc-pwCVQZ-DK",
        "ccpwcv5zdk": "cc-pwCV5Z-DK",
        "augccpvd+dz": "aug-cc-pV(D+d)Z",
        "augccpvt+dz": "aug-cc-pV(T+d)Z",
        "augccpvq+dz": "aug-cc-pV(Q+d)Z",
        "augccpv5+dz": "aug-cc-pV(5+d)Z",
        "augccpv6+dz": "aug-cc-pV(6+d)Z",
        "w1opt": "W1-Opt",
        "w1qz": "W1-QZ",
        "augccpwvdzdk": "aug-cc-pwVDZ-DK",
        "augccpwvtzdk": "aug-cc-pwVTZ-DK",
        "augccpwvqzdk": "aug-cc-pwVQZ-DK",
        "augccpwv5zdk": "aug-cc-pwV5Z-DK",
        "partridge1": "Partridge-1",
        "partridge2": "Partridge-2",
        "partridge3": "Partridge-3",
        "partridge4": "Partridge-4",
        "anosz": "ANO-SZ",
        "lanl08": "LANL08",
        "lanl08f": "LANL08(f)",
        "lanl2dz": "LANL2DZ",
        "lanl2tz": "LANL2TZ",
        "lanl2tzf": "LANL2TZ(f)",
        "sapporodzp2012": "Sapporo-DZP-2012",
        "sapporotzp2012": "Sapporo-TZP-2012",
        "sapporoqzp2012": "Sapporo-QZP-2012",
        "sapporodkh3dzp2012": "Sapporo-DKH3-DZP-2012",
        "sapporodkh3tzp2012": "Sapporo-DKH3-TZP-2012",
        "sapporodkh3qzp2012": "Sapporo-DKH3-QZP-2012",
        "sarcdkhsvp": "SARC-DKH-SVP",
        "sarcdkhtzvp": "SARC-DKH-TZVP",
        "sarcdkhtzvpp": "SARC-DKH-TZVPP",
        "sarc2dkhqzv": "SARC2-DKH-QZV",
        "sarc2dkhqzvp": "SARC2-DKH-QZVP",
        "sarc2zoraqzv": "SARC2-ZORA-QZV",
        "sarc2zoraqzvp": "SARC2-ZORA-QZVP",
        "d95": "D95",
        "d95p": "D95p",
        "eprii": "EPR-II",
        "epriii": "EPR-III",
        "igloii": "IGLO-II",
        "igloiii": "IGLO-III",
        "ugbs": "UGBS",
        "cp": "CP",
        "cpppp": "CP(PPP)",
        "wachters+f": "Wachters+f",
        "ccpvdzf12": "cc-pVDZ-F12",
        "ccpvtzf12": "cc-pVTZ-F12",
        "ccpvqzf12": "cc-pVQZ-F12",
        "ccpvdzppf12": "cc-pVDZ-PP-F12",
        "ccpvtzppf12": "cc-pVTZ-PP-F12",
        "ccpvqzppf12": "cc-pVQZ-PP-F12",
        "ccpcvdzf12": "cc-pCVDZ-F12",
        "ccpcvtzf12": "cc-pCVTZ-F12",
        "ccpcvqzf12": "cc-pCVQZ-F12",
        # Auxiliary
        "def2": "Def2",
        "def2mtzvp": "Def2-mTZVP",
        "def2mtzvpp": "Def2-mTZVPP",
        "x2c": "x2c",
        "sarc": "SARC",
        "w1mtsmall": "W1-mtsmall",
        "w1dz": "W1-DZ",
        "w1tz": "W1-TZ",
        "w1qz": "W1-QZ",
        "w1opt": "W1-Opt",
    },
    "gaussian": {
        "sto3g": "STO-3G",
        "midix": "MidiX",
        "321g": "3-21G",
        "431g": "4-31G",
        "621g": "6-21G",
        "631g": "6-31G",
        # Missing: 6-31G dagger from Petersson
        "631gd": "6-31G(d)",
        "631gdp": "6-31G(d,p)",
        "631g2d": "6-31G(2d)",
        "631g2dp": "6-31G(2d,p)",
        "631g2d2p": "6-31G(2d,2p)",
        "631g2df": "6-31G(2df)",
        "631g2df2p": "6-31G(2df,2p)",
        "631g2df2pd": "6-31G(2df,2pd)",
        "631+gd": "6-31+G(d)",
        "631+gdp": "6-31+G(d,p)",
        "631+g2d": "6-31+G(2d)",
        "631+g2dp": "6-31+G(2d,p)",
        "631+g2d2p": "6-31+G(2d,2p)",
        "631+g2df": "6-31+G(2df)",
        "631+g2df2p": "6-31+G(2df,2p)",
        "631+g2df2pd": "6-31+G(2df,2pd)",
        "631++gdp": "6-31++G(d,p)",
        "631++g2dp": "6-31++G(2d,p)",
        "631++g2d2p": "6-31++G(2d,2p)",
        "631++g2df": "6-31++G(2df)",
        "631++g2df2p": "6-31++G(2df,2p)",
        "631++g2df2pd": "6-31++G(2df,2pd)",
        "6311gd": "6-311G(d)",
        "6311gdp": "6-311G(d,p)",
        "6311g2d": "6-311G(2d)",
        "6311g2dp": "6-311G(2d,p)",
        "6311g2d2p": "6-311G(2d,2p)",
        "6311g2df": "6-311G(2df)",
        "6311g2df2p": "6-311G(2df,2p)",
        "6311g2df2pd": "6-311G(2df,2pd)",
        "6311g3df": "6-311G(3df)",
        "6311g3df3pd": "6-311G(3df,3pd)",
        "6311+gd": "6-311+G(d)",
        "6311+gdp": "6-311+G(d,p)",
        "6311+g2d": "6-311+G(2d)",
        "6311+g2dp": "6-311+G(2d,p)",
        "6311+g2d2p": "6-311+G(2d,2p)",
        "6311+g2df": "6-311+G(2df)",
        "6311+g2df2p": "6-311+G(2df,2p)",
        "6311+g2df2pd": "6-311+G(2df,2pd)",
        "6311+g3df": "6-311+G(3df)",
        "6311+g3df2p": "6-311+G(3df,2p)",
        "6311+g3df3pd": "6-311+G(3df,3pd)",
        "6311++gdp": "6-311++G(d,p)",
        "6311++g2d": "6-311++G(2d)",
        "6311++g2dp": "6-311++G(2d,p)",
        "6311++g2d2p": "6-311++G(2d,2p)",
        "6311++g2df": "6-311++G(2df)",
        "6311++g2df2p": "6-311++G(2df,2p)",
        "6311++g2df2pd": "6-311++G(2df,2pd)",
        "6311++g3df": "6-311++G(3df)",
        "6311++g3df3pd": "6-311++G(3df,3pd)",
        "w1mtsmall": "MTSmall",
        "sv": "SV",
        "svp": "SVP",
        "tzv": "TZV",
        "tzvp": "TZVP",
        "qzvp": "QZVP",
        "def2sv": "Def2SV",
        "def2svp": "Def2SVP",
        "def2svpp": "Def2SVPP",
        "def2tzv": "Def2TZV",
        "def2tzvp": "Def2TZVP",
        "def2tzvpp": "Def2TZVPP",
        "def2qzv": "Def2QZV",
        "def2qzvp": "Def2QZVP",
        "def2qzvpp": "Def2QZVPP",
        "ccpvdz": "cc-pVDZ",
        "ccpvtz": "cc-pVTZ",
        "ccpvqz": "cc-pVQZ",
        "ccpv5z": "cc-pV5Z",
        "ccpv6z": "cc-pV6Z",
        "augccpvdz": "aug-cc-pVDZ",
        "augccpvtz": "aug-cc-pVTZ",
        "augccpvqz": "aug-cc-pVQZ",
        "augccpv5z": "aug-cc-pV5Z",
        "augccpv6z": "aug-cc-pV6Z",
        "augccpvd+dz": "dAug-cc-pVDZ",  # to verify
        "augccpvt+dz": "dAug-cc-pVTZ",  # to verify
        "augccpvq+dz": "dAug-cc-pVQZ",  # to verify
        "augccpv5+dz": "dAug-cc-pV5Z",  # to verify
        "lanl2mb": "LANL2MB",
        "lanl2dz": "LANL2DZ",
        "d95": "D95",
        "d95v": "D95V",
        "eprii": "EPR-II",
        "epriii": "EPR-III",
        "ugbs": "UGBS",  # polarization missing
        "shc": "SHC",
        "cep4g": "CEP-4G",
        "cep31g": "CEP-31G",
        "cep121g": "CEP-121G",
        "sdd": "SDD",
        "sddall": "SDDAll",
        "dgdzvp": "DGDZVP",
        "dgdzvp2": "DGDZVP2",
        "gdtzvp": "DGTZVP",
        "cbsb7": "cbsb7",
    },
}

SOFTWARE_SOLVENTS = {
    "gaussian": {
        "water": "water",
        "acetonitrile": "acetonitrile",
        "methanol": "methanol",
        "ethanol": "ethanol",
        "isoquinoline": "isoquinoline",
        "quinoline": "quinoline",
        "chloroform": "chloroform",
        "diethylether": "diethylether",
        "dichloromethane": "dichloromethane",
        "dichloroethane": "dichloroethane",
        "carbontetrachloride": "carbontetrachloride",
        "benzene": "benzene",
        "toluene": "toluene",
        "chlorobenzene": "chlorobenzene",
        "nitromethane": "nitromethane",
        "heptane": "heptane",
        "cyclohexane": "cyclohexane",
        "aniline": "aniline",
        "acetone": "acetone",
        "tetrahydrofuran": "tetrahydrofuran",
        "dimethylsulfoxide": "dimethylsulfoxide",
        "argon": "argon",
        "krypton": "krypton",
        "xenon": "xenon",
        "noctanol": "n-octanol",
        "111trichloroethane": "1,1,1-trichloroethane",
        "112trichloroethane": "1,1,2-trichloroethane",
        "124trimethylbenzene": "1,2,4-trimethylbenzene",
        "12dibromoethane": "1,2-dibromoethane",
        "12ethanediol": "1,2-ethanediol",
        "14dioxane": "1,4-dioxane",
        "1bromo-2-methylpropane": "1-bromo-2-methylpropane",
        "1bromooctane": "1-bromooctane",
        "1bromopentane": "1-bromopentane",
        "1bromopropane": "1-bromopropane",
        "1butanol": "1-butanol",
        "1chlorohexane": "1-chlorohexane",
        "1chloropentane": "1-chloropentane",
        "1chloropropane": "1-chloropropane",
        "1decanol": "1-decanol",
        "1fluorooctane": "1-fluorooctane",
        "1heptanol": "1-heptanol",
        "1hexanol": "1-hexanol",
        "1hexene": "1-hexene",
        "1hexyne": "1-hexyne",
        "1iodobutane": "1-iodobutane",
        "1iodohexadecane": "1-iodohexadecane",
        "1iodopentane": "1-iodopentane",
        "1iodopropane": "1-iodopropane",
        "1nitropropane": "1-nitropropane",
        "1nonanol": "1-nonanol",
        "1pentanol": "1-pentanol",
        "1pentene": "1-pentene",
        "1propanol": "1-propanol",
        "222trifluoroethanol": "2,2,2-trifluoroethanol",
        "224trimethylpentane": "2,2,4-trimethylpentane",
        "24dimethylpentane": "2,4-dimethylpentane",
        "24dimethylpyridine": "2,4-dimethylpyridine",
        "26dimethylpyridine": "2,6-dimethylpyridine",
        "2bromopropane": "2-bromopropane",
        "2butanol": "2-butanol",
        "2chlorobutane": "2-chlorobutane",
        "2heptanone": "2-heptanone",
        "2hexanone": "2-hexanone",
        "2methoxyethanol": "2-methoxyethanol",
        "2methyl1propanol": "2-methyl-1-propanol",
        "2methyl2propanol": "2-methyl-2-propanol",
        "2methylpentane": "2-methylpentane",
        "2methylpyridine": "2-methylpyridine",
        "2nitropropane": "2-nitropropane",
        "2octanone": "2-octanone",
        "2pentanone": "2-pentanone",
        "2propanol": "2-propanol",
        "2propen1ol": "2-propen-1-ol",
        "3methylpyridine": "3-methylpyridine",
        "3pentanone": "3-pentanone",
        "4heptanone": "4-heptanone",
        "4methyl-2-pentanone": "4-methyl-2-pentanone",
        "4methylpyridine": "4-methylpyridine",
        "5nonanone": "5-nonanone",
        "aceticacid": "aceticacid",
        "acetophenone": "acetophenone",
        "achlorotoluene": "a-chlorotoluene",
        "anisole": "anisole",
        "benzaldehyde": "benzaldehyde",
        "benzonitrile": "benzonitrile",
        "benzylalcohol": "benzylalcohol",
        "bromobenzene": "bromobenzene",
        "bromoethane": "bromoethane",
        "bromoform": "bromoform",
        "butanal": "butanal",
        "butanoicacid": "butanoicacid",
        "butanone": "butanone",
        "butanonitrile": "butanonitrile",
        "butylamine": "butylamine",
        "butylethanoate": "butylethanoate",
        "carbondisulfide": "carbondisulfide",
        "cis12dimethylcyclohexane": "cis-1,2-dimethylcyclohexane",
        "cisdecalin": "cis-decalin",
        "cyclohexanone": "cyclohexanone",
        "cyclopentane": "cyclopentane",
        "cyclopentanol": "cyclopentanol",
        "cyclopentanone": "cyclopentanone",
        "decalin-mixture": "decalin-mixture",
        "dibromomethane": "dibromomethane",
        "dibutylether": "dibutylether",
        "diethylamine": "diethylamine",
        "diethylsulfide": "diethylsulfide",
        "diiodomethane": "diiodomethane",
        "diisopropylether": "diisopropylether",
        "dimethyldisulfide": "dimethyldisulfide",
        "diphenylether": "diphenylether",
        "dipropylamine": "dipropylamine",
        "e12dichloroethene": "e-1,2-dichloroethene",
        "e2pentene": "e-2-pentene",
        "ethanethiol": "ethanethiol",
        "ethylbenzene": "ethylbenzene",
        "ethylethanoate": "ethylethanoate",
        "ethylmethanoate": "ethylmethanoate",
        "ethylphenylether": "ethylphenylether",
        "fluorobenzene": "fluorobenzene",
        "formamide": "formamide",
        "formicacid": "formicacid",
        "hexanoicacid": "hexanoicacid",
        "iodobenzene": "iodobenzene",
        "iodoethane": "iodoethane",
        "iodomethane": "iodomethane",
        "isopropylbenzene": "isopropylbenzene",
        "mcresol": "m-cresol",
        "mesitylene": "mesitylene",
        "methylbenzoate": "methylbenzoate",
        "methylbutanoate": "methylbutanoate",
        "methylcyclohexane": "methylcyclohexane",
        "methylethanoate": "methylethanoate",
        "methylmethanoate": "methylmethanoate",
        "methylpropanoate": "methylpropanoate",
        "mxylene": "m-xylene",
        "nbutylbenzene": "n-butylbenzene",
        "ndecane": "n-decane",
        "ndodecane": "n-dodecane",
        "nhexadecane": "n-hexadecane",
        "nhexane": "n-hexane",
        "nitrobenzene": "nitrobenzene",
        "nitroethane": "nitroethane",
        "nmethylaniline": "n-methylaniline",
        "nmethylformamide-mixture": "n-methylformamide-mixture",
        "nndimethylacetamide": "n,n-dimethylacetamide",
        "nndimethylformamide": "n,n-dimethylformamide",
        "nnonane": "n-nonane",
        "noctane": "n-octane",
        "npentadecane": "n-pentadecane",
        "npentane": "n-pentane",
        "nundecane": "n-undecane",
        "ochlorotoluene": "o-chlorotoluene",
        "ocresol": "o-cresol",
        "odichlorobenzene": "o-dichlorobenzene",
        "onitrotoluene": "o-nitrotoluene",
        "oxylene": "o-xylene",
        "pentanal": "pentanal",
        "pentanoicacid": "pentanoicacid",
        "pentylamine": "pentylamine",
        "pentylethanoate": "pentylethanoate",
        "perfluorobenzene": "perfluorobenzene",
        "pisopropyltoluene": "p-isopropyltoluene",
        "propanal": "propanal",
        "propanoicacid": "propanoicacid",
        "propanonitrile": "propanonitrile",
        "propylamine": "propylamine",
        "propylethanoate": "propylethanoate",
        "pxylene": "p-xylene",
        "pyridine": "pyridine",
        "secbutylbenzene": "sec-butylbenzene",
        "tertbutylbenzene": "tert-butylbenzene",
        "tetrachloroethene": "tetrachloroethene",
        "tetrahydrothiophene-s,s-dioxide": "tetrahydrothiophene-s,s-dioxide",
        "tetralin": "tetralin",
        "thiophene": "thiophene",
        "thiophenol": "thiophenol",
        "transdecalin": "trans-decalin",
        "tributylphosphate": "tributylphosphate",
        "trichloroethene": "trichloroethene",
        "triethylamine": "triethylamine",
        "xylenemixture": "xylene-mixture",
        "z12dichloroethene": "z-1,2-dichloroethene",
    },
    "orca": {
        "water": "water",
        "acetone": "acetone",
        "acetonitrile": "acetonitrile",
        "ammonia": "ammonia",
        "benzene": "benzene",
        "carbontetrachloride": "ccl4",
        "dichloromethane": "ch2cl2",
        "dichloroethane": "1,2-dichloroethane",
        "chloroform": "chloroform",
        "cyclohexane": "cyclohexane",
        "dimethylformamide": "dmf",
        "dimethylsulfoxide": "dmso",
        "ethanol": "ethanol",
        "nhexane": "hexane",
        "methanol": "methanol",
        "noctanol": "octanol",  # CPCM keyword, see calculation_helper.py
        "pyridine": "pyridine",
        "tetrahydrofuran": "thf",
        "toluene": "toluene",
        "111trichloroethane": "1,1,1-trichloroethane",
        "112trichloroethane": "1,1,2-trichloroethane",
        "124trimethylbenzene": "1,2,4-trimethylbenzene",
        "12dibromoethane": "1,2-dibromoethane",
        "12ethanediol": "1,2-ethanediol",
        "14dioxane": "1,4-dioxane",
        "1bromo2methylpropane": "1-bromo-2-methylpropane",
        "1bromooctane": "1-bromooctane",
        "1bromopentane": "1-bromopentane",
        "1bromopropane": "1-bromopropane",
        "1butanol": "1-butanol",
        "1chlorohexane": "1-chlorohexane",
        "1chloropentane": "1-chloropentane",
        "1chloropropane": "1-chloropropane",
        "1decanol": "1-decanol",
        "1fluorooctane": "1-fluorooctane",
        "1heptanol": "1-heptanol",
        "1hexanol": "1-hexanol",
        "1hexene": "1-hexene",
        "1hexyne": "1-hexyne",
        "1iodobutane": "1-iodobutane",
        "1iodohexadecane": "1-iodohexadecane",
        "1iodopentane": "1-iodopentane",
        "1iodopropane": "1-iodopropane",
        "1nitropropane": "1-nitropropane",
        "1nonanol": "1-nonanol",
        "1octanol": "1-octanol",
        "1pentanol": "1-pentanol",
        "1pentene": "1-pentene",
        "1propanol": "1-propanol",
        "222trifluoroethanol": "2,2,2-trifluoroethanol",
        "224trimethylpentane": "2,2,4-trimethylpentane",
        "24dimethylpentane": "2,4-dimethylpentane",
        "24dimethylpyridine": "2,4-dimethylpyridine",
        "26dimethylpyridine": "2,6-dimethylpyridine",
        "2bromopropane": "2-bromopropane",
        "2butanol": "2-butanol",
        "cyclopentane": "cyclopentane",
        "cyclopentanol": "cyclopentanol",
        "cyclopentanone": "cyclopentanone",
        "decalincistransmixture": "decalin (cis/trans mixture)",
        "cisdecalin": "cis-decalin",
        "ndecane": "n-decane",
        "dibromomethane": "dibromomethane",
        "dibutylether": "dibutylether",
        "odichlorobenzene": "o-dichlorobenzene",
        "e12dichloroethene": "e-1,2-dichloroethene",
        "z12dichloroethene": "z-1,2-dichloroethene",
        "diethylsulfide": "diethyl sulfide",
        "diethylamine": "diethylamine",
        "diiodomethane": "diiodomethane",
        "diisopropylether": "diisopropyl ether",
        "cis12dimethylcyclohexane": "cis-1,2-dimethylcyclohexane",
        "dimethyldisulfide": "dimethyl disulfide",
        "nndimethylacetamide": "n,n-dimethylacetamide",
        "nndimethylformamide": "n,n-dimethylformamide",
        "dimethylsulfoxide": "dimethylsulfoxide",
        "diphenylether": "diphenylether",
        "dipropylamine": "dipropylamine",
        "ndodecane": "n-dodecane",
        "ethanethiol": "ethanethiol",
        "ethanol": "ethanol",
        "ethylethanoate": "ethyl ethanoate",
        "ethylmethanoate": "ethyl methanoate",
        "ethylphenylether": "ethyl phenyl ether",
        "ethylbenzene": "ethylbenzene",
        "fluorobenzene": "fluorobenzene",
        "formamide": "formamide",
        "formicacid": "formic acid",
        "nheptane": "n-heptane",
        "nhexadecane": "n-hexadecane",
        "hexanoicacid": "hexanoic acid",
        "2chlorobutane": "2-chlorobutane",
        "2heptanone": "2-heptanone",
        "2hexanone": "2-hexanone",
        "2methoxyethanol": "2-methoxyethanol",
        "2methyl1propanol": "2-methyl-1-propanol",
        "2methyl2propanol": "2-methyl-2-propanol",
        "2methylpentane": "2-methylpentane",
        "2methylpyridine": "2-methylpyridine",
        "2nitropropane": "2-nitropropane",
        "2octanone": "2-octanone",
        "2pentanone": "2-pentanone",
        "2propanol": "2-propanol",
        "2propen1ol": "2-propen-1-ol",
        "e2pentene": "e-2-pentene",
        "3methylpyridine": "3-methylpyridine",
        "3pentanone": "3-pentanone",
        "4heptanone": "4-heptanone",
        "4methyl2pentanone": "4-methyl-2-pentanone",
        "4methylpyridine": "4-methylpyridine",
        "5nonanone": "5-nonanone",
        "aceticacid": "acetic acid",
        "acetophenone": "acetophenone",
        "aniline": "aniline",
        "anisole": "anisole",
        "benzaldehyde": "benzaldehyde",
        "benzonitrile": "benzonitrile",
        "benzylalcohol": "benzyl alcohol",
        "bromobenzene": "bromobenzene",
        "bromoethane": "bromoethane",
        "bromoform": "bromoform",
        "butanal": "butanal",
        "butanoicacid": "butanoic acid",
        "butanone": "butanone",
        "butanonitrile": "butanonitrile",
        "butylethanoate": "butyl ethanoate",
        "butylamine": "butylamine",
        "nbutylbenzene": "n-butylbenzene",
        "secbutylbenzene": "sec-butylbenzene",
        "tertbutylbenzene": "tert-butylbenzene",
        "carbondisulfide": "carbon disulfide",
        "carbontetrachloride": "carbon tetrachloride",
        "chlorobenzene": "chlorobenzene",
        "chloroform": "chloroform",
        "achlorotoluene": "a-chlorotoluene",
        "ochlorotoluene": "o-chlorotoluene",
        "mcresol": "m-cresol",
        "ocresol": "o-cresol",
        "cyclohexane": "cyclohexane",
        "cyclohexanone": "cyclohexanone",
        "iodobenzene": "iodobenzene",
        "iodoethane": "iodoethane",
        "iodomethane": "iodomethane",
        "isopropylbenzene": "isopropylbenzene",
        "pisopropyltoluene": "p-isopropyltoluene",
        "mesitylene": "mesitylene",
        "methylbenzoate": "methyl benzoate",
        "methylbutanoate": "methyl butanoate",
        "methylethanoate": "methyl ethanoate",
        "methylmethanoate": "methyl methanoate",
        "methylpropanoate": "methyl propanoate",
        "nmethylaniline": "n-methylaniline",
        "methylcyclohexane": "methylcyclohexane",
        "nmethylformamideezmixture": "n-methylformamide (e/z mixture)",
        "nitrobenzene": "nitrobenzene",
        "nitroethane": "nitroethane",
        "nitromethane": "nitromethane",
        "onitrotoluene": "o-nitrotoluene",
        "nnonane": "n-nonane",
        "noctane": "n-octane",
        "npentadecane": "n-pentadecane",
        "pentanal": "pentanal",
        "npentane": "n-pentane",
        "pentanoicacid": "pentanoic acid",
        "pentylethanoate": "pentyl ethanoate",
        "pentylamine": "pentylamine",
        "perfluorobenzene": "perfluorobenzene",
        "propanal": "propanal",
        "propanoicacid": "propanoic acid",
        "propanonitrile": "propanonitrile",
        "propylethanoate": "propyl ethanoate",
        "propylamine": "propylamine",
        "tetrachloroethene": "tetrachloroethene",
        "tetrahydrothiophenessdioxide": "tetrahydrothiophene-s,s-dioxide",
        "tetralin": "tetralin",
        "thiophene": "thiophene",
        "thiophenol": "thiophenol",
        "transdecalin": "trans-decalin",
        "tributylphosphate": "tributylphosphate",
        "trichloroethene": "trichloroethene",
        "triethylamine": "triethylamine",
        "nundecane": "n-undecane",
        "xylenemixture": "xylene (mixture)",
        "mxylene": "m-xylene",
        "oxylene": "o-xylene",
        "pxylene": "p-xylene",
    },
    "xtb": {
        "acetone": "acetone",
        "acetonitrile": "acetonitrile",
        "benzene": "benzene",
        "dichloromethane": "ch2cl2",
        "chloroform": "chcl3",
        "carbon disulfide": "cs2",
        "dimethylformamide": "dmf",
        "dimethylsulfoxide": "dmso",
        "diethylether": "ether",
        "water": "h2o",
        "methanol": "methanol",
        "nhexane": "n-hexane",
        "tetrahydrofuran": "thf",
        "toluene": "toluene",
    },
}

# The two lists below use the "absolute" method identifications.
# See SYN_METHODS for the full list of names and their synonyms.

# These lists contain all the functionals with D3 parameters developed
# by the Grimme group, which are not software specific. As such, the
# parameters might not be available in every QM package (although it could).

# Some functionals were not found in the list of known functionals or synonyms.
# These are indicated with the question mark.
FUNCTIONALS_WITH_DISPERSION_PARAMETERS = {
    "d3": [
        "b1b95",
        "b2gp-plyp",
        "b3lyp",
        "b97-d",
        "bhlyp",  # BHandHLYP?
        "blyp",
        "bp86",  # ?
        "bpbe",  # HISSbPBE?
        "mpwlyp",
        "pbe",
        "pbe0",
        "pw6b95",
        "pwb6k",  # ?
        "revpbe",
        "tpss",
        "tpss0",
        "tpssh",
        "bop",  # ?
        "mpw1b95",  # ?
        "mpwb1k",  # ?
        "olyp",
        "opbe",
        "otpss",  # ?
        "pbe38",  # ?
        "pbesol",  # ?
        "revssb",  # ?
        "ssb",  # ?
        "b3pw91",  # ?
        "bmk",
        "camb3lyp",
        "lcwpbe",  # ?
        "m052x",
        "m05",
        "m062x",
        "m06hf",
        "m06l",
        "m06",
        "hcth120",  # ?
        "b2plyp",
        "dsdblyp",  # ?
        "ptpss",  # ?
        "pwpb95",
        "revpbe0",  # ?
        "revpbe38",  # ?
        "rpw86pbe",
    ],
    "d3bj": [
        "1b95",
        "b2gp-plyp",
        "b3pw91",  # ?
        "bhlyp",  # ?
        "bmk",
        "bop",  # ?
        "bpbe",  # ?
        "camb3lyp",
        "lcwpbe",  # ?
        "mpw1b95",  # ?
        "mpwb1k",  # ?
        "mpwlyp",
        "olyp",
        "opbe",
        "otpss",  # ?
        "pbe38",  # ?
        "pbesol",  # ?
        "ptpss",  # ?
        "pwb6k",  # ?
        "revssb",  # ?
        "ssb",  # ?
        "tpssh",
        "hcth120",  # ?
        "b2plyp",
        "b3lyp",
        "b97-d",
        "blyp",
        "bp86",  # ?
        "dsdblyp",  # ?
        "pbe0",
        "pbe",
        "pw6b95",
        "pwpb95",
        "revpbe0",  # ?
        "revpbe38",  # ?
        "revpbe",
        "rpw86pbe",
        "tpss0",
        "tpss",
    ],
}
