"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const assert_1 = require("@aws-cdk/assert");
const mediastore = require("@aws-cdk/aws-mediastore");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const lib_1 = require("../lib");
// --------------------------------------------------------------
// Test the default deployment snapshot
// --------------------------------------------------------------
test('Test the default deployment snapshot', () => {
    // Initial setup
    const stack = new core_1.Stack();
    new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {});
    // Assertion
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test the default deployment pattern variables
// --------------------------------------------------------------
test('Test the default deployment pattern variables', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {});
    // Assertion
    expect(cloudFrontToMediaStore.cloudFrontWebDistribution).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.mediaStoreContainer).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontLoggingBucket).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontOriginRequestPolicy).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontFunction).not.toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment without HTTP security headers
// --------------------------------------------------------------
test('Test the deployment without HTTP security headers', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        insertHttpSecurityHeaders: false
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                    OriginCustomHeaders: [
                        {
                            HeaderName: 'User-Agent',
                            HeaderValue: {
                                Ref: 'testcloudfrontmediastoreCloudFrontOriginAccessIdentity966405A0'
                            }
                        }
                    ]
                }
            ]
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontFunction).toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment with existing MediaStore container
// --------------------------------------------------------------
test('Test the deployment with existing MediaStore container', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const mediaStoreContainer = new mediastore.CfnContainer(stack, 'MyMediaStoreContainer', {
        containerName: 'MyMediaStoreContainer'
    });
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        existingMediaStoreContainerObj: mediaStoreContainer
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "testcloudfrontmediastoreSetHttpSecurityHeaders9995A63D",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'MyMediaStoreContainer',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D'
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        ContainerName: 'MyMediaStoreContainer'
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment with the user provided MediaStore properties
// --------------------------------------------------------------
test('Test the deployment with the user provided MediaStore properties', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        mediaStoreContainerProps: {
            containerName: 'MyMediaStoreContainer',
            policy: '{}',
            lifecyclePolicy: '{}',
            corsPolicy: [],
            metricPolicy: {
                containerLevelMetrics: 'DISABLED'
            }
        }
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "testcloudfrontmediastoreSetHttpSecurityHeaders9995A63D",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D'
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        ContainerName: 'MyMediaStoreContainer',
        Policy: '{}',
        LifecyclePolicy: '{}',
        CorsPolicy: [],
        MetricPolicy: {
            ContainerLevelMetrics: 'DISABLED'
        }
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment with the user provided CloudFront properties
// --------------------------------------------------------------
test('Test the deployment with the user provided CloudFront properties', () => {
    // Initial setup
    const stack = new core_1.Stack();
    new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        cloudFrontDistributionProps: {
            defaultBehavior: {
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.HTTPS_ONLY,
                allowedMethods: cloudfront.AllowedMethods.ALLOW_ALL,
                cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD
            }
        }
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS',
                    'PUT',
                    'PATCH',
                    'POST',
                    'DELETE'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD'
                ],
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "testcloudfrontmediastoreSetHttpSecurityHeaders9995A63D",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'https-only'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                    OriginCustomHeaders: [
                        {
                            HeaderName: 'User-Agent',
                            HeaderValue: {
                                Ref: 'testcloudfrontmediastoreCloudFrontOriginAccessIdentity966405A0'
                            }
                        }
                    ]
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        AccessLoggingEnabled: true,
        ContainerName: {
            Ref: 'AWS::StackName'
        },
        CorsPolicy: [
            {
                AllowedHeaders: ['*'],
                AllowedMethods: ['GET'],
                AllowedOrigins: ['*'],
                ExposeHeaders: ["*"],
                MaxAgeSeconds: 3000
            }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::CloudFrontOriginAccessIdentity', {
        CloudFrontOriginAccessIdentityConfig: {
            Comment: {
                'Fn::Join': [
                    '',
                    [
                        'access-identity-',
                        {
                            Ref: 'AWS::Region',
                        },
                        '-',
                        {
                            Ref: 'AWS::StackName',
                        }
                    ]
                ]
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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