# PPshare使用指南（by@zemengchuan）
GitHub链接：htpp.://github.com/zemengchuan/PPshare

## 用途：

PPshare是一个长期专注于科研数据的应用性平台，平台提供直接从互联网爬取、社区采取和团队整理数据并将其存入数据库，通过对数据质量进行严格控制为用户提供高质量科研数据。

## 安装方式

```python
pip install PPshare
```

## 方法库

## 快速获取目标数据的函数

方法：get_methods

描述：通过向get_methods方法传入目标数据的描述，可以快速获取你想要的函数名称，并且可以自行选择需要匹配的函数数量，以及是否需要调用

输入参数：

| 名称     | 类型 | 描述                |
| -------- | ---- | ------------------- |
| describe | str  | 目标数据的名字      |
| limit    | int  | 匹配的数量，默认为5 |


输出参数：你需要的数据


方法示例：

```python
import PPshare as pp

method = pp.get_methods("商品价格指数")
print(method)
```

结果示例：

```
您希望获取的数据有可能是:
0       " 企业商品价格指数"，其调用方法为China_CGPI
1       "商品零售价格指数"，其调用方法为China_retail_price_index
2       "大宗商品价格"，其调用方法为China_commodity_price_index
3       "建材价格指数"，其调用方法为China_construction_price_index
4       "工业品出厂价格指数"，其调用方法为China_PPI

请选择需要调用的函数编号：（不调用则填quit）0

您选择的数据是:“ 企业商品价格指数”
[            月份  总指数-指数值  总指数-同比增长  总指数-环比增长  农产品-指数值   农产品-同比增长  农产品-环比增长  矿产品-指数值   矿产品-同比增长  矿产品-环比增长  煤油电-指数值   煤油电-同比增长  煤油电-环
比增长
0    2023年01月份    100.4 -5.639098  0.199601    107.4  15.359828  2.972196    101.9  -1.450677 -0.682261    105.8 -10.338983 -2.037037
1    2022年12月份    100.2 -6.790698  0.400802    104.3   7.971014 -0.666667    102.6  -1.535509  1.886792    108.0  -8.937605  0.185529
2    2022年11月份     99.8 -9.107468 -0.299700    105.0   6.924644 -6.166220    100.7  -6.586271  2.650357    107.8 -12.924071 -0.185185
3    2022年10月份    100.1 -9.082652 -1.573255    111.9  21.498371 -0.089286     98.1 -11.938959  2.400835    108.0 -12.691997 -3.398927
4    2022年09月份    101.7 -5.746061 -0.586510    112.0  24.721604  4.089219     95.8 -16.258741  3.232759    111.8  -4.931973 -2.272727
..         ...      ...       ...       ...      ...        ...       ...      ...        ...       ...      ...        ...       ...
195  2006年10月份    103.8  2.843555  0.193050    106.3   8.116355  0.662879    116.0   9.002067  2.112676    107.0 -11.401838 -2.550091
196  2006年09月份    103.6  2.139406  0.680272    105.6   6.850147  1.343570    113.6   4.048361  0.709220    109.8  -9.495549 -0.453309
197  2006年08月份    102.9  0.753941  0.390244    104.2   4.628979  1.165049    112.8   1.768315  0.624442    110.3 -10.492575 -0.090580
198  2006年07月份    102.5 -0.214174  0.195503    103.0   2.763644  0.292113    112.1   1.009191  1.356239    110.4 -10.657927  0.272480
199  2006年06月份    102.3 -0.689253  0.788177    102.7   2.392822  1.884921    110.6  -0.887176  1.589051    110.1 -10.356619 -0.524033

[200 rows x 13 columns]]
```



## 中国宏观

### 中国宏观杠杆率

接口: macro_cnbs

描述: 中国国家金融与发展实验室-中国宏观杠杆率数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称           | 类型    | 描述        |
| -------------- | ------- | ----------- |
| 年份           | object  | 日期, 年-月 |
| 居民部门       | float64 | -           |
| 非金融企业部门 | float64 | -           |
| 政府部门       | float64 | -           |
| 中央政府       | float64 | -           |
| 地方政府       | float64 | -           |
| 实体经济部门   | float64 | -           |
| 金融部门资产方 | float64 | -           |
| 金融部门负债方 | float64 | -           |

接口示例

```python
import PPshare as pp

macro_cnbs_df = pp.macro_cnbs()
print(macro_cnbs_df)
```

数据示例

```
    年份      居民部门    非金融企业部门  ... 实体经济部门 金融部门资产方 金融部门负债方
0   1993-12   8.311222   91.658000  ...  107.791459   8.896441   7.128428
1   1994-12   7.808230   82.411703  ...   98.354271   9.808787   6.796868
2   1995-12   8.240004   80.950106  ...   97.850373  10.009081   7.006151
3   1996-03   8.403456   81.711651  ...   99.241521  10.183896   7.186300
4   1996-06   8.581114   82.051373  ...   99.679459  10.379730   7.380510
..      ...        ...         ...  ...         ...        ...        ...
93  2018-09  52.575456  155.641011  ...  245.227043  61.350917  60.645733
94  2018-12  53.198837  153.553140  ...  243.702122  60.638348  60.936158
95  2019-03  54.277928  156.881879  ...  248.828108  60.542178  59.417322
96  2019-06  55.304291  155.743313  ...  249.533412  58.736094  58.727086
97  2019-09  56.314848  155.618498  ...  251.147265  55.820243  59.358625
```

### 国民经济运行状况

#### 经济状况

##### 企业商品价格指数

接口: China_CGPI

描述: 东方财富-经济数据一览-中国-企业商品价格指数, 数据区间从 20050101-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称            | 类型    | 描述        |
| --------------- | ------- | ----------- |
| 月份            | object  | -           |
| 总指数-指数值   | float64 | -           |
| 总指数-同比增长 | float64 | 注意单位: % |
| 总指数-环比增长 | float64 | 注意单位: % |
| 农产品-指数值   | float64 | -           |
| 农产品-同比增长 | float64 | 注意单位: % |
| 农产品-环比增长 | float64 | 注意单位: % |
| 矿产品-指数值   | float64 | -           |
| 矿产品-同比增长 | float64 | 注意单位: % |
| 矿产品-环比增长 | float64 | 注意单位: % |
| 煤油电-指数值   | float64 | -           |
| 煤油电-同比增长 | float64 | 注意单位: % |
| 煤油电-环比增长 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_CGPI_df = pp.China_CGPI()
print(China_CGPI_df)
```

数据示例

```
     月份  总指数-指数值  总指数-同比增长  ...  煤油电-指数值   煤油电-同比增长  煤油电-环比增长
0    2022年09月份   101.70 -5.746061  ...   111.80  -4.931973 -2.272727
1    2022年08月份   102.30 -4.925651  ...   114.40  -0.608167 -1.970865
2    2022年07月份   103.10 -4.448563  ...   116.70   1.214224 -3.233831
3    2022年06月份   105.20 -2.682701  ...   120.60   4.960836  0.249377
4    2022年05月份   105.10 -3.666361  ...   120.30   4.427083 -1.635323
..         ...      ...       ...  ...      ...        ...       ...
208  2005年05月份   103.16 -5.738304  ...   122.53  11.168572  1.625612
209  2005年04月份   102.81 -5.903350  ...   120.57  12.640135  0.920733
210  2005年03月份   103.48 -4.485878  ...   119.47  14.292548  0.538585
211  2005年02月份   104.82 -2.046538  ...   118.83  13.974679  2.590003
212  2005年01月份   104.67 -1.957662  ...   115.83   8.903723 -1.814021
```

##### 外商直接投资数据

接口: China_FDI

描述: 东方财富-经济数据一览-中国-外商直接投资数据, 数据区间从 200801-202012

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称          | 类型    | 描述           |
| ------------- | ------- | -------------- |
| 月份          | object  | -              |
| 当月          | int64   | -              |
| 当月-同比增长 | float64 | 注意单位: 美元 |
| 当月-环比增长 | float64 | 注意单位: %    |
| 累计          | float64 | 注意单位: 美元 |
| 累计-同比增长 | float64 | 注意单位: %    |

接口示例

```python
import PPshare as pp

China_FDI_df = pp.China_FDI()
print(China_FDI_df)
```

数据示例

```
      月份        当月     当月-同比增长    当月-环比增长  累计  累计-同比增长
0    2022年08月份  14490000   11.119632  25.237684  138410000    20.20
1    2022年07月份  11570000   18.302658 -52.929211  123920000    21.50
2    2022年06月份  24580000   26.831785  84.812030  112350000    21.80
3    2022年05月份  13300000    6.060606 -13.524057   87770000    22.60
4    2022年04月份  15380000    8.462623 -27.555346   74470000    26.10
..         ...       ...         ...        ...        ...      ...
171  2008年05月份   7760960   58.421856   2.079078   42777890    54.97
172  2008年04月份   7602890   70.214971 -18.121715   35016930    59.32
173  2008年03月份   9285600   50.171752  34.026347   27414040    61.26
174  2008年02月份   6928190   52.801205 -38.142541   18128440    75.19
175  2008年01月份  11200250  116.422424 -14.460445   11200250   109.78
```

##### **贷款市场报价利率**

接口:LPR()

描述: 中国 LPR 品种数据, 数据区间从 2013.10.25-至今

限量: 单次返回一年（364天）的数据量，超过则会返回空表

输入参数

| 名称       | 类型                | 描述                                    |
| ---------- | ------------------- | --------------------------------------- |
| start_date | str / datetime.date | 起始时间，默认为终止时间前一年（364天） |
| end_date   | str / datetime.date | 终止时间，默认为今天                    |

输出参数

| 名称 | 类型  | 描述 |
| ---- | ----- | ---- |
| 日期 | str   | 日期 |
| 1Y   | float | 1年  |

接口示例

```python
import PPshare as pp

China_lpr_df = pp.LPR(start_date='2013-10-01',end_date='2014-9-30')
print(China_lpr_df)
```

数据示例

```
             日期    1Y
0    2014-09-30  5.76
1    2014-09-29  5.76
2    2014-09-28  5.76
3    2014-09-26  5.76
4    2014-09-25  5.76
..          ...   ...
231  2013-10-31  5.72
232  2013-10-30  5.72
233  2013-10-29  5.72
234  2013-10-28  5.72
235  2013-10-25  5.71
```

##### 社会融资规模增量统计

接口: China_ISFC

描述: 商务数据中心-国内贸易-社会融资规模增量统计, 数据区间从 201501-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称                        | 类型    | 描述                       |
| --------------------------- | ------- | -------------------------- |
| 月份                        | object  | 年月                       |
| 社会融资规模增量            | float64 | 注意单位: 亿元             |
| 其中-人民币贷款             | float64 | 注意单位: 亿元             |
| 其中-委托贷款外币贷款       | float64 | 注意单位: 折合人民币, 亿元 |
| 其中-委托贷款               | float64 | 注意单位: 亿元             |
| 其中-信托贷款               | float64 | 注意单位: 亿元             |
| 其中-未贴现银行承兑汇票     | float64 | 注意单位: 亿元             |
| 其中-企业债券               | float64 | 注意单位: 亿元             |
| 其中-非金融企业境内股票融资 | float64 | 注意单位: 亿元             |

接口示例

```python
import PPshare as pp

China_ISFC_df = pp.China_ISFC()
print(China_ISFC_df)
```

数据示例

```
        月份  社会融资规模增量  其中-人民币贷款  ...  其中-未贴现银行承兑汇票  其中-企业债券  其中-非金融企业境内股票融资
0   201501     20516     14708  ...          1946     1868           526.0
1   201502     13609     11437  ...          -592      716           542.0
2   201503     12433      9920  ...          -910     1344           639.0
3   201504     10582      8045  ...           -74     1616           597.0
4   201505     12397      8510  ...           961     1710           584.0
..     ...       ...       ...  ...           ...      ...             ...
82  202111     25983     13021  ...          -383     4006          1294.0
83  202112     23682     10350  ...         -1419     2167          2075.0
84  202201     61750     41988  ...          4733     5829          1439.0
85  202202     11928      9084  ...         -4228     3377           585.0
86  202203     46531     32291  ...           287     3573           958.0
```

##### 中国 GDP 年率

接口: China_GDP_yearly

描述: 金十数据中心-中国GDP年率报告, 数据区间从 20110120-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称  | 类型    | 描述 |
| ----- | ------- | ---- |
| index | object  | 年月 |
| value | float64 | 年率 |

接口示例

```python
import PPshare as pp

China_GDP_yearly_df = pp.China_GDP_yearly()
print(China_GDP_yearly_df)
```

数据示例

```
          date  value
0   2011-01-20    9.8
1   2011-04-15    9.7
2   2011-07-13    9.5
3   2011-10-18    9.1
4   2012-01-17    8.9
5   2012-04-13    8.1
6   2012-07-13    7.6
7   2012-10-18    7.4
8   2013-01-18    7.9
9   2013-04-15    7.7
10  2013-07-15    7.5
11  2013-10-18    7.8
12  2014-01-20    7.7
13  2014-04-16    7.4
14  2014-07-16    7.5
15  2014-10-21    7.3
16  2015-01-20    7.3
17  2015-04-15    7.0
18  2015-07-15    7.0
19  2015-10-19    6.9
20  2016-01-19    6.8
21  2016-04-15    6.7
22  2016-07-15    6.7
23  2016-10-19    6.7
24  2017-01-20    6.8
25  2017-04-17    6.9
26  2017-07-17    6.9
28  2017-10-19    6.8
30  2018-01-18    6.8
32  2018-04-17    6.8
34  2018-07-16    6.7
36  2018-10-19    6.5
38  2019-01-21    6.4
40  2019-04-17    6.4
42  2019-07-15    6.2
44  2019-10-18    6.0
46  2020-01-17    6.0
48  2020-04-17   -6.8
50  2020-07-16    3.2
52  2020-10-19    4.9
54  2021-01-18    6.5
56  2021-04-16   18.3
58  2021-07-15    7.9
59  2021-10-18    4.9
60  2022-01-17    4.0
61  2022-04-18    4.8
```

#### 物价水平

##### 中国 CPI 年率报告

接口: China_CPI_yearly

描述: 中国年度 CPI 数据, 数据区间从 19860201-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称  | 类型    | 描述 |
| ----- | ------- | ---- |
| index | object  | 日期 |
| value | float64 | 年率 |

接口示例

```python
import PPshare as pp

China_CPI_yearly_df = pp.China_CPI_yearly()
print(China_CPI_yearly_df)
```

数据示例

```
           date  value
0    1986-02-01    7.1
1    1986-03-01    7.1
2    1986-04-01    7.1
3    1986-05-01    7.1
4    1986-06-01    7.1
..          ...    ...
434  2022-04-11    1.5
435  2022-05-11    2.1
436  2022-06-10    2.1
437  2022-07-09    2.5
438  2022-08-10    2.7
```

##### 中国 CPI 月率报告

接口: China_CPI_monthly

描述: 中国月度 CPI 数据, 数据区间从 19960201-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| 无   | 无   | 无   |

输出参数

| 名称  | 类型  | 描述      |
| ----- | ----- | --------- |
| index | str   | 日期-索引 |
| value | float | 月率      |

接口示例

```python
import PPshare as pp

China_CPI_monthly_df = pp.China_CPI_monthly()
print(China_CPI_monthly_df)
print(China_CPI_monthly_df.name)
```

数据示例

China_CPI_monthly_df.name:

```
CPI
```

China_CPI_monthly_df:

```
1996-02-01     2.1
1996-03-01     2.3
1996-04-01     0.6
1996-05-01     0.7
2019-07-10    -0.1
2019-08-09     0.4
2019-09-10     0.7
2019-10-15     0.9
2019-11-09     0.9
2019-12-10     0.4
2020-01-09       0
```

##### 中国 PPI 年率报告

接口: China_PPI_yearly

描述: 获取中国年度PPI数据, 数据区间从19950801-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称  | 类型    | 描述      |
| ----- | ------- | --------- |
| index | object  | 日期-索引 |
| value | float64 | 年率      |

接口示例

```python
import PPshare as pp

China_PPI_yearly_df = pp.China_PPI_yearly()
print(China_PPI_yearly_df)
print(China_PPI_yearly_df.name)
```

数据示例

China_PPI_yearly_df.name:

```
PPI
```

China_PPI_yearly_df:

```
1995-08-01    13.5
1995-09-01      13
1995-10-01    12.9
1995-11-01    12.5
1995-12-01    11.1
2019-08-09    -0.3
2019-09-10    -0.8
2019-10-15    -1.2
2019-11-09    -1.6
2019-12-10    -1.4
2020-01-09    -0.5
```

### 贸易状况

#### 以美元计算出口年率

接口: China_exports_yoy

描述: 中国以美元计算出口年率报告, 数据区间从 19820201-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称  | 类型  | 默认显示 | 描述      |
| ----- | ----- | -------- | --------- |
| index | str   | Y        | 日期-索引 |
| value | float | Y        | 年率      |

接口示例

```python
import PPshare as pp

China_exports_yoy_df = pp.China_exports_yoy()
print(China_exports_yoy_df)
print(China_exports_yoy_df.name)
```

数据示例

China_exports_yoy_df.name:

```
china_exports_yoy
```

China_exports_yoy_df:

```
1982-02-01      8.7
1982-03-01     23.2
1982-04-01     12.2
1982-05-01     -2.5
1982-06-01     41.5
2019-03-08    -20.8
2019-04-12     14.2
2019-05-08     -2.7
2019-06-10      1.1
2019-07-12     -1.3
2019-08-08      3.3
2019-09-08       -1
2019-10-14     -3.2
2019-11-08     -0.9
2019-12-08     -1.1
2020-01-14        0
```

#### 以美元计算进口年率

接口: China_imports_yoy

描述: 获取中国以美元计算进口年率报告, 数据区间从19960201-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称  | 类型  | 默认显示 | 描述      |
| ----- | ----- | -------- | --------- |
| index | str   | Y        | 日期-索引 |
| value | float | Y        | 年率      |

接口示例

```python
import PPshare as pp

China_imports_yoy_df = pp.China_imports_yoy()
print(China_imports_yoy_df)
print(China_imports_yoy_df.name)
```

数据示例

China_imports_yoy_df.name:

```
china_imports_yoy
```

China_imports_yoy_df:

```
1996-02-01    55.8
1996-03-01    14.2
1996-04-01     8.7
1996-05-01     6.4
1996-06-01     4.5
2019-06-10    -8.5
2019-07-12    -7.3
2019-08-08    -5.6
2019-09-08    -5.6
2019-10-14    -8.5
2019-11-08    -6.5
2019-12-08     0.3
2020-01-14       0
```

#### 以美元计算贸易帐(亿美元)

接口: China_trade_balance

描述: 获取中国以美元计算贸易帐报告, 数据区间从19810201-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称  | 类型  | 默认显示 | 描述      |
| ----- | ----- | -------- | --------- |
| index | str   | Y        | 日期-索引 |
| value | float | Y        | 年率      |

接口示例

```python
import PPshare as pp

China_trade_balance_df = pp.China_trade_balance()
print(China_trade_balance_df)
print(China_trade_balance_df.name)
```

数据示例

China_trade_balance_df.name:

```
china_trade_balance
```

China_trade_balance_df:

```
1981-02-01     -2.2
1981-03-01     -4.9
1981-04-01     -7.4
1981-05-01     -4.8
2019-10-14    396.5
2019-11-08    430.2
2019-12-08    387.3
2019-12-09        0
2020-01-14        0
```

### 产业指标

#### 工业增加值增长

接口: China_gyzjz

描述: 东方财富-中国工业增加值增长, 数据区间从 2008 - 至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| 无   | 无   | 无   |

输出参数

| 名称     | 类型    | 描述        |
| -------- | ------- | ----------- |
| 月份     | object  | -           |
| 同比增长 | float64 | 注意单位: % |
| 累计增长 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_gyzjz_df = pp.China_gyzjz
print(China_gyzjz_df)
```

数据示例

```
     月份  同比增长  累计增长
0    2021-05-01   8.8  17.8
1    2021-04-01   9.8  20.3
2    2021-03-01  14.1  24.5
3    2021-02-01   0.0  35.1
4    2020-12-01   7.3   2.8
..          ...   ...   ...
142  2008-06-01  16.0  16.3
143  2008-05-01  16.0  16.3
144  2008-04-01  15.7  16.3
145  2008-03-01  17.8  16.4
146  2008-02-01  15.4  15.4
```

#### 规模以上工业增加值年率

接口: China_industrial_production_yoy

描述: 获取中国规模以上工业增加值年率报告, 数据区间从19900301-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称  | 类型  | 默认显示 | 描述      |
| ----- | ----- | -------- | --------- |
| index | str   | Y        | 日期-索引 |
| value | float | Y        | 年率      |

接口示例

```python
import PPshare as pp

China_industrial_production_yoy_df = pp.China_industrial_production_yoy()
print(China_industrial_production_yoy_df)
print(China_industrial_production_yoy_df.name)
```

数据示例

China_industrial_production_yoy_df.name:

```
china_industrial_production_yoy
```

China_industrial_production_yoy_df:

```
1990-03-01      5
1990-04-01    0.8
1990-05-01    1.7
1990-06-01    3.3
1990-07-01      5
             ...
2019-09-16    4.4
2019-10-18    5.8
2019-11-14    4.7
2019-12-16    6.2
2020-01-17      0
```

#### 官方制造业PMI

接口: China_pmi_yearly

描述: 获取中国年度PMI数据, 数据区间从20050201-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称  | 类型  | 默认显示 | 描述      |
| ----- | ----- | -------- | --------- |
| index | str   | Y        | 日期-索引 |
| value | float | Y        | 年率      |

接口示例

```python
import PPshare as pp

China_pmi_yearly_df = pp.China_pmi_yearly()
print(China_pmi_yearly_df)
print(China_pmi_yearly_df.name)
```

数据示例

China_pmi_yearly_df.name:

```
pmi
```

China_pmi_yearly_df:

```
2005-02-01    54.7
2005-03-01    54.5
2005-04-01    57.9
2005-05-01    56.7
2005-06-01    52.9
              ...
2019-09-30    49.8
2019-10-31    49.3
2019-11-30    50.2
2019-12-31    50.2
2020-01-31       0
```

#### 财新制造业PMI终值

接口: China_cx_pmi_yearly

描述: 获取中国年度财新 PMI 数据, 数据区间从20120120-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称  | 类型  | 默认显示 | 描述      |
| ----- | ----- | -------- | --------- |
| index | str   | Y        | 日期-索引 |
| value | float | Y        | 年率      |

接口示例

```python
import PPshare as pp

China_cx_pmi_yearly_df = pp.China_cx_pmi_yearly()
print(China_cx_pmi_yearly_df)
print(China_cx_pmi_yearly_df.name)
```

数据示例

China_cx_pmi_yearly_df.name:

```
cx_pmi
```

China_cx_pmi_yearly_df:

```
2012-01-20    48.8
2012-02-22    49.6
2012-03-22    48.3
2012-04-23    49.1
2012-05-02    49.3
              ...
2019-09-02    50.4
2019-09-30    51.4
2019-11-01    51.7
2019-12-02    51.8
2020-01-02    51.5
```

#### 财新服务业PMI

接口: China_cx_services_pmi_yearly

描述: 获取中国财新服务业PMI报告, 数据区间从20120405-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称  | 类型  | 默认显示 | 描述      |
| ----- | ----- | -------- | --------- |
| index | str   | Y        | 日期-索引 |
| value | float | Y        | 年率      |

接口示例

```python
import PPshare as pp

China_cx_services_pmi_yearly_df = pp.China_cx_services_pmi_yearly()
print(China_cx_services_pmi_yearly_df)
print(China_cx_services_pmi_yearly_df.name)
```

数据示例

China_cx_services_pmi_yearly_df.name:

```
cx_services_pmi
```

China_cx_services_pmi_yearly_df:

```
2012-04-05    53.3
2012-05-04    54.1
2012-06-05    54.7
2012-07-04    52.3
2012-08-03    53.1
              ...
2019-09-04    52.1
2019-10-08    51.3
2019-11-05    51.1
2019-12-04    53.5
2020-01-06    52.5
```

#### 中国官方非制造业PMI

接口: China_non_man_pmi

描述: 获取中国官方非制造业PMI, 数据区间从20160101-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称  | 类型  | 默认显示 | 描述      |
| ----- | ----- | -------- | --------- |
| index | str   | Y        | 日期-索引 |
| value | float | Y        | 年率      |

接口示例

```python
import PPshare as pp

China_non_man_pmi_df = pp.China_non_man_pmi()
print(China_non_man_pmi_df)
print(China_non_man_pmi_df.name)
```

数据示例

China_non_man_pmi_df.name:

```
cx_services_pmi
```

China_non_man_pmi_df:

```
2007-02-01    60.4
2007-03-01    60.6
2007-04-01    58.2
2007-05-01    60.4
2007-06-01    62.2
              ...
2019-09-30    53.7
2019-10-31    52.8
2019-11-30    54.4
2019-12-31    53.5
2020-01-31       0
```

### 金融指标

#### 外汇储备(亿美元)

接口: China_fx_reserves_yearly

描述: 获取中国年度外汇储备数据, 数据区间从20140115-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称  | 类型  | 默认显示 | 描述      |
| ----- | ----- | -------- | --------- |
| index | str   | Y        | 日期-索引 |
| value | float | Y        | 年率      |

接口示例

```python
import PPshare as pp

China_fx_reserves_yearly_df = pp.China_fx_reserves_yearly()
print(China_fx_reserves_yearly_df)
print(China_fx_reserves_yearly_df.name)
```

数据示例

China_fx_reserves_yearly_df.name:

```
fx_reserves
```

China_fx_reserves_yearly_df:

```
2014-01-15    39500
2014-07-15    39900
2015-01-15    32300
2016-03-07    32020
2016-04-07    32100
              ...
2019-09-07    31070
2019-10-08    30920
2019-11-07    31050
2019-12-07    30960
2020-01-07    31080
```

#### M2货币供应年率

接口: China_M2_yearly

描述: 获取中国年度M2数据, 数据区间从19980201-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称  | 类型  | 默认显示 | 描述      |
| ----- | ----- | -------- | --------- |
| index | str   | Y        | 日期-索引 |
| value | float | Y        | 年率      |

接口示例

```python
import PPshare as pp

China_M2_yearly_df = pp.China_M2_yearly()
print(China_M2_yearly_df)
print(China_M2_yearly_df.name)
```

数据示例

China_M2_yearly_df.name:

```
M2
```

China_M2_yearly_df:

```
1998-02-01    17.4
1998-03-01    16.7
1998-04-01    15.4
1998-05-01    14.6
1998-06-01    15.5
              ...
2019-11-11     8.4
2019-11-14       0
2019-12-10     8.2
2019-12-13       0
2020-01-14       0
```

#### 新房价指数

接口: China_new_house_price

描述: 中国新房价指数月度数据, 数据区间从 201101-至今

限量: 单次返回指定城市的所有历史数据

输入参数

| 名称        | 类型 | 描述                                   |
| ----------- | ---- | -------------------------------------- |
| city_first  | str  | city_first="北京"; 城市列表见目标网站  |
| city_second | str  | city_second="上海"; 城市列表见目标网站 |

输出参数

| 名称                      | 类型    | 描述 |
| ------------------------- | ------- | ---- |
| 日期                      | object  | 日期 |
| 城市                      | object  | -    |
| 新建商品住宅价格指数-环比 | float64 | -    |
| 新建商品住宅价格指数-同比 | float64 | -    |
| 新建商品住宅价格指数-定基 | float64 | -    |
| 二手住宅价格指数-环比     | float64 | -    |
| 二手住宅价格指数-同比     | float64 | -    |
| 二手住宅价格指数-定基     | float64 | -    |

接口示例

```python
import PPshare as pp

China_new_house_price_df = pp.China_new_house_price()
print(China_new_house_price_df)
```

数据示例

```
             日期  城市  新建商品住宅价格指数-同比  ...  二手住宅价格指数-环比  二手住宅价格指数-同比  二手住宅价格指数-定基
0    2022-10-01  北京          105.9  ...        105.2        100.1        116.7
1    2022-10-01  上海          104.0  ...        103.9         99.6        113.7
2    2022-09-01  北京          106.1  ...        104.6        100.4        116.5
3    2022-09-01  上海          103.8  ...        103.9        100.5        114.2
4    2022-08-01  北京          105.8  ...        103.9        100.2        116.0
..          ...  ..            ...  ...          ...          ...          ...
279  2011-03-01  上海          102.0  ...        100.5        100.4        101.4
280  2011-02-01  北京          108.4  ...        102.9        100.4        101.5
281  2011-02-01  上海          102.8  ...        102.0        100.4        101.0
282  2011-01-01  北京          109.1  ...        102.6        100.3        101.2
283  2011-01-01  上海          101.8  ...        101.7        100.5        100.6
```

#### 企业景气及企业家信心指数

接口: China_enterprise_boom_index

描述: 中国企业景气及企业家信心指数数据, 数据区间从 2005 一季度-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称                | 类型    | 描述        |
| ------------------- | ------- | ----------- |
| 季度                | object  | 日期        |
| 企业景气指数-指数   | float64 | -           |
| 企业景气指数-同比   | float64 | 注意单位: % |
| 企业景气指数-环比   | float64 | 注意单位: % |
| 企业家信心指数-指数 | float64 | -           |
| 企业家信心指数-同比 | float64 | 注意单位: % |
| 企业家信心指数-环比 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_enterprise_boom_index_df = pp.China_enterprise_boom_index()
print(China_enterprise_boom_index_df)
```

数据示例

```
           季度  企业景气指数-指数  企业景气指数-同比  ...  企业家信心指数-指数  企业家信心指数-同比  企业家信心指数-环比
0   2022年第2季度     101.80       1.80  ...         NaN         NaN         NaN
1   2022年第1季度     112.70      12.70  ...         NaN         NaN         NaN
2   2021年第4季度     119.20      19.20  ...         NaN         NaN         NaN
3   2021年第3季度     119.20      19.20  ...      120.90       20.90       -5.10
4   2021年第2季度     123.80      23.80  ...      126.00       26.00       -1.80
..        ...        ...        ...  ...         ...         ...         ...
65  2006年第1季度     131.50      31.50  ...      133.10       33.10        7.70
66  2005年第4季度     131.70      31.70  ...      125.40       25.40       -2.20
67  2005年第3季度     132.00      32.00  ...      127.60       27.60       -0.90
68  2005年第2季度     131.71      31.71  ...      128.50       28.50       -7.35
69  2005年第1季度     132.46      32.46  ...      135.85       35.85        5.04
```

#### 全国税收收入

接口: China_national_tax_receipts

描述: 中国全国税收收入数据, 数据区间从 2005 一季度-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称         | 类型    | 描述           |
| ------------ | ------- | -------------- |
| 季度         | object  | 日期           |
| 税收收入合计 | float64 | 注意单位: 亿元 |
| 较上年同期   | float64 | 注意单位: %    |
| 季度环比     | float64 | -              |

接口示例

```python
import PPshare as pp

China_national_tax_receipts_df = pp.China_national_tax_receipts()
print(China_national_tax_receipts_df)
```

数据示例

```
             季度     税收收入合计  较上年同期      季度环比
0   2022年第1-3季度  124365.00  -11.6  0.171811
1   2022年第1-2季度   85564.00  -14.8 -0.368718
2     2022年第1季度   52452.00    7.7  0.637641
3   2021年第1-4季度  172731.00   11.9 -0.204070
4   2021年第1-3季度  140702.00   18.4 -0.222216
..          ...        ...    ...       ...
64  2006年第1-2季度   18484.80   22.0  0.142837
65    2006年第1季度    8626.32   19.0       NaN
66  2005年第1-3季度   21855.35   15.9 -0.151249
67  2005年第1-2季度   15149.73   13.4  0.089860
68    2005年第1季度    7249.16   10.7       NaN
```

#### 银行理财产品发行数量

接口: China_bank_financing

描述: 银行理财产品发行数量, 数据区间从 2000 一月-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_bank_financing_df = pp.China_bank_financing()
print(China_bank_financing_df)
```

数据示例

```
        日期   最新值         涨跌幅  ...   近1年涨跌幅   近2年涨跌幅  近3年涨跌幅
0    2000-03-01     4         NaN  ...         NaN         NaN         NaN
1    2004-02-01     2  -50.000000  ...  -50.000000  -50.000000  -50.000000
2    2004-03-01     8  300.000000  ...  100.000000  100.000000  100.000000
3    2004-04-01     3  -62.500000  ...  -25.000000  -25.000000  -25.000000
4    2004-05-01     6  100.000000  ...   50.000000   50.000000   50.000000
..          ...   ...         ...  ...         ...         ...         ...
213  2021-10-01  2640  -27.829415  ...  -51.710262  -70.054446  -74.346516
214  2021-11-01  3424   29.696970  ...  -45.347167  -59.541534  -70.644719
215  2021-12-01  3876   13.200935  ...  -44.422139  -44.117647  -64.904020
216  2022-01-01  2793  -27.941176  ...  -45.406568  -50.478723  -77.035027
217  2022-02-01  1779  -36.305048  ...  -54.893509  -66.204407  -78.496313
```

#### 原保险保费收入

接口: China_insurance_income

描述: 原保险保费收入, 数据区间从 200407-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_insurance_income_df = pp.China_insurance_income()
print(China_insurance_income_df)
```

数据示例

```
      日期        最新值        涨跌幅  ...    近1年涨跌幅    近2年涨跌幅     近3年涨跌幅
0    2004-07-01  0.000000e+00        NaN  ...       NaN       NaN        NaN
1    2004-09-01  2.453616e+07        NaN  ...       NaN       NaN        NaN
2    2004-10-01  2.704315e+07  10.217507  ...       NaN       NaN        NaN
3    2004-11-01  2.956397e+07   9.321471  ...       NaN       NaN        NaN
4    2004-12-01  3.193586e+07   8.022916  ...       NaN       NaN        NaN
..          ...           ...        ...  ...       ...       ...        ...
205  2021-09-01  1.979124e+08   9.327015  ... -1.959838  2.790277  -9.435956
206  2021-10-01  2.094363e+08   5.822728  ... -2.107731  2.866552 -10.337719
207  2021-11-01  2.205047e+08   5.284853  ... -2.053675  2.876131 -11.053454
208  2021-12-01  2.357184e+08   6.899490  ... -1.710283  3.594269 -10.239691
209  2022-01-01  7.049640e+07 -70.092958  ... -5.768089  5.429366  12.398597
```

#### 手机出货量

接口: China_mobile_number

描述: 手机出货量, 数据区间从 201201-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_mobile_number_df = pp.China_mobile_number()
print(China_mobile_number_df)
```

数据示例

```
        日期     最新值     涨跌幅  ...     近1年涨跌幅      近2年涨跌幅     近3年涨跌幅
0    2012-01-01  2400.6        NaN  ...        NaN         NaN        NaN
1    2012-02-01  2998.9  24.922936  ...        NaN         NaN        NaN
2    2012-03-01  3958.6  32.001734  ...        NaN         NaN        NaN
3    2012-04-01  3528.7 -10.859900  ...        NaN         NaN        NaN
4    2012-05-01  3586.8   1.646499  ...        NaN         NaN        NaN
..          ...     ...        ...  ...        ...         ...        ...
117  2021-10-01  3357.5  56.599813  ...  28.379153   -6.655731 -12.866893
118  2021-11-01  3525.2   4.994788  ...  19.159005    1.176741  -0.333616
119  2021-12-01  3340.1  -5.250766  ...  25.591277    9.712916  -6.371587
120  2022-01-01  3302.2  -1.134697  ... -17.691924   58.660453  -3.013393
121  2022-02-01  1486.4 -54.987584  ... -31.688037  132.832080   2.432637
```

#### 菜篮子产品批发价格指数

接口: China_vegetable_basket

描述: 菜篮子产品批发价格指数, 数据区间从 20050927-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_vegetable_basket_df = pp.China_vegetable_basket()
print(China_vegetable_basket_df)
```

数据示例

```
        日期     最新值       涨跌幅  ...    近1年涨跌幅  近2年涨跌幅  近3年涨跌幅
0     2005-09-27  123.80       NaN  ...       NaN       NaN        NaN
1     2005-09-28  123.00 -0.646204  ...       NaN       NaN        NaN
2     2005-09-29  123.10  0.081301  ...       NaN       NaN        NaN
3     2005-09-30  124.10  0.812348  ...       NaN       NaN        NaN
4     2005-10-08  122.20 -1.531023  ...       NaN       NaN        NaN
          ...     ...       ...  ...       ...       ...        ...
4100  2022-03-28  137.93  0.561388  ...  6.026597  6.608440  15.470908
4101  2022-03-29  138.45  0.377003  ...  6.952491  7.010357  15.712495
4102  2022-03-30  138.85  0.288913  ...  7.410846  7.978847  16.046803
4103  2022-03-31  139.38  0.381707  ...  8.180689  8.289954  16.489762
4104  2022-04-01  139.70  0.229588  ...  8.775208  8.622969  16.484616
```

#### 农产品批发价格总指数

接口: China_agricultural_product

描述: 农产品批发价格总指数, 数据区间从 20050927-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_agricultural_product_df = pp.China_agricultural_product()
print(China_agricultural_product_df)
```

数据示例

```
          日期     最新值    涨跌幅  ...    近1年涨跌幅  近2年涨跌幅  近3年涨跌幅
0     2005-09-27  125.50       NaN  ...       NaN       NaN        NaN
1     2005-09-28  125.00 -0.398406  ...       NaN       NaN        NaN
2     2005-09-29  125.00  0.000000  ...       NaN       NaN        NaN
3     2005-09-30  125.80  0.640000  ...       NaN       NaN        NaN
4     2005-10-08  124.20 -1.271860  ...       NaN       NaN        NaN
          ...     ...       ...  ...       ...       ...        ...
4100  2022-03-28  134.83  0.499404  ...  6.349582  7.374373  15.259019
4101  2022-03-29  135.30  0.348587  ...  7.100451  7.748666  15.492958
4102  2022-03-30  135.03 -0.199557  ...  7.098668  8.127803  15.262484
4103  2022-03-31  136.10  0.792416  ...  8.265054  8.862582  16.175843
4104  2022-04-01  136.38  0.205731  ...  8.790683  9.156395  16.176846
```

#### 农副指数

接口: China_agricultural_index

描述: 农副指数数据, 数据区间从 20111205-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_agricultural_index_df = pp.China_agricultural_index()
print(China_agricultural_index_df)
```

数据示例

```
          日期   最新值   涨跌幅  ...    近1年涨跌幅   近2年涨跌幅   近3年涨跌幅
0     2011-12-05   995       NaN  ...       NaN        NaN        NaN
1     2011-12-12   986 -0.904523  ...       NaN        NaN        NaN
2     2011-12-19   990  0.405680  ...       NaN        NaN        NaN
3     2011-12-26   988 -0.202020  ...       NaN        NaN        NaN
4     2012-01-02   993  0.506073  ...       NaN        NaN        NaN
          ...   ...       ...  ...       ...        ...        ...
2971  2022-03-28  1316 -0.679245  ...  7.428571  24.385633  55.739645
2972  2022-03-29  1311 -0.379939  ...  7.635468  23.913043  55.331754
2973  2022-03-30  1305 -0.457666  ...  7.495881  22.881356  54.620853
2974  2022-03-31  1303 -0.153257  ...  8.222591  22.347418  54.383886
2975  2022-04-01  1289 -1.074444  ...  6.090535  21.489161  52.906287
```

#### 能源指数

接口: China_energy_index

描述: 能源指数数据, 数据区间从 20111205-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_energy_index_df = pp.China_energy_index()
print(China_energy_index_df)
```

数据示例

```
         日期   最新值    涨跌幅  ...     近1年涨跌幅    近2年涨跌幅  近3年涨跌幅
0     2011-12-05  1003       NaN  ...        NaN        NaN        NaN
1     2011-12-12   995 -0.797607  ...        NaN        NaN        NaN
2     2011-12-19   987 -0.804020  ...        NaN        NaN        NaN
3     2011-12-26   983 -0.405268  ...        NaN        NaN        NaN
4     2012-01-02   984  0.101729  ...        NaN        NaN        NaN
          ...   ...       ...  ...        ...        ...        ...
2972  2022-03-29  1208 -0.247729  ...  48.220859  84.992343  48.768473
2973  2022-03-30  1206 -0.165563  ...  47.252747  88.437500  48.522167
2974  2022-03-31  1207  0.082919  ...  47.735618  90.378549  48.645320
2975  2022-04-01  1207  0.000000  ...  48.098160  91.283677  48.828607
2976  2022-04-02  1208  0.082850  ...  47.858017  91.442155  47.858017
```

#### 大宗商品价格

接口: China_commodity_price_index

描述: 大宗商品价格数据, 数据区间从 20111205-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_commodity_price_index_df = pp.China_commodity_price_index()
print(China_commodity_price_index_df)
```

数据示例

```
          日期   最新值    涨跌幅  ...     近1年涨跌幅   近2年涨跌幅   近3年涨跌幅
0     2011-12-05   999       NaN  ...        NaN        NaN        NaN
1     2011-12-12   991 -0.800801  ...        NaN        NaN        NaN
2     2011-12-19   990 -0.100908  ...        NaN        NaN        NaN
3     2011-12-26   988 -0.202020  ...        NaN        NaN        NaN
4     2012-01-02   992  0.404858  ...        NaN        NaN        NaN
          ...   ...       ...  ...        ...        ...        ...
2975  2022-03-31  1213  0.000000  ...  21.543086  68.005540  40.556199
2976  2022-04-01  1212 -0.082440  ...  20.958084  69.037657  40.277778
2977  2022-04-02  1212  0.000000  ...  20.717131  70.224719  40.440324
2978  2022-04-03  1212  0.000000  ...  20.837488  70.464135  40.440324
2979  2022-04-04  1212  0.000000  ...  20.837488  70.704225  40.440324
```

#### 费城半导体指数

接口: macro_global_sox_index

描述: 费城半导体指数数据, 数据区间从 19940504-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

macro_global_sox_index_df = pp.macro_global_sox_index()
print(macro_global_sox_index_df)
```

数据示例

```
         日期     最新值    涨跌幅  ...     近1年涨跌幅   近2年涨跌幅      近3年涨跌幅
0     1994-05-04   119.0       NaN  ...        NaN         NaN         NaN
1     1994-05-05   118.3 -0.588235  ...        NaN         NaN         NaN
2     1994-05-06   117.7 -0.507185  ...        NaN         NaN         NaN
3     1994-05-09   115.6 -1.784197  ...        NaN         NaN         NaN
4     1994-05-10   117.4  1.557093  ...        NaN         NaN         NaN
          ...     ...       ...  ...        ...         ...         ...
6957  2022-03-25  3525.3 -0.209472  ...  19.102801  139.397788  154.948472
6958  2022-03-28  3547.2  0.621224  ...  14.194471  138.267003  158.203523
6959  2022-03-29  3625.6  2.210194  ...  18.691953  143.533165  159.804659
6960  2022-03-30  3508.6 -3.227052  ...  15.241990  127.048295  151.420628
6961  2022-03-31  3429.0 -2.268711  ...   9.730107  126.997398  145.716620
```

#### 义乌小商品指数-电子元器件

接口: China_yw_electronic_index

描述: 义乌小商品指数-电子元器件数据, 数据区间从 20060911-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_yw_electronic_index_df = pp.China_yw_electronic_index()
print(China_yw_electronic_index_df)
```

数据示例

```
        日期     最新值   涨跌幅    近3月涨跌幅 近6月涨跌幅 近1年涨跌幅  近2年涨跌幅 近3年涨跌幅
0    2006-09-11  101.34       NaN       NaN     NaN     NaN       NaN     NaN
1    2006-09-18  102.94  1.578844       NaN     NaN     NaN       NaN     NaN
2    2006-09-25  103.34  0.388576       NaN     NaN     NaN       NaN     NaN
3    2006-10-02  103.21 -0.125798       NaN     NaN     NaN       NaN     NaN
4    2006-10-09  103.06 -0.145335       NaN     NaN     NaN       NaN     NaN
..          ...     ...       ...       ...     ...     ...       ...     ...
755  2022-03-07  100.18 -0.169407  0.089919    0.18    0.18 -0.802060    0.18
756  2022-03-14  100.09 -0.089838  0.039980    0.09    0.09 -0.407960    0.09
757  2022-03-21  106.11  6.014587  0.463927    6.11    6.11  5.845387    6.11
758  2022-03-28  103.06 -2.874376  0.243167    3.06    3.06  2.936476    3.06
759  2022-04-04  101.53 -1.484572  0.118332    1.53    1.53  1.469119    1.53
```

#### 建材指数

接口: China_construction_index

描述: 建材指数数据, 数据区间从 20111205-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_construction_index_df = pp.China_construction_index()
print(China_construction_index_df)
```

数据示例

```
       日期   最新值       涨跌幅  ...   近1年涨跌幅  近2年涨跌幅  近3年涨跌幅
0     2011-12-05  1014       NaN  ...       NaN        NaN       NaN
1     2011-12-12  1013 -0.098619  ...       NaN        NaN       NaN
2     2011-12-19  1005 -0.789733  ...       NaN        NaN       NaN
3     2011-12-26   993 -1.194030  ...       NaN        NaN       NaN
4     2012-01-02   991 -0.201410  ...       NaN        NaN       NaN
          ...   ...       ...  ...       ...        ...       ...
2455  2022-04-03  1287  0.000000  ...  1.658768  42.209945  6.628003
2456  2022-04-04  1287  0.000000  ...  1.658768  42.209945  6.628003
2457  2022-04-05  1287  0.000000  ...  1.658768  42.209945  6.628003
2458  2022-04-06  1289  0.155400  ...  1.018809  42.430939  6.793703
2459  2022-04-07  1287 -0.155159  ...  0.704225  43.638393  6.628003
```

#### 建材价格指数

接口: China_construction_price_index

描述: 建材价格指数数据, 数据区间从 20100615-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_construction_price_index_df = pp.China_construction_price_index()
print(China_construction_price_index_df)
```

数据示例

```
       日期        最新值    涨跌幅  ...    近1年涨跌幅   近2年涨跌幅  近3年涨跌幅
0     2010-06-15  1230.40       NaN  ...       NaN       NaN       NaN
1     2010-06-16  1245.10  1.194733  ...       NaN       NaN       NaN
2     2010-08-01  1281.60  2.931491  ...       NaN       NaN       NaN
3     2010-08-02  1214.90 -5.204432  ...       NaN       NaN       NaN
4     2010-08-03  1218.90  0.329245  ...       NaN       NaN       NaN
          ...      ...       ...  ...       ...       ...       ...
3841  2022-02-25  1099.01  1.567395  ...  1.467995  1.493295  2.177410
3842  2022-02-26  1088.05 -0.997261  ...  0.456094  0.481142  1.521824
3843  2022-02-27  1097.41  0.860255  ...  1.120479  1.345536  2.198733
3844  2022-02-28  1088.05 -0.852917  ...  0.977244  0.481142  1.549302
3845  2022-03-01  1097.41  0.860255  ...  1.419528  0.749139  2.051425
```

#### 物流景气指数

接口: China_lpi_index

描述: 物流景气指数数据, 数据区间从 20130701-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_lpi_index_df = pp.China_lpi_index()
print(China_lpi_index_df)
```

数据示例

```
         日期   最新值   涨跌幅  ...    近1年涨跌幅   近2年涨跌幅 近3年涨跌幅
0    2013-07-01  52.4       NaN  ...       NaN        NaN       NaN
1    2013-08-01  52.9  0.954198  ...       NaN        NaN       NaN
2    2013-09-01  53.3  0.756144  ...       NaN        NaN       NaN
3    2013-10-01  53.6  0.562852  ...       NaN        NaN       NaN
4    2013-11-01  53.0 -1.119403  ...       NaN        NaN       NaN
..          ...   ...       ...  ...       ...        ...       ...
100  2021-11-01  53.6  0.186916  ... -6.782609  -8.998302  7.200000
101  2021-12-01  52.6 -1.865672  ... -7.557118 -10.238908 -3.839122
102  2022-01-01  51.1 -2.851711  ... -6.066176   2.404810 -4.841713
103  2022-02-01  51.2  0.195695  ...  2.811245  95.419847  3.225806
104  2022-03-01  48.7 -4.882812  ... -6.165703  -5.436893 -7.414449
```

#### 原油运输指数

接口: China_bdti_index

描述: 原油运输指数数据, 数据区间从 20011227-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_bdti_index_df = pp.China_bdti_index()
print(China_bdti_index_df)
```

数据示例

```
           日期   最新值   涨跌幅  ...      近1年涨跌幅     近2年涨跌幅      近3年涨跌幅
0     2001-12-27   849       NaN  ...         NaN        NaN         NaN
1     2001-12-28   850  0.117786  ...         NaN        NaN         NaN
2     2002-01-02   845 -0.588235  ...         NaN        NaN         NaN
3     2002-01-03   826 -2.248521  ...         NaN        NaN         NaN
4     2002-01-04   811 -1.815981  ...         NaN        NaN         NaN
          ...   ...       ...  ...         ...        ...         ...
4921  2022-04-05  1469  7.304602  ...  110.157368   6.218366  138.087520
4922  2022-04-06  1547  5.309735  ...  125.839416  24.758065  150.729335
4923  2022-04-07  1653  6.851972  ...  144.888889  47.326203  167.909238
4924  2022-04-08  1677  1.451906  ...  154.863222  56.582633  174.918033
4925  2022-04-11  1730  3.160405  ...  167.801858  55.296230  172.870662
```

#### 超灵便型船运价指数

接口: China_bsi_index

描述: 超灵便型船运价指数数据, 数据区间从 20060103-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_bsi_index_df = pp.China_bsi_index()
print(China_bsi_index_df)
```

数据示例

```
          日期   最新值   涨跌幅  ...     近1年涨跌幅    近2年涨跌幅    近3年涨跌幅
0     2006-01-03  1819       NaN  ...        NaN         NaN         NaN
1     2006-01-05  1825  0.329852  ...        NaN         NaN         NaN
2     2006-01-09  1814 -0.602740  ...        NaN         NaN         NaN
3     2006-01-11  1784 -1.653804  ...        NaN         NaN         NaN
4     2006-01-12  1759 -1.401345  ...        NaN         NaN         NaN
          ...   ...       ...  ...        ...         ...         ...
3984  2022-04-06  2605 -2.067669  ...  44.002211  437.113402  251.078167
3985  2022-04-07  2547 -2.226488  ...  42.131696  439.618644  243.261456
3986  2022-04-08  2502 -1.766784  ...  40.089586  451.101322  241.336971
3987  2022-04-11  2473 -1.159073  ...  38.932584  460.770975  245.874126
3988  2022-04-12  2448 -1.010918  ...  37.993236  455.102041  244.303797
```

#### 海岬型运费指数

接口: macro_shiPPIng_bci

描述: 海岬型运费指数, 数据区间从 19990430-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

macro_shiPPIng_bci_df = pp.macro_shiPPIng_bci()
print(macro_shiPPIng_bci_df)
```

数据示例

```
      日期   最新值   涨跌幅  ...     近1年涨跌幅     近2年涨跌幅      近3年涨跌幅
0     1999-04-30   940        NaN  ...        NaN        NaN         NaN
1     1999-05-04   947   0.744681  ...        NaN        NaN         NaN
2     1999-05-05   960   1.372756  ...        NaN        NaN         NaN
3     1999-05-06   969   0.937500  ...        NaN        NaN         NaN
4     1999-05-07   981   1.238390  ...        NaN        NaN         NaN
          ...   ...        ...  ...        ...        ...         ...
5988  2023-03-03  1195  19.500000  ... -27.089689 -28.741801  445.375723
5989  2023-03-06  1329  11.213389  ... -18.715596 -25.504484  525.961538
5990  2023-03-07  1471  10.684725  ... -15.894797 -17.544843  571.474359
5991  2023-03-08  1550   5.370496  ... -18.248945 -15.068493  596.794872
5992  2023-03-09  1662   7.225806  ... -27.169150 -14.769231  546.774194
```

#### 波罗的海干散货指数

接口: macro_shiPPIng_bdi

描述: 波罗的海干散货指数, 数据区间从 19881019-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

macro_shiPPIng_bdi_df = pp.macro_shiPPIng_bdi()
print(macro_shiPPIng_bdi_df)
```

数据示例

```
      日期   最新值    涨跌幅  ...     近1年涨跌幅     近2年涨跌幅      近3年涨跌幅
0     1988-10-19  1317       NaN  ...        NaN        NaN         NaN
1     1988-10-20  1316 -0.075930  ...        NaN        NaN         NaN
2     1988-10-21  1328  0.911854  ...        NaN        NaN         NaN
3     1988-10-24  1361  2.484940  ...        NaN        NaN         NaN
4     1988-10-25  1363  0.146951  ...        NaN        NaN         NaN
          ...   ...       ...  ...        ...        ...         ...
8577  2023-03-03  1211  5.764192  ... -42.442966 -31.310267  120.582878
8578  2023-03-06  1258  3.881090  ... -41.433892 -31.219245  103.889789
8579  2023-03-07  1298  3.179650  ... -41.923937 -29.032258  110.372771
8580  2023-03-08  1327  2.234206  ... -43.579932 -28.386400  115.072934
8581  2023-03-09  1379  3.918613  ... -46.090696 -27.459232  123.863636
```

#### 巴拿马型运费指数

接口: macro_shiPPIng_bpi

描述: 巴拿马型运费指数, 数据区间从 19981231-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

macro_shiPPIng_bpi_df = pp.macro_shiPPIng_bpi()
print(macro_shiPPIng_bpi_df)
```

数据示例

```
     日期   最新值       涨跌幅  ...     近1年涨跌幅     近2年涨跌幅     近3年涨跌幅
0     1998-12-31   732       NaN  ...        NaN        NaN        NaN
1     1999-01-04   717 -2.049180  ...        NaN        NaN        NaN
2     1999-01-05   709 -1.115760  ...        NaN        NaN        NaN
3     1999-01-06   729  2.820874  ...        NaN        NaN        NaN
4     1999-01-07   734  0.685871  ...        NaN        NaN        NaN
          ...   ...       ...  ...        ...        ...        ...
5951  2023-03-03  1565  0.578406  ... -42.015561 -27.579824  67.379679
5952  2023-03-06  1582  1.086262  ... -43.195691 -29.406515  48.127341
5953  2023-03-07  1580 -0.126422  ... -45.536022 -29.495761  47.940075
5954  2023-03-08  1592  0.759494  ... -47.648800 -29.495128  49.063670
5955  2023-03-09  1624  2.010050  ... -49.154665 -27.435210  52.202437
```

#### 成品油运输指数

接口: macro_shiPPIng_bcti

描述: 成品油运输指数, 数据区间从 20011217-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述        |
| ----------- | ------- | ----------- |
| 日期        | object  | -           |
| 最新值      | int64   | -           |
| 涨跌幅      | float64 | 注意单位: % |
| 近3月涨跌幅 | float64 | 注意单位: % |
| 近6月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅 | float64 | 注意单位: % |
| 近2年涨跌幅 | float64 | 注意单位: % |
| 近3年涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

macro_shiPPIng_bcti_df = pp.macro_shiPPIng_bcti()
print(macro_shiPPIng_bcti_df)
```

数据示例

```
    日期  最新值       涨跌幅  ...     近1年涨跌幅     近2年涨跌幅     近3年涨跌幅
0     2001-12-27  693       NaN  ...        NaN        NaN        NaN
1     2001-12-28  691 -0.288600  ...        NaN        NaN        NaN
2     2002-01-02  688 -0.434153  ...        NaN        NaN        NaN
3     2002-01-03  687 -0.145349  ...        NaN        NaN        NaN
4     2002-01-04  687  0.000000  ...        NaN        NaN        NaN
          ...  ...       ...  ...        ...        ...        ...
5148  2023-03-03  789 -1.743462  ... -20.383451  62.012320  22.136223
5149  2023-03-06  782 -0.887199  ... -20.930233  60.245902  14.160584
5150  2023-03-07  784  0.255754  ... -26.797386  60.655738  14.452555
5151  2023-03-08  827  5.484694  ... -21.163012  67.748479  20.729927
5152  2023-03-09  871  5.320435  ... -15.682478  73.161034  21.478382
```

#### 新增信贷数据

接口: China_new_financial_credit

描述: 中国新增信贷数据数据, 数据区间从 200801 至今, 月度数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称          | 类型    | 描述           |
| ------------- | ------- | -------------- |
| 月份          | object  | -              |
| 当月          | float64 | 注意单位: 亿元 |
| 当月-同比增长 | float64 | 注意单位: %    |
| 当月-环比增长 | float64 | 注意单位: %    |
| 累计          | float64 | 注意单位: 亿元 |
| 累计-同比增长 | float64 | 注意单位: %    |

接口示例

```python
import PPshare as pp

China_new_financial_credit_df = pp.China_new_financial_credit()
print(China_new_financial_credit_df)
```

数据示例

```
     月份       当月    当月-同比增长      当月-环比增长        累计    累计-同比增长
0    2022年10月份   4431.0 -42.840557   -82.749358  183298.0   4.127067
1    2022年09月份  25686.0  44.669107    92.491007  178867.0   6.290669
2    2022年08月份  13344.0   4.963423   226.418787  153181.0   1.763815
3    2022年07月份   4088.0 -51.281135   -86.614276  139837.0   1.468657
4    2022年06月份  30540.0  31.740143    67.526056  135749.0   4.888659
..         ...      ...        ...          ...       ...        ...
173  2008年05月份   3185.0  28.790942   -32.089552   21201.0   1.386830
174  2008年04月份   4690.0  11.137441    65.490473   18016.0  -2.288751
175  2008年03月份   2834.0 -35.838805    16.433854   13326.0  -6.273738
176  2008年02月份   2434.0 -41.179314   -69.793994   10492.0   7.050301
177  2008年01月份   8058.0  42.292071  1561.443299    8058.0  42.292071
```

#### 居民消费价格指数

接口: China_CPI

描述: 中国居民消费价格指数, 数据区间从 200801 至今, 月度数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称          | 类型    | 描述        |
| ------------- | ------- | ----------- |
| 月份          | object  | -           |
| 全国-当月     | float64 | -           |
| 全国-同比增长 | float64 | 注意单位: % |
| 全国-环比增长 | float64 | 注意单位: % |
| 全国-累计     | float64 | -           |
| 城市-当月     | float64 | -           |
| 城市-同比增长 | float64 | 注意单位: % |
| 城市-环比增长 | float64 | 注意单位: % |
| 城市-累计     | float64 | -           |
| 农村-当月     | float64 | -           |
| 农村-同比增长 | float64 | 注意单位: % |
| 农村-环比增长 | float64 | 注意单位: % |
| 农村-累计     | float64 | -           |

接口示例

```python
import PPshare as pp

China_CPI_df = pp.China_CPI()
print(China_CPI_df)
```

数据示例

```
     月份     全国-当月  全国-同比增长  ...  农村-同比增长  农村-环比增长     农村-累计
0    2022年10月份  102.1000   2.1000  ...   2.5000      0.1  102.0000
1    2022年09月份  102.8000   2.8000  ...   3.1000      0.4  102.0000
2    2022年08月份  102.5000   2.5000  ...   2.7000     -0.1  101.8000
3    2022年07月份  102.7000   2.7000  ...   3.0000      0.5  101.7000
4    2022年06月份  102.5000   2.5000  ...   2.6000      0.0  101.5000
..         ...       ...      ...  ...      ...      ...       ...
173  2008年05月份  107.7163   7.7163  ...   8.5481     -0.3  108.7612
174  2008年04月份  108.4829   8.4829  ...   9.2737      0.1  108.8147
175  2008年03月份  108.3097   8.3097  ...   9.0330     -0.5  108.6618
176  2008年02月份  108.7443   8.7443  ...   9.2344      2.4  108.4812
177  2008年01月份  107.0781   7.0781  ...   7.7209      1.2  107.7209
```

#### 国内生产总值

接口: China_GDP

描述: 中国国内生产总值, 数据区间从 200601 至今, 月度数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称                  | 类型    | 描述           |
| --------------------- | ------- | -------------- |
| 季度                  | object  | -              |
| 国内生产总值-绝对值   | float64 | 注意单位: 亿元 |
| 国内生产总值-同比增长 | float64 | 注意单位: %    |
| 第一产业-绝对值       | float64 | 注意单位: 亿元 |
| 第一产业-同比增长     | float64 | 注意单位: %    |
| 第二产业-绝对值       | float64 | 注意单位: 亿元 |
| 第二产业-同比增长     | float64 | 注意单位: %    |
| 第三产业-绝对值       | float64 | 注意单位: 亿元 |
| 第三产业-同比增长     | float64 | 注意单位: %    |

接口示例

```python
import PPshare as pp

China_GDP_df = pp.China_GDP()
print(China_GDP_df)
```

数据示例

```
   季度  国内生产总值-绝对值  国内生产总值-同比增长  ...  第二产业-同比增长  第三产业-绝对值  第三产业-同比增长
0   2021-03-01    249310.0         18.3  ...       24.4  145355.0       15.6
1   2020-12-01   1015986.2          2.3  ...        2.6  553976.8        2.1
2   2020-09-01    719688.4          0.7  ...        0.9  401249.1        0.4
3   2020-06-01    454712.1         -1.6  ...       -1.9  258427.4       -1.6
4   2020-03-01    205727.0         -6.8  ...       -9.6  123008.5       -5.2
..         ...         ...          ...  ...        ...       ...        ...
56  2007-03-01     57159.3         13.8  ...       14.8   27703.2       14.1
57  2006-12-01    219438.5         12.7  ...       13.5   91762.2       14.1
58  2006-09-01    155816.8         12.8  ...       13.7   67187.0       13.7
59  2006-06-01     99752.2         13.1  ...       14.2   44996.5       13.6
60  2006-03-01     47078.9         12.5  ...       13.1   22648.0       13.1
```

#### 工业品出厂价格指数

接口: China_PPI

描述: 工业品出厂价格指数, 数据区间从 200601 至今, 月度数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称         | 类型    | 描述        |
| ------------ | ------- | ----------- |
| 月份         | object  | -           |
| 当月         | float64 | -           |
| 当月同比增长 | float64 | 注意单位: % |
| 累计         | float64 | -           |

接口示例

```python
import PPshare as pp

China_PPI_df = pp.China_PPI()
print(China_PPI_df)
```

数据示例

```
            月份       当月  当月同比增长        累计
0    2022年10月份   98.700   -1.30  105.2000
1    2022年09月份  100.948    0.90  105.9397
2    2022年08月份  102.300    2.30  106.6000
3    2022年07月份  104.200    4.20  107.2000
4    2022年06月份  106.100    6.10  107.7000
..         ...      ...     ...       ...
197  2006年05月份  102.430    2.43  102.5700
198  2006年04月份  101.870    1.87  102.6000
199  2006年03月份  102.490    2.49  102.9000
200  2006年02月份  103.010    3.01  103.0000
201  2006年01月份  103.050    3.05  103.0500
```

#### 采购经理人指数

接口: China_pmi

描述: 采购经理人指数, 数据区间从 200801 至今, 月度数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称              | 类型    | 描述        |
| ----------------- | ------- | ----------- |
| 月份              | object  | -           |
| 制造业-指数       | float64 | -           |
| 制造业-同比增长   | float64 | 注意单位: % |
| 非制造业-指数     | float64 | -           |
| 非制造业-同比增长 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_pmi_df = pp.China_pmi()
print(China_pmi_df)
```

数据示例

```
    月份  制造业-指数  制造业-同比增长  非制造业-指数  非制造业-同比增长
0    2022年10月份    49.2  0.000000     48.7  -7.061069
1    2022年09月份    50.1  1.008065     50.6  -4.887218
2    2022年08月份    49.4 -1.397206     52.6  10.736842
3    2022年07月份    49.0 -2.777778     53.8   0.938086
4    2022年06月份    50.2 -1.375246     54.7   2.242991
..         ...     ...       ...      ...        ...
173  2008年05月份    53.3 -4.308797     57.4  -7.717042
174  2008年04月份    59.2  1.023891     58.4  -3.311258
175  2008年03月份    58.4  4.099822     58.9   1.202749
176  2008年02月份    53.4  0.564972     59.3  -2.145215
177  2008年01月份    53.0 -3.811252     60.2  -0.331126
```

#### 中国城镇固定资产投资

接口: China_gdzctz

描述: 中国城镇固定资产投资, 数据区间从 200802 至今, 月度数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称       | 类型    | 描述           |
| ---------- | ------- | -------------- |
| 月份       | object  | -              |
| 当月       | float64 | 注意单位: 亿元 |
| 同比增长   | float64 | 注意单位: %    |
| 环比增长   | float64 | 注意单位: %    |
| 自年初累计 | float64 | 注意单位: 亿元 |

接口示例

```python
import PPshare as pp

China_gdzctz_df = pp.China_gdzctz()
print(China_gdzctz_df)
```

数据示例

```
        月份        当月   同比增长   环比增长      自年初累计
0    2022年10月份  50047.00   4.27  -7.84  471459.00
1    2022年09月份  54306.00   6.66  14.83  421412.00
2    2022年08月份  47294.00   6.57  -2.25  367106.00
3    2022年07月份  48382.00   3.75 -26.10  319812.00
4    2022年06月份  65466.00   5.62  24.89  271430.00
..         ...       ...    ...    ...        ...
158  2008年06月份  18171.78  29.49  53.29   58435.98
159  2008年05月份  11854.13  25.44  17.45   40264.20
160  2008年04月份  10093.14  25.37  -1.01   28410.07
161  2008年03月份  10195.65  27.31    NaN   18316.94
162  2008年02月份       NaN    NaN    NaN    8121.29
```

#### 海关进出口增减情况

接口: China_hgjck

描述: 中国海关进出口增减情况一览表, 数据区间从 200801 至今, 月度数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称                | 类型    | 描述             |
| ------------------- | ------- | ---------------- |
| 月份                | object  | -                |
| 当月出口额-金额     | float64 | 注意单位: 亿美元 |
| 当月出口额-同比增长 | float64 | 注意单位: %      |
| 当月出口额-环比增长 | float64 | 注意单位: %      |
| 当月进口额-金额     | float64 | 注意单位: 亿美元 |
| 当月进口额-同比增长 | float64 | 注意单位: %      |
| 当月进口额-环比增长 | float64 | 注意单位: %      |
| 累计出口额-金额     | float64 | 注意单位: 亿美元 |
| 累计出口额-同比增长 | float64 | 注意单位: %      |
| 累计进口额-金额     | float64 | 注意单位: 亿美元 |
| 累计进口额-同比增长 | float64 | 注意单位: %      |

接口示例

```python
import PPshare as pp

China_hgjck_df = pp.China_hgjck()
print(China_hgjck_df)
```

数据示例

```
     月份      当月出口额-金额  当月出口额-同比增长  ...  累计出口额-同比增长      累计进口额-金额  累计进口额-同比增长
0    2022年10月份  2.983717e+08        -0.3  ...        11.1  2.264551e+09         3.5
1    2022年09月份  3.226903e+08         5.9  ...        12.5  2.051334e+09         4.0
2    2022年08月份  3.143794e+08         7.2  ...        13.5  1.813451e+09         4.5
3    2022年07月份  3.318411e+08        18.0  ...        14.5  1.578288e+09         5.2
4    2022年06月份  3.281211e+08        17.1  ...        13.9  1.347358e+09         5.7
..         ...           ...         ...  ...         ...           ...         ...
173  2008年05月份  1.204965e+08        28.1  ...        22.9  4.670271e+08        30.4
174  2008年04月份  1.187067e+08        21.8  ...        21.5  3.665725e+08        27.9
175  2008年03月份  1.089629e+08        30.6  ...        21.4  2.644787e+08        28.6
176  2008年02月份  8.736780e+07         6.5  ...        16.8  1.689377e+08        30.9
177  2008年01月份  1.096400e+08        26.6  ...        26.6  9.017445e+07        27.6
```

#### 财政收入

接口: China_czsr

描述: 中国财政收入, 数据区间从 200801 至今, 月度数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称          | 类型    | 描述           |
| ------------- | ------- | -------------- |
| 月份          | object  | -              |
| 当月          | float64 | 注意单位: 亿元 |
| 当月-同比增长 | float64 | 注意单位: %    |
| 当月-环比增长 | float64 | 注意单位: %    |
| 累计          | float64 | 注意单位: 亿元 |
| 累计-同比增长 | float64 | 注意单位: %    |

接口示例

```python
import PPshare as pp

China_czsr_df = pp.China_czsr()
print(China_czsr_df)
```

数据示例

```
     月份        当月    当月-同比增长     当月-环比增长         累计  累计-同比增长
0    2022年10月份  20246.00  15.651777   34.008472  173397.00     -4.5
1    2022年09月份  15108.00   8.440999   15.663757  153151.00     -6.6
2    2022年08月份  13062.00   5.577110  -33.896761  138043.00     -8.0
3    2022年07月份  19760.00  -4.077670    6.914836  124981.00     -9.2
4    2022年06月份  18482.00 -10.550770   48.497509  105221.00    -10.2
..         ...       ...        ...         ...        ...      ...
169  2008年05月份   6268.15  52.602556   -8.157360   29064.37     33.8
170  2008年04月份   6824.88  17.016465   54.550663   22796.22     29.4
171  2008年03月份   4415.95  24.671449    6.184551   15971.34     35.5
172  2008年02月份   4158.75  36.552139  -43.775147   11555.39     40.2
173  2008年01月份   7396.64  42.352305  135.212710    7396.64     42.4
```

#### 外汇贷款数据

接口: China_whxd

描述: 外汇贷款数据, 数据区间从 200802 至今, 月度数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称     | 类型    | 描述             |
| -------- | ------- | ---------------- |
| 月份     | object  | -                |
| 当月     | float64 | 注意单位: 亿美元 |
| 同比增长 | float64 | 注意单位: %      |
| 环比增长 | float64 | 注意单位: %      |
| 累计     | float64 | 注意单位: 亿美元 |

接口示例

```python
import PPshare as pp

China_whxd_df = pp.China_whxd()
print(China_whxd_df)
```

数据示例

```
       月份      当月     同比增长     环比增长       累计
0    2022年10月份 -264.00  -303.08  -142.20  7895.00
1    2022年09月份 -109.00  -581.25    68.59  8159.00
2    2022年08月份 -347.00 -2378.57   -11.58  8268.00
3    2022年07月份 -311.00  -972.41   -79.77  8615.00
4    2022年06月份 -173.00  -269.61   -50.43  8926.00
..         ...     ...      ...      ...      ...
173  2008年05月份   30.04   -13.15    40.70  2739.34
174  2008年04月份   21.35    39.82   -79.40  2709.30
175  2008年03月份  103.62   279.28   -52.25  2687.95
176  2008年02月份  217.02   754.75    28.20  2584.33
177  2008年01月份  169.28  4508.33  1789.29  2367.31
```

#### 本外币存款

接口: China_wbck

描述: 本外币存款, 数据区间从 200802 至今, 月度数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称     | 类型    | 描述           |
| -------- | ------- | -------------- |
| 月份     | object  | -              |
| 当月     | float64 | 注意单位: 亿元 |
| 同比增长 | float64 | 注意单位: %    |
| 环比增长 | float64 | 注意单位: %    |
| 累计     | float64 | 注意单位: 亿元 |

接口示例

```python
import PPshare as pp

China_wbck_df = pp.China_wbck()
print(China_wbck_df)
```

数据示例

```
      月份        当月        同比增长        环比增长          累计
0    2022年10月份  -2650.78 -132.917535 -110.068013  2610227.88
1    2022年09月份  26328.73   14.546846  134.206364  2612878.66
2    2022年08月份  11241.68  -19.737487  871.473473  2586549.93
3    2022年07月份  -1457.17   88.621966 -102.976736  2575308.25
4    2022年06月份  48951.94   22.914621   63.520287  2576765.42
..         ...       ...         ...         ...         ...
173  2008年05月份   8763.12  370.965835   22.035258   442540.86
174  2008年04月份   7180.81   69.117415  -33.125126   433777.74
175  2008年03月份  10737.68   25.845804  -19.689339   426596.93
176  2008年02月份  13370.18  152.665137  829.976560   415859.25
177  2008年01月份   1437.69  -72.142704  -59.893939   402489.07
```

#### 货币净投放与净回笼

提示: 该接口暂停更新

接口: China_hb

描述: 央行币净投放与净回笼, 数据区间从 19970310 至今, 周或则月频率

限量: 单次返回指定 symbol 的所有历史数据

输入参数

| 名称   | 类型 | 描述                                             |
| ------ | ---- | ------------------------------------------------ |
| symbol | str  | symbol="weekly"; choice of {"weekly", "monthly"} |

输出参数-weekly

| 名称     | 类型    | 描述         |
| -------- | ------- | ------------ |
| 日期     | object  | -            |
| 开始日期 | object  | -            |
| 结束日期 | object  | -            |
| 投放量   | float64 | 注意单位: 亿 |
| 回笼量   | float64 | 注意单位: 亿 |
| 净投放   | float64 | 注意单位: 亿 |

接口示例-weekly

```
import PPshare as pp

China_hb_df = pp.China_hb(symbol="weekly")
print(China_hb_df)
```

数据示例-weekly

```
      日期      开始日期    结束日期     投放量     回笼量     净投放
0     1997-11  1997-03-10  1997-03-16     0.0     0.0     0.0
1     1997-12  1997-03-17  1997-03-23     0.0     0.0     0.0
2     1997-13  1997-03-24  1997-03-30     0.0     0.0     0.0
3     1997-14  1997-03-31  1997-04-06     0.0     0.0     0.0
4     1997-15  1997-04-07  1997-04-13     0.0     0.0     0.0
       ...         ...         ...     ...     ...     ...
1287  2021-50  2021-12-06  2021-12-12   500.0  2300.0 -1800.0
1288  2021-51  2021-12-13  2021-12-19   500.0   500.0     0.0
1289  2021-52  2021-12-20  2021-12-26  1000.0   500.0   500.0
1290  2022-01  2021-12-27  2022-01-02  6500.0   500.0  6000.0
1291  2022-02  2022-01-03  2022-01-09   300.0  7000.0 -6700.0
```

输出参数-monthly

| 名称   | 类型    | 描述         |
| ------ | ------- | ------------ |
| 日期   | object  | -            |
| 投放量 | float64 | 注意单位: 亿 |
| 回笼量 | float64 | 注意单位: 亿 |
| 净投放 | float64 | 注意单位: 亿 |

接口示例-monthly

```
import PPshare as pp

China_hb_df = pp.China_hb(symbol="monthly")
print(China_hb_df)
```

数据示例-monthly

```
      日期      投放量      回笼量     净投放
0    1997-12      0.0      0.0     0.0
1    1997-11      0.0      0.0     0.0
2    1997-10      0.0      0.0     0.0
3    1997-09      0.0      0.0     0.0
4    1997-08      0.0      0.0     0.0
..       ...      ...      ...     ...
294  2021-04   2200.0   2100.0   100.0
295  2021-03   2400.0   2500.0  -100.0
296  2021-02   8300.0  11440.0 -3140.0
297  2021-01  10740.0  12450.0 -1710.0
298  2022-01    300.0   7300.0 -7000.0
```
#### 新债发行

接口: China_bond_public

描述: 中国外汇交易中心暨全国银行间同业拆借中心-债券信息披露-新债发行; 近期债券发行数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称       | 类型    | 描述           |
| ---------- | ------- | -------------- |
| 债券全称   | object  | -              |
| 债券类型   | object  | -              |
| 发行日期   | object  | -              |
| 计息方式   | object  | -              |
| 价格       | float64 | 注意单位: 元   |
| 债券期限   | object  | -              |
| 计划发行量 | float64 | 注意单位: 亿元 |
| 债券评级   | object  | -              |

接口示例

```python
import PPshare as pp

China_bond_public_df = pp.China_bond_public()
print(China_bond_public_df)
```

数据示例

```
                                 债券全称    债券类型   发行日期  ...   债券期限  计划发行量  债券评级
0    2022年杭州市临安区新锦产业发展集团有限公司公司债券(第一期)     企业债  01-07  ...   3+4年    7.0   AA+
1      无锡市锡西新城产业发展集团有限公司2022年度第一期中期票据    中期票据  01-06  ...     3年    5.0  None
2             宁波舜农集团有限公司2022年度第一期中期票据    中期票据  01-06  ...     3年   10.0  None
3     2022年第一期淮北市建投控股集团有限公司小微企业增信集合债券     企业债  01-06  ...   3+1年   10.0   AAA
4          融腾2022年第一期个人汽车贷款资产支持证券(次级)  资产支持证券  01-06  ...  1507日    5.7  None
..                                ...     ...    ...  ...    ...    ...   ...
229         山东高速集团有限公司2021年度第六期超短期融资券  超短期融资券  12-30  ...   269日   20.0  None
230     浪潮电子信息产业股份有限公司2021年度第四期超短期融资券  超短期融资券  12-30  ...   268日   10.0  None
231      陕西煤业化工集团有限责任公司2021年度第十二期中期票据    中期票据  12-30  ...     3年   20.0  None
232     陕西延长石油(集团)有限责任公司2021年度第四期中期票据    中期票据  12-30  ...     3年   20.0  None
233     昆明滇池投资有限责任公司2021年度第四期绿色超短期融资券  超短期融资券  12-29  ...   270日   10.0  None
```

#### 消费者信心指数

接口: China_xfzxx

描述: 东方财富网-消费者信心指数

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称                    | 类型    | 描述        |
| ----------------------- | ------- | ----------- |
| 月份                    | object  | -           |
| 消费者信心指数-指数值   | float64 | -           |
| 消费者信心指数-同比增长 | float64 | 注意单位: % |
| 消费者信心指数-环比增长 | float64 | 注意单位: % |
| 消费者满意指数-指数值   | float64 | -           |
| 消费者满意指数-同比增长 | float64 | 注意单位: % |
| 消费者满意指数-环比增长 | float64 | 注意单位: % |
| 消费者预期指数-指数值   | float64 | -           |
| 消费者预期指数-同比增长 | float64 | 注意单位: % |
| 消费者预期指数-环比增长 | float64 | 注意单位: % |

接口示例

```python
import PPshare as pp

China_xfzxx_df = pp.China_xfzxx()
print(China_xfzxx_df)
```

数据示例

```
      月份  消费者信心指数-指数值  ...  消费者预期指数-同比增长  消费者预期指数-环比增长
0    2022年09月份         87.2  ...    -28.995984      0.568828
1    2022年08月份         87.0  ...    -26.134454     -1.897321
2    2022年07月份         87.9  ...    -25.581395     -1.538462
3    2022年06月份         88.9  ...    -27.949327      3.762828
4    2022年05月份         86.8  ...    -30.063796      1.036866
..         ...          ...  ...           ...           ...
184  2007年05月份        112.8  ...      2.975654      0.351494
185  2007年04月份        112.3  ...      2.430243      1.426025
186  2007年03月份        111.0  ...      0.718133     -1.232394
187  2007年02月份        111.8  ...      2.805430     -0.525394
188  2007年01月份        112.4  ...      3.442029     -0.609225
```

#### 存款准备金率

接口: China_reserve_requirement_ratio

描述: 国家统计局-存款准备金率

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称                      | 类型    | 描述        |
| ------------------------- | ------- | ----------- |
| 公布时间                  | object  | XXXX年X月   |
| 生效时间                  | object  | XXXX年X月   |
| 大型金融机构-调整前       | float64 | 注意单位: % |
| 大型金融机构-调整后       | float64 | 注意单位: % |
| 大型金融机构-调整幅度     | float64 | 注意单位: % |
| 中小金融机构-调整前       | float64 | 注意单位: % |
| 中小金融机构-调整后       | float64 | 注意单位: % |
| 中小金融机构-调整幅度     | float64 | 注意单位: % |
| 消息公布次日指数涨跌-上证 | float64 | 注意单位: % |
| 消息公布次日指数涨跌-深证 | float64 | 注意单位: % |
| 备注                      | object  | -           |

接口示例

```python
import PPshare as pp

China_reserve_requirement_ratio_df = pp.China_reserve_requirement_ratio()
print(China_reserve_requirement_ratio_df)
```

数据示例

```
           公布时间  ...                                                 备注
0   2022年11月25日  ...  为保持流动性合理充裕,促进综合融资成本稳中有降,落实稳经济一揽子政策措施,巩固经济回稳向上基...
1   2022年04月15日  ...  为支持实体经济发展,促进综合融资成本稳中有降,中国人民银行决定于2022年4月25日下调金融...
2   2021年12月06日  ...  为支持实体经济发展,促进综合融资成本稳中有降,中国人民银行决定于2021年12月15日下调金...
3   2021年07月09日  ...  为支持实体经济发展,促进综合融资成本稳中有降,中国人民银行决定于2021年7月15日下调金融...
4   2020年04月03日  ...  为支持实体经济发展,促进加大对中小微企业的支持力度,降低社会融资实际成本,中国人民银行决定对...
5   2020年04月03日  ...  为支持实体经济发展,促进加大对中小微企业的支持力度,降低社会融资实际成本,中国人民银行决定对...
6   2020年01月01日  ...  为支持实体经济发展,降低社会融资实际成本,中国人民银行决定于2020年1月6日下调金融机构存...
7   2019年09月06日  ...  为支持实体经济发展,降低社会融资实际成本,中国人民银行决定于2019年9月16日全面下调金融...
8   2019年01月04日  ...  为进一步支持实体经济发展,优化流动性结构,降低融资成本,中国人民银行决定下调金融机构存款准备...
9   2019年01月04日  ...  为进一步支持实体经济发展,优化流动性结构,降低融资成本,中国人民银行决定下调金融机构存款准备...
10  2018年10月07日  ...  从2018年10月15日起,下调大型商业银行、股份制商业银行、城市商业银行、非县域农村商业银...
11  2018年06月24日  ...  从2018年7月5日起,下调大型商业银行、股份制商业银行、邮政储蓄银行、城市商业银行、非县域...
12  2018年04月17日  ...  从2018年4月25日起,下调大型商业银行、股份制商业银行、城市商业银行、非县域农村商业银行...
13  2016年02月29日  ...  自2016年3月1日起,普遍下调金融机构人民币存款准备金率0.5个百分点,以保持金融体系流动...
14  2015年10月23日  ...  2015-10-24,自同日起,下调金融机构人民币存款准备金率0.5个百分点,以保持银行体系...
15  2015年08月25日  ...  自2015年9月6日起,为进一步增强金融机构支持“三农”和小微企业的能力,额外降低县域农村商...
16  2015年04月19日  ...  自4月20日起对农信社、村镇银行等农村金融机构额外降低人民币存款准备金率1个百分点,并统一下...
17  2015年02月04日  ...  同时,为进一步增强金融机构支持结构调整的能力,加大对小微企业、“三农”以及重大水利工程建设的...
18  2012年05月12日  ...                                               None
19  2012年02月18日  ...                                               None
20  2011年11月30日  ...                                               None
21  2011年06月14日  ...                                               None
22  2011年05月12日  ...                                               None
23  2011年04月17日  ...                                               None
24  2011年03月18日  ...                                               None
25  2011年02月18日  ...                                               None
26  2011年01月14日  ...                                               None
27  2010年12月10日  ...                                               None
28  2010年11月19日  ...                                               None
29  2010年11月10日  ...                                               None
30  2010年05月02日  ...  从2010年5月10日起,上调存款类金融机构人民币存款准备金率0.5个百分点,农村信用社、村...
31  2010年02月12日  ...  从2010年2月25日起,上调存款类金融机构人民币存款准备金率0.5个百分点。为加大对“三农...
32  2010年01月12日  ...  从2010年1月18日起,上调存款类金融机构人民币存款准备金率0.5个百分点。为增强支农资金...
33  2008年12月22日  ...                                               None
34  2008年11月26日  ...                                               None
35  2008年10月08日  ...                                               None
36  2008年09月15日  ...  从2008年9月25日起,除工商银行、农业银行、中国银行、建设银行、交通银行、邮政储蓄银行暂...
37  2008年06月07日  ...                                               None
38  2008年06月07日  ...  中国人民银行决定上调存款类金融机构人民币存款准备金率1个百分点,于2008年6月15日和25...
39  2008年05月12日  ...                                               None
40  2008年04月16日  ...                                               None
41  2008年03月18日  ...                                               None
42  2008年01月16日  ...                                               None
43  2007年12月08日  ...                                               None
44  2007年11月10日  ...                                               None
45  2007年10月13日  ...                                               None
46  2007年09月06日  ...                                               None
47  2007年07月30日  ...                                               None
48  2007年05月18日  ...                                               None
49  2007年04月29日  ...                                               None
50  2007年04月05日  ...                                               None
51  2007年02月16日  ...                                               None
52  2007年01月05日  ...                                               None
```

#### 社会消费品零售总额

接口: China_consumer_goods_retail

描述: 东方财富-经济数据-社会消费品零售总额

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称          | 类型    | 描述           |
| ------------- | ------- | -------------- |
| 月份          | object  | -              |
| 当月          | float64 | 注意单位: 亿元 |
| 同比增长      | float64 | 注意单位: %    |
| 环比增长      | float64 | 注意单位: %    |
| 累计          | float64 | 注意单位: 亿元 |
| 累计-同比增长 | float64 | 注意单位: %    |

接口示例

```python
import PPshare as pp

China_consumer_goods_retail_df = pp.China_consumer_goods_retail()
print(China_consumer_goods_retail_df)
```

数据示例

```
     月份       当月  同比增长       环比增长        累计  累计-同比增长
0    2022年10月份  40271.0  -0.5   6.692277  360575.0      0.6
1    2022年09月份  37745.0   2.5   4.101164  320305.0      0.7
2    2022年08月份  36258.0   5.4   1.081684  282560.0      0.5
3    2022年07月份  35870.0   2.7  -7.413143  246302.0     -0.2
4    2022年06月份  38742.0   3.1  15.485736  210432.0     -0.7
..         ...      ...   ...        ...       ...      ...
162  2008年05月份   8703.5  21.6   6.896340   42400.7     21.1
163  2008年04月份   8142.0  22.0   0.231436   33697.2     21.0
164  2008年03月份   8123.2  21.5  -2.770895   25555.2     20.6
165  2008年02月份   8354.7  19.1  -7.960517   17432.0     20.2
166  2008年01月份   9077.3  21.2   0.687720    9077.3     21.2
```

#### 全社会用电分类情况表

接口: China_society_electricity

描述: 国家统计局-全社会用电分类情况表

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称                       | 类型  | 默认显示 | 描述               |
| -------------------------- | ----- | -------- | ------------------ |
| 统计时间                   | str   | Y        | -                  |
| 全社会用电量               | float | Y        | 注意单位: 万千瓦时 |
| 全社会用电量同比           | float | Y        | 注意单位: %        |
| 各行业用电量合计           | float | Y        | 注意单位: 万千瓦时 |
| 各行业用电量合计同比       | float | Y        | 注意单位: %        |
| 第一产业用电量             | float | Y        | 注意单位: 万千瓦时 |
| 第一产业用电量同比         | float | Y        | 注意单位: %        |
| 第二产业用电量             | float | Y        | 注意单位: 万千瓦时 |
| 第二产业用电量同比         | float | Y        | 注意单位: %        |
| 第三产业用电量             | float | Y        | 注意单位: 万千瓦时 |
| 第三产业用电量同比         | float | Y        | 注意单位: %        |
| 城乡居民生活用电量合计     | float | Y        | 注意单位: 万千瓦时 |
| 城乡居民生活用电量合计同比 | float | Y        | 注意单位: %        |
| 城镇居民用电量             | float | Y        | 注意单位: 万千瓦时 |
| 城镇居民用电量同比         | float | Y        | 注意单位: %        |
| 乡村居民用电量             | float | Y        | 注意单位: 万千瓦时 |
| 乡村居民用电量同比         | float | Y        | 注意单位: %        |

接口示例

```python
import PPshare as pp

China_society_electricity_df = pp.China_society_electricity()
print(China_society_electricity_df)
```

数据示例

```
     统计时间  全社会用电量 全社会用电量同比  ... 城镇居民用电量同比     乡村居民用电量 乡村居民用电量同比
0     2020.8  476760000.00     0.50  ...      None        None      None
1     2020.7  403810000.00    -0.70  ...      None        None      None
2     2020.6  335470000.00    -1.30  ...      None        None      None
3     2020.5  271970000.00    -2.80  ...      None        None      None
4     2020.4  212700000.00    -4.70  ...      None        None      None
..       ...           ...      ...  ...       ...         ...       ...
171  2004.11  194584023.00    15.13  ...      8.95  8408040.00     10.48
172  2004.10  175828690.00    15.17  ...      8.74  7652223.00      8.99
173   2004.9  157131146.00    14.92  ...      8.58  6814158.00      9.80
174   2004.3   48045510.00    15.70  ...      9.85  2187609.00     11.68
175  2003.12  188912117.00    15.29  ...     16.12  8806708.00      5.79
```

#### 全社会客货运输量

接口: China_society_traffic_volume

描述: 国家统计局-全社会客货运输量-非累计

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称                           | 类型  | 描述           |
| ------------------------------ | ----- | -------------- |
| 统计时间                       | str   | 年月           |
| 统计对象                       | float | -              |
| 货运量                         | float | 注意单位: 亿吨 |
| 货运量同比增长                 | float | 注意单位: %    |
| 货物周转量                     | float | 注意单位: 亿   |
| 公里货物周转量同比增长         | float | 注意单位: %    |
| 客运量                         | float | 注意单位: 亿人 |
| 客运量同比增长                 | float | 注意单位: %    |
| 旅客周转量                     | float | 注意单位: 亿   |
| 公里旅客周转量同比增长         | float | 注意单位: %    |
| 沿海主要港口货物吞吐量         | float | 注意单位: 亿吨 |
| 沿海主要港口货物吞吐量同比增长 | float | 注意单位: %    |
| 其中:外贸货物吞吐量            | float | 注意单位: 亿吨 |
| 其中:外贸货物吞吐量同比增长    | float | 注意单位: %    |
| 民航总周转量                   | float | 注意单位: 亿   |
| 公里民航总周转                 | float | 注意单位: %    |

接口示例

```python
import PPshare as pp

China_society_traffic_volume_df = pp.China_society_traffic_volume()
print(China_society_traffic_volume_df)
```

数据示例

```
         统计时间 统计对象    货运量 货运量同比增长  ... 其中:外贸货物吞吐量 其中:外贸货物吞吐量同比增长 民航总周转量 公里民航总周转
0      2020.8   民航  54.94  -12.70  ...       None           None   None    None
1      2020.8   水运   6.72    0.60  ...       None           None   None    None
2      2020.8   公路  32.53    5.50  ...       None           None   None    None
3      2020.8   铁路   3.84    6.50  ...       None           None   None    None
4      2020.8   合计  43.10    4.80  ...       None           None   None    None
       ...  ...    ...     ...  ...        ...            ...    ...     ...
2088  1952.12   水运   None    None  ...       None           None   None    None
2089  1952.12   公路   None    None  ...       None           None   None    None
2090  1952.12   铁路   None    None  ...       None           None   None    None
2091  1952.12   合计   None    None  ...       None           None   None    None
2092  1952.12   民航   None    None  ...       None           None   None    None
```

#### 邮电业务基本情况

接口: China_postal_telecommunicational

描述: 国家统计局-邮电业务基本情况-非累计

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称                       | 类型  | 描述           |
| -------------------------- | ----- | -------------- |
| 统计时间                   | str   | 年月           |
| 邮电业务总量               | float | 注意单位: 亿元 |
| 邮电业务总量同比增长       | float | 注意单位: %    |
| 邮政业务总量               | float | 注意单位: 亿元 |
| 邮政业务总量同比增长       | float | 注意单位: %    |
| 电信业务总量               | float | 注意单位: 亿元 |
| 电信业务总量同比增长       | float | 注意单位: %    |
| 函件总数                   | float | 注意单位: 万件 |
| 函件总数同比增长           | float | 注意单位: %    |
| 包件                       | float | 注意单位: 万件 |
| 包件同比增长               | float | 注意单位: %    |
| 特快专递                   | float | 注意单位: 万件 |
| 特快专递同比增长           | float | 注意单位: %    |
| 汇票                       | float | 注意单位: 万张 |
| 汇票同比增长               | float | 注意单位: %    |
| 订销报纸累计数             | float | 注意单位: 万份 |
| 订销报纸累计数同比增长     | float | 注意单位: %    |
| 订销杂志累计数             | float | 注意单位: 万份 |
| 订销杂志累计数同比增长     | float | 注意单位: %    |
| 集邮业务                   | float | 注意单位: 万枚 |
| 集邮业务同比增长           | float | 注意单位: %    |
| 邮政储蓄期末余额           | float | 注意单位: 亿元 |
| 邮政储蓄期末余额同比增长   | float | 注意单位: %    |
| 长途电话通话时长           | float | 注意单位: 万   |
| 钟长途电话通话时长同比增长 | float | 注意单位: %    |
| 本地电话期末用户数         | float | 注意单位: %    |
| 本地电话期末用户数同比增长 | float | 注意单位: %    |
| 城市电话用户数             | float | 注意单位: 万户 |
| 城市电话用户数同比增长     | float | 注意单位: %    |
| 乡村电话用户数             | float | 注意单位: 万户 |
| 乡村电话用户数同比增长     | float | 注意单位: %    |
| 无线寻呼用户数             | float | 注意单位: 万户 |
| 无线寻呼用户数同比增长     | float | 注意单位: %    |
| 移动电话用户数             | float | 注意单位: 万户 |
| 移动电话用户数同比增长     | float | 注意单位: %    |
| 固定电话用                 | float | 注意单位: 万户 |
| 固定电话用户数同比增长     | float | 注意单位: %    |
| 城市住宅电话用户           | float | 注意单位: 万户 |
| 城市住宅电话用户同比增长   | float | 注意单位: %    |
| 乡村住宅电话用户           | float | 注意单位: 万户 |
| 乡村住宅电话用户同比增长   | float | 注意单位: %    |

接口示例

```python
import PPshare as pp

China_postal_telecommunicational_df = pp.China_postal_telecommunicational()
print(China_postal_telecommunicational_df)
```

数据示例

```
      统计时间 邮电业务总量 邮电业务总量同比增长  ... 城市住宅电话用户同比增长 乡村住宅电话用户 乡村住宅电话用户同比增长
0     2020.8   None       None  ...         None     None         None
1     2020.7   None       None  ...         None     None         None
2     2020.6   None       None  ...         None     None         None
3     2020.5   None       None  ...         None     None         None
4     2020.4   None       None  ...         None     None         None
..       ...    ...        ...  ...          ...      ...          ...
267  1970.12   None       None  ...         None     None         None
268  1965.12   None       None  ...         None     None         None
269  1962.12   None       None  ...         None     None         None
270  1957.12   None       None  ...         None     None         None
271  1952.12   None       None  ...         None     None         None
```

#### 国际旅游外汇收入构成

接口: China_international_tourism_fx

描述: 国家统计局-国际旅游外汇收入构成

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称     | 类型  | 默认显示 | 描述               |
| -------- | ----- | -------- | ------------------ |
| 统计年度 | str   | Y        | 年                 |
| 指标     | float | Y        | -                  |
| 数量     | float | Y        | 注意单位: 百万美元 |
| 比重     | float | Y        | 注意单位: %        |

接口示例

```python
import PPshare as pp

China_international_tourism_fx_df = pp.China_international_tourism_fx()
print(China_international_tourism_fx_df)
```

数据示例

```
     统计年度    指标         数量      比重
0    2019    总计  131254.00    None
1    2018  其他服务   11454.26    9.00
2    2018    住宿   18108.80   14.00
3    2018    总计  127103.00  100.00
4    2018  商品销售   32761.35   26.00
..    ...   ...        ...     ...
238  1996    餐饮    1376.00   13.50
239  1995    总计    8733.00  100.00
240  1994    总计    7323.00  100.00
241  1993    总计    4683.00  100.00
242  1992    总计    3947.00  100.00
```

#### 民航客座率及载运率

接口: China_passenger_load_factor

描述: 国家统计局-民航客座率及载运率

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称     | 类型  | 默认显示 | 描述        |
| -------- | ----- | -------- | ----------- |
| 统计年度 | str   | Y        | 年月        |
| 客座率   | float | Y        | 注意单位: % |
| 载运率   | float | Y        | 注意单位: % |

接口示例

```python
import PPshare as pp

China_passenger_load_factor_df = pp.China_passenger_load_factor()
print(China_passenger_load_factor_df)
```

数据示例

```
    统计时间    客座率    载运率
0    2020.7  74.80  67.80
1    2020.6  70.30  67.60
2    2020.5  68.50  66.10
3    2020.4  66.30  66.80
4    2020.3  58.60  62.70
..      ...    ...    ...
167  2006.6  72.00  64.60
168  2006.5  71.30  64.40
169  2006.4  76.30  69.00
170  2006.3  72.40  67.50
171  2006.2  72.70  64.60
```

#### 航贸运价指数

接口: China_freight_index

描述: 国家统计局-航贸运价指数

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| 无   | 无   | 无   |

输出参数

| 名称                           | 类型    | 描述 |
| ------------------------------ | ------- | ---- |
| 截止日期                       | object  | 年月 |
| 波罗的海好望角型船运价指数BCI  | float64 | -    |
| 灵便型船综合运价指数BHMI       | float64 | -    |
| 波罗的海超级大灵便型船BSI指数  | float64 | -    |
| 波罗的海综合运价指数BDI        | float64 | -    |
| HRCI国际集装箱租船指数         | float64 | -    |
| 油轮运价指数成品油运价指数BCTI | float64 | -    |
| 油轮运价指数原油运价指数BDTI   | float64 | -    |

接口示例

```python
import PPshare as pp

China_freight_index_df = pp.China_freight_index()
print(China_freight_index_df)
```

数据示例

```
      截止日期  波罗的海好望角型船运价指数BCI  ...  油轮运价指数成品油运价指数BCTI  油轮运价指数原油运价指数BDTI
0     2021-08-10            4328.0  ...                NaN               NaN
1     2021-08-09            4342.0  ...                NaN               NaN
2     2021-08-06            4359.0  ...                NaN               NaN
3     2021-08-05            4414.0  ...                NaN               NaN
4     2021-08-04            4302.0  ...                NaN               NaN
          ...               ...  ...                ...               ...
3845  2005-12-13            3459.0  ...             1326.0            2063.0
3846  2005-12-12            3476.0  ...             1286.0            2051.0
3847  2005-12-09            3509.0  ...             1261.0            2061.0
3848  2005-12-08            3553.0  ...                NaN               NaN
3849  2005-12-07            3709.0  ...                NaN               NaN
```

#### 央行货币当局资产负债

接口: China_central_bank_balance

描述: 国家统计局-央行货币当局资产负债

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称                 | 类型  | 默认显示 | 描述           |
| -------------------- | ----- | -------- | -------------- |
| 统计时间             | str   | Y        | 年月           |
| 国外资产             | float | Y        | 注意单位: 亿元 |
| 外汇                 | float | Y        | 注意单位: 亿元 |
| 货币黄金             | float | Y        | 注意单位: 亿元 |
| 其他国外资产         | float | Y        | 注意单位: 亿元 |
| 对政府债权           | float | Y        | 注意单位: 亿元 |
| 其中:中央政府        | float | Y        | 注意单位: 亿元 |
| 对其他存款性公司债权 | float | Y        | 注意单位: 亿元 |
| 对其他金融性公司债权 | float | Y        | 注意单位: 亿元 |
| 对非货币金融机构债权 | float | Y        | 注意单位: 亿元 |
| 对非金融性公司债权   | float | Y        | 注意单位: 亿元 |
| 其他资产             | float | Y        | 注意单位: 亿元 |
| 总资产               | float | Y        | 注意单位: 亿元 |
| 储备货币             | float | Y        | 注意单位: 亿元 |
| 发行货币             | float | Y        | 注意单位: 亿元 |
| 金融性公司存款       | float | Y        | 注意单位: 亿元 |
| 其他存款性公司       | float | Y        | 注意单位: 亿元 |
| 其他金融性公司       | float | Y        | 注意单位: 亿元 |
| 对金融机构负债       | float | Y        | 注意单位: 亿元 |
| 准备金存款           | float | Y        | 注意单位: 亿元 |
| 非金融性公司存款     | float | Y        | 注意单位: 亿元 |
| 活期存款             | float | Y        | 注意单位: 亿元 |
| 债券                 | float | Y        | 注意单位: 亿元 |
| 国外负债             | float | Y        | 注意单位: 亿元 |
| 政府存款             | float | Y        | 注意单位: 亿元 |
| 自有资金             | float | Y        | 注意单位: 亿元 |
| 其他负债             | float | Y        | 注意单位: 亿元 |
| 总负债               | float | Y        | 注意单位: 亿元 |

接口示例

```python
import PPshare as pp

China_central_bank_balance_df = pp.China_central_bank_balance()
print(China_central_bank_balance_df)
```

数据示例

```
        统计时间       国外资产         外汇  ...    自有资金      其他负债        总负债
0     2020.8  218361.92  211684.91  ...  219.75  11750.36  365052.29
1     2020.7  218374.92  211723.16  ...  219.75  11068.35  357924.90
2     2020.6  218332.59  211742.47  ...  219.75  10222.08  363931.30
3     2020.5  218333.25  211802.63  ...  219.75   9494.26  367870.74
4     2020.4  218310.70  211914.83  ...  219.75   8914.45  360347.67
..       ...        ...        ...  ...     ...       ...        ...
279   1994.3    2437.60    2253.80  ...  280.50   -298.60       None
280  1993.12    1549.50    1431.80  ...  310.30   -255.00       None
281   1993.9    1451.00    1332.20  ...  310.30   -286.50       None
282   1993.6    1286.50    1183.00  ...  310.30   -411.60       None
283   1993.3    1324.00    1222.00  ...  310.30   -384.10       None
```

#### 保险业经营情况

接口: China_insurance

描述: 国家统计局-保险业经营情况

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称                             | 类型  | 默认显示 | 描述           |
| -------------------------------- | ----- | -------- | -------------- |
| 统计时间                         | str   | Y        | 年月           |
| 省市地区                         | float | Y        | -              |
| 原保险保费收入                   | float | Y        | 注意单位: 万元 |
| 财产险保费收入                   | float | Y        | 注意单位: 万元 |
| 人身险保费收入                   | float | Y        | 注意单位: 万元 |
| 人身险-寿险保费收入              | float | Y        | 注意单位: 万元 |
| 人身险-健康险保费收入            | float | Y        | 注意单位: 万元 |
| 人身险-意外险保费收入            | float | Y        | 注意单位: 万元 |
| 养老保险公司企业年金缴费         | float | Y        | 注意单位: 万元 |
| 原保险赔付支出                   | float | Y        | 注意单位: 万元 |
| 财产险保费赔付支出               | float | Y        | 注意单位: 万元 |
| 人身险保费赔付支出               | float | Y        | 注意单位: 万元 |
| 人身险-寿险赔付支出              | float | Y        | 注意单位: 万元 |
| 人身险-健康险赔付支出            | float | Y        | 注意单位: 万元 |
| 人身险-意外险赔付支出            | float | Y        | 注意单位: 万元 |
| 业务及管理费                     | float | Y        | 注意单位: 万元 |
| 银行存款                         | float | Y        | 注意单位: 万元 |
| 投资                             | float | Y        | 注意单位: 万元 |
| 资产总额                         | float | Y        | 注意单位: 万元 |
| 养老保险公司企业年金受托管理资产 | float | Y        | 注意单位: 万元 |
| 养老保险公司企业年金投资管理资产 | float | Y        | 注意单位: 万元 |

接口示例

```python
import PPshare as pp

China_insurance_df = pp.China_insurance()
print(China_insurance_df)
```

数据示例

```
      统计时间 省市地区     原保险保费收入  ...         资产总额 养老保险公司企业年金受托管理资产 养老保险公司企业年金投资管理资产
0     2020.8   新疆  5176100.00  ...         None             None             None
1     2020.8   宁夏  1564700.00  ...         None             None             None
2     2020.8   青海   751400.00  ...         None             None             None
3     2020.8   甘肃  3679100.00  ...         None             None             None
4     2020.8   陕西  8222700.00  ...         None             None             None
      ...  ...         ...  ...          ...              ...              ...
6755  1999.5   全国  5289491.00  ...         None             None             None
6756  1999.4   全国  4161147.00  ...         None             None             None
6757  1999.3   全国  2837194.00  ...  21408761.00             None             None
6758  1999.2   全国  1543697.00  ...         None             None             None
6759  1999.1   全国   885113.00  ...         None             None             None
```

#### 货币供应量

接口: China_supply_of_money

描述: 国家统计局-货币供应量

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称                               | 类型  | 默认显示 | 描述           |
| ---------------------------------- | ----- | -------- | -------------- |
| 统计时间                           | str   | Y        | 年月           |
| 货币和准货币（广义货币M2）         | float | Y        | 注意单位: 亿元 |
| 货币和准货币（广义货币M2）同比增长 | float | Y        | 注意单位: %    |
| 货币(狭义货币M1)                   | float | Y        | 注意单位: 亿元 |
| 货币(狭义货币M1)同比增长           | float | Y        | 注意单位: %    |
| 流通中现金(M0)                     | float | Y        | 注意单位: 亿元 |
| 流通中现金(M0)同比增长             | float | Y        | 注意单位: %    |
| 活期存款                           | float | Y        | 注意单位: 亿元 |
| 活期存款同比增长                   | float | Y        | 注意单位: %    |
| 准货币                             | float | Y        | 注意单位: 亿元 |
| 准货币同比增长                     | float | Y        | 注意单位: %    |
| 定期存款                           | float | Y        | 注意单位: 亿元 |
| 定期存款同比增长                   | float | Y        | 注意单位: %    |
| 储蓄存款出                         | float | Y        | 注意单位: 亿元 |
| 储蓄存款同比增长                   | float | Y        | 注意单位: %    |
| 其他存款                           | float | Y        | 注意单位: 亿元 |
| 其他存款同比增长                   | float | Y        | 注意单位: %    |

接口示例

```python
import PPshare as pp

China_supply_of_money_df = pp.China_supply_of_money()
print(China_supply_of_money_df)
```

数据示例

```
       统计时间 货币和准货币（广义货币M2） 货币和准货币（广义货币M2）同比增长  ... 储蓄存款同比增长       其他存款 其他存款同比增长
0    2020.8     2136800.00              10.40  ...     None  235344.24     None
1    2020.7     2125458.46              10.70  ...     None  240538.49     None
2    2020.6     2134948.66              11.10  ...     None  228402.91     None
3    2020.5     2100183.74              11.10  ...     None  233222.73     None
4    2020.4     2093533.83              11.10  ...     None  241313.38     None
..      ...            ...                ...  ...      ...        ...      ...
507  1978.5           None               None  ...     None       None     None
508  1978.4           None               None  ...     None       None     None
509  1978.3           None               None  ...     None       None     None
510  1978.2           None               None  ...     None       None     None
511  1978.1           None               None  ...     None       None     None
```

#### FR007利率互换曲线历史数据

接口: China_swap_rate

描述: 国家统计局-FR007 利率互换曲线历史数据

限量: 单次返回所有历史数据, 该接口只能获取近一年的数据的数据，其中每次只能获取一个月的数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称     | 类型    | 描述 |
| -------- | ------- | ---- |
| 日期     | object  | -    |
| 曲线名称 | object  | -    |
| 时刻     | object  | -    |
| 价格类型 | object  | -    |
| 1M       | float64 | -    |
| 3M       | float64 | -    |
| 6M       | float64 | -    |
| 9M       | float64 | -    |
| 1Y       | float64 | -    |
| 2Y       | float64 | -    |
| 3Y       | float64 | -    |
| 4Y       | float64 | -    |
| 5Y       | float64 | -    |
| 7Y       | float64 | -    |
| 10Y      | float64 | -    |

接口示例

```python
import PPshare as pp

China_swap_rate_df = pp.China_swap_rate(start_date="20220212", end_date="20220312")
print(China_swap_rate_df)
```

数据示例

```
             日期           曲线名称     时刻 价格类型  ...      4Y      5Y      7Y     10Y
0    2022-03-11  FR007利率互换收盘曲线  16:30   报买  ...  2.4435  2.5505  2.6617  2.8240
1    2022-03-11  FR007利率互换收盘曲线  16:30   均值  ...  2.4565  2.5518  2.6907  2.8511
2    2022-03-11  FR007利率互换收盘曲线  16:30   报卖  ...  2.4695  2.5532  2.7197  2.8782
3    2022-03-11  FR007利率互换行情曲线  16:00   报买  ...  2.4800  2.5765  2.7200  2.8758
4    2022-03-11  FR007利率互换行情曲线  16:00   均值  ...  2.4850  2.5783  2.7324  2.8891
..          ...            ...    ...  ...  ...     ...     ...     ...     ...
895  2022-02-14  FR007利率互换行情曲线  10:00   均值  ...  2.4238  2.5236  2.6710  2.8269
896  2022-02-14  FR007利率互换行情曲线  10:00   报卖  ...  2.4329  2.5251  2.6825  2.8398
897  2022-02-14  FR007利率互换行情曲线   9:30   报买  ...  2.4045  2.5160  2.6461  2.7976
898  2022-02-14  FR007利率互换行情曲线   9:30   均值  ...  2.4212  2.5212  2.6647  2.8197
899  2022-02-14  FR007利率互换行情曲线   9:30   报卖  ...  2.4379  2.5264  2.6833  2.8419
```

#### 央行黄金和外汇储备

接口: China_foreign_exchange_gold

描述: 国家统计局-央行黄金和外汇储备, 比东财接口数据时间长

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称         | 类型 | 默认显示 | 描述             |
| ------------ | ---- | -------- | ---------------- |
| 统计时间     | str  | Y        | 年月             |
| 黄金储备     | str  | Y        | 注意单位: 万盎司 |
| 国家外汇储备 | str  | Y        | 注意单位: 亿美元 |

接口示例

```python
import PPshare as pp

China_foreign_exchange_gold_df = pp.China_foreign_exchange_gold()
print(China_foreign_exchange_gold_df)
```

数据示例

```
      统计时间     黄金储备    国家外汇储备
0     2020.9  6264.00  31425.62
1     2020.8  6264.00  31646.09
2     2020.7  6264.00  31543.91
3     2020.6  6264.00  31123.28
4     2020.5  6264.00  31016.92
..       ...      ...       ...
343  1982.12  1267.00     69.86
344  1981.12  1267.00     27.08
345  1980.12  1280.00    -12.96
346  1979.12  1280.00      8.40
347  1978.12  1280.00      1.67
```

#### 商品零售价格指数

接口: China_retail_price_index

描述: 国家统计局-商品零售价格指数

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称             | 类型  | 默认显示 | 描述 |
| ---------------- | ----- | -------- | ---- |
| 统计月份         | str   | Y        | 年月 |
| 居民消费项目     | str   | Y        | -    |
| 零售商品价格指数 | float | Y        | -    |

接口示例

```python
import PPshare as pp

China_retail_price_index_df = pp.China_retail_price_index()
print(China_retail_price_index_df)
```

数据示例

```
     统计月份       居民消费项目 零售商品价格指数
0     2020.8    建筑材料及五金电料   100.10
1     2020.8           燃料    90.70
2     2020.8   书报杂志及电子出版物   101.70
3     2020.8  中西药品及医疗保健用品   100.30
4     2020.8         金银珠宝   122.90
      ...          ...      ...
3777  2002.1         机电产品    94.10
3778  2002.1    家用电器及音像器材    94.20
3779  2002.1       体育娱乐用品    98.80
3780  2002.1  中西药品及医疗保健用品    97.30
3781  2002.1   书报杂志及电子出版物   101.00
```

#### 国房景气指数

接口: China_real_estate

描述: 国家统计局-国房景气指数

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称        | 类型    | 描述 |
| ----------- | ------- | ---- |
| 日期        | object  | -    |
| 最新值      | float64 | -    |
| 涨跌幅      | float64 | -    |
| 近3月涨跌幅 | float64 | -    |
| 近6月涨跌幅 | float64 | -    |
| 近1年涨跌幅 | float64 | -    |
| 近2年涨跌幅 | float64 | -    |
| 近3年涨跌幅 | float64 | -    |

接口示例

```python
import PPshare as pp

China_real_estate_df = pp.China_real_estate()
print(China_real_estate_df)
```

数据示例

```
     日期     最新值       涨跌幅  ...    近1年涨跌幅    近2年涨跌幅    近3年涨跌幅
0    1998-01-01   98.60       NaN  ...       NaN       NaN       NaN
1    1998-02-01   98.99  0.395538  ...       NaN       NaN       NaN
2    1998-03-01   99.05  0.060612  ...       NaN       NaN       NaN
3    1998-04-01  100.81  1.776880  ...       NaN       NaN       NaN
4    1998-05-01  101.68  0.863010  ...       NaN       NaN       NaN
..          ...     ...       ...  ...       ...       ...       ...
280  2022-03-01   96.66 -0.278552  ... -4.618117 -1.598290 -4.306504
281  2022-04-01   95.89 -0.796607  ... -5.349916 -3.023867 -5.190825
282  2022-05-01   95.60 -0.302430  ... -5.533597 -3.774534 -5.402731
283  2022-06-01   95.40 -0.209205  ... -5.619311 -4.456685 -5.628648
284  2022-07-01   95.26 -0.146751  ... -5.683168 -4.825657 -5.757816
```

#### 外汇和黄金储备

接口: China_fx_gold

描述: 中国外汇和黄金储备, 数据区间从 200801 至今, 月度数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称              | 类型    | 描述             |
| ----------------- | ------- | ---------------- |
| 月份              | object  | 年度和月份       |
| 黄金储备-数值     | float64 | 注意单位: 万盎司 |
| 黄金储备-同比     | float64 | 注意单位: 万盎司 |
| 黄金储备-环比     | float64 | 注意单位: 万盎司 |
| 国家外汇储备-数值 | float64 | 注意单位: 亿美元 |
| 国家外汇储备-同比 | float64 | 注意单位: 亿美元 |
| 国家外汇储备-环比 | float64 | 注意单位: 亿美元 |

接口示例

```python
import PPshare as pp

China_fx_gold_df = pp.China_fx_gold()
print(China_fx_gold_df)
```

数据示例

```
       月份  黄金储备-数值    黄金储备-同比   黄金储备-环比   国家外汇储备-数值  国家外汇储备-同比  国家外汇储备-环比
0    2008年01月份      NaN        NaN       NaN  15898.1040  43.914387   4.028224
1    2008年02月份      NaN        NaN       NaN  16471.3371  42.316677   3.605670
2    2008年03月份      NaN        NaN       NaN  16821.7700  39.944557   2.127532
3    2008年04月份      NaN        NaN       NaN  17566.5514  40.919594   4.427485
4    2008年05月份      NaN        NaN       NaN  17969.6074  39.011455   2.294451
..         ...      ...        ...       ...         ...        ...        ...
170  2022年03月份  1216.63  14.849008  1.690906  31879.9400   0.566714  -0.803808
171  2022年04月份  1197.31   8.126829 -1.587993  31197.2000  -2.453270  -2.141598
172  2022年05月份  1151.83  -3.223828 -3.798515  31277.8000  -2.918335   0.258357
173  2022年06月份  1138.23   3.053871 -1.180730  30712.7200  -4.441119  -1.806649
174  2022年07月份  1098.39  -3.963383 -3.500171  31040.7100  -4.073655   1.067929
```

#### 中国货币供应量

接口: China_money_supply

描述: 东方财富-经济数据-中国宏观-中国货币供应量; 数据区间从 200801 至今, 月度数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称                        | 类型    | 描述       |
| --------------------------- | ------- | ---------- |
| 月份                        | object  | 年度和月份 |
| 货币和准货币(M2)-数量(亿元) | float64 | -          |
| 货币和准货币(M2)-同比增长   | float64 | -          |
| 货币和准货币(M2)-环比增长   | float64 | -          |
| 货币(M1)-数量(亿元)         | float64 | -          |
| 货币(M1)-同比增长           | float64 | -          |
| 货币(M1)-环比增长           | float64 | -          |
| 流通中的现金(M0)-数量(亿元) | float64 | -          |
| 流通中的现金(M0)-同比增长   | float64 | -          |
| 流通中的现金(M0)-环比增长   | float64 | -          |

接口示例

```python
import PPshare as pp

China_money_supply_df = pp.China_money_supply()
print(China_money_supply_df)
```

数据示例

```
            月份  货币和准货币(M2)-数量(亿元)  ...  流通中的现金(M0)-同比增长  流通中的现金(M0)-环比增长
0    2022年10月份         2612914.57  ...            14.30        -0.258787
1    2022年09月份         2626600.92  ...            13.60         1.482068
2    2022年08月份         2595068.27  ...            14.30         0.747950
3    2022年07月份         2578078.57  ...            13.90         0.518710
4    2022年06月份         2581451.20  ...            13.80         0.485950
..         ...                ...  ...              ...              ...
173  2008年05月份          436221.60  ...            12.88        -2.014673
174  2008年04月份          429313.72  ...            10.70         1.171555
175  2008年03月份          423054.53  ...            11.12        -6.228418
176  2008年02月份          421037.84  ...             5.96       -11.503457
177  2008年01月份          417846.17  ...            31.21        20.896562
```

#### 全国股票交易统计表

接口: China_stock_market_cap

描述: 全国股票交易统计表, 数据区间从 200801 至今, 月度数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称                     | 类型    | 描述           |
| ------------------------ | ------- | -------------- |
| 数据日期                 | object  | 年度和月份     |
| 发行总股本-上海          | float64 | 注意单位: 亿元 |
| 发行总股本-深圳          | float64 | 注意单位: 亿元 |
| 市价总值-上海            | float64 | 注意单位: 亿元 |
| 市价总值-深圳            | float64 | 注意单位: 亿元 |
| 成交金额-上海            | float64 | 注意单位: 亿元 |
| 成交金额-深圳            | float64 | 注意单位: 亿元 |
| 成交量-上海              | float64 | -              |
| 成交量-深圳              | float64 | -              |
| A股最高综合股价指数-上海 | float64 | -              |
| A股最高综合股价指数-深圳 | float64 | -              |
| A股最低综合股价指数-上海 | float64 | -              |
| A股最低综合股价指数-深圳 | float64 | -              |

接口示例

```python
import PPshare as pp

China_stock_market_cap_df = pp.China_stock_market_cap()
print(China_stock_market_cap_df)
```

数据示例

```
          数据日期  发行总股本-上海  ...  A股最低综合股价指数-上海  A股最低综合股价指数-深圳
0    2022年11月份       NaN  ...      2896.7572      2032.0920
1    2022年10月份  47240.00  ...      2885.0894      1974.9663
2    2022年09月份  47205.00  ...      3021.9309      2032.3677
3    2022年08月份  47038.00  ...      3155.1866      2231.3209
4    2022年07月份  46908.00  ...      3226.2315      2284.0597
..         ...       ...  ...            ...            ...
174  2008年05月份  14980.19  ...      3333.9510      1086.0100
175  2008年04月份  14785.95  ...      2990.7880       963.8400
176  2008年03月份  14476.47  ...      3357.2290      1186.6000
177  2008年02月份  14300.84  ...      4123.3050      1385.0400
178  2008年01月份  14198.16  ...      4330.6970      1424.3500
```

#### 上海银行业同业拆借报告

接口: shibor

描述: 上海银行业同业拆借报告, 数据区间从2006.0108-至今

限量: 单次返回一年（364天）的数据，超过则会返回空表

输入参数

| 名称       | 类型                | 描述                                                         |
| ---------- | ------------------- | ------------------------------------------------------------ |
| mode       | str                 | 选择模式，默认为‘data’。可选模式有：<br>data：Shibor数据<br>queto:Shibor报价数据<br>avg：Shibor均值数据 |
| start_date | str / datetime.date | 起始时间，默认为终止时间的一年前（364天）                    |
| end_date   | str / datetime.date | 终止时间，默认为今天                                         |

输出参数

| 名称 | 类型  | 描述      |
| ---- | ----- | --------- |
| 日期 | str   | 日期-索引 |
| O/N  | float | -         |
| 1W   | float | 1周       |
| 2W   | float | 2周       |
| 1M   | float | 1月       |
| 3M   | float | 3月       |
| 6M   | float | 6月       |
| 9M   | float | 9月       |
| 1Y   | float | 1年       |

接口示例

```python
import PPshare as pp

China_shibor_df = pp.shibor(mode='data',end_date='2007-01-01')
print(China_shibor_df)
```

数据示例

```
            日期     O/N      1W      2W      1M      3M      6M      9M      1Y
0   2006-12-31  1.5675  1.6321  1.9753  2.5833  2.8082  2.8685  2.9301  3.0021
1   2006-12-30  1.5651  1.9182  2.0744  2.5848  2.8081  2.8683  2.9301  3.0021
2   2006-12-29  1.7203  1.9641  2.2236  2.5852  2.8080  2.8682  2.9301  3.0020
3   2006-12-28  1.7107  2.0753  2.3773  2.5832  2.8079  2.8681  2.9301  3.0020
4   2006-12-27  1.7185  2.2785  2.4817  2.5802  2.8076  2.8680  2.9300  3.0020
..         ...     ...     ...     ...     ...     ...     ...     ...     ...
58  2006-10-12  2.0943  2.2904  2.5240  2.5900  2.6380  2.7470  2.8570  2.9550
59  2006-10-11  2.0955  2.2932  2.4930  2.5864  2.6338  2.7475  2.8547  2.9559
60  2006-10-10  2.0922  2.2971  2.4236  2.5739  2.6325  2.7454  2.8544  2.9531
61  2006-10-09  2.0990  2.2960  2.3972  2.5522  2.6248  2.7431  2.8514  2.9549
62  2006-10-08  2.1184  2.2930  2.3848  2.5319  2.6110  2.7404  2.8521  2.9543
```

#### 人民币香港银行同业拆息

接口: China_hk_market_info

描述: 香港同业拆借报告, 数据区间从 20170320-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称       | 类型    | 描述     |
| ---------- | ------- | -------- |
| 日期       | object  | 日期     |
| O/N_定价   | float64 | -        |
| O/N_涨跌幅 | float64 | 单位: 点 |
| 1W_定价    | float64 | -        |
| 1W_涨跌幅  | float64 | 单位: 点 |
| 2W_定价    | float64 | -        |
| 2W_涨跌幅  | float64 | 单位: 点 |
| 1M_定价    | float64 | -        |
| 1M_涨跌幅  | float64 | 单位: 点 |
| 3M_定价    | float64 | -        |
| 3M_涨跌幅  | float64 | 单位: 点 |
| 6M_定价    | float64 | -        |
| 6M_涨跌幅  | float64 | 单位: 点 |
| 9M_定价    | float64 | -        |
| 9M_涨跌幅  | float64 | 单位: 点 |
| 1Y_定价    | float64 | -        |
| 1Y_涨跌幅  | float64 | 单位: 点 |
| ON_定价    | float64 | -        |
| ON_涨跌幅  | float64 | 单位: 点 |
| 2M_定价    | float64 | -        |
| 2M_涨跌幅  | float64 | 单位: 点 |

接口示例

```python
import PPshare as pp

China_hk_market_info_df = pp.China_hk_market_info()
print(China_hk_market_info_df)
```

数据示例

```
      日期   1W_定价  1W_涨跌幅   2W_定价  ...   ON_定价  ON_涨跌幅   2M_定价  2M_涨跌幅
0     2017-03-20  3.9428     NaN  4.2417  ...  2.0283     NaN  4.5915     NaN
1     2017-03-21  3.9094   -3.34  4.1647  ...  2.3883   36.00  4.6113    1.98
2     2017-03-22  4.3795   47.01  4.5853  ...  2.3100   -7.83  4.8365   22.52
3     2017-03-23  4.3538   -2.57  4.3930  ...  2.2263   -8.37  4.5410  -29.55
4     2017-03-24  4.3542    0.03  4.4410  ...  2.2438    1.75  4.6405    9.95
          ...     ...     ...     ...  ...     ...     ...     ...     ...
1327  2022-08-09  1.3648   -5.27  1.4678  ...  1.1943   -3.39  2.0330   -0.90
1328  2022-08-10  1.5283   16.35  1.6553  ...  1.3379   14.37  2.0668    3.38
1329  2022-08-11  1.6290   10.07  1.6748  ...  1.3677    2.97  2.0758    0.90
1330  2022-08-12  1.6195   -0.95  1.6733  ...  1.5007   13.30  2.0658   -1.00
1331  2022-08-15  1.5908   -2.87  1.6330  ...  1.3665  -13.42  2.0623   -0.35
```

### 其他指标

#### 中国日度沿海六大电库存

接口: China_daily_energy

描述: 获取中国日度沿海六大电库存数据, 数据区间从20160101-至今, 不再更新, 只能获得历史数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称 | 类型  | 默认显示 | 描述      |
| ---- | ----- | -------- | --------- |
| 日期 | str   | Y        | 日期-索引 |
| 数值 | float | Y        | energy    |

接口示例

```python
import PPshare as pp

China_daily_energy_df = pp.China_daily_energy()
print(China_daily_energy_df)
```

数据示例

China_daily_energy_df:

```
          沿海六大电库存      日耗 存煤可用天数
2016-01-01  1167.60   64.20   18.19
2016-01-02  1162.90   63.40   18.34
2016-01-03  1160.80   62.60   18.54
2016-01-04  1185.30   57.60   20.58
2016-01-05  1150.20   57.20   20.11
              ...     ...    ...
2019-05-17   1639.47   61.71  26.56
2019-05-21   1591.92   62.67  25.40
2019-05-22   1578.63   59.54  26.51
2019-05-24   1671.83   60.65  27.56
2019-06-21   1786.64   66.57  26.84
```

#### 人民币汇率中间价报告

接口: China_rmb

描述: 中国人民币汇率中间价报告, 数据区间从 20170103-20210513

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
| ---- | ---- | ---- |
| -    | -    | -    |

输出参数

| 名称                         | 类型    | 描述       |
| ---------------------------- | ------- | ---------- |
| 日期                         | object  | 日期       |
| 美元/人民币_中间价           | float64 | -          |
| 美元/人民币_涨跌幅           | float64 | 单位: 点   |
| 欧元/人民币_中间价           | float64 | -          |
| 欧元/人民币_涨跌幅           | float64 | 单位: 点   |
| 100日元/人民币_中间价        | float64 | -          |
| 100日元/人民币_涨跌幅        | float64 | 单位: 点对 |
| 港元/人民币_中间价           | float64 | -          |
| 港元/人民币_涨跌幅           | float64 | 单位: 点   |
| 英镑/人民币_中间价           | float64 | -          |
| 英镑/人民币_涨跌幅           | float64 | 单位: 点   |
| 澳元/人民币_中间价           | float64 | -          |
| 澳元/人民币_涨跌幅           | float64 | 单位: 点   |
| 新西兰元/人民币_中间价       | float64 | -          |
| 新西兰元/人民币_涨跌幅       | float64 | 单位: 点   |
| 新加坡元/人民币_中间价       | float64 | -          |
| 新加坡元/人民币_涨跌幅       | float64 | 单位: 点   |
| 瑞郎/人民币_中间价           | float64 | -          |
| 瑞郎/人民币_涨跌幅           | float64 | 单位: 点   |
| 加元/人民币_中间价           | float64 | -          |
| 加元/人民币_涨跌幅           | float64 | 单位: 点   |
| 人民币/马来西亚林吉特_中间价 | float64 | -          |
| 人民币/马来西亚林吉特_涨跌幅 | float64 | 单位: 点   |
| 人民币/俄罗斯卢布_中间价     | float64 | -          |
| 人民币/俄罗斯卢布_涨跌幅     | float64 | 单位: 点   |
| 人民币/南非兰特_中间价       | float64 | -          |
| 人民币/南非兰特_涨跌幅       | float64 | 单位: 点   |
| 人民币/韩元_中间价           | float64 | -          |
| 人民币/韩元_涨跌幅           | float64 | 单位: 点   |
| 人民币/阿联酋迪拉姆_中间价   | float64 | -          |
| 人民币/阿联酋迪拉姆_涨跌幅   | float64 | 单位: 点   |
| 人民币/沙特里亚尔_中间价     | float64 | -          |
| 人民币/沙特里亚尔_涨跌幅     | float64 | 单位: 点   |
| 人民币/匈牙利福林_中间价     | float64 | -          |
| 人民币/匈牙利福林_涨跌幅     | float64 | 单位: 点   |
| 人民币/波兰兹罗提_中间价     | float64 | -          |
| 人民币/波兰兹罗提_涨跌幅     | float64 | 单位: 点   |
| 人民币/丹麦克朗_中间价       | float64 | -          |
| 人民币/丹麦克朗_涨跌幅       | float64 | 单位: 点   |
| 人民币/瑞典克朗_中间价       | float64 | -          |
| 人民币/瑞典克朗_涨跌幅       | float64 | 单位: 点   |
| 人民币/丹麦克朗_中间价       | float64 | -          |
| 人民币/丹麦克朗_涨跌幅       | float64 | 单位: 点   |
| 人民币/挪威克朗_中间价       | float64 | -          |
| 人民币/挪威克朗_涨跌幅       | float64 | 单位: 点   |
| 人民币/土耳其里拉_中间价     | float64 | -          |
| 人民币/土耳其里拉_涨跌幅     | float64 | 单位: 点   |
| 人民币/墨西哥比索_中间价     | float64 | -          |
| 人民币/墨西哥比索_涨跌幅     | float64 | 单位: 点   |
| 人民币/泰铢_中间价           | float64 | -          |
| 人民币/泰铢_涨跌幅           | float64 | 单位: 点   |

接口示例

```python
import PPshare as pp

China_rmb_df = pp.China_rmb()
print(China_rmb_df)
```

数据示例

```
     日期  美元/人民币_中间价  美元/人民币_涨跌幅  ...  人民币/墨西哥比索_涨跌幅  人民币/泰铢_定价  人民币/泰铢_涨跌幅
0    2018-02-06      6.3072         0.0  ...            0.0     5.0191         0.0
1    2018-02-07      6.2882      -190.0  ...         -184.0     5.0178       -13.0
2    2018-02-08      6.2822       -60.0  ...          286.0     5.0429       251.0
3    2018-02-09      6.3194       372.0  ...          -64.0     5.0406       -23.0
4    2018-02-12      6.3001      -193.0  ...         -146.0     5.0310       -96.0
..          ...         ...         ...  ...            ...        ...         ...
786  2021-05-07      6.4678      -217.0  ...         -118.0     4.8283       310.0
787  2021-05-10      6.4425      -253.0  ...         -171.0     4.8333        50.0
788  2021-05-11      6.4254      -171.0  ...           85.0     4.8434       101.0
789  2021-05-12      6.4258         4.0  ...           65.0     4.8503        69.0
790  2021-05-13      6.4612       354.0  ...          107.0     4.8427       -76.0
```

#### 深圳融资融券报告

接口: China_market_margin_sz


描述: 获取深圳融资融券报告, 数据区间从20100331-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称             | 类型  | 默认显示 | 描述      |
| ---------------- | ----- | -------- | --------- |
| 日期             | str   | Y        | 日期-索引 |
| 融资买入额(元)   | float | Y        | -         |
| 融资余额(元)     | float | Y        | -         |
| 融券卖出量(股)   | float | Y        | -         |
| 融券余量(股)     | float | Y        | -         |
| 融券余额(元)     | float | Y        | -         |
| 融资融券余额(元) | float | Y        | -         |

接口示例

```python
import PPshare as pp

China_market_margin_sz_df = pp.China_market_margin_sz()
print(China_market_margin_sz_df)
```

数据示例

```
               融资买入额(元)       融资余额(元)  融券卖出量(股)    融券余量(股)     融券余额(元)  \
2010-03-31       684569        670796      4000       3900       70895
2010-04-08      6713260      14467758      2100       3100       56023
2010-04-09      9357095      19732998      6700       5400      108362
2010-04-12     10406563      24813027      2200       1000        8100
2010-04-15     16607172      47980287      4200       5200       97676
                 ...           ...       ...        ...         ...
2019-12-12  25190412075  423457288662  29769255  209557883  2504593151
2019-12-13  29636811209  423422868505  32820867  206092170  2509424768
2019-12-16  39166060634  428851154451  44000215  217123568  2647520178
2019-12-17  46930557203  433966722200  40492711  220945538  2750371397
2019-12-18  41043515833  438511398249  39150376  224554586  2761303194
               融资融券余额(元)
2010-03-31        741691
2010-04-08      14523781
2010-04-09      19841360
2010-04-12      24821127
2010-04-15      48077963
                  ...
2019-12-12  425961881813
2019-12-13  425932293273
2019-12-16  431498674629
2019-12-17  436717093597
2019-12-18  441272701443
```

#### 上海融资融券报告

接口: China_market_margin_sh


描述: 获取上海融资融券报告, 数据区间从20100331-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称         | 类型  | 默认显示 | 描述         |
| ------------ | ----- | -------- | ------------ |
| 日期         | str   | Y        | 日期-索引    |
| 融资余额     | float | Y        | 注意单位：元 |
| 融资买入额   | float | Y        | 注意单位：元 |
| 融券余量     | float | Y        | 注意单位：股 |
| 融券余额     | float | Y        | 注意单位：元 |
| 融券卖出量   | float | Y        | 注意单位：股 |
| 融资融券余额 | float | Y        | 注意单位：元 |

接口示例

```python
import PPshare as pp

China_market_margin_sh_df = pp.China_market_margin_sh()
print(China_market_margin_sh_df)
```

数据示例

```
                    融资余额         融资买入额  ...        融券卖出量        融资融券余额
2010-03-31  5.824813e+06  5.866316e+06  ...       3100.0  5.848955e+06
2010-04-01  6.842114e+06  1.054024e+06  ...          0.0  6.859439e+06
2010-04-02  6.762781e+06  2.075160e+05  ...          0.0  6.774710e+06
2010-04-06  1.009124e+07  3.329461e+06  ...          0.0  1.010151e+07
2010-04-07  2.508683e+07  1.514140e+07  ...       1400.0  2.512524e+07
                  ...           ...  ...          ...           ...
2020-08-17  7.320815e+11  6.532228e+10  ...  311910304.0  7.735001e+11
2020-08-18  7.371393e+11  5.588143e+10  ...  236068903.0  7.784736e+11
2020-08-19  7.381023e+11  5.470145e+10  ...  302792889.0  7.794299e+11
2020-08-20  7.368085e+11  3.763784e+10  ...  203660295.0  7.778967e+11
2020-08-21  7.352210e+11  3.599241e+10  ...  196118235.0  7.765762e+11
```

#### 上海黄金交易所报告

接口: China_au_report


描述: 获取上海黄金交易所报告, 数据区间从20140905-至今

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称       | 类型  | 默认显示 | 描述      |
| ---------- | ----- | -------- | --------- |
| 日期       | str   | Y        | 日期-索引 |
| 商品       | float | Y        | -         |
| 开盘价     | float | Y        | -         |
| 最高价     | float | Y        | -         |
| 最低价     | float | Y        | -         |
| 收盘价     | float | Y        | -         |
| 涨跌       | float | Y        | -         |
| 涨跌幅     | float | Y        | -         |
| 加权平均价 | float | Y        | -         |
| 成交量     | float | Y        | -         |
| 成交金额   | float | Y        | -         |
| 持仓量     | float | Y        | -         |
| 交收方向   | float | Y        | -         |
| 交收量     | float | Y        | -         |

接口示例

```python
import PPshare as pp

China_au_report_df = pp.China_au_report()
print(China_au_report_df)
```

数据示例

```
                  商品      开盘价      最高价  ...          持仓量   交收方向    交收量
日期                                      ...
2014-09-05    Pt9995   293.50   293.50  ...          NaN           NaN
2014-09-05   Ag(T+D)  4159.00  4197.00  ...  5.25224e+06   空支付多    NaN
2014-09-05   Ag99.99  4218.00  4218.00  ...          322           NaN
2014-09-05   Au(T+D)   250.52   252.48  ...       144574   多支付空  11848
2014-09-05  mAu(T+D)   250.87   252.50  ...       130808   多支付空  23644
              ...      ...      ...  ...          ...    ...    ...
2020-04-22  NYAuTN12   379.80   385.00  ...        63446           NaN
2020-04-22    PGC30g   375.83   379.50  ...          NaN           NaN
2020-04-22   Pt99.95   173.30   181.96  ...          NaN           NaN
2020-04-22   Au99.99   372.00   374.45  ...          NaN           NaN
2020-04-22   Ag(T+D)  3554.00  3620.00  ...  1.06485e+07  多支付给空  67050
```

### 中国电煤价格指数

#### 全国综合电煤价格指数

接口: China_ctci


描述: 获取中国电煤价格指数-全国综合电煤价格指数图, 20140101-至今的所有历史数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称  | 类型  | 默认显示 | 描述 |
| ----- | ----- | -------- | ---- |
| date  | str   | Y        | 日期 |
| value | float | Y        | -    |

接口示例

```python
import PPshare as pp

China_ctci_df = pp.China_ctci()
print(China_ctci_df)
```

数据示例

```
        date   value
0   20140201  495.79
1   20140301  484.88
2   20140401  470.61
3   20140501  461.02
4   20140601  454.23
5   20140701  443.48
6   20140801  430.77
7   20140901  417.76
8   20141001  412.30
9   20141101  415.58
67  20190901  486.79
68  20191001  489.88
69  20191101  492.01
70  20191201  483.09
```

#### 各价区电煤价格指数

接口: China_ctci_detail


描述: 获取中国电煤价格指数-各价区电煤价格指数, 具体年具体月的历史数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述 |
| ---- | ---- | ---- | ---- |
| 无   | 无   | 无   | 无   |

输出参数

| 名称 | 类型  | 默认显示 | 描述 |
| ---- | ----- | -------- | ---- |
| 环比 | str   | Y        | 日期 |
| 上期 | float | Y        | -    |
| 同比 | float | Y        | -    |
| 本期 | float | Y        | -    |

接口示例

```python
import PPshare as pp

China_ctci_detail_df = pp.China_ctci_detail()
print(China_ctci_detail_df)
```

数据示例

```
          环比      上期      同比      本期
全国     -1.81  492.01   -7.11  483.09
天津市    -1.72  482.96  -12.83  474.65
冀北      0.56  453.80   -6.82  456.32
冀南     -0.89  471.71   -8.22  467.51
山西省    -2.43  344.46  -11.06  336.08
蒙西      0.51  292.07    4.31  293.56
蒙东     -0.04  274.00   11.53  273.90
辽宁省    -1.25  545.90   -6.91  539.09
吉林省    -0.97  556.83   -5.14  551.43
黑龙江省   -0.27  540.51   -0.84  539.07
上海市    -1.43  519.98  -10.75  512.55
江苏省    -0.85  535.04   -9.17  530.51
浙江省    -0.29  571.41   -2.57  569.78
安徽省    -1.12  574.23   -8.08  567.78
福建省    -1.01  532.02  -13.31  526.62
江西省     0.41  649.13   -5.46  651.79
山东省    -1.62  547.18  -12.24  538.33
河南省    -2.10  499.34  -18.86  488.87
湖北省    -0.75  583.65  -10.48  579.30
湖南省    -0.55  616.56   -9.44  613.18
广东省    -1.38  562.91   -8.65  555.12
广西自治区   0.48  682.33   -6.23  685.59
海南省     3.02  505.27   -4.46  520.53
重庆市    -5.14  613.98   -4.42  582.43
四川省    -0.10  570.25   -8.56  569.70
贵州省     0.85  488.61   -1.55  492.78
云南省    -3.38  457.68  -19.56  442.19
陕西省    -2.02  393.92  -10.11  385.95
甘肃省     0.30  463.03   -9.62  464.42
青海省     2.49  529.57    2.57  542.76
宁夏自治区  -2.46  362.00   -3.44  353.08
新疆自治区  -0.33  258.74    0.09  257.89
```

#### 历史电煤价格指数

接口: China_ctci_detail_hist


描述: 获取中国电煤价格指数-历史电煤价格指数, 具体年的历史数据

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 必选 | 描述                         |
| ---- | ---- | ---- | ---------------------------- |
| year | str  | Y    | year="2018", 从 2014-2019 年 |

输出参数

| 名称 | 类型  | 默认显示 | 描述 |
| ---- | ----- | -------- | ---- |
| 地区 | str   | Y        | 日期 |
| 月份 | float | Y        | -    |
| ---  | ---   | ---      | -    |
| 月份 | float | Y        | -    |

接口示例

```python
import PPshare as pp

China_ctci_detail_hist_df = pp.China_ctci_detail_hist(year="2018")
print(China_ctci_detail_hist_df)
```

数据示例

```
       地区 2018年01月 2018年02月 2018年03月 2018年04月 2018年05月 2018年06月 2018年07月  \
0      全国   549.12   567.21   546.58   522.78   515.39   528.57   532.53
1     天津市   561.23   583.85   585.94   551.54   524.41   528.97   540.94
2      冀北   510.57   534.05   517.08   474.42   453.55   461.06   467.58
3      冀南   530.74   540.18   532.12   514.28   503.97   497.25   493.77
4     山西省   393.98   408.07   396.38   370.90   364.30   374.28   376.38
5      蒙西   301.63   301.46   292.33   276.21   275.37   286.48   281.19
6      蒙东   230.38   227.76   242.35   246.00   248.76   250.37   242.71
7     辽宁省   563.70   563.07   558.31   538.79   530.65   537.27   553.26
8     吉林省   543.02   548.27   535.59   513.15   506.41   522.02   544.06
9    黑龙江省   519.31   522.83   495.48   477.43   470.00   472.85   483.66
10    上海市   645.34   643.98   647.36   600.11   578.45   584.03   602.29
11    江苏省   644.64   651.17   624.19   588.70   576.74   583.94   589.85
12    浙江省   636.39   637.83   625.57   592.52   609.26   618.92   586.79
13    安徽省   647.37   661.77   635.92   604.14   587.82   592.20   603.95
14    福建省   656.82   650.69   649.71   613.20   573.53   600.93   618.63
15    江西省   751.64   765.61   739.68   691.16   662.62   677.83   690.77
16    山东省   628.53   649.27   634.08   592.92   573.53   586.10   594.91
17    河南省   613.92   631.25   621.77   594.66   593.18   601.12   608.91
18    湖北省   677.92   703.40   686.57   632.80   610.36   621.44   642.14
19    湖南省   718.96   737.04   719.14   668.50   640.31   649.64   678.89
20    广东省   651.36   657.91   649.06   634.28   630.47   640.65   639.35
21  广西自治区   756.91   754.45   757.12   745.06   716.98   721.96   705.18
22    海南省   614.98   625.04   637.07   593.45   525.72   565.77   587.75
23    重庆市   619.37   629.52   620.51   598.80   571.14   570.81   611.41
24    四川省   648.10   654.02   651.93   619.14   601.58   605.26   612.76
25    贵州省   518.60   505.77   500.04   497.16   493.63   486.77   489.90
26    云南省   467.09   388.74   421.98   447.52   482.76   513.48   510.29
27    陕西省   463.31   478.76   465.24   442.26   435.63   436.38   428.95
28    甘肃省   505.24   509.11   500.83   489.16   487.85   468.13   464.82
29    青海省   544.92   549.28   548.58   536.46   524.28   510.17   516.74
30  宁夏自治区   400.22   398.22   391.12   370.74   370.42   382.99   378.98
31  新疆自治区   254.34   269.80   250.79   235.72   238.54   245.57   234.11
   2018年08月 2018年09月 2018年10月 2018年11月 2018年12月
0    522.40   522.16   523.47   520.09   522.20
1    544.64   541.17   548.94   544.48   541.99
2    466.16   471.52   485.16   489.72   490.16
3    500.48   499.84   505.87   509.36   513.28
4    371.11   371.23   378.49   377.88   375.88
5    273.10   282.96   286.80   281.42   284.51
6    228.15   242.71   251.47   245.58   250.44
7    563.19   573.65   577.80   579.09   563.81
8    573.05   588.78   592.40   581.29   568.79
9    520.07   539.22   537.95   543.64   535.24
10   607.19   585.45   574.96   574.26   555.67
11   583.12   577.07   582.36   584.04   576.59
12   575.81   583.42   587.16   584.79   587.46
13   599.81   598.09   605.65   617.72   610.69
14   609.29   604.87   601.59   607.44   608.93
15   690.62   691.52   695.14   689.43   678.53
16   590.17   588.25   596.28   613.43   616.25
17   605.13   595.11   597.53   602.47   599.72
18   638.43   625.53   629.82   647.10   653.36
19   677.36   671.32   678.18   677.13   668.14
20   629.72   626.41   616.66   607.70   612.66
21   712.84   724.77   729.57   731.13   716.45
22   587.61   590.36   558.11   544.82   534.33
23   594.85   597.89   608.23   609.34   618.87
24   622.66   608.49   601.19   623.04   642.92
25   502.76   503.33   498.03   500.56   513.44
26   530.24   540.00   541.58   549.68   569.41
27   412.37   418.11   420.12   429.35   434.98
28   454.77   482.04   502.39   513.85   501.55
29   522.63   516.94   517.64   529.17   546.25
30   374.67   376.04   372.16   365.67   368.41
31   229.33   251.85   254.20   257.66   265.38
```

#### 中国银行同业间拆借市场数据（chinbor）

接口: get_chinbor(period)

描述: 中国银行同业间拆借市场数据报告, 数据区间从 2004.05.25-至今

限量: 单次返回所有数据

输入参数

| 名称   | 类型 | 描述                                                         |
| ------ | ---- | ------------------------------------------------------------ |
| period | str  | 选择时期，可选时期<br>'ON':'隔夜'<br/>'1W':'1周'<br/>'2W':'2周'<br/>'3W':'3周'<br/>'1M':'1月'<br/>'2M':'2月' <br/>'3M':'3月'<br/>'4M':'4月'<br/>'6M':'6月'<br/>'9M':'9月'<br/>'1Y':'1年', |

输出参数

| 名称       | 类型  | 描述 |
| ---------- | ----- | ---- |
| 日期       | str   | -    |
| 时期       | str   | -    |
| 利率（%）  | float | -    |
| 涨跌（BP） | float | -    |

接口示例

```python
import PPshare as pp

China_chibor_df = pp.get_chinbor(period='1M')
print(China_chibor_df)
```

数据示例

```
             日期      时期   利率（%）  涨跌（BP）
0    2023-03-21  1月(1M)  3.1716  -63.37
0    2023-03-20  1月(1M)  3.8053   68.53
0    2023-03-17  1月(1M)  3.1200   -6.82
0    2023-03-16  1月(1M)  3.1882   52.46
0    2023-03-15  1月(1M)  2.6636   16.37
..          ...     ...     ...     ...
99   2004-05-31  1月(1M)  2.7500  -55.00
100  2004-05-28  1月(1M)  3.3000  131.48
101  2004-05-27  1月(1M)  1.9852 -107.73
102  2004-05-26  1月(1M)  3.0625   31.98
103  2004-05-25  1月(1M)  2.7427    0.00
```

#### 伦敦银行同业间拆借市场数据（Libor）

接口: get_libor(period)

描述: 伦敦银行同业间拆借市场数据报告, 数据区间从 1986.01.02-至今

限量: 单次返回所有数据

输入参数

| 名称      | 类型 | 描述                                                         |
| --------- | ---- | ------------------------------------------------------------ |
| curr_type | str  | 货币类型，可选类型<br>'USD':'美元'<br/>'GBP':'英镑'<br/>'JPY':'日元'<br/>'EUR':'欧元' |
| period    | str  | 选择时期，可选时期<br>'ON':'隔夜'<br/>'1W':'1周'<br/>'1M':'1月'<br/>'2M':'2月' <br/>'3M':'3月'<br/>8M':'8月' |

输出参数

| 名称       | 类型  | 描述 |
| ---------- | ----- | ---- |
| 日期       | str   | -    |
| 时期       | str   | -    |
| 利率（%）  | float | -    |
| 涨跌（BP） | float | -    |

接口示例

```python
import PPshare as pp

df = pp.get_libor(curr_type='USD', period='1M')
print(df)
```

数据示例

```
             日期      时期    利率（%）  涨跌（BP）
0    2023-03-20  1月(1M)  4.75229  -2.542
0    2023-03-17  1月(1M)  4.77771   1.628
0    2023-03-16  1月(1M)  4.76143   5.286
0    2023-03-15  1月(1M)  4.70857  -1.914
0    2023-03-14  1月(1M)  4.72771   4.328
..          ...     ...      ...     ...
448  1986-01-08  1月(1M)  8.00000 -12.500
449  1986-01-07  1月(1M)  8.12500  -6.250
450  1986-01-06  1月(1M)  8.18750   6.250
451  1986-01-03  1月(1M)  8.12500   0.000
452  1986-01-02  1月(1M)  8.12500   0.000
```



#### 获取欧洲银行同业间拆借市场数据（eurinbor）

接口: get_euribor(period)

描述: 欧洲银行同业间拆借市场数据报告, 数据区间从 1998.12.30-至今

限量: 单次所有数据

输入参数

| 名称   | 类型 | 描述                                                         |
| ------ | ---- | ------------------------------------------------------------ |
| period | str  | 选择时期，可选时期<br/>'ON':'隔夜'<br/>'1W':'1周'<br/>'2W':'2周'<br/>'3W':'3周'<br/>'1M':'1月'<br/>'2M':'2月' <br/>'3M':'3月'<br/>'4M':'4月'<br/>'5M':'5月'<br/>'6M':'6月'<br/>'7M':'7月'<br/>'8M':'8月'<br/>'9M':'9月'<br/>'10M':'10月'<br/>'11M':'11月'<br/>'1Y':'1年', |

输出参数

| 名称       | 类型  | 描述 |
| ---------- | ----- | ---- |
| 日期       | str   | -    |
| 时期       | str   | -    |
| 利率（%）  | float | -    |
| 涨跌（BP） | float | -    |

接口示例

```python
import PPshare as pp

df = pp.get_euribor(period='1Y')
print(df)
```

数据示例

```
             日期      时期    利率（%）  涨跌（BP）
0    2023-03-20  1月(1M)  4.75229  -2.542
0    2023-03-17  1月(1M)  4.77771   1.628
0    2023-03-16  1月(1M)  4.76143   5.286
0    2023-03-15  1月(1M)  4.70857  -1.914
0    2023-03-14  1月(1M)  4.72771   4.328
..          ...     ...      ...     ...
448  1986-01-08  1月(1M)  8.00000 -12.500
449  1986-01-07  1月(1M)  8.12500  -6.250
450  1986-01-06  1月(1M)  8.18750   6.250
451  1986-01-03  1月(1M)  8.12500   0.000
452  1986-01-02  1月(1M)  8.12500   0.000
```



#### 香港银行同业间拆借市场数据（hibor）

接口: get_hibor(curr_type, period)

描述: 香港银行同业间拆借市场数据报告, 数据区间从2013.03.22-至今

限量: 单次返回所有数据

输入参数

| 名称      | 类型 | 描述                                                         |
| --------- | ---- | ------------------------------------------------------------ |
| curr_type | str  | 货币类型，可选类型<br>'USD':'美元'<br/>'HKD':'港币'<br/>'CNH':'人民币' |
| period    | str  | 选择时期，可选时期<br>'ON':'隔夜'<br/>'1W':'1周'<br/>'2W':'2周'<br/>'1M':'1月'<br/>'2M':'2月' <br/>'3M':'3月'<br/>'4M':'4月'<br/>'6M':'6月'<br/>'9M':'9月'<br/>'1Y:'1Y' |

输出参数

| 名称       | 类型  | 描述 |
| ---------- | ----- | ---- |
| 日期       | str   | -    |
| 时期       | str   | -    |
| 利率（%）  | float | -    |
| 涨跌（BP） | float | -    |

接口示例

```python
import PPshare as pp

df = pp.get_hibor(curr_type='CNH', period='6M')
print(df)
```

数据示例

```
             日期      时期    利率（%）  涨跌（BP）
0    2023-03-22  6月(6M)  2.97924  -1.394
0    2023-03-21  6月(6M)  2.99318   3.924
0    2023-03-20  6月(6M)  2.95394   0.773
0    2023-03-17  6月(6M)  2.94621  -0.046
0    2023-03-16  6月(6M)  2.94667  -0.454
..          ...     ...      ...     ...
453  2013-03-28  6月(6M)  2.73330   0.000
454  2013-03-27  6月(6M)  2.73330  -1.000
455  2013-03-26  6月(6M)  2.74330   1.000
456  2013-03-25  6月(6M)  2.73330   0.000
457  2013-03-22  6月(6M)  2.73330   0.000
```

#### 新加坡银行同业间拆借市场数据（sibor）

接口: get_hibor(curr_type, period)

描述: 新加坡银行同业间拆借市场数据报告, 数据区间从2007.01.05-至今

限量: 单次返回所有数据

输入参数

| 名称      | 类型 | 描述                                                         |
| --------- | ---- | ------------------------------------------------------------ |
| curr_type | str  | 货币类型，可选类型<br>'USD':'美元'<br/>'SGD':'星元'          |
| period    | str  | 选择时期，可选时期<br>'1M':'1月'<br/>'2M':'2月' <br/>'3M':'3月'<br/>'6M':'6月'<br/>'9M':'9月'<br/>'1Y:'1Y' |

输出参数

| 名称       | 类型  | 描述 |
| ---------- | ----- | ---- |
| 日期       | str   | -    |
| 时期       | str   | -    |
| 利率（%）  | float | -    |
| 涨跌（BP） | float | -    |

接口示例

```python
import PPshare as pp

df = pp.get_sibor(curr_type='USD', period='2M')
print(df)
```

数据示例

```
             日期      时期    利率（%）   涨跌（BP）
0    2015-09-29  2月(2M) -0.75900   -0.200
0    2015-08-17  2月(2M) -0.75700    5.000
0    2015-05-11  2月(2M) -0.80700 -100.620
0    2014-11-04  2月(2M)  0.19920    0.645
0    2014-07-14  2月(2M)  0.19275   -0.460
..          ...     ...      ...      ...
178  2007-01-11  2月(2M)  5.34927   -0.073
179  2007-01-10  2月(2M)  5.35000    0.000
180  2007-01-09  2月(2M)  5.35000    0.000
181  2007-01-08  2月(2M)  5.35000    0.300
182  2007-01-05  2月(2M)  5.34700    0.000
```

