---
layout: post
title: "Backtesting Portfolios of Leveraged ETFs in Python with Backtrader"
---
In my [last post](/2019/04/simulating-historical-performance-of-leveraged-etfs-in-python/) we discussed simulation of the 3x leveraged S&P 500 ETF, UPRO, and demonstrated why a 100% long UPRO portfolio may not be the best idea. In this post we will analyze the simulated historical performance of another 3x leveraged ETF, TMF, and explore a leveraged variation of Jack Bogle's 60 / 40 equity/bond allocation.

First lets import the libraries we need.


```python
import pandas as pd
import pandas_datareader.data as web
import datetime
import backtrader as bt
import numpy as np
%matplotlib inline
import matplotlib.pyplot as plt
plt.rcParams["figure.figsize"] = (10, 6) # (w, h)
```

We'll also need the `sim_leverage` function from my [last post](/2019/04/simulating-historical-performance-of-leveraged-etfs-in-python/).


```python
def sim_leverage(proxy, leverage=1, expense_ratio = 0.0, initial_value=1.0):
    """
    Simulates a leverage ETF given its proxy, leverage, and expense ratio.
    
    Daily percent change is calculated by taking the daily percent change of
    the proxy, subtracting the daily expense ratio, then multiplying by the leverage.
    """
    pct_change = proxy.pct_change(1)
    pct_change = (pct_change - expense_ratio / 252) * leverage
    sim = (1 + pct_change).cumprod() * initial_value
    sim[0] = initial_value
    return sim
```

For this article we will be using two leveraged ETFs: **UPRO**, a 3x leveraged S&P 500 ETF featured in my last post, and **TMF**, a 3x leveraged U.S. Treasury 20+ Year Bond Index. We can create simulations for both UPRO and TMF using the following values:

| ETF | Expense Ratio | Proxy | Proxy Inception Date |
|-----|---------------|-------|----------------------|
|UPRO | 0.92%         |VFINX  |  08/31/1976          |
|TMF  | 1.09%         |VUSTX  |  05/19/1986          |

We'll use 05/19/1986 as our start date as that is when we'll have data for both proxies.


```python
start = datetime.datetime(1986, 5, 19)
end = datetime.datetime(2019, 1, 1)

vfinx = web.DataReader("VFINX", "yahoo", start, end)["Adj Close"]
vustx = web.DataReader("VUSTX", "yahoo", start, end)["Adj Close"]

upro_sim = sim_leverage(vfinx, leverage=3.0, expense_ratio=0.0092).to_frame("close")
tmf_sim = sim_leverage(vustx, leverage=3.0, expense_ratio=0.0109).to_frame("close")
```

## Backtesting

Before we look at a multi-asset strategy, lets see how each of the assets perform with a simple buy-and-hold strategy. For backtesting our strategies, we will be using [Backtrader](https://www.backtrader.com/), a popular Python backtesting libray that also supports live trading.

In order for our data to work with Backtrader, we will have to fill in the open, high, low, and volume columns. For simplicity we will copy the close price to all columns, since we will only be trading at market close.


```python
for column in ["open", "high", "low"]:
    upro_sim[column] = upro_sim["close"]
    tmf_sim[column] = tmf_sim["close"]
    
upro_sim["volume"] = 0
tmf_sim["volume"] = 0

upro_sim = bt.feeds.PandasData(dataname=upro_sim)
tmf_sim = bt.feeds.PandasData(dataname=tmf_sim)
vfinx = bt.feeds.YahooFinanceData(dataname="VFINX", fromdate=start, todate=end)
```

Now lets write our buy-and-hold strategy:


```python
class BuyAndHold(bt.Strategy):
    def next(self):
        if not self.getposition(self.data).size:
            self.order_target_percent(self.data, target=1.0)
```

We'll also write a simple helper function that runs the backtest and returns important metrics. We will be using the [Sharpe ratio](https://en.wikipedia.org/wiki/Sharpe_ratio) to rate our strategies' performance as it is a good way of measuring risk adjusted returns.


```python
def backtest(datas, strategy, plot=False, **kwargs):
    cerebro = bt.Cerebro()
    for data in datas:
        cerebro.adddata(data)
    cerebro.addanalyzer(bt.analyzers.SharpeRatio, riskfreerate=0.0)
    cerebro.addanalyzer(bt.analyzers.Returns)
    cerebro.addanalyzer(bt.analyzers.DrawDown)
    cerebro.addstrategy(strategy, **kwargs)
    results = cerebro.run()
    if plot:
        cerebro.plot()
    return (results[0].analyzers.drawdown.get_analysis()['max']['drawdown'],
            results[0].analyzers.returns.get_analysis()['rnorm100'],
            results[0].analyzers.sharperatio.get_analysis()['sharperatio'])
```

We'll test our buy-and-hold strategy using VFINX, the S&P 500 ETF as our benchmark:


```python
dd, cagr, sharpe = backtest([vfinx], BuyAndHold, plot=True)
print(f"Max Drawdown: {dd:.2f}%\nCAGR: {cagr:.2f}%\nSharpe: {sharpe:.3f}")
```

<img src="data:image/png;base64,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" width="720">


    Max Drawdown: 55.23%
    CAGR: 9.55%
    Sharpe: 0.652


We can see that, as intended, the strategy performs a single buy, then holds the asset for the remaining years.

Lets now run the buy-and-hold strategy on UPRO.


```python
dd, cagr, sharpe = backtest([upro_sim], BuyAndHold)
print(f"Max Drawdown: {dd:.2f}%\nCAGR: {cagr:.2f}%\nSharpe: {sharpe:.3f}")
```

    Max Drawdown: 97.11%
    CAGR: 15.37%
    Sharpe: 0.541


These numbers do not exactly match those of the last post because of the slightly shorter time period. We do, however, still have the massive 97% max drawdown.

Finally lets test buy-and-hold with TMF.


```python
dd, cagr, sharpe = backtest([tmf_sim], BuyAndHold)
print(f"Max Drawdown: {dd:.2f}%\nCAGR: {cagr:.2f}%\nSharpe: {sharpe:.3f}")
```

    Max Drawdown: 50.27%
    CAGR: 16.12%
    Sharpe: 0.575


In terms of Sharpe ratio, UPRO and TMF both underperform the S&P 500 ETF. Lets see what happens when we put them together!

### Multi Asset Allocation

Vanguard founder Jack Bogle had long advocated for an portfolio solely consisting of 60% U.S. Stocks and 40% bonds. We'll use the same logic to create a 60 / 40 UPRO/TMF portfolio, rebalancing every 20 trading days. The exact allocation percent will remain a parameter so we can tinker with it later.


```python
class AssetAllocation(bt.Strategy):
    params = (
        ('equity',0.6),
    )
    def __init__(self):
        self.UPRO = self.datas[0]
        self.TMF = self.datas[1]
        self.counter = 0
        
    def next(self):
        if  self.counter % 20 == 0:
            self.order_target_percent(self.UPRO, target=self.params.equity)
            self.order_target_percent(self.TMF, target=(1 - self.params.equity))
        self.counter += 1
```

Now lets test it!


```python
dd, cagr, sharpe = backtest([upro_sim, tmf_sim], AssetAllocation, plot=True, equity=0.6)
print(f"Max Drawdown: {dd:.2f}%\nCAGR: {cagr:.2f}%\nSharpe: {sharpe:.3f}")
```

<img src="data:image/png;base64,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" width="720">


    Max Drawdown: 61.62%
    CAGR: 18.35%
    Sharpe: 0.631


The Sharpe ratio is now higher than our S&P 500 benchmark. Lets see if we can improve it any more by optimizing the allocations.

### Optimization

*Note: When optimizing parameters, one must be wary of overfitting. From the Backtrader website: "There is plenty of literature about Optimization and associated pros and cons. But the advice will always point in the same direction: do not overoptimize. If a trading idea is not sound, optimizing may end producing a positive result which is only valid for the backtested dataset."*

Lets run our backtest for all allocations of UPRO and TMF in 5% Intervals, and take note of each resulting Sharpe ratio. Backtrader does have built in [parameter optimization](https://www.backtrader.com/docu/quickstart/quickstart.html#let-s-optimize) functionality, it requires multithreading, which does not work within Jupyter Notebooks.


```python
sharpes = {}
for perc_equity in range(0, 101, 5):
    sharpes[perc_equity] = backtest([upro_sim, tmf_sim], AssetAllocation, equity=(perc_equity / 100.0))[2]
```

This may take a minute or two. Once we have the results, we can graph how the portfolio allocation effects the Sharpe ratio, and find the optimal allocation.


```python
series = pd.Series(sharpes)
ax = series.plot(title="UPRO/TMF allocation vs Sharpe")
ax.set_ylabel("Sharpe Ratio")
ax.set_xlabel("Percent Portfolio UPRO");
print(f"Max Sharpe of {series.max():.3f} at {series.idxmax()}% UPRO")
```

<img src="data:image/png;base64,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" width="720">


    Max Sharpe of 0.743 at 40% UPRO


In order to achieve the best Sharpe ratio in the backtest the best UPRO/TMF allocation is 40 / 60. Lets run one more backtest with this allocation.


```python
dd, cagr, sharpe = backtest([upro_sim, tmf_sim], AssetAllocation, plot=True, equity=0.4)
print(f"Max Drawdown: {dd:.2f}%\nCAGR: {cagr:.2f}%\nSharpe: {sharpe:.3f}")
```


<img src="data:image/png;base64,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" width="720">


    Max Drawdown: 43.08%
    CAGR: 20.13%
    Sharpe: 0.743


## Conclusion

As we can see above, our simulated 40 / 60 UPRO/TMF portfolio more than doubled the annual returns of the S&P 500 ETF, all while producing a greater Sharpe ratio, and lesser max drawdown. While going long a single 3x leveraged ETF is probably not a good idea, a leveraged multi-asset strategy could provide significant gains while reducing risk.  
