"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LoadBalancer = exports.LoadBalancerGeneration = void 0;
/**
 * The generations of AWS load balancing solutions.
 */
var LoadBalancerGeneration;
(function (LoadBalancerGeneration) {
    LoadBalancerGeneration[LoadBalancerGeneration["FIRST"] = 0] = "FIRST";
    LoadBalancerGeneration[LoadBalancerGeneration["SECOND"] = 1] = "SECOND";
})(LoadBalancerGeneration = exports.LoadBalancerGeneration || (exports.LoadBalancerGeneration = {}));
/**
 * An interface of an abstract load balancer, as needed by CodeDeploy.
 *
 * Create instances using the static factory methods:
 * {@link #classic}, {@link #application} and {@link #network}.
 */
class LoadBalancer {
    /**
     * Creates a new CodeDeploy load balancer from a Classic ELB Load Balancer.
     *
     * @param loadBalancer a classic ELB Load Balancer.
     */
    static classic(loadBalancer) {
        class ClassicLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.FIRST;
                this.name = loadBalancer.loadBalancerName;
            }
        }
        return new ClassicLoadBalancer();
    }
    /**
     * Creates a new CodeDeploy load balancer from an Application Load Balancer Target Group.
     *
     * @param albTargetGroup an ALB Target Group.
     */
    static application(albTargetGroup) {
        class AlbLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.SECOND;
                this.name = albTargetGroup.targetGroupName;
            }
        }
        return new AlbLoadBalancer();
    }
    /**
     * Creates a new CodeDeploy load balancer from a Network Load Balancer Target Group.
     *
     * @param nlbTargetGroup an NLB Target Group.
     */
    static network(nlbTargetGroup) {
        class NlbLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.SECOND;
                this.name = nlbTargetGroup.targetGroupName;
            }
        }
        return new NlbLoadBalancer();
    }
}
exports.LoadBalancer = LoadBalancer;
//# sourceMappingURL=data:application/json;base64,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