"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.TLSSecurityPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const url_1 = require("url");
const acm = require("@aws-cdk/aws-certificatemanager");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const route53 = require("@aws-cdk/aws-route53");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
const opensearch_access_policy_1 = require("./opensearch-access-policy");
const opensearchservice_generated_1 = require("./opensearchservice.generated");
const perms = require("./perms");
/**
 * The minimum TLS version required for traffic to the domain.
 *
 * @stability stable
 */
var TLSSecurityPolicy;
(function (TLSSecurityPolicy) {
    TLSSecurityPolicy["TLS_1_0"] = "Policy-Min-TLS-1-0-2019-07";
    TLSSecurityPolicy["TLS_1_2"] = "Policy-Min-TLS-1-2-2019-07";
})(TLSSecurityPolicy = exports.TLSSecurityPolicy || (exports.TLSSecurityPolicy = {}));
/**
 * A new or imported domain.
 */
class DomainBase extends cdk.Resource {
    /**
     * Grant read permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantRead(identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantWrite(identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read/write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantReadWrite(identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexRead(index, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexWrite(index, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read/write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexReadWrite(index, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathRead(path, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathWrite(path, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant read/write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathReadWrite(path, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Return the given named metric for this domain.
     *
     * @stability stable
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ES',
            metricName,
            dimensions: {
                DomainName: this.domainName,
                ClientId: this.stack.account,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricClusterStatusRed(props) {
        return this.metric('ClusterStatus.red', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricClusterStatusYellow(props) {
        return this.metric('ClusterStatus.yellow', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     * @stability stable
     */
    metricFreeStorageSpace(props) {
        return this.metric('FreeStorageSpace', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...props,
        });
    }
    /**
     * Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     * @stability stable
     */
    metricClusterIndexWritesBlocked(props) {
        return this.metric('ClusterIndexWritesBlocked', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     * @stability stable
     */
    metricNodes(props) {
        return this.metric('Nodes', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            period: cdk.Duration.hours(1),
            ...props,
        });
    }
    /**
     * Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricAutomatedSnapshotFailure(props) {
        return this.metric('AutomatedSnapshotFailure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricCPUUtilization(props) {
        return this.metric('CPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricJVMMemoryPressure(props) {
        return this.metric('JVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricMasterCPUUtilization(props) {
        return this.metric('MasterCPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricMasterJVMMemoryPressure(props) {
        return this.metric('MasterJVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricKMSKeyError(props) {
        return this.metric('KMSKeyError', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricKMSKeyInaccessible(props) {
        return this.metric('KMSKeyInaccessible', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricSearchableDocuments(props) {
        return this.metric('SearchableDocuments', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for search latency.
     *
     * @default p99 over 5 minutes
     * @stability stable
     */
    metricSearchLatency(props) {
        return this.metric('SearchLatency', { statistic: 'p99', ...props });
    }
    /**
     * Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     * @stability stable
     */
    metricIndexingLatency(props) {
        return this.metric('IndexingLatency', { statistic: 'p99', ...props });
    }
    grant(grantee, domainActions, resourceArn, ...otherResourceArns) {
        const resourceArns = [resourceArn, ...otherResourceArns];
        const grant = iam.Grant.addToPrincipal({
            grantee,
            actions: domainActions,
            resourceArns,
            scope: this,
        });
        return grant;
    }
}
/**
 * Provides an Amazon OpenSearch Service domain.
 *
 * @stability stable
 */
class Domain extends DomainBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23, _24, _25, _26, _27, _28, _29, _30, _31, _32, _33, _34, _35, _36, _37;
        super(scope, id, {
            physicalName: props.domainName,
        });
        const defaultInstanceType = 'r5.large.search';
        const warmDefaultInstanceType = 'ultrawarm1.medium.search';
        const dedicatedMasterType = (_d = (_c = (_b = props.capacity) === null || _b === void 0 ? void 0 : _b.masterNodeInstanceType) === null || _c === void 0 ? void 0 : _c.toLowerCase()) !== null && _d !== void 0 ? _d : defaultInstanceType;
        const dedicatedMasterCount = (_f = (_e = props.capacity) === null || _e === void 0 ? void 0 : _e.masterNodes) !== null && _f !== void 0 ? _f : 0;
        const dedicatedMasterEnabled = dedicatedMasterCount > 0;
        const instanceType = (_j = (_h = (_g = props.capacity) === null || _g === void 0 ? void 0 : _g.dataNodeInstanceType) === null || _h === void 0 ? void 0 : _h.toLowerCase()) !== null && _j !== void 0 ? _j : defaultInstanceType;
        const instanceCount = (_l = (_k = props.capacity) === null || _k === void 0 ? void 0 : _k.dataNodes) !== null && _l !== void 0 ? _l : 1;
        const warmType = (_p = (_o = (_m = props.capacity) === null || _m === void 0 ? void 0 : _m.warmInstanceType) === null || _o === void 0 ? void 0 : _o.toLowerCase()) !== null && _p !== void 0 ? _p : warmDefaultInstanceType;
        const warmCount = (_r = (_q = props.capacity) === null || _q === void 0 ? void 0 : _q.warmNodes) !== null && _r !== void 0 ? _r : 0;
        const warmEnabled = warmCount > 0;
        const availabilityZoneCount = (_t = (_s = props.zoneAwareness) === null || _s === void 0 ? void 0 : _s.availabilityZoneCount) !== null && _t !== void 0 ? _t : 2;
        if (![2, 3].includes(availabilityZoneCount)) {
            throw new Error('Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3');
        }
        const zoneAwarenessEnabled = (_v = (_u = props.zoneAwareness) === null || _u === void 0 ? void 0 : _u.enabled) !== null && _v !== void 0 ? _v : ((_w = props.zoneAwareness) === null || _w === void 0 ? void 0 : _w.availabilityZoneCount) != null;
        let securityGroups;
        let subnets;
        if (props.vpc) {
            subnets = selectSubnets(props.vpc, (_x = props.vpcSubnets) !== null && _x !== void 0 ? _x : [{ subnetType: ec2.SubnetType.PRIVATE }]);
            securityGroups = (_y = props.securityGroups) !== null && _y !== void 0 ? _y : [new ec2.SecurityGroup(this, 'SecurityGroup', {
                    vpc: props.vpc,
                    description: `Security group for domain ${this.node.id}`,
                })];
            this._connections = new ec2.Connections({ securityGroups });
        }
        // If VPC options are supplied ensure that the number of subnets matches the number AZ
        if (subnets && zoneAwarenessEnabled && new Set(subnets.map((subnet) => subnet.availabilityZone)).size < availabilityZoneCount) {
            throw new Error('When providing vpc options you need to provide a subnet for each AZ you are using');
        }
        if ([dedicatedMasterType, instanceType, warmType].some(t => !t.endsWith('.search'))) {
            throw new Error('Master, data and UltraWarm node instance types must end with ".search".');
        }
        if (!warmType.startsWith('ultrawarm')) {
            throw new Error('UltraWarm node instance type must start with "ultrawarm".');
        }
        const unsignedBasicAuthEnabled = (_z = props.useUnsignedBasicAuth) !== null && _z !== void 0 ? _z : false;
        if (unsignedBasicAuthEnabled) {
            if (props.enforceHttps == false) {
                throw new Error('You cannot disable HTTPS and use unsigned basic auth');
            }
            if (props.nodeToNodeEncryption == false) {
                throw new Error('You cannot disable node to node encryption and use unsigned basic auth');
            }
            if (((_0 = props.encryptionAtRest) === null || _0 === void 0 ? void 0 : _0.enabled) == false) {
                throw new Error('You cannot disable encryption at rest and use unsigned basic auth');
            }
        }
        const unsignedAccessPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:ESHttp*'],
            principals: [new iam.AnyPrincipal()],
            resources: [cdk.Lazy.string({ produce: () => `${this.domainArn}/*` })],
        });
        const masterUserArn = (_1 = props.fineGrainedAccessControl) === null || _1 === void 0 ? void 0 : _1.masterUserArn;
        const masterUserNameProps = (_2 = props.fineGrainedAccessControl) === null || _2 === void 0 ? void 0 : _2.masterUserName;
        // If basic auth is enabled set the user name to admin if no other user info is supplied.
        const masterUserName = unsignedBasicAuthEnabled
            ? (masterUserArn == null ? (masterUserNameProps !== null && masterUserNameProps !== void 0 ? masterUserNameProps : 'admin') : undefined)
            : masterUserNameProps;
        if (masterUserArn != null && masterUserName != null) {
            throw new Error('Invalid fine grained access control settings. Only provide one of master user ARN or master user name. Not both.');
        }
        const advancedSecurityEnabled = (masterUserArn !== null && masterUserArn !== void 0 ? masterUserArn : masterUserName) != null;
        const internalUserDatabaseEnabled = masterUserName != null;
        const masterUserPasswordProp = (_3 = props.fineGrainedAccessControl) === null || _3 === void 0 ? void 0 : _3.masterUserPassword;
        const createMasterUserPassword = () => {
            return new secretsmanager.Secret(this, 'MasterUser', {
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({
                        username: masterUserName,
                    }),
                    generateStringKey: 'password',
                    excludeCharacters: "{}'\\*[]()`",
                },
            })
                .secretValueFromJson('password');
        };
        this.masterUserPassword = internalUserDatabaseEnabled ?
            (masterUserPasswordProp !== null && masterUserPasswordProp !== void 0 ? masterUserPasswordProp : createMasterUserPassword())
            : undefined;
        const encryptionAtRestEnabled = (_5 = (_4 = props.encryptionAtRest) === null || _4 === void 0 ? void 0 : _4.enabled) !== null && _5 !== void 0 ? _5 : (((_6 = props.encryptionAtRest) === null || _6 === void 0 ? void 0 : _6.kmsKey) != null || unsignedBasicAuthEnabled);
        const nodeToNodeEncryptionEnabled = (_7 = props.nodeToNodeEncryption) !== null && _7 !== void 0 ? _7 : unsignedBasicAuthEnabled;
        const volumeSize = (_9 = (_8 = props.ebs) === null || _8 === void 0 ? void 0 : _8.volumeSize) !== null && _9 !== void 0 ? _9 : 10;
        const volumeType = (_11 = (_10 = props.ebs) === null || _10 === void 0 ? void 0 : _10.volumeType) !== null && _11 !== void 0 ? _11 : ec2.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
        const ebsEnabled = (_13 = (_12 = props.ebs) === null || _12 === void 0 ? void 0 : _12.enabled) !== null && _13 !== void 0 ? _13 : true;
        const enforceHttps = (_14 = props.enforceHttps) !== null && _14 !== void 0 ? _14 : unsignedBasicAuthEnabled;
        function isInstanceType(t) {
            return dedicatedMasterType.startsWith(t) || instanceType.startsWith(t);
        }
        ;
        function isSomeInstanceType(...instanceTypes) {
            return instanceTypes.some(isInstanceType);
        }
        ;
        function isEveryInstanceType(...instanceTypes) {
            return instanceTypes.some(t => dedicatedMasterType.startsWith(t))
                && instanceTypes.some(t => instanceType.startsWith(t));
        }
        ;
        // Validate feature support for the given Elasticsearch/OpenSearch version, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/features-by-version.html
        const { versionNum: versionNum, isElasticsearchVersion } = parseVersion(props.version);
        if (isElasticsearchVersion) {
            if (versionNum <= 7.7 &&
                ![
                    1.5, 2.3, 5.1, 5.3, 5.5, 5.6, 6.0,
                    6.2, 6.3, 6.4, 6.5, 6.7, 6.8, 7.1, 7.4,
                    7.7,
                ].includes(versionNum)) {
                throw new Error(`Unknown Elasticsearch version: ${versionNum}`);
            }
            if (versionNum < 5.1) {
                if ((_15 = props.logging) === null || _15 === void 0 ? void 0 : _15.appLogEnabled) {
                    throw new Error('Error logs publishing requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if ((_16 = props.encryptionAtRest) === null || _16 === void 0 ? void 0 : _16.enabled) {
                    throw new Error('Encryption of data at rest requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if (props.cognitoDashboardsAuth != null) {
                    throw new Error('Cognito authentication for OpenSearch Dashboards requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if (isSomeInstanceType('c5', 'i3', 'm5', 'r5')) {
                    throw new Error('C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.0) {
                if (props.nodeToNodeEncryption) {
                    throw new Error('Node-to-node encryption requires Elasticsearch version 6.0 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.7) {
                if (unsignedBasicAuthEnabled) {
                    throw new Error('Using unsigned basic auth requires Elasticsearch version 6.7 or later or OpenSearch version 1.0 or later.');
                }
                if (advancedSecurityEnabled) {
                    throw new Error('Fine-grained access control requires Elasticsearch version 6.7 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.8 && warmEnabled) {
                throw new Error('UltraWarm requires Elasticsearch version 6.8 or later or OpenSearch version 1.0 or later.');
            }
        }
        // Validate against instance type restrictions, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/supported-instance-types.html
        if (isSomeInstanceType('i3', 'r6gd') && ebsEnabled) {
            throw new Error('I3 and R6GD instance types do not support EBS storage volumes.');
        }
        if (isSomeInstanceType('m3', 'r3', 't2') && encryptionAtRestEnabled) {
            throw new Error('M3, R3, and T2 instance types do not support encryption of data at rest.');
        }
        if (isInstanceType('t2.micro') && !(isElasticsearchVersion && versionNum <= 2.3)) {
            throw new Error('The t2.micro.search instance type supports only Elasticsearch versions 1.5 and 2.3.');
        }
        if (isSomeInstanceType('t2', 't3') && warmEnabled) {
            throw new Error('T2 and T3 instance types do not support UltraWarm storage.');
        }
        // Only R3, I3 and r6gd support instance storage, per
        // https://aws.amazon.com/opensearch-service/pricing/
        if (!ebsEnabled && !isEveryInstanceType('r3', 'i3', 'r6gd')) {
            throw new Error('EBS volumes are required when using instance types other than r3, i3 or r6gd.');
        }
        // Fine-grained access control requires node-to-node encryption, encryption at rest,
        // and enforced HTTPS.
        if (advancedSecurityEnabled) {
            if (!nodeToNodeEncryptionEnabled) {
                throw new Error('Node-to-node encryption is required when fine-grained access control is enabled.');
            }
            if (!encryptionAtRestEnabled) {
                throw new Error('Encryption-at-rest is required when fine-grained access control is enabled.');
            }
            if (!enforceHttps) {
                throw new Error('Enforce HTTPS is required when fine-grained access control is enabled.');
            }
        }
        // Validate fine grained access control enabled for audit logs, per
        // https://aws.amazon.com/about-aws/whats-new/2020/09/elasticsearch-audit-logs-now-available-on-amazon-elasticsearch-service/
        if (((_17 = props.logging) === null || _17 === void 0 ? void 0 : _17.auditLogEnabled) && !advancedSecurityEnabled) {
            throw new Error('Fine-grained access control is required when audit logs publishing is enabled.');
        }
        // Validate UltraWarm requirement for dedicated master nodes, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/ultrawarm.html
        if (warmEnabled && !dedicatedMasterEnabled) {
            throw new Error('Dedicated master node is required when UltraWarm storage is enabled.');
        }
        let cfnVpcOptions;
        if (securityGroups && subnets) {
            cfnVpcOptions = {
                securityGroupIds: securityGroups.map((sg) => sg.securityGroupId),
                subnetIds: subnets.map((subnet) => subnet.subnetId),
            };
        }
        // Setup logging
        const logGroups = [];
        if ((_18 = props.logging) === null || _18 === void 0 ? void 0 : _18.slowSearchLogEnabled) {
            this.slowSearchLogGroup = (_19 = props.logging.slowSearchLogGroup) !== null && _19 !== void 0 ? _19 : new logs.LogGroup(this, 'SlowSearchLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowSearchLogGroup);
        }
        ;
        if ((_20 = props.logging) === null || _20 === void 0 ? void 0 : _20.slowIndexLogEnabled) {
            this.slowIndexLogGroup = (_21 = props.logging.slowIndexLogGroup) !== null && _21 !== void 0 ? _21 : new logs.LogGroup(this, 'SlowIndexLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowIndexLogGroup);
        }
        ;
        if ((_22 = props.logging) === null || _22 === void 0 ? void 0 : _22.appLogEnabled) {
            this.appLogGroup = (_23 = props.logging.appLogGroup) !== null && _23 !== void 0 ? _23 : new logs.LogGroup(this, 'AppLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.appLogGroup);
        }
        ;
        if ((_24 = props.logging) === null || _24 === void 0 ? void 0 : _24.auditLogEnabled) {
            this.auditLogGroup = (_25 = props.logging.auditLogGroup) !== null && _25 !== void 0 ? _25 : new logs.LogGroup(this, 'AuditLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.auditLogGroup);
        }
        ;
        let logGroupResourcePolicy = null;
        if (logGroups.length > 0) {
            const logPolicyStatement = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: logGroups.map((lg) => lg.logGroupArn),
                principals: [new iam.ServicePrincipal('es.amazonaws.com')],
            });
            // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
            // https://github.com/aws/aws-cdk/issues/5343
            logGroupResourcePolicy = new log_group_resource_policy_1.LogGroupResourcePolicy(this, `ESLogGroupPolicy${this.node.addr}`, {
                // create a cloudwatch logs resource policy name that is unique to this domain instance
                policyName: `ESLogPolicy${this.node.addr}`,
                policyStatements: [logPolicyStatement],
            });
        }
        const logPublishing = {};
        if (this.appLogGroup) {
            logPublishing.ES_APPLICATION_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.appLogGroup.logGroupArn,
            };
        }
        if (this.slowSearchLogGroup) {
            logPublishing.SEARCH_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowSearchLogGroup.logGroupArn,
            };
        }
        if (this.slowIndexLogGroup) {
            logPublishing.INDEX_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowIndexLogGroup.logGroupArn,
            };
        }
        if (this.auditLogGroup) {
            logPublishing.AUDIT_LOGS = {
                enabled: this.auditLogGroup != null,
                cloudWatchLogsLogGroupArn: (_26 = this.auditLogGroup) === null || _26 === void 0 ? void 0 : _26.logGroupArn,
            };
        }
        let customEndpointCertificate;
        if (props.customEndpoint) {
            if (props.customEndpoint.certificate) {
                customEndpointCertificate = props.customEndpoint.certificate;
            }
            else {
                customEndpointCertificate = new acm.Certificate(this, 'CustomEndpointCertificate', {
                    domainName: props.customEndpoint.domainName,
                    validation: props.customEndpoint.hostedZone ? acm.CertificateValidation.fromDns(props.customEndpoint.hostedZone) : undefined,
                });
            }
        }
        // Create the domain
        this.domain = new opensearchservice_generated_1.CfnDomain(this, 'Resource', {
            domainName: this.physicalName,
            engineVersion: props.version.version,
            clusterConfig: {
                dedicatedMasterEnabled,
                dedicatedMasterCount: dedicatedMasterEnabled
                    ? dedicatedMasterCount
                    : undefined,
                dedicatedMasterType: dedicatedMasterEnabled
                    ? dedicatedMasterType
                    : undefined,
                instanceCount,
                instanceType,
                warmEnabled: warmEnabled
                    ? warmEnabled
                    : undefined,
                warmCount: warmEnabled
                    ? warmCount
                    : undefined,
                warmType: warmEnabled
                    ? warmType
                    : undefined,
                zoneAwarenessEnabled,
                zoneAwarenessConfig: zoneAwarenessEnabled
                    ? { availabilityZoneCount }
                    : undefined,
            },
            ebsOptions: {
                ebsEnabled,
                volumeSize: ebsEnabled ? volumeSize : undefined,
                volumeType: ebsEnabled ? volumeType : undefined,
                iops: ebsEnabled ? (_27 = props.ebs) === null || _27 === void 0 ? void 0 : _27.iops : undefined,
            },
            encryptionAtRestOptions: {
                enabled: encryptionAtRestEnabled,
                kmsKeyId: encryptionAtRestEnabled
                    ? (_29 = (_28 = props.encryptionAtRest) === null || _28 === void 0 ? void 0 : _28.kmsKey) === null || _29 === void 0 ? void 0 : _29.keyId : undefined,
            },
            nodeToNodeEncryptionOptions: { enabled: nodeToNodeEncryptionEnabled },
            logPublishingOptions: logPublishing,
            cognitoOptions: {
                enabled: props.cognitoDashboardsAuth != null,
                identityPoolId: (_30 = props.cognitoDashboardsAuth) === null || _30 === void 0 ? void 0 : _30.identityPoolId,
                roleArn: (_31 = props.cognitoDashboardsAuth) === null || _31 === void 0 ? void 0 : _31.role.roleArn,
                userPoolId: (_32 = props.cognitoDashboardsAuth) === null || _32 === void 0 ? void 0 : _32.userPoolId,
            },
            vpcOptions: cfnVpcOptions,
            snapshotOptions: props.automatedSnapshotStartHour
                ? { automatedSnapshotStartHour: props.automatedSnapshotStartHour }
                : undefined,
            domainEndpointOptions: {
                enforceHttps,
                tlsSecurityPolicy: (_33 = props.tlsSecurityPolicy) !== null && _33 !== void 0 ? _33 : TLSSecurityPolicy.TLS_1_0,
                ...props.customEndpoint && {
                    customEndpointEnabled: true,
                    customEndpoint: props.customEndpoint.domainName,
                    customEndpointCertificateArn: customEndpointCertificate.certificateArn,
                },
            },
            advancedSecurityOptions: advancedSecurityEnabled
                ? {
                    enabled: true,
                    internalUserDatabaseEnabled,
                    masterUserOptions: {
                        masterUserArn: masterUserArn,
                        masterUserName: masterUserName,
                        masterUserPassword: (_34 = this.masterUserPassword) === null || _34 === void 0 ? void 0 : _34.toString(),
                    },
                }
                : undefined,
            advancedOptions: props.advancedOptions,
        });
        this.domain.applyRemovalPolicy(props.removalPolicy);
        if (props.enableVersionUpgrade) {
            this.domain.cfnOptions.updatePolicy = {
                ...this.domain.cfnOptions.updatePolicy,
                enableVersionUpgrade: props.enableVersionUpgrade,
            };
        }
        if (logGroupResourcePolicy) {
            this.domain.node.addDependency(logGroupResourcePolicy);
        }
        if (props.domainName) {
            if (!cdk.Token.isUnresolved(props.domainName)) {
                // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/configuration-api.html#configuration-api-datatypes-domainname
                if (!props.domainName.match(/^[a-z0-9\-]+$/)) {
                    throw new Error(`Invalid domainName '${props.domainName}'. Valid characters are a-z (lowercase only), 0-9, and – (hyphen).`);
                }
                if (props.domainName.length < 3 || props.domainName.length > 28) {
                    throw new Error(`Invalid domainName '${props.domainName}'. It must be between 3 and 28 characters`);
                }
                if (props.domainName[0] < 'a' || props.domainName[0] > 'z') {
                    throw new Error(`Invalid domainName '${props.domainName}'. It must start with a lowercase letter`);
                }
            }
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.domainName);
        }
        this.domainName = this.getResourceNameAttribute(this.domain.ref);
        this.domainId = this.domain.getAtt('Id').toString();
        this.domainEndpoint = this.domain.getAtt('DomainEndpoint').toString();
        this.domainArn = this.getResourceArnAttribute(this.domain.attrArn, {
            service: 'es',
            resource: 'domain',
            resourceName: this.physicalName,
        });
        if ((_35 = props.customEndpoint) === null || _35 === void 0 ? void 0 : _35.hostedZone) {
            new route53.CnameRecord(this, 'CnameRecord', {
                recordName: props.customEndpoint.domainName,
                zone: props.customEndpoint.hostedZone,
                domainName: this.domainEndpoint,
            });
        }
        const accessPolicyStatements = unsignedBasicAuthEnabled
            ? ((_36 = props.accessPolicies) !== null && _36 !== void 0 ? _36 : []).concat(unsignedAccessPolicy)
            : props.accessPolicies;
        if (accessPolicyStatements != null) {
            const accessPolicy = new opensearch_access_policy_1.OpenSearchAccessPolicy(this, 'Access Policy', {
                domainName: this.domainName,
                domainArn: this.domainArn,
                accessPolicies: accessPolicyStatements,
            });
            if ((_37 = props.encryptionAtRest) === null || _37 === void 0 ? void 0 : _37.kmsKey) {
                // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/encryption-at-rest.html
                // these permissions are documented as required during domain creation.
                // while not strictly documented for updates as well, it stands to reason that an update
                // operation might require these in case the cluster uses a kms key.
                // empircal evidence shows this is indeed required: https://github.com/aws/aws-cdk/issues/11412
                accessPolicy.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                    actions: ['kms:List*', 'kms:Describe*', 'kms:CreateGrant'],
                    resources: [props.encryptionAtRest.kmsKey.keyArn],
                    effect: iam.Effect.ALLOW,
                }));
            }
            accessPolicy.node.addDependency(this.domain);
        }
    }
    /**
     * Creates a domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     * @stability stable
     */
    static fromDomainEndpoint(scope, id, domainEndpoint) {
        const stack = cdk.Stack.of(scope);
        const domainName = extractNameFromEndpoint(domainEndpoint);
        const domainArn = stack.formatArn({
            service: 'es',
            resource: 'domain',
            resourceName: domainName,
        });
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainEndpoint,
        });
    }
    /**
     * Creates a domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     * @stability stable
     */
    static fromDomainAttributes(scope, id, attrs) {
        var _b;
        const { domainArn, domainEndpoint } = attrs;
        const domainName = (_b = cdk.Stack.of(scope).parseArn(domainArn).resourceName) !== null && _b !== void 0 ? _b : extractNameFromEndpoint(domainEndpoint);
        return new class extends DomainBase {
            constructor() {
                super(scope, id);
                this.domainArn = domainArn;
                this.domainName = domainName;
                this.domainId = domainName;
                this.domainEndpoint = domainEndpoint;
            }
        };
    }
    /**
     * Manages network connections to the domain.
     *
     * This will throw an error in case the domain
     * is not placed inside a VPC.
     *
     * @stability stable
     */
    get connections() {
        if (!this._connections) {
            throw new Error("Connections are only available on VPC enabled domains. Use the 'vpc' property to place a domain inside a VPC");
        }
        return this._connections;
    }
}
exports.Domain = Domain;
_a = JSII_RTTI_SYMBOL_1;
Domain[_a] = { fqn: "@aws-cdk/aws-opensearchservice.Domain", version: "1.129.0" };
/**
 * Given an Amazon OpenSearch Service domain endpoint, returns a CloudFormation expression that
 * extracts the domain name.
 *
 * Domain endpoints look like this:
 *
 *   https://example-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com
 *   https://<domain-name>-<suffix>.<region>.es.amazonaws.com
 *
 * ..which means that in order to extract the domain name from the endpoint, we can
 * split the endpoint using "-<suffix>" and select the component in index 0.
 *
 * @param domainEndpoint The Amazon OpenSearch Service domain endpoint
 */
function extractNameFromEndpoint(domainEndpoint) {
    const { hostname } = new url_1.URL(domainEndpoint);
    const domain = hostname.split('.')[0];
    const suffix = '-' + domain.split('-').slice(-1)[0];
    return domain.split(suffix)[0];
}
/**
 * Converts an engine version into a into a decimal number with major and minor version i.e x.y.
 *
 * @param version The engine version object
 */
function parseVersion(version) {
    const elasticsearchPrefix = 'Elasticsearch_';
    const openSearchPrefix = 'OpenSearch_';
    const isElasticsearchVersion = version.version.startsWith(elasticsearchPrefix);
    const versionStr = isElasticsearchVersion
        ? version.version.substring(elasticsearchPrefix.length)
        : version.version.substring(openSearchPrefix.length);
    const firstDot = versionStr.indexOf('.');
    if (firstDot < 1) {
        throw new Error(`Invalid engine version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
    const secondDot = versionStr.indexOf('.', firstDot + 1);
    try {
        if (secondDot == -1) {
            return { versionNum: parseFloat(versionStr), isElasticsearchVersion };
        }
        else {
            return { versionNum: parseFloat(versionStr.substring(0, secondDot)), isElasticsearchVersion };
        }
    }
    catch (error) {
        throw new Error(`Invalid engine version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
}
function selectSubnets(vpc, vpcSubnets) {
    const selected = [];
    for (const selection of vpcSubnets) {
        selected.push(...vpc.selectSubnets(selection).subnets);
    }
    return selected;
}
//# sourceMappingURL=data:application/json;base64,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