"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("@aws-cdk/core");
require("@aws-cdk/assert/jest");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
test('check for default props', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "process solutions constructs messages",
                sql: "SELECT * FROM 'solutions/constructs'",
                actions: []
            }
        }
    };
    const construct = new lib_1.IotToS3(stack, 'test-iot-s3-integration', props);
    // Check whether construct has two s3 buckets for storing msgs and logging
    expect(stack).toCountResources('AWS::S3::Bucket', 2);
    // Check for IoT Topic Rule Definition
    expect(stack).toHaveResource('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    S3: {
                        BucketName: {
                            Ref: "testiots3integrationS3Bucket9B8B180C"
                        },
                        Key: "${topic()}/${timestamp()}",
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiots3integrationiotactionsrole04473665",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "process solutions constructs messages",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'solutions/constructs'"
        }
    });
    // Check for IAM policy to have access to s3 bucket
    /**
     * Due to difference in CDK V1 and V2 Synth, the policy documents doesn't match, hence checking only for number of policies
     */
    expect(stack).toCountResources('AWS::IAM::Policy', 1);
    // Check for properties
    expect(construct.s3Bucket).toBeDefined();
    expect(construct.s3BucketInterface).toBeDefined();
    expect(construct.s3LoggingBucket).toBeDefined();
    expect(construct.iotActionsRole).toBeDefined();
    expect(construct.iotTopicRule).toBeDefined();
});
test('check for overriden props', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: true,
                description: "process solutions constructs messages",
                sql: "SELECT * FROM 'test/constructs'",
                actions: []
            }
        },
        s3Key: 'test/key',
        bucketProps: {
            encryption: s3.BucketEncryption.KMS
        },
        loggingBucketProps: {
            encryption: s3.BucketEncryption.KMS_MANAGED
        }
    };
    const construct = new lib_1.IotToS3(stack, 'test-iot-s3-integration', props);
    // Check whether construct has two s3 buckets for storing msgs and logging
    expect(stack).toCountResources('AWS::S3::Bucket', 2);
    // Check logging bucket encryption type to be KMS_Managed
    expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "aws:kms"
                    }
                }
            ]
        }
    });
    // Check for bucket to have KMS CMK Encryption
    expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        KMSMasterKeyID: {
                            "Fn::GetAtt": [
                                "testiots3integrationS3BucketKey127368C9",
                                "Arn"
                            ]
                        },
                        SSEAlgorithm: "aws:kms"
                    }
                }
            ]
        },
    });
    // Check for IoT Topic Rule Definition
    expect(stack).toHaveResource('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    S3: {
                        BucketName: {
                            Ref: "testiots3integrationS3Bucket9B8B180C"
                        },
                        Key: "test/key",
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiots3integrationiotactionsrole04473665",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "process solutions constructs messages",
            RuleDisabled: true,
            Sql: "SELECT * FROM 'test/constructs'"
        }
    });
    /**
     * Due to difference in CDK V1 and V2 Synth, the policy documents doesn't match, hence checking only for number of policies
     */
    // Check for automatically created CMK KMS Key
    expect(stack).toCountResources('AWS::KMS::Key', 1);
    // Check for IoT Topic Rule permissions to KMS key to store msgs to S3 Bucket and access to put data to s3 bucket
    expect(stack).toCountResources('AWS::IAM::Policy', 1);
    // Check for properties
    expect(construct.s3Bucket).toBeDefined();
    expect(construct.s3BucketInterface).toBeDefined();
    expect(construct.s3LoggingBucket).toBeDefined();
    expect(construct.iotActionsRole).toBeDefined();
    expect(construct.iotTopicRule).toBeDefined();
});
test('check for existing bucket', () => {
    const stack = new cdk.Stack();
    const existingBucket = new s3.Bucket(stack, `existingBucket`);
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "process solutions constructs messages",
                sql: "SELECT * FROM 'test/constructs'",
                actions: []
            }
        },
        s3Key: 'existingtest/key',
        existingBucketInterface: existingBucket
    };
    const construct = new lib_1.IotToS3(stack, 'test-iot-s3-integration', props);
    // Check whether construct has a single s3 bucket, no logging bucket should exist since existing bucket is supplied
    expect(stack).toCountResources('AWS::S3::Bucket', 1);
    // Check for IoT Topic Rule Definition with existing Bucket Ref
    expect(stack).toHaveResource('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    S3: {
                        BucketName: {
                            Ref: "existingBucket9529822F"
                        },
                        Key: "existingtest/key",
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiots3integrationiotactionsrole04473665",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "process solutions constructs messages",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/constructs'"
        }
    });
    /**
     * Due to difference in CDK V1 and V2 Synth, the policy documents doesn't match, hence checking only for number of policies
     */
    // Check for IAM policy to have access to s3 bucket
    expect(stack).toCountResources('AWS::IAM::Policy', 1);
    // since existing bucket is supplied, no key should exist
    expect(stack).not.toHaveResource('AWS::KMS::Key', {});
    // Check for IoT Topic Rule permissions to KMS key to store msgs to S3 Bucket
    expect(stack).toCountResources("AWS::IAM::Policy", 1);
    // Check for properties
    expect(construct.s3Bucket).toBeUndefined();
    expect(construct.s3BucketInterface).toBeDefined();
    expect(construct.s3LoggingBucket).toBeUndefined();
    expect(construct.iotActionsRole).toBeDefined();
    expect(construct.iotTopicRule).toBeDefined();
});
test('check for both bucketProps and existingBucket', () => {
    const stack = new cdk.Stack();
    const existingBucket = new s3.Bucket(stack, `existingBucket`, { encryption: s3.BucketEncryption.KMS });
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "process solutions constructs messages",
                sql: "SELECT * FROM 'test/constructs'",
                actions: []
            }
        },
        bucketProps: {
            encryption: s3.BucketEncryption.KMS_MANAGED
        },
        existingBucketInterface: existingBucket
    };
    // since bucketprops and existing bucket is supplied, this should result in error
    try {
        new lib_1.IotToS3(stack, 'test-iot-s3-integration', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check for name collision', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "process solutions constructs messages",
                sql: "SELECT * FROM 'test/constructs'",
                actions: []
            }
        },
        bucketProps: {
            autoDeleteObjects: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY
        }
    };
    // since bucketprops and existing bucket is supplied, this should result in error
    new lib_1.IotToS3(stack, 'test-iot-s3-integration', props);
    new lib_1.IotToS3(stack, 'test-iot-s3-integration1', props);
    expect(stack).toCountResources('AWS::IoT::TopicRule', 2);
    expect(stack).toCountResources('AWS::S3::Bucket', 4);
});
test('check for chaining of resource', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "process solutions constructs messages",
                sql: "SELECT * FROM 'test/constructs'",
                actions: []
            }
        }
    };
    // since bucketprops and existing bucket is supplied, this should result in error
    const construct = new lib_1.IotToS3(stack, 'test-iot-s3-integration', props);
    const props1 = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "process solutions constructs messages",
                sql: "SELECT * FROM 'test/constructs'",
                actions: []
            }
        },
        existingBucketInterface: construct.s3Bucket
    };
    new lib_1.IotToS3(stack, 'test-iot-s3-integration1', props1);
    expect(stack).toCountResources('AWS::IoT::TopicRule', 2);
    expect(stack).toCountResources('AWS::S3::Bucket', 2);
});
//# sourceMappingURL=data:application/json;base64,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