# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['requests_oauth2client',
 'requests_oauth2client.flask',
 'requests_oauth2client.vendor_specific',
 'tests',
 'tests.unit_tests']

package_data = \
{'': ['*']}

install_requires = \
['cryptography>=3.4.8,<4.0.0',
 'flake8-rst-docstrings[test]>=0.2.3,<0.3.0',
 'furl>=2.1.2,<3.0.0',
 'requests>=2.26.0,<3.0.0']

extras_require = \
{'dev': ['tox>=3.20.1,<4.0.0',
         'virtualenv>=20.2.2,<21.0.0',
         'pip>=20.3.1,<21.0.0',
         'twine>=3.3.0,<4.0.0',
         'pre-commit>=2.12.0,<3.0.0',
         'toml>=0.10.2,<0.11.0'],
 'doc': ['mkdocs>=1.1.2,<2.0.0',
         'mkdocs-include-markdown-plugin>=3.2.2,<4.0.0',
         'mkdocs-material>=7.2.6,<8.0.0',
         'mkdocstrings>=0.15.2,<0.16.0',
         'mkdocs-autorefs>=0.2.1,<0.3.0',
         'livereload>=2.6.3,<3.0.0'],
 'test': ['black==21.8b0',
          'isort==5.9.3',
          'flake8==3.9.2',
          'flake8-docstrings>=1.6.0,<2.0.0',
          'pytest==6.2.5',
          'pytest-cov==2.12.1',
          'requests-mock>=1.9.3,<2.0.0',
          'flask>=2.0.1,<3.0.0',
          'mypy>=0.910,<0.911',
          'types-requests>=2.25.6,<3.0.0',
          'types-cryptography>=3.3.5,<4.0.0']}

setup_kwargs = {
    'name': 'requests-oauth2client',
    'version': '1.0.0a0',
    'description': 'An OAuth2.x Client based on and extending requests..',
    'long_description': 'A Python OAuth 2.x client, able to obtain, refresh and revoke tokens\nfrom any OAuth2.x/OIDC compliant Authorization Server.\n\nIt can act as an OAuth 2.0/2.1 client, to automatically get and renew\naccess tokens, based on the Client Credentials, Authorization Code,\nRefresh token, Device Authorization, or CIBA grants.\n\nIt comes with a [requests] add-on to handle OAuth 2.0 Bearer\nToken based authorization when accessing APIs.\n\nIt also supports OpenID Connect, PKCE, Client Assertions, Token\nRevocation, Exchange, and Introspection, Backchannel Authentication\nrequests, as well as using custom params to any endpoint, and other\nimportant features that are often overlooked in other client libraries.\n\nAnd it also includes a wrapper around [requests.Session](https://docs.python-requests.org/en/master/api/#request-sessions)\nthat makes it super easy to use REST-style APIs, with or without OAuth 2.0.\n\n# Installation\n\nAs easy as:\n\n    pip install requests_oauth2client\n\n# Usage\n\nImport it like this:\n\n    from requests_oauth2client import *\n\n\n## Calling APIs with an access token\n\nIf you already managed to obtain an access token, you can simply use the\n[BearerAuth] Auth Handler for `requests`:\n\n    token = "an_access_token"\n    resp = requests.get("https://my.protected.api/endpoint", auth=BearerAuth(token))\n\nThis authentication handler will add a properly formatted `Authorization`\nheader in the request, with your access token according to RFC6750.\n\n## Using an OAuth2Client\n\n[OAuth2Client] offers several methods that implement the\ncommunication to the various endpoints that are standardised by OAuth\n2.0 and its extensions. Those endpoints include the Token Endpoint, the Revocation, Introspection, UserInfo, BackChannel Authentication and Device Authorization Endpoints.\n\nTo initialize an [OAuth2Client], you only need a Token\nEndpoint URI, and the credentials for your application, which are often a `client_id` and a `client_secret`:\n\n    oauth2client = OAuth2Client("https://myas.local/token_endpoint", ("client_id", "client_secret"))\n\nThe Token Endpoint is the only Endpoint that is mandatory to obtain\ntokens. Credentials are used to authenticate the client everytime it\nsends a request to its Authorization Server. Usually, those are a static\nClient ID and Secret, which are the direct equivalent of a username and\na password, but meant for an application instead of for a human user. The\ndefault authentication method used by OAuth2Client is *Client Secret\nPost*, but other standardised methods such as *Client Secret Basic*,\n*Client Secret JWT* or *Private Key JWT* are supported as well. See below.\n\n## Obtaining tokens\n\n[OAuth2Client] has methods to send requests to the Token\nEndpoint using the different standardised (and/or custom) grants. Since\nthe token endpoint and authentication method are already declared for\nthe client, the only required parameters are those that will be sent in\nthe request to the Token Endpoint.\n\nThose methods directly return a [BearerToken] if the request\nis successful, or raise an exception if it fails.\n[BearerToken] will manage the token expiration, will contain\nthe eventual refresh token that matches the access token, and will keep\ntrack of other associated metadata as well. You can create such a\n[BearerToken] yourself if you need:\n\n    bearer_token = BearerToken(access_token="an_access_token", expires_in=60)\n    print(bearer_token)\n    > {\'access_token\': \'an_access_token\', \'expires_in\': 55, \'token_type\': \'Bearer\'}\n    print(bearer_token.expires_at)\n    > datetime.datetime(2021, 8, 20, 9, 56, 59, 498793)\n\nNote that the `expires_in` indicator here is not static. It keeps\ntrack of the token lifetime and is calculated as the time flies. You can\ncheck if a token is expired with\n[bearer_token.is_expired()](/api/#requests_oauth2client.tokens.BearerToken.is_expired).\n\nYou can use a [BearerToken] instance everywhere you can\nsupply an access_token as string.\n\n### Using OAuth2Client as a requests Auth Handler\n\nWhile using [OAuth2Client] directly is great for testing or debugging\nOAuth2.0 flows, it is not a viable option for actual applications where\ntokens must be obtained, used during their lifetime then obtained again\nor refreshed once they are expired. `requests_oauth2client` contains\nseveral [requests] compatible Auth Handler (subclasses of\n[requests.auth.AuthBase](https://docs.python-requests.org/en/master/user/advanced/#custom-authentication), that will take care of obtaining\ntokens when required, then will cache those tokens until they are\nexpired, and will obtain new ones (or refresh them, when possible), once\nthe initial token is expired. Those are best used with a\n[requests.Session], or an [ApiClient] which is a\nSession Subclass with a few enhancements as described below.\n\n### Client Credentials grant\n\nTo send a request using the Client Credentials grant, use the aptly\nnamed [.client_credentials()](/api/#requests_oauth2client.client.OAuth2Client.client_credentials) method:\n\n    token = oauth2client.client_credentials(\n        scope="myscope",\n        resource="https://myapi.local"\n        # you may pass additional kw params such as audience, or whatever your AS needs\n    )\n\n#### As Auth Handler\n\nYou can use the [OAuth2ClientCredentials](/api/#requests_oauth2client.auth.OAuth2ClientCredentialsAuth) auth handler. It\ntakes an [OAuth2Client] as parameter, and the additional kwargs to pass to\nthe token endpoint:\n\n    api_client = ApiClient(\n        \'https://myapi.local/resource\',\n        auth=OAuth2ClientCredentials(oauth2client, scope=\'myscope\', resource="https://myapi.local")\n    )\n\n    resp = api_client.get() # when you send your first request to the API, it will fetch an access token first.\n\n### Authorization Code Grant\n\nObtaining tokens with the Authorization code grant is made in 3 steps:\n\n1. your application must open specific url called the *Authentication\n    Request* in a browser.\n\n2. your application must obtain and validate the *Authorization\nResponse*, which is a redirection back to your application that contains\nan *Authorization Code* as parameter.\n\n3. your application must then exchange this Authorization Code for an\n*Access Token*, with a request to the Token Endpoint.\n\n[OAuth2Client] doesn\'t implement anything that is related\nto the Authorization Request or Response. It is only able to exchange\nthe Authorization Code for a Token in step 3. But\n`requests_oauth2client` has other classes to help you with\nsteps 1 and 2, as described below:\n\n#### Generating Authorization Requests\n\nYou can generate valid authorization requests with the\n[AuthorizationRequest](http://127.0.0.1:8000/api/#requests_oauth2client.authorization_request.AuthorizationRequest) class:\n\n    auth_request = AuthorizationRequest(\n        authorization_endpoint,\n        client_id,\n        redirect_uri=redirect_uri,\n        scope=scope,\n        resource=resource, # not mandatory\n    ) # add any other param that needs to be sent to your AS\n    print(auth_request) # redirect the user to that URL to get a code\n\nThis request will look like this (with line breaks for display purposes only):\n\n    https://myas.local/authorize\n    ?client_id=my_client_id\n    &redirect_uri=http%3A%2F%2Flocalhost%2Fcallback\n    &response_type=code\n    &state=kHWL4VwcbUbtPR4mtht6yMAGG_S-ZcBh5RxI_IGDmJc\n    &nonce=mSGOS1M3LYU9ncTvvutoqUR4n1EtmaC_sQ3db4dyMAc\n    &scope=openid+email+profile\n    &code_challenge=Dk11ttaDb_Hyq1dObMqQcTIlfYYRVblFMC9lFM3UWW8\n    &code_challenge_method=S256\n    &resource=https%3A%2F%2Fmy.resource.local%2Fapi\n\n[AuthorizationRequest] supports PKCE and uses it by default. You can avoid\nit by passing `code_challenge_method=None` to\n[AuthenticationRequest]. You can obtain the generated\ncode_verifier from `auth_request.code_verifier`.\n\nRedirecting or otherwise sending the user to this url is your\napplication responsibility, as well as obtaining the Authorization\nResponse url.\n\n#### Validating the Authorization Response\n\nOnce the user is successfully authenticated and authorized, the AS will\nrespond with a redirection to your redirect_uri. That is the\n*Authorization Response*. It contains several parameters that must be\nretrieved by your client. The authorization code is one of those\nparameters, but you must also validate that the *state* matches your\nrequest. You can do this with:\n\n    params = input("Please enter the full url and/or params obtained on the redirect_uri: ")\n    code = auth_request.validate_callback(params)\n\n#### Exchanging code for tokens\n\nTo exchange a code for Access and/or ID tokens, use the\n[OAuth2Client.authorization_code()](/api/#requests_oauth2client.client.OAuth2Client.authorization_code) method:\n\n    token = oauth2client.authorization_code(\n        code=code,\n        code_verifier=auth_request.code_verifier,\n        redirect_uri=redirect_uri) # redirect_uri is not always mandatory, but some AS still requires it\n\n#### As Auth Handler\n\nThe [OAuth2AuthorizationCodeAuth](/api/#requests_oauth2client.auth.OAuth2AuthorizationCodeAuth) handler takes an\n[OAuth2Client] and an authorization code as parameter, plus whatever\nadditional keyword parameters are required by your Authorization Server:\n\n    api_client = ApiClient(\n        "https://your.protected.api/endpoint",\n        auth=OAuth2AuthorizationCodeAuth(\n            client, code,\n            code_verifier=auth_request.code_verifier, redirect_uri=redirect_uri)\n\n    resp = api_client.post(data={...}) # first call will exchange the code for an initial access/refresh tokens\n\n[OAuth2AuthorizationCodeAuth](/api/#requests_oauth2client.auth.OAuth2AuthorizationCodeAuth) will take care of refreshing\nthe token automatically once it is expired, using the refresh token, if\navailable.\n\n### Device Authorization Grant\n\nHelpers for the Device Authorization Grant are also included. To get\ndevice and user codes:\n\n    client = OAuth2Client(\n        token_endpoint="https://myas.local/token",\n        device_authorization_endpoint="https://myas.local/device",\n        auth=(client_id, client_secret),\n    )\n\n    da_resp = client.authorize_device()\n\n`da_resp` contains the Device Code, User Code, Verification\nURI and other info returned by the AS:\n\n    da_resp.device_code\n    da_resp.user_code\n    da_resp.verification_uri\n    da_resp.verification_uri_complete\n    da_resp.expires_at # just like for BearerToken, expiration is tracked by requests_oauth2client\n    da_resp.interval\n\nSend/show the Verification Uri and User Code to the user. He must use a\nbrowser to visit that url, authenticate and input the User Code. You can\nthen request the Token endpoint to check if the user successfully\nauthorized you using an \\`OAuth2Client\\`:\n\n    token = client.device_code(da_resp.device_code)\n\nThis will raise an exception, either [AuthorizationPending](/api/#requests_oauth2client.exceptions.AuthorizationPending),\n[SlowDown](/api/#requests_oauth2client.exceptions.SlowDown), [ExpiredToken](/api/#requests_oauth2client.exceptions.ExpiredToken), or\n[AccessDenied](/api/#requests_oauth2client.exceptions.AccessDenied) if the user did not yet finish authorizing\nyour device, if you should increase your pooling period, or if the\ndevice code is no longer valid, or the user finally denied your access,\nrespectively. Other exceptions may be raised depending on the error code\nthat the AS responds with. If the user did finish authorizing\nsuccessfully, `token` will contain your access token.\n\nTo make pooling easier, you can use a\n[DeviceAuthorizationPoolingJob](/api/#requests_oauth2client.device_authorization.DeviceAuthorizationPoolingJob) like this:\n\n    pool_job = DeviceAuthorizationPoolingJob(\n        client,\n        device_auth_resp.device_code,\n        interval=device_auth_resp.interval\n    )\n\n    resp = None\n    while resp is None:\n        resp = pool_job()\n\n    assert isinstance(resp, BearerToken)\n\n[DeviceAuthorizationPoolingJob](/api/#requests_oauth2client.device_authorization.DeviceAuthorizationPoolingJob) will automatically obey the\npooling period. Everytime you call pool_job(), it will wait the\nappropriate number of seconds as indicated by the AS, and will apply\nslow_down requests.\n\n#### As Auth Handler\n\nUse [OAuth2DeviceCodeAuth](/api/#requests_oauth2client.auth.OAuth2DeviceCodeAuth) as auth handler to exchange a\ndevice code for an access token:\n\n    api_client = ApiClient(\n        "https://your.protected.api/endpoint",\n        auth=OAuth2DeviceCodeAuth(\n            client, device_auth_resp.device_code,\n            interval=device_auth_resp.interval, expires_in=device_auth_resp.expires_in\n        )\n\n    resp = api_client.post(data={...}) # first call will hang until the user authorizes your app and the token endpoint returns a token.\n\n## Client-Initiated Backchannel Authentication (CIBA)\n\nTo initiate a BackChannel Authentication against the dedicated endpoint:\n\n    client = OAuth2Client(\n        token_endpoint="https://myas.local/token",\n        backchannel_authentication_endpoint="https://myas.local/backchannel_authorize",\n        auth=(client_id, client_secret)\n    )\n\n    ba_resp = client.backchannel_authentication_request(\n        scope="openid email profile",\n        login_hint="user@example.net",\n    )\n\n`ba_resp` will contain the response attributes as returned\nby the AS, including an `auth_req_id`:\n\n    ba_resp.auth_req_id\n    ba_resp.expires_in # decreases as times fly\n    ba_resp.expires_at # a datetime to keep track of the expiration date, based on the "expires_in" returned by the AS\n    ba_resp.interval # the pooling interval indicated by the AS\n    ba_resp.custom # if the AS respond with additional attributes, they are also accessible\n\nTo pool the Token Endpoint until the end-user successfully\nauthenticates:\n\n    pool_job = BackChannelAuthenticationPoolingJob(\n        client=client,\n        auth_req_id=ba_resp.auth_req_id,\n        interval=bca_resp.interval,\n    )\n\n    resp = None\n    while resp is None:\n        resp = pool_job()\n\n    assert isinstance(resp, BearerToken)\n\n## Supported Client Authentication Methods\n\n`requests_oauth2client` supports multiple client\nauthentication methods, as defined in multiple OAuth2.x standards. You\nselect the appropriate method to use when initializing your\n[OAuth2Client], with the `auth` parameter. Once initialized, a\nclient will automatically use the configured authentication method every\ntime it sends a requested to an endpoint that requires client\nauthentication. You don\'t have anything else to do afterwards.\n\n- **client_secret_basic**: client_id and client_secret are included in\n    clear-text in the Authorization header. To use it, just pass a\n    [ClientSecretBasic(client_id, client_secret)](/api/#requests_oauth2client.client_authentication.ClientSecretBasic)} as `auth` parameter:\n\n        client = OAuth2Client(token_endpoint, auth=ClientSecretBasic(client_id, client_secret))\n\n- **client_secret_post**: client_id and client_secret are included as\n    part of the body form data. To use it, pass a\n    [ClientSecretPost(client_id, client_secret)](/api/#requests_oauth2client.client_authentication.ClientSecretPost) as `auth`\n    parameter. This also what is being used as default when you pass a\n    tuple `(client_id, client_secret)` as `auth`:\n\n        client = OAuth2Client(token_endpoint, auth=ClientSecretPost(client_id, client_secret))\n        # or\n        client = OAuth2Client(token_endpoint, auth=(client_id, client_secret))\n\n- **client_secret_jwt**: client generates an ephemeral JWT assertion\n    including information about itself (client_id), the AS (url of the\n    endpoint), and expiration date. To use it, pass a\n    [ClientSecretJWT(client_id, client_secret)](/api/#requests_oauth2client.client_authentication.ClientSecretJWT) as `auth`\n    parameter. Assertion generation is entirely automatic, you don\'t\n    have anything to do:\n\n        client = OAuth2Client(token_endpoint, auth=ClientSecretJWT(client_id, client_secret))\n\n- **private_key_jwt**: client uses a JWT assertion like\n    _client_secret_jwt_, but it is signed with an _asymmetric_ key. To use\n    it, you need a private signing key, in a `dict` that\n    matches the JWK format. The matching public key must be registered\n    for your client on AS side. Once you have that, using this auth\n    method is as simple with the [PrivateKeyJWT(client_id, private_jwk)](/api/#requests_oauth2client.client_authentication.PrivateKeyJWT) auth\n    handler:\n\n        private_jwk = {\n            "kid": "mykid",\n            "kty": "RSA",\n            "e": "AQAB", "n": "...", "d": "...", "p": "...",\n            "q": "...", "dp": "...", "dq": "...", "qi": "...",\n        }\n\n        client = OAuth2Client(\n            "https://myas.local/token",\n             auth=PrivateKeyJWT(client_id, private_jwk)\n        )\n\n- **none**: client only presents its client_id in body form data to\n    the AS, without any authentication credentials. Use\n    [PublicApp(client_id)](/api/#requests_oauth2client.client_authentication.PublicApp):\n\n        client = OAuth2Client(token_endpoint, auth=PublicApp(client_id, client_secret))\n\n## Token Exchange\n\nTo send a token exchange request, use the\n[OAuth2Client.token_exchange()](/api/#requests_oauth2client.client.OAuth2Client.token_exchange) method:\n\n    client = OAuth2Client(token_endpoint, auth=...)\n    token = client.token_exchange(\n        subject_token=\'your_token_value\',\n        subject_token_type="urn:ietf:params:oauth:token-type:access_token"\n    )\n\nAs with the other grant-type specific methods, you may specify\nadditional keyword parameters, that will be passed to the token\nendpoint, including any standardised attribute like\n`actor_token` or `actor_token_type`, or any\ncustom parameter. There are short names for token types, that will be\nautomatically translated to standardised types:\n\n    token = client.token_exchange(\n        subject_token=\'your_token_value\',\n        subject_token_type="access_token", # will be automatically replaced by "urn:ietf:params:oauth:token-type:access_token"\n        actor_token=\'your_actor_token\',\n        actor_token_type=\'id_token\', # will be automatically replaced by "urn:ietf:params:oauth:token-type:id_token"\n    )\n\nOr to make it even easier, types can be guessed based on the supplied\nsubject or actor token:\n\n    token = client.token_exchange(\n        subject_token=BearerToken(\'your_token_value\'),  # subject_token_type will be "urn:ietf:params:oauth:token-type:access_token"\n        actor_token=IdToken(\'your_actor_token\'), # actor_token_type will be "urn:ietf:params:oauth:token-type:id_token"\n    )\n\n## Token Revocation\n\n[OAuth2Client] can send revocation requests to a Revocation\nEndpoint. You need to provide a Revocation Endpoint URI when creating\nthe \\`OAuth2Client\\`:\n\n    oauth2client = OAuth2Client(\n        token_endpoint,\n        revocation_endpoint=revocation_endpoint,\n        auth=ClientSecretJWT("client_id", "client_secret"))\n\nThe [OAuth2Client.revoke_token()](/api/#requests_oauth2client.client.OAuth2Client.revoke_token) method and its specialized aliases\n[.revoke_access_token()](/api/#requests_oauth2client.client.OAuth2Client.revoke_access_token) and\n[.revoke_refresh_token()](/api/#requests_oauth2client.client.OAuth2Client.revoke_refresh_token) are then available:\n\n    oauth2client.revoke_token("mytoken", token_type_hint="access_token")\n    oauth2client.revoke_access_token("mytoken") # will automatically add token_type_hint=access_token\n    oauth2client.revoke_refresh_token("mytoken") # will automatically add token_type_hint=refresh_token\n\nBecause Revocation Endpoints usually don\'t return meaningful responses,\nthose methods return a boolean. This boolean indicates that a request\nwas successfully sent and no error was returned. If the Authorization\nServer actually returns a standardised error, an exception will be\nraised instead.\n\n## Token Introspection\n\n[OAuth2Client] can send requests to a Token Introspection\nEndpoint. You need to provide an Introspection Endpoint URI when\ncreating the `OAuth2Client`:\n\n    oauth2client = OAuth2Client(\n       token_endpoint,\n       introspection_endpoint=introspection_endpoint,\n       auth=ClientSecretJWT("client_id", "client_secret"))\n\nThe [OAuth2Client.introspect_token()](/api/#requests_oauth2client.client.OAuth2Client.instrospect_token()) method is then available:\n\n    resp = oauth2client.introspect_token("mytoken", token_type_hint="access_token")\n\nIt returns whatever data is returned by the introspection endpoint (if\nit is a JSON, its content is returned decoded).\n\n## UserInfo Requests\n\n[OAuth2Client] can send requests to an UserInfo Endpoint.\nYou need to provide an UserInfo Endpoint URI when creating the\n`OAuth2Client`:\n\n    oauth2client = OAuth2Client(\n       token_endpoint,\n       userinfo_endpoint=userinfo_endpoint,\n       auth=ClientSecretJWT("client_id", "client_secret"))\n\nThe [OAuth2Client.userinfo()](/api/#requests_oauth2client.client.OAuth2Client.userinfo)) method is then available:\n\n    resp = oauth2client.userinfo("mytoken")\n\nIt returns whatever data is returned by the userinfo endpoint (if it is\na JSON, its content is returned decoded).\n\n## Initializing an OAuth2Client from a discovery document\n\nYou can initialize an [OAuth2Client] with the endpoint URIs mentioned in\na standardised discovery document with the [OAuth2Client.from_discovery_endpoint()](/api/#requests_oauth2client.client.OAuth2Client.from_discovery_document) class method:\n\n    oauth2client = OAuth2Client.from_discovery_endpoint("https://myas.local/.well-known/openid-configuration")\n\nThis will fetch the document from the specified URI, then will decode it\nand initialize an [OAuth2Client] pointing to the appropriate endpoint\nURIs.\n\n## Specialized API Client\n\nUsing APIs usually involves multiple endpoints under the same root url,\nwith a common authentication method. To make it easier,\n`requests_oauth2client` includes a specialized\n[requests.Session] subclass called ApiClient, which takes a\nroot url as parameter on initialization. You can then send requests to\ndifferent endpoints by passing their relative path instead of the full\nurl. [ApiClient] also accepts an `auth` parameter with an\nAuthHandler. You can pass any of the OAuth2 Auth Handler from this\nmodule, or any [requests]-compatible\n[Authentication Handler](https://docs.python-requests.org/en/master/user/advanced/#custom-authentication). Which makes it very easy to call APIs that\nare protected with an OAuth2 Client Credentials Grant:\n\n    oauth2client = OAuth2Client("https://myas.local/token", (client_id, client_secret))\n    api = ApiClient("https://myapi.local/root", auth=OAuth2ClientCredentialsAuth(oauth2client))\n    resp = api.get("/resource/foo") # will actually send a GET to https://myapi.local/root/resource/foo\n\nNote that [ApiClient] will never send requests "outside"\nits configured root url, unless you specifically give it full url at\nrequest time. The leading `/` in `/resource` above is\noptional. A leading `/` will not "reset" the url path to root, which\nmeans that you can also write the relative path without the `/` and it\nwill automatically be included:\n\n    api.get("resource/foo") # will actually send a GET to https://myapi.local/root/resource/foo\n\nYou may also pass the path as an iterable of strings (or string-able\nobjects), in which case they will be joined with a `/` and appended to the\nurl path:\n\n    api.get(["resource", "foo"]) # will actually send a GET to https://myapi.local/root/resource/foo\n    api.get(["users", 1234, "details"]) # will actually send a GET to https://myapi.local/root/users/1234/details\n\n[ApiClient] will, by default, raise exceptions whenever a\nrequest returns an error status. You can disable that by passing\n`raise_for_status=False` when initializing your\n[ApiClient]:\n\n    api = ApiClient(\n        "http://httpstat.us",\n         raise_for_status=False # this defaults to True\n    )\n    resp = api.get("500") # without raise_for_status=False, this would raise a requests.exceptions.HTTPError\n\nYou may override this at request time:\n\n    resp = api.get("500", raise_for_status=True) # raise_for_status at request-time overrides raise_for_status defined at init-time\n\n## Vendor-Specific clients\n\n`requests_oauth2client` being flexible enough to handle most\nuse cases, you should be able to use any AS by any vendor as long as it\nsupports OAuth 2.0.\n\nYou can however create a subclass of [OAuth2Client] or [ApiClient] to make it easier to\nuse with specific Authorization Servers or APIs.\nThe sub-module `requests_oauth2client.vendor_specific` includes such\nclasses for Auth0:\n\n    from requests_oauth2client.vendor_specific import Auth0Client\n\n    a0client = Auth0Client("mytenant.eu", (client_id, client_secret))\n    # this will automatically initialize the token endpoint to https://mytenant.eu.auth0.com/oauth/token\n    # so you can use it directly\n    token = a0client.client_credentials(audience="audience")\n\n    # this is a wrapper around Auth0 Management API\n    a0mgmt = Auth0ManagementApiClient("mytenant.eu", (client_id, client_secret))\n    myusers = a0mgmt.get("users")\n\n\n[requests]: https://docs.python-requests.org/en/master/\n[OAuth2Client]: /api/#requests_oauth2client.client.OAuth2Client\n[BearerAuth]: /api/#requests_oauth2client.auth.BearerAuth\n[BearerToken]: http://127.0.0.1:8000/api/#requests_oauth2client.tokens.BearerToken\n[ApiClient]: /api/#requests_oauth2client.api_client.ApiClient\n',
    'author': 'Guillaume Pujol',
    'author_email': 'guill.p.linux@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/guillp/requests_oauth2client',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6.2,<4.0',
}


setup(**setup_kwargs)
