"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.logic = void 0;
const ddbRepository = require("./ddb-repository");
const errors_1 = require("./errors");
const util_1 = require("./util");
const parseResourceProperties = (event) => {
    const { priority, rulePath } = event.ResourceProperties;
    if (!rulePath) {
        throw new Error('Rule path is required.');
    }
    return {
        priority: priority ? parseInt(priority) : undefined,
        rulePath,
    };
};
async function onDelete(ctx, event) {
    const props = parseResourceProperties(event);
    const priority = props.priority ??
        (await ddbRepository.fetchListenerRulePriority(ctx, {
            rulePath: props.rulePath,
        }));
    await ddbRepository.destroyListenerRule(ctx, {
        ...props,
        priority,
    });
}
async function onCreate(ctx, event) {
    const props = parseResourceProperties(event);
    let priority = props.priority;
    if (priority) {
        await ddbRepository.createNewListenerRule(ctx, {
            ...props,
            priority,
        });
    }
    else {
        priority = await ddbRepository.fetchFreePriority(ctx);
        while (true) {
            try {
                await ddbRepository.createNewListenerRule(ctx, {
                    ...props,
                    priority,
                });
                break;
            }
            catch (e) {
                if (e instanceof errors_1.ListenerRulePriorityAlreadyTakenError) {
                    priority = await ddbRepository.fetchFreePriority(ctx);
                }
                else if (e instanceof errors_1.ListenerRuleAlreadyExistsError) {
                    // deallocate priority if rule already exists
                    await ddbRepository.deallocPriority(ctx, priority);
                    throw e;
                }
                else {
                    throw e;
                }
            }
        }
    }
    return { priority };
}
async function onUpdate(ctx, event) {
    const props = parseResourceProperties(event);
    const persistedPriority = await ddbRepository.fetchListenerRulePriority(ctx, props);
    // if priority is not changed, do nothing
    // if priority was explicit and is now unknown, returns the persisted value
    if (props.priority === persistedPriority || !props.priority) {
        return {
            priority: props.priority || persistedPriority,
        };
    }
    await ddbRepository.replaceListenerRulePriority(ctx, {
        ...props,
        priority: props.priority,
        oldPriority: persistedPriority,
    });
    return {
        priority: props.priority,
    };
}
async function logic(ctx, event) {
    switch (event.RequestType) {
        case 'Delete':
            await onDelete(ctx, event);
            break;
        case 'Create':
            return {
                Data: await onCreate(ctx, event),
            };
        case 'Update':
            const oldProps = event.OldResourceProperties;
            const props = event.ResourceProperties;
            if (oldProps.listenerArn !== props.listenerArn || oldProps.rulePath !== props.rulePath) {
                return {
                    PhysicalResourceId: (0, util_1.generatePhysicalResourceId)(`${props.listenerArn}-${props.rulePath}`),
                    Data: await onCreate(ctx, event),
                };
            }
            return {
                Data: await onUpdate(ctx, event),
            };
    }
    return {};
}
exports.logic = logic;
//# sourceMappingURL=data:application/json;base64,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