"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MonitoringFacade = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aspects_1 = require("./aspects");
/**
 * The MonitoringFacade creates a Cloudwatch dashboard and applies monitoring aspects to resources.
 * These aspects will scan for resources, create alarms and add metrics to the MonitoringFacade dashboard.
 *
 * This allow to have a centralized monitoring configuration for all resources in the stack.
 *
 * Additionally, the `config*` methods allow to override the default configuration for a specific resource.
 *
 * @example
 * class MyStack extends cdk.Stack {
 *   constructor(scope: Construct, id: string, props: cdk.StackProps) {
 *     super(scope, id, props);
 *
 *     const cluster = new AuroraCluster(this, 'DatabaseCluster', { ... });
 *
 *     // Even if the MonitoringFacade is built after the AuroraCluster, the cluster will be monitored, because the aspects are executed after the stack is built.
 *     const monitoring = new MonitoringFacade(this, { topicArn: 'arn:aws:sns:us-east-1:123456789012:MyTopic' });
 *
 *     const cluster2 = new AuroraCluster(this, 'DatabaseCluster2', { ... });
 *     // The monitoring configuration for the second cluster is modified so that the CPU utilization alarm is triggered when the utilization is over the 10%.
 *     monitoring.configRdsCluster(cluster2, {
 *       cpuUtilizationThreshold: 0.1,
 *     });
 *   }
 * }
 */
class MonitoringFacade {
    static of(scope) {
        return MonitoringFacade.instances.get(cdk.Stack.of(scope).node.path);
    }
    constructor(scope, props) {
        if (MonitoringFacade.instances.has(scope.node.path)) {
            throw new Error(`CondenseMonitoringFacade already exists for ${scope.node.path}`);
        }
        MonitoringFacade.instances.set(scope.node.path, this);
        this.dashboard = new aws_cdk_lib_1.aws_cloudwatch.Dashboard(scope, 'Dashboard', {
            dashboardName: props?.dashboardName,
        });
        this.alarmTopic = aws_cdk_lib_1.aws_sns.Topic.fromTopicArn(scope, 'ObservabilityTopic', props.topicArn);
        const aspects = cdk.Aspects.of(scope);
        aspects.add((this.rdsInstanceAspect = new aspects_1.RdsInstanceMonitoringAspect(this)));
        aspects.add((this.cacheClusterAspect = new aspects_1.CacheClusterMonitoringAspect(this)));
        aspects.add((this.targetGroupAspect = new aspects_1.TargetGroupMonitoringAspect(this)));
        aspects.add((this.fargateServiceAspect = new aspects_1.FargateServiceMonitoringAspect(this)));
        aspects.add((this.albAspect = new aspects_1.ApplicationLoadBalancerMonitoringAspect(this)));
        aspects.add((this.rdsClusterAspect = new aspects_1.RdsClusterMonitoringAspect(this)));
    }
    addAlarm(alarm) {
        alarm.addAlarmAction(new aws_cdk_lib_1.aws_cloudwatch_actions.SnsAction(this.alarmTopic));
    }
    /**
     * Overrides the default configuration for a specific Elasticache cluster.
     * @param resource The elasticache cluster to monitor
     * @param config The configuration to apply
     */
    configCacheCluster(resource, config) {
        this.cacheClusterAspect.overrideConfig(resource, config);
    }
    /**
     * Overrides the default configuration for a specific ELBv2 Target Group.
     * @param resource The target group to monitor
     * @param config The configuration to apply
     */
    configTargetGroup(resource, config) {
        this.targetGroupAspect.overrideConfig(resource, config);
    }
    /**
     * Overrides the default configuration for a specific RDS instance.
     * @param resource The RDS instance to monitor
     * @param config The configuration to apply
     */
    configRdsInstance(resource, config) {
        this.rdsInstanceAspect.overrideConfig(resource, config);
    }
    /**
     * Overrides the default configuration for a specific ECS Fargate service.
     * @param resource The Fargate service to monitor
     * @param config The configuration to apply
     */
    configFargateService(resource, config) {
        this.fargateServiceAspect.overrideConfig(resource, config);
    }
    /**
     * Overrides the default configuration for a specific Application Load Balancer.
     * @param resource The ALB to monitor
     * @param config The configuration to apply
     */
    configApplicationLoadBalancer(resource, config) {
        this.albAspect.overrideConfig(resource, config);
    }
    /**
     * Overrides the default configuration for a specific RDS cluster.
     * @param resource The RDS cluster to monitor
     * @param config The configuration to apply
     */
    configRdsCluster(resource, config) {
        this.rdsClusterAspect.overrideConfig(resource, config);
    }
}
exports.MonitoringFacade = MonitoringFacade;
_a = JSII_RTTI_SYMBOL_1;
MonitoringFacade[_a] = { fqn: "@condensetech/cdk-constructs.MonitoringFacade", version: "0.5.2" };
MonitoringFacade.instances = new Map();
//# sourceMappingURL=data:application/json;base64,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